/*
gcc -Wall -O -lpthread `pkg-config --cflags --libs glib-2.0` -std=c99 *.c
↑これでコンパイル可能。

glibを使用しているので、
sudo apt-get install libgtk2.0-dev
が必要。

今後の予定
・移動モードの実装

インデントはタブ文字。タブの幅は半角スペース4文字。
*/

#define _XOPEN_SOURCE 600
#define _GNU_SOURCE

#include "global.h"
#include "struct.h"
#include "enum.h"

#include "ch_time_mod.h"
#include "compare.h"
#include "dir_traverse.h"
#include "file_list_maker.h"
#include "hash_function.h"
#include "help.h"
#include "list_append.h"
#include "lets_begin.h"
#include "log.h"
#include "option.h"
#include "rw_thread.h"
#include "same_or_different.h"

#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <signal.h>
#include <getopt.h>
#include <malloc.h>
#include <glib.h>

char today[11]; // 今日の日付

// 設定ファイルを保存するフォルダ等のパス
char *config_dir;
char *settings_dir;
char *config_file;
char *log_dir;
char *compare_error_dir;
char *compare_success_dir;
char *copy_error_dir;
char *copy_success_dir;

// 現在書き込み中のファイルのパス
char *write_now = NULL;

int buffer_size = 0;
int compare_mode = 0; // 1か2
int current_fd = 0;

// 関数プロトタイプ
static inline __attribute__((always_inline)) void get_today(void);
static inline __attribute__((always_inline)) void get_settings_dir(void);
static inline __attribute__((always_inline)) void buffer_size_set(const int tmp);
void handler(int sig);

/*******************************************************************************

*******************************************************************************/
int main(int argc, char *argv[])
{
	// 引数無しの場合、helpを表示する
	if(argc == 1)
	{
		help();
		return 0;
	}

	// 今日の日付を取得。コンペアのログ保存用。
	get_today();
	// ログや設定を保存するフォルダ名を作成
	get_settings_dir();

	// 引数が1つで、-oの場合、オプションを設定、保存する。
	if((argc == 2) && (strcmp(argv[1], "-o") == 0))
	{
		option_set();
		return 0;
	}

	int result;
	enum opt_chk option;

	B = C = D = I = L = M = N = V = option = NOT;
	copy_mode = SAME;
	UNList *hdd_list = NULL;
	UNList *cp_list = NULL;

	// allocaなのでfreeするべからず
	struct argument *arg_from = alloca(sizeof(struct argument));
	struct argument *arg_to = alloca(sizeof(struct argument));
	arg_from->path = alloca(PATH_LEN);
	arg_to->path = alloca(PATH_LEN);

	// 初期設定ではバッファは64MBまで
	buffer_size_set(64);

	// 設定ファイルが存在する場合は読み込む
	if(g_file_test(config_file, G_FILE_TEST_EXISTS) == TRUE)
		option_get();

	// 引数解析
	while((result = getopt(argc, argv, "1:5:0b:c:dilmnov")) != -1)
	{
		switch(result)
		{
		// SHA1SUMでチェックする
		case '1':
			option = EXIST;
			C = ONE;
			{
				char *endptr = NULL;
				// atoiは使うべきではない、らしい
				compare_mode = (int)strtol(optarg, &endptr, 10);
			}
			if((compare_mode == 1) || (compare_mode == 2))
			{
				break;
			}
			else
			{
				puts("SHA1SUMモードの指定値が正しくありません");
				exit(-1);
			}
			break;
		// MD5SUMでチェックする
		case '5':
			option = EXIST;
			C = FIVE;
			{
				char *endptr = NULL;
				// atoiは使うべきではない、らしい
				compare_mode = (int)strtol(optarg, &endptr, 10);
			}
			if((compare_mode == 1) || (compare_mode == 2))
			{
				break;
			}
			else
			{
				puts("MD5SUMモードの指定値が正しくありません");
				exit(-1);
			}
			break;
		// コンペアを行わない
		case '0':
			option = EXIST;
			C = NOT;
			compare_mode = 0;
			break;
		// バッファサイズを指定
		case 'b':
			option = EXIST;
			B = BUFFER;
			{
				char *endptr = NULL;
				errno = 0;
				// atoiは使うべきではない、らしい
				int b_size = (int)strtol(optarg, &endptr, 10);
				if((endptr[0] != '\0') || (errno != 0))
				{
					puts("バッファの値が正しくありません");
					exit(-1);
				}
				buffer_size_set(b_size);
			}
			break;
		// memcmpでチェックする
		case 'c':
			option = EXIST;
			C = COMPARE;
			{
				char *endptr = NULL;
				// atoiは使うべきではない、らしい
				compare_mode = (int)strtol(optarg, &endptr, 10);
			}
			if((compare_mode == 1) || (compare_mode == 2))
			{
				break;
			}
			else
			{
				puts("コンペアモードの指定値が正しくありません");
				exit(-1);
			}
			break;
		// ダイレクト I/Oを使用する
		case 'd':
			option = EXIST;
			D = DIRECT_IO;
			break;
		// 上書き時に確認する
		case 'i':
			option = EXIST;
			I = INTERACTIVE;
			N = NOT;
			break;
		// コンペアのログを保存する
		case 'l':
			option = EXIST;
			L = LOG;
			break;
		// 移動モード
		// 未実装
		case 'm':
			option = EXIST;
			M = MOVE;
			break;
		// 非上書きモード
		case 'n':
			option = EXIST;
			N = NOT_OVERWRITE;
			I = NOT;
			break;
		// オプションの設定
		case 'o':
			option = EXIST;
			option_set();
			return 0;
			break;
		// 表示モード
		case 'v':
			option = EXIST;
			V = VERBOS;
			break;
		// 未知のオプション
		case '?':
			puts("オプションが正しくありません");
			exit(-1);
			break;
		}
	}

	if(optind == (argc - 1))
	{
		puts("引数が足りません");
		exit(-1);
	}
	else if((option == EXIST) && (optind == argc))
	{
		puts("引数が正しくありません");
		exit(-1);
	}
	else if((C == COMPARE) && (L == LOG))
	{
		puts("-lオプションはSHA-1、もしくはMD5で比較する場合のみ有効です");
		exit(-1);
	}

	FILE *mtab;
	if((mtab = fopen("/etc/mtab", "r")) == NULL)
	{
		puts("mtabのオープンに失敗しました");
		exit(-1);
	}
	// マウントされているHDDを確認
	char mtab_buf[PATH_LEN];
	while(fgets(mtab_buf, PATH_LEN, mtab) != NULL)
	{
		if(strncmp("/dev/sd", mtab_buf, 7) == 0)
		{
			char *c;
			// alloca使ってるのでfree厳禁
			struct hdd_path *tmp = alloca(sizeof(struct hdd_path));
			c = strtok(mtab_buf, " ");
			tmp->hdd_dev = alloca(strlen(c) + 1);
			strcpy(tmp->hdd_dev, c);
			c = strtok(NULL, " ");
			tmp->hdd_mount = alloca(strlen(c) + 1);
			strcpy(tmp->hdd_mount, c);
			c = tmp->hdd_dev;
			c[strlen(c) - 1] = '\0';
			hdd_list = list_append(hdd_list, (void *)tmp);
		}
	}
	fclose(mtab);

	// シグナルハンドラ
	struct sigaction act;
	act.sa_handler = handler;
	act.sa_flags = 0;
	sigaction(SIGINT, &act, NULL);
	sigaction(SIGTERM, &act, NULL);
	sigaction(SIGSEGV, &act, NULL);
	sigaction(SIGBUS, &act, NULL);
	sigaction(SIGKILL, &act, NULL);
	sigaction(SIGABRT, &act, NULL);
	sigaction(SIGQUIT, &act, NULL);
	sigaction(SIGHUP, &act, NULL);

	// ページサイズの取得。sysconf()でも可。
	p_size = getpagesize();

	// ログフォルダとカレントを行き来するための
	if((current_fd = open(".", O_RDONLY | O_NOATIME)) == -1)
	{
		fprintf(stderr, "カレントディレクトリのオープンに失敗しました\n");
		exit(-1);
	}

	// コピー元の数をチェック
	int argv_count = 0;
	for(int tmp = optind; tmp < (argc - 1); tmp++)
	{
		argv_count++;
	}

//	if(V == VERBOS)
		puts("コピー対象の一覧を作成中です...");

	// コピーするファイルのリストを作成する
	cp_list = file_list_maker(cp_list, hdd_list, arg_from, arg_to, argv_count, optind, argc, argv);

//	if(V == VERBOS)
		puts("コピーを開始します...");

	// コピーを実行
	lets_begin(cp_list);

	// フォルダのタイムスタンプをコピーする
	{
		UNList *list_temp = cp_list;
		for(int loop = 0;;)
		{
			struct cp_target *tmp = list_temp->data;

			if(tmp->file_type == DIRECTORY)
			{
				if(V == VERBOS)
					printf("%s のタイムスタンプをコピーします\n", tmp->to);

				ch_time_mod_dir(tmp->from, tmp->to);
			}

			loop++;
			if(loop == cp_file_count_all)
				break;
			list_temp = list_temp->next;
		}
	}

//	if(V == VERBOS)
		puts("コピーを完了しました...");

	// コンペアその2。
	// 全てのコピー終了後に実行。
	if(compare_mode == 2)
	{
//		if(V == VERBOS)
			puts("コンペアを開始します...");

		UNList *list_temp = cp_list;
		for(int loop = 0;;)
		{
			struct cp_target *tmp = list_temp->data;

			if((tmp->file_type == REGULAR) && (tmp->write == WRITE))
			{
				if(V == VERBOS)
					printf("%sのコンペアを開始します\n", tmp->to);

				if(C == COMPARE)
				{
					compare(tmp->from, tmp->to);
				}
				else if(C == ONE)
				{
					hash_function(tmp->from, tmp->to, tmp->copy_mode, SHA1SUM);
				}
				else if(C == FIVE)
				{
					hash_function(tmp->from, tmp->to, tmp->copy_mode, MD5SUM);
				}
			}

			loop++;
			if(loop == cp_file_count_all)
				break;
			list_temp = (UNList *)list_temp->next;
		}

//		if(V == VERBOS)
			puts("コンペアを完了しました...");
	}

	log_errors_count();

	/*
	freeする必要があるもの
	UNList *hdd_list
	UNList *cp_list
	struct cp_target
	char *config_dir;
	char *settings_dir;
	char *log_dir;
	char *compare_error_dir;
	char *compare_success_dir;
	char *copy_error_dir;
	char *copy_success_dir;
	*/

	printf("Source : %lld MB (%lld Byte)\n", total_byte / (1024 * 1024), total_byte);
	printf("Read   : %lld MB (%lld Byte)\n", total_read / (1024 * 1024), total_read);
	printf("Write  : %lld MB (%lld Byte)\n", total_write / (1024 * 1024), total_write);

	if(C != NOT)
		close(current_fd);

	return 0;
}

/*******************************************************************************
今日の日付を文字列に変換
*******************************************************************************/
void get_today()
{
	time_t timer;
	struct tm *t_st;

	time(&timer);
	t_st = localtime(&timer);
	sprintf(today, "%d%s%d%s%d", t_st->tm_year + 1900, "-", t_st->tm_mon + 1, "-", t_st->tm_mday);
}

/*******************************************************************************
設定やログを保存するフォルダのパスを作成
*******************************************************************************/
void get_settings_dir()
{
	const char *homedir = g_getenv("HOME");
	if(!homedir)
		homedir = g_get_home_dir();

	// ~/.config
	config_dir = g_build_path(G_DIR_SEPARATOR_S, homedir, ".config", NULL);

	// ~/.config/snowcp
	settings_dir = g_build_path(G_DIR_SEPARATOR_S, config_dir, "snowcp", NULL);

	// ~/.config/snowcp/snowcp.conf
	config_file = g_build_path(G_DIR_SEPARATOR_S, settings_dir, "snowcp.conf", NULL);

	// ~/.config/snowcp/log
	log_dir = g_build_path(G_DIR_SEPARATOR_S, settings_dir, "log", NULL);

	// ~/.config/snowcp/log/compare_error
	compare_error_dir = g_build_path(G_DIR_SEPARATOR_S, log_dir, "compare_error", NULL);

	// ~/.config/snowcp/log/compare_success
	compare_success_dir = g_build_path(G_DIR_SEPARATOR_S, log_dir, "compare_success", NULL);

	// ~/.config/snowcp/log/copy_error
	copy_error_dir = g_build_path(G_DIR_SEPARATOR_S, log_dir, "copy_error", NULL);

	// ~/.config/snowcp/log/copy_success
	copy_success_dir = g_build_path(G_DIR_SEPARATOR_S, log_dir, "copy_success", NULL);
}

/*******************************************************************************
Disk I/O用のバッファサイズを確保
*******************************************************************************/
static void buffer_size_set(const int tmp)
{
	if(tmp <= 0)
	{
		puts("バッファのサイズが不正です");
		exit(-1);
	}
	buffer_size = 1024 * 1024 * tmp;
}

/*******************************************************************************
シグナルを受け取ったら、コピー中のファイルを削除して終了する
*******************************************************************************/
void handler(int sig)
{
	puts("ERROR!");
	printf("signal number = %d\n", sig);
	if(write_now != NULL)
	{
		unlink(write_now);
		printf("書き込み中の %s を削除しました\n", write_now);
	}
	exit(-1);
}
