#include "AfxWin.h"
#include <stdlib.h>
#include <assert.h>
#include "dispatch.h"

enum {
	CLIENT_STATUS_LOCK		= 1 << 0,
	CLIENT_STATUS_DESTROYED	= 1 << 1,
} ;

typedef struct tagDISPATCH_CLIENT {
	int							m_nEvent ;
	long						m_lEventMask ;
	WINDOWPROC					m_pWindowProc ;
	void*						m_pClosure ;
	unsigned int				m_uStatus ;
	struct tagDISPATCH_CLIENT*	m_pNext ;
}	DISPATCH_CLIENT ;

typedef struct tagDISPATCHED {
	Window						m_window ;
	int							m_nEventMin, m_nEventMax ;
	long						m_lEventMask ;
	struct tagDISPATCH_CLIENT*	m_lstClient ;
	struct tagDISPATCHED*		m_pLeft ;
	struct tagDISPATCHED*		m_pRight ;
}	DISPATCHED ;


static	DISPATCHED*			dispatcher_searchNode	(Window, DISPATCHED**) ;
static	DISPATCHED*			dispatcher_newNode		(Window) ;
static	DISPATCH_CLIENT*	dispatcher_searchClient	(DISPATCHED*, int, DISPATCH_CLIENT**) ;
static	DISPATCH_CLIENT*	dispatcher_newClient	(void*, int, long, WINDOWPROC) ;
static	void				dispatcher_removeNode	(DISPATCHED*, DISPATCHED*) ;
static	void				dispatcher_removeClient	(DISPATCHED*, DISPATCH_CLIENT*) ;


static	DISPATCHED*			spRootOfDispatchedTree		= NULL ;
static	DISPATCHED*			slstFreeDispatched			= NULL ;
static	DISPATCH_CLIENT*	slstFreeDispatchClient		= NULL ;

Boolean
AfxRegisterWindow (
	register Display*	pDisplay,
	register Window		window,
	register int		nEvent,
	register long		lEventMask,
	register void*		pClosure,
	register WINDOWPROC	pWindowProc)
{
	register DISPATCHED*		pDispatched ;
	DISPATCHED*					pParent ;
	register DISPATCH_CLIENT*	pClient ;
	DISPATCH_CLIENT*			pPrevious ;

	pDispatched	= dispatcher_searchNode (window, &pParent) ;
	if (pDispatched == NULL) {
		pDispatched	= dispatcher_newNode (window) ;
		if (pParent == NULL) {
			spRootOfDispatchedTree	= pDispatched ;
		} else {
			if (window > pParent->m_window) {
				pParent->m_pRight	= pDispatched ;
			} else {
				pParent->m_pLeft	= pDispatched ;
			}
		}
	}
	pClient	= dispatcher_searchClient (pDispatched, nEvent, &pPrevious) ;
	if (pClient != NULL) {
		while (pClient != NULL && pClient->m_nEvent == nEvent) {
			if (pClient->m_pWindowProc == pWindowProc &&
				pClient->m_pClosure    == pClosure) {
				pClient->m_lEventMask	|= lEventMask ;
				break ;
			}
		}
	} else {
		pClient	= dispatcher_newClient (pClosure, nEvent, lEventMask, pWindowProc) ;
		if (pPrevious != NULL) {
			pClient->m_pNext			= pPrevious->m_pNext ;
			pPrevious->m_pNext			= pClient ;
		} else {
			pClient->m_pNext			= pDispatched->m_lstClient ;
			pDispatched->m_lstClient	= pClient ;
		}
		pDispatched->m_nEventMin	= (nEvent < pDispatched->m_nEventMin)? nEvent : pDispatched->m_nEventMin ;
		pDispatched->m_nEventMax	= (nEvent > pDispatched->m_nEventMax)? nEvent : pDispatched->m_nEventMax ;
	}
	pDispatched->m_lEventMask	|= lEventMask ;
	XSelectInput (pDisplay, window, pDispatched->m_lEventMask) ;
	return	True ;
}

Boolean
AfxUnregisterWindow (
	register Window		window,
	register int		nEvent,
	register long		lEventMask,
	register void*		pClosure,
	register WINDOWPROC	pWindowProc)
{
	register DISPATCHED*		pDispatched ;
	DISPATCHED*					pParent ;
	register DISPATCH_CLIENT*	pClient ;
	DISPATCH_CLIENT*			pPrevClient ;
	register DISPATCH_CLIENT*	pNextClient ;

	pDispatched	= dispatcher_searchNode (window, &pParent) ;
	if (pDispatched == NULL) 
		return	False ;
	pClient	= dispatcher_searchClient (pDispatched, nEvent, &pPrevClient) ;
	if (pClient == NULL) 
		return	False ;

	while (pClient != NULL && pClient->m_nEvent == nEvent) {
		pNextClient	= pClient->m_pNext ;
		if (pClient->m_pWindowProc == pWindowProc &&
			pClient->m_pClosure    == pClosure) {
			pClient->m_lEventMask	&= ~lEventMask ;
			if (pClient->m_lEventMask == 0L) {
				dispatcher_removeClient (pDispatched, pClient) ;
			}
		}
		pClient		= pNextClient ;
	}
	if (pDispatched->m_lstClient == NULL) {
		/*XSelectInput (pDisplay, window, 0) ;*/
		dispatcher_removeNode (pDispatched, pParent) ;
	}
	return	False ;
}

Boolean
AfxDispatchEvent (
	register XEvent*	pEvent)
{
	register DISPATCHED*		pNode ;
	register DISPATCH_CLIENT*	pClient ;
	register Window				window	= pEvent->xany.window ;

	pNode	= dispatcher_searchNode (window, NULL) ;
	if (pNode != NULL) {
#if defined (DEBUG)
		fprintf (stderr, "Event(%d), Window(%lx), Serial(%lu), Node(%p,%d<<%d)\n",
				 pEvent->type, 
				 pEvent->xany.window,
				 pEvent->xany.serial,
				 pNode,
				 pNode->m_nEventMin,
				 pNode->m_nEventMax) ;
#endif
		if (pNode->m_nEventMin <= pEvent->type &&
			pNode->m_nEventMax >= pEvent->type) {
			register DISPATCH_CLIENT*	pNextClient ;

			pClient	= dispatcher_searchClient (pNode, pEvent->type, NULL) ;
			while (pClient != NULL && pClient->m_nEvent == pEvent->type) {
				assert (pClient->m_lEventMask != 0) ;

				pClient->m_uStatus	|= CLIENT_STATUS_LOCK ;
				/*
				 *	$B$3$3$N(B handle $BCf$K(B pClient $B$,(B free $B$5$l$FIe$k2DG=@-(B
				 *	$B$,$"$k!#$d$i$l$?!#(B
				 */
				(pClient->m_pWindowProc)(pClient->m_pClosure, pEvent) ;

				pClient->m_uStatus	&= ~CLIENT_STATUS_LOCK ;
				pNextClient			= pClient->m_pNext ;

				if (pClient->m_uStatus & CLIENT_STATUS_DESTROYED) 
					dispatcher_removeClient (pNode, pClient) ;
				pClient				= pNextClient ;
			}
			if (pNode->m_lstClient == NULL) {
				register DISPATCHED*	pTmpNode ;
				DISPATCHED*				pParent ;
				
				pTmpNode	= dispatcher_searchNode (window, &pParent) ;
				assert (pTmpNode == pNode) ;
				dispatcher_removeNode (pNode, pParent) ;
			}
		}
	}
	return	XtDispatchEvent (pEvent) ;
}


DISPATCHED*
dispatcher_searchNode (
	register Window			window,
	register DISPATCHED**	ppParent)
{
	register DISPATCHED*		pDispatched ;
	register DISPATCHED*		pParent ;

	pDispatched		= spRootOfDispatchedTree ;
	pParent			= NULL ;
	while (pDispatched != NULL) {
		if (window == pDispatched->m_window)
			break ;
		
		pParent		= pDispatched ;
		if (window > pDispatched->m_window) {
			pDispatched	= pDispatched->m_pRight ;
		} else {
			pDispatched	= pDispatched->m_pLeft ;
		}
	}
	if (ppParent != NULL)
		*ppParent	= pParent ;
	return	pDispatched ;
}

DISPATCHED*
dispatcher_newNode (
	register Window		window)
{
	DISPATCHED*	pDispatched ;

	if (slstFreeDispatched != NULL) {
		pDispatched			= slstFreeDispatched ;
		slstFreeDispatched	= slstFreeDispatched->m_pRight ;
	} else {
		pDispatched			= MALLOC (sizeof (DISPATCHED)) ;
		assert (pDispatched != NULL) ;
	}
	pDispatched->m_lEventMask	= 0L ;
	pDispatched->m_nEventMax	= 0 ;
	pDispatched->m_nEventMin	= LASTEvent ;
	pDispatched->m_pLeft		= NULL ;
	pDispatched->m_pRight		= NULL ;
	pDispatched->m_lstClient	= NULL ;
	pDispatched->m_window		= window ;
	return	pDispatched ;
}

DISPATCH_CLIENT*
dispatcher_newClient (
	register void*			pClosure,
	register int			nEvent,
	register long			lEventMask,
	register WINDOWPROC		pWindowProc)
{
	register DISPATCH_CLIENT*	pClient ;

	if (slstFreeDispatchClient != NULL) {
		pClient					= slstFreeDispatchClient ;
		slstFreeDispatchClient	= slstFreeDispatchClient->m_pNext ;
	} else {
		pClient		= MALLOC (sizeof (DISPATCH_CLIENT)) ;
	}
	pClient->m_nEvent		= nEvent ;
	pClient->m_lEventMask	= lEventMask ;
	pClient->m_pWindowProc	= pWindowProc ;
	pClient->m_pClosure		= pClosure ;
	pClient->m_uStatus		= 0 ;
	pClient->m_pNext		= NULL ;
	return	pClient ;
}

DISPATCH_CLIENT*
dispatcher_searchClient (
	register DISPATCHED*			pDispatched,
	register int					nEvent,
	register DISPATCH_CLIENT**		ppPrevious)
{
	register DISPATCH_CLIENT*	pClient ;
	register DISPATCH_CLIENT*	pPrevious ;

	assert (pDispatched != NULL) ;

	pClient		= pDispatched->m_lstClient ;
	pPrevious	= NULL ;
	while (pClient != NULL && pClient->m_nEvent < nEvent) {
		pPrevious	= pClient ;
		pClient		= pClient->m_pNext ;
	}
	if (ppPrevious != NULL)
		*ppPrevious	= pPrevious ;
	if (pClient == NULL || pClient->m_nEvent != nEvent)
		return	NULL ;
	return	pClient ;
}

void
dispatcher_removeNode (
	register DISPATCHED*		pNode,
	register DISPATCHED*		pParent)
{
	register DISPATCHED*	pNewChild ;

	pNewChild	= pNode->m_pLeft ;
	if (pNewChild != NULL) {
		if (pNode->m_pRight != NULL) {
			register DISPATCHED*	pNewChildRight ;

			pNewChildRight	= pNewChild->m_pRight ;
			if (pNewChildRight != NULL) {
				while (pNewChildRight->m_pRight != NULL) 
					pNewChildRight	= pNewChildRight->m_pRight ;
				pNewChildRight->m_pRight	= pNode->m_pRight ;
			} else {
				pNewChild->m_pRight	= pNode->m_pRight ;
			}
		}
	}  else {
		pNewChild	= pNode->m_pRight ;
	}

	if (pParent == NULL) {
		assert (spRootOfDispatchedTree == pNode) ;
		spRootOfDispatchedTree	= pNewChild ;
	} else if (pParent->m_pRight == pNode) {
		pParent->m_pRight	= pNewChild ;
	} else {
		assert (pParent->m_pLeft == pNode) ;
		pParent->m_pLeft	= pNewChild ;
	}
	pNode->m_pRight		= slstFreeDispatched ;
	pNode->m_pLeft		= NULL ;
	slstFreeDispatched	= pNode ;
	return ;
}

void
dispatcher_removeClient (
	register DISPATCHED*		pDispatched,
	register DISPATCH_CLIENT*	pRemoveClient)
{
	register DISPATCH_CLIENT*	pClient ;
	register DISPATCH_CLIENT*	pPrevious ;
	
	if (pRemoveClient->m_uStatus & CLIENT_STATUS_LOCK) {
		pRemoveClient->m_uStatus	|= CLIENT_STATUS_DESTROYED ;
		return ;
	}
	pClient		= pDispatched->m_lstClient ;
	pPrevious	= NULL ;
	while (pClient != pRemoveClient && pClient != NULL) {
		pPrevious	= pClient ;
		pClient		= pClient->m_pNext ;
	}
	if (pClient != pRemoveClient) 
		return ;
	if (pPrevious == NULL) {
		pDispatched->m_lstClient	= pClient->m_pNext ;
	} else {
		pPrevious->m_pNext			= pClient->m_pNext ;
	}
	pClient->m_pNext		= slstFreeDispatchClient ;
	slstFreeDispatchClient	= pClient ;
	return ;
}

