#include "AfxWin.h"
#include <string.h>
#include "TFontSet.h"
#include "varbuffer.h"

static	Boolean		tFontSet_loadFont (Display*, TFontSet*, const char*) ;
static	int			tFontSet_getCharsetOfFontName	(const char*) ;
static	Boolean		tFontSet_calcMetric				(TFontSet*) ;
static	void		tFontSet_adjustAsciiFont		(TFontSet*) ;

static const char*	rstrFontCharsetTable []	= {
	"-iso646.1991-irv",	"-iso8859-1",		"-iso8859-2",		"-iso8859-3", 
	"-iso8859-4",		"-iso8859-5",		"-iso8859-6",		"-iso8859-7",
	"-iso8859-8",		"-iso8859-9",		"-iso8859-14",		"-iso8859-15",
	"-jisx0201.1976-0",	"-jisx0208.1978-0",	"-jisx0208.1983-0",	"-jisx0212.1990-0",
	"-gb2312.1980-0",	"-gb12345.90-0",	"-gb7589.87-0",		"-gb7590.87-0",
	"-gb13131.91-0",	"-gb13132.91-0",	"-ksc5601.1987-0",	"-ksc5601.1992-0",
} ;

void
TFontSet_Initialize (
	register TFontSet*		pFontSet)
{
	register int	i ;

	assert (pFontSet != NULL) ;

	for (i = 0 ; i < MAX_CHARSET ; i ++) 
		pFontSet->m_af [i]	= NULL ;
	pFontSet->m_iHeight	= 0 ;
	pFontSet->m_iAscent	= 0 ;
	return ;
}

Boolean
TFontSet_Load (
	register Display*		pDisplay,
	register TFontSet*		pFontSet,
	register const char*	pStrFontSet,
	register Boolean			fAppend)
{
	if (!fAppend)
		TFontSet_Clear (pFontSet) ;

	tFontSet_loadFont (pDisplay, pFontSet, pStrFontSet) ;
	tFontSet_calcMetric (pFontSet) ;
	return	True ;
}

Boolean
TFontSet_LoadAtom (
	register Display*		pDisplay,
	register TFontSet*		pFontSet,
	register int			nCount,
	register Atom*			patFontName,
	register Boolean		fAppend)
{
	register int	i ;
	char*	apNames [nCount] ;

	if (!fAppend)
		TFontSet_Clear (pFontSet) ;

	if (!XGetAtomNames (pDisplay, patFontName, nCount, apNames))
		return	False ;

	for (i = 0 ; i < nCount ; i ++) {
		tFontSet_loadFont (pDisplay, pFontSet, apNames [i]) ;
		XFree (apNames [i]) ;
	}
	tFontSet_calcMetric (pFontSet) ;
	return	True ;
}

void
TFontSet_Clear (
	register TFontSet*		pFontSet)
{
	register int 	i ;

	for (i = 0 ; i < MAX_CHARSET ; i ++) {
		if (pFontSet->m_af [i] != NULL) {
			TFont_Unload (pFontSet->m_af [i]) ;
			pFontSet->m_af [i]	= NULL ;
		}
	}
	return ;
}

void
TFontSet_Destroy (
	register TFontSet*	pFontSet)
{
	assert (pFontSet != NULL) ;

	TFontSet_Clear (pFontSet) ;
	return ;
}

void
TFontSet_Copy (
	register TFontSet*			pFSDest,
	register const TFontSet*	pFSSrc)
{
	register int 	i ;

	for (i = 0 ; i < MAX_CHARSET ; i ++) {
		if (pFSDest->m_af [i] != NULL) 
			TFont_Unload (pFSDest->m_af [i]) ;

		if (pFSSrc->m_af [i] != NULL) {
			pFSDest->m_af [i]	= TFont_Clone (pFSSrc->m_af [i]) ;
		} else {
			pFSDest->m_af [i]	= NULL ;
		}
	}
	pFSDest->m_iAscent		= pFSSrc->m_iAscent ;
	pFSDest->m_iHeight		= pFSSrc->m_iHeight ;
	pFSDest->m_iAvgWidth	= pFSSrc->m_iAvgWidth ;
	return ;
}

int
TFontSet_GetAscent (
	register TFontSet*	pFontSet)
{
	assert (pFontSet != NULL) ;
	return	pFontSet->m_iAscent ;
}

int
TFontSet_GetHeight (
	register TFontSet*	pFontSet)
{
	assert (pFontSet != NULL) ;
	return	pFontSet->m_iHeight ;
}

int
TFontSet_GetDescent (
	register TFontSet*	pFontSet)
{
	assert (pFontSet != NULL) ;
	return	(pFontSet->m_iHeight - pFontSet->m_iAscent) ;
}

/*
 *	եȽ礫ꤵ줿ʸΥեȤ롣
 *	ASCII ξISO8859-1  ISO8859-14, JISX0201-1976 ΥեȤ
 *	ѤǤ褦ˤƤ롣
 */
TFont*
TFontSet_GetFont (
	register TFontSet*	pFontSet,
	register int		nSet)
{
	assert (pFontSet != NULL) ;
	assert (0 <= nSet && nSet < MAX_CHARSET) ;

	if (nSet == KCHARSET_ASCII && pFontSet->m_af [nSet] == NULL) {
		nSet	++ ;
		while (nSet < KCHARSET_2BYTES_CHARACTER) {
			if (pFontSet->m_af [nSet] != NULL)
				return	pFontSet->m_af [nSet] ;
		}
		return	NULL ;
	}

	return	pFontSet->m_af [nSet] ;
}

void
TFontSet_Show (
	register TFontSet*	pFontSet)
{
	register int	i ;

	assert (pFontSet != NULL) ;

	for (i = 0 ; i < MAX_CHARSET ; i ++) 
		fprintf (stderr, "Charset(%d) => Font:(%p)\n", i, pFontSet->m_af [i]) ;
	fprintf (stderr, "Height (%d), Ascent (%d)\n",
			 pFontSet->m_iHeight, pFontSet->m_iAscent) ;
	return ;
}

/*	礱Ƥ Font ưŪ䤦
 */
void
TFontSet_Adjust (
	register Display*	pDisplay,
	register TFontSet*	pFS,
	register int		nDefaultHeight)
{
	char			rbufFontName [256] ;
	register int	i, nHeight ;
	int				iActualCount ;
	register char**	pListedFont ;

	assert (pDisplay != NULL) ;
	assert (pFS != NULL) ;
	assert (nDefaultHeight > 0) ;

	nHeight	= TFontSet_GetHeight (pFS) ;
#if defined (DEBUG) || 0
	fprintf (stderr, "FontHeight = %d\n", nHeight) ;
#endif
	if (nHeight <= 0)
		nHeight	= nDefaultHeight ;

	/*	ASCII  ISO8859-1 ꤷƤ뤳Ȥ¿Τǡޤ
	 *	θ롣
	 */
	tFontSet_adjustAsciiFont (pFS) ;

	for (i = 0 ; i < NELEMENTS (rstrFontCharsetTable) ; i ++) {
		if (pFS->m_af [i] != NULL)
			continue ;
		snprintf (rbufFontName, NELEMENTS (rbufFontName) - 1, "-*--%d-*%s", nHeight, rstrFontCharsetTable [i]) ;
		rbufFontName [NELEMENTS (rbufFontName) - 1]	= '\0' ;

		pListedFont	= XListFonts (pDisplay, rbufFontName, 1, &iActualCount) ;
		if (pListedFont != NULL) {
			pFS->m_af [i]	= TFont_Load (pDisplay, pListedFont [0]) ;
#if defined (DEBUG) || 0
			fprintf (stderr, "  --> \"%s\"\n", pListedFont [0]) ;
#endif
		}
		XFreeFontNames (pListedFont) ;
	}
	tFontSet_adjustAsciiFont (pFS) ;
	return ;
}

/*========================================================================*
 *	ؿ
 *========================================================================*/

Boolean
tFontSet_loadFont (
	register Display*		pDisplay,
	register TFontSet*		pFontSet,
	register const char*	pFontSetString)
{
	TVarbuffer	vbPattern ;
	int			iActualCount ;
	char		cc ;
	TFontSet	fs ;
	register const char*	ptr ;
	register char**			pListedFont ;
	register int			i ;
	register int			nCharset ;

	if (TFAILED (TVarbuffer_Initialize (&vbPattern, sizeof (char))))
		return	False ;

	TFontSet_Initialize (&fs) ;
	cc	= '\0' ;

	while (*pFontSetString != '\0') {
		ptr	= pFontSetString ;
		while (*ptr != ',' && *ptr != '\0') 
			ptr	++ ;

		if (ptr == pFontSetString) 
			break ;

		TVarbuffer_Add (&vbPattern, pFontSetString, ptr - pFontSetString) ;
		TVarbuffer_Add (&vbPattern, &cc, 1) ;
#if defined (DEBUG) || 0
		fprintf (stderr, "tFontSet_loadFont: try: \"%s\"\n",
				 (const char*)TVarbuffer_GetBuffer (&vbPattern)) ;
#endif

		pListedFont	= XListFonts (pDisplay, TVarbuffer_GetBuffer (&vbPattern), TFONTSET_MAX_NAMES, &iActualCount) ;
		if (pListedFont != NULL) {
			for (i = 0 ; i < iActualCount ; i ++) {
				nCharset	= tFontSet_getCharsetOfFontName (pListedFont [i]) ;
				if (nCharset < KCHARSET_ASCII || nCharset >= MAX_CHARSET)
					continue ;

				if (fs.m_af [nCharset] == NULL) {
					fs.m_af [nCharset]	= TFont_Load (pDisplay, pListedFont [i]) ;
#if defined (DEBUG) || 0
					fprintf (stderr, "  --> \"%s\"\n", pListedFont [i]) ;
#endif
				}
			}
			XFreeFontNames (pListedFont) ;
		}

		while (*ptr == ',') {
			ptr	++ ;
			while (*ptr == ' ' || *ptr == '\t' || *ptr == '\n' || *ptr == '\r')
				ptr	++ ;
		}
		pFontSetString	= ptr ;
		TVarbuffer_Clear (&vbPattern) ;
	}
	TVarbuffer_Uninitialize (&vbPattern) ;

	for (i = 0 ; i < MAX_CHARSET ; i ++) {
		if (fs.m_af [i] != NULL) {
			if (pFontSet->m_af [i] != NULL)
				TFont_Unload (pFontSet->m_af [i]) ;
			pFontSet->m_af [i]	= fs.m_af [i] ;
		}
	}
	return	TRUE ;
}

int
tFontSet_getCharsetOfFontName (
	register const char*	pFontName)
{
	register int		i ;
	register int		nFontName ;
	register int		nLength ;

	nFontName	= strlen (pFontName) ;
	for (i = 0 ; i < NELEMENTS (rstrFontCharsetTable) ; i ++) {
		nLength	= strlen (rstrFontCharsetTable [i]) ;
		/*
		 *	ס(postfix )
		 */
		if (nLength < nFontName && strcmp (pFontName + nFontName - nLength, rstrFontCharsetTable [i]) == 0) 
			return	i ;
	}
	return	KCHARSET_NOTHING ;
}

Boolean
tFontSet_calcMetric (
	register TFontSet*	pFontSet)
{
	register XFontStruct*	pFS ;
	register int			nSet ;
	register int			iAscent ;
	register int			iDescent ;
	register int			nFonts ;
	register int			nWidth ;

	iAscent		= 0 ;
	iDescent	= 0 ;
	nFonts		= 0 ;
	nWidth		= 0 ; 

	for (nSet = 0 ; nSet < MAX_CHARSET ; nSet ++) {
		if (pFontSet->m_af [nSet] == NULL) 
			continue ;
		pFS	= TFont_GetFontStruct (pFontSet->m_af [nSet]) ;
		if (pFS == NULL)
			continue ;
		if (iAscent < pFS->ascent)
			iAscent		= pFS->ascent ;
		if (iDescent < pFS->descent)
			iDescent	= pFS->descent ;
		nWidth	+= (pFS->min_bounds.width + pFS->max_bounds.width) / 2 ;
		nFonts	++ ;
	}
	pFontSet->m_iAscent		= iAscent ;
	pFontSet->m_iHeight		= iAscent + iDescent ;

	if (nFonts > 0) {
		pFontSet->m_iAvgWidth	= nWidth / nFonts ;
	} else {
		pFontSet->m_iAvgWidth	= 0 ;
	}
	return	TRUE ;
}

void
tFontSet_adjustAsciiFont (
	register TFontSet*	pFS)
{
	register int	i ;

	if (pFS->m_af [KCHARSET_ASCII] != NULL) 
		return ;

	for (i = KCHARSET_ISO8859_1 ; i < KCHARSET_2BYTES_CHARACTER ; i ++) {
		if (pFS->m_af [i] != NULL) {
			pFS->m_af [KCHARSET_ASCII]	= TFont_Clone (pFS->m_af [i]) ;
			break ;
		}
	}
	return ;
}

