#include <assert.h>
#include "Char.h"
#include "kanji.h"

/*========================================================================
 *	
 *========================================================================*/
enum {
	KCHARSET_JISX0201_1976_KATAKANA		= (MAX_CHARSET + 1),

	KCODING_SYSTEM_CHECK_False			= (0),
	KCODING_SYSTEM_CHECK_True			= (1),
	KCODING_SYSTEM_CHECK_True_EXCLUSIVE	= (2),
} ;

enum {
	ISO2022JPSTATE_ESCAPE	= 0,
	ISO2022JPSTATE_G0_94_2_CHARSET1,
	ISO2022JPSTATE_G0_94_2_CHARSET2,
	ISO2022JPSTATE_G0_94_1_CHARSET,
	ISO2022JPSTATE_G2_96_1_CHARSET,
	ISO2022JPSTATE_SINGLE_SHIFT,
} ;

enum {
	CTEXTSTATE_ESCAPE			= 0,
	CTEXTSTATE_GL_94CHARSET,
	CTEXTSTATE_94NCHARSET,
	CTEXTSTATE_GL_94NCHARSET,
	CTEXTSTATE_GR_94NCHARSET,
	CTEXTSTATE_GR_94CHARSET,
	CTEXTSTATE_GR_96CHARSET,
} ;

/*========================================================================*
 *	
 *========================================================================*/
typedef struct {
	int		m_nLineFeed ;		/* 0x0A */
	int		m_nCarriageReturn ;	/* 0x0D */
	int		m_nCRLF ;
	Boolean	m_fPostCR ;
}	NEWLINEINFO ;


/*========================================================================*
 *	ץȥ
 *========================================================================*/
static	int		checkShiftJisCodingSystem			(char*, int*, int) ;
static	int		checkEucJpCodingSystem				(char*, int*, int) ;
static	int		checkIso2022Jp2CodingSystem			(char*, int*, int) ;
static	Boolean	checkNewline						(Char, NEWLINEINFO*) ;
static	int		restartEucJpStateMachine			(PKANJISTATEMACHINE) ;
static	int		transferEucJpStateMachine			(PKANJISTATEMACHINE, int, Char*) ;
static	int		rtransferEucJpStateMachine			(PKANJISTATEMACHINE, Char, char*) ;
static	int		restartIso2022JpStateMachine		(PKANJISTATEMACHINE) ;
static	int		transferIso2022JpStateMachine		(PKANJISTATEMACHINE, int, Char*) ;
static	int		iso2022jpEscapeSequenceState0		(PISO2022STATE, int) ;
static	int		iso2022jpEscapeSequenceState1		(PISO2022STATE, int) ;
static	int		iso2022jpEscapeSequenceState2		(PISO2022STATE, int) ;
static	int		iso2022jpEscapeSequenceState3		(PISO2022STATE, int) ;
static	int		iso2022jpEscapeSequenceState4		(PISO2022STATE, int) ;
static	int		iso2022jpEscapeSequenceState5 		(PISO2022STATE, int) ;
static	int		iso2022jpEscapeSequenceDefault		(PISO2022STATE, int) ;
static	int		rtransferIso2022JpStateMachine		(PKANJISTATEMACHINE, Char, char*) ;
static	int		restartShiftJisStateMachine			(PKANJISTATEMACHINE) ;
static	int		transferShiftJisStateMachine		(PKANJISTATEMACHINE, int, Char*) ;
static	int		rtransferShiftJisStateMachine		(PKANJISTATEMACHINE, Char, char*) ;
static	int		restartCompoundTextStateMachine		(PKANJISTATEMACHINE) ;
static	int		transferCompoundTextStateMachine	(PKANJISTATEMACHINE, int, Char*) ;
static	int		ctextEscapeSequenceState0			(PCTEXTSTATE, int) ;
static	int		ctextEscapeSequenceState1			(PCTEXTSTATE, int) ;
static	int		ctextEscapeSequenceState2			(PCTEXTSTATE, int) ;
static	int		ctextEscapeSequenceState3			(PCTEXTSTATE, int) ;
static	int		ctextEscapeSequenceState4			(PCTEXTSTATE, int) ;
static	int		ctextEscapeSequenceState5			(PCTEXTSTATE, int) ;
static	int		ctextEscapeSequenceState6			(PCTEXTSTATE, int) ;
static	int		ctextEscapeSequenceDefault			(PCTEXTSTATE, int) ;
static	int		rtransferCompoundTextStateMachine	(PKANJISTATEMACHINE, Char, char*) ;


/*========================================================================
 *	Хؿ
 *========================================================================*/
/*
 *	ɲϾֵνԤ
 *[]
 *	pKSM			ܵ
 *	iCodingSystem	ǥ󥰼ˡ(EUC/SHIFTJIS/...)
 */
int
InitializeKanjiFiniteStateMachine (
	register PKANJISTATEMACHINE	pKSM,
	register int				iCodingSystem)
{
	static	int		(*pRestartFuncTbl [])(PKANJISTATEMACHINE pKSM)	= {
		restartEucJpStateMachine,
		restartIso2022JpStateMachine,
		restartShiftJisStateMachine,
		restartCompoundTextStateMachine,
	} ;
	static	int		(*pTransferFuncTbl [])(PKANJISTATEMACHINE pKSM, int iChara, Char* pOutput)	= {
		transferEucJpStateMachine,
		transferIso2022JpStateMachine,
		transferShiftJisStateMachine,
		transferCompoundTextStateMachine,
	} ;
	static	int		(*pRtransferFuncTbl [])(PKANJISTATEMACHINE pKSM, Char cc, char* pOutput)	= {
		rtransferEucJpStateMachine,
		rtransferIso2022JpStateMachine,
		rtransferShiftJisStateMachine,
		rtransferCompoundTextStateMachine,
	} ;

	assert (pKSM != NULL) ;
	assert (KCODING_SYSTEM_EUCJP <= iCodingSystem && iCodingSystem < MAX_KCODING_SYSTEM) ;

	pKSM->m_pRestartFunc	= pRestartFuncTbl   [iCodingSystem] ;
	pKSM->m_pTransferFunc	= pTransferFuncTbl  [iCodingSystem] ;
	pKSM->m_pRtransferFunc	= pRtransferFuncTbl [iCodingSystem] ;
	return	(pKSM->m_pRestartFunc)(pKSM) ;
}

/*
 *	ɾ֤᤹ܵ/ꤹ롣
 *[]
 *	pKSM		ܵ
 */
int
RestartKanjiFiniteStateMachine (
	register PKANJISTATEMACHINE	pKSM)
{
	assert (pKSM != NULL) ;
	assert (pKSM->m_pRestartFunc != NULL) ;
	return	(pKSM->m_pRestartFunc)(pKSM) ;
}

/*
 *	ɾܵ1ʸϤͿܤ롣
 *[]
 *	pKSM		ܵ
 *	iChara		ϥܥ
 *	pOutput		ϥܥ
 *[֤]
 *	False		ϥܥ¸ߤʤ
 *	True		ϥܥ뤬¸ߤ롣
 *	-1			顼ȯ
 */
int
TransferKanjiFiniteStateMachine (
	register PKANJISTATEMACHINE	pKSM,
	register int				iChara,
	register Char*				pOutput)
{
	assert (pKSM != NULL) ;
	assert (pKSM->m_pTransferFunc != NULL) ;
	assert (pOutput != NULL) ;
	return	(pKSM->m_pTransferFunc)(pKSM, iChara, pOutput) ;
}

/*
 *
 */
int
RtransferKanjiFiniteStateMachine (
	register PKANJISTATEMACHINE	pKSM,
	register Char				cc,
	register char*				pOutput)
{
	assert (pKSM != NULL) ;
	assert (pKSM->m_pRtransferFunc != NULL) ;
	assert (pOutput != NULL) ;
	return	(pKSM->m_pRtransferFunc)(pKSM, cc, pOutput) ;
}

/*
 *	եʸ沽Ƚ̤ޤ
 *	ºݤˤϷԲǽʾ礬¸ߤΤǤɤˤ⤳ˤ⡣
 *	⡢ߤȽϴְäƤġ
 */
int
DetectKanjiCodingSystem (
	register const char*		pBytes,
	register int				nCheckBytes,
	register int*				pNewlineType)
{
	register int	cc ;
	register int	i ;
	Boolean			fProbabilityCodingSystem [MAX_KCODING_SYSTEM] ;
	char			bufShiftJisChar [8] ;
	int				iBufShiftJisUsage ;
	char			bufEucJpChar [8] ;
	int				iBufEucJpUsage ;
	char			bufIso2022JpChar [8] ;
	int				iBufIso2022JpUsage ;
	NEWLINEINFO		newlineinfo ;
	register int	iNewlineType ;

	/*
	 *	ɤ̥ͥơ֥롣
	 */
	const int				priorityListOfCodingSystem [MAX_KCODING_SYSTEM] = {
		KCODING_SYSTEM_EUCJP,
		KCODING_SYSTEM_SHIFTJIS,
		KCODING_SYSTEM_ISO2022JP2,
	} ;

	assert (pBytes != NULL || nCheckBytes <= 0) ;

	/*
	 *	ɤʸɤƱͤ˳Τ餷פȤȤˤޤ
	 */
	for (i = 0 ; i < MAX_KCODING_SYSTEM ; i ++)
		fProbabilityCodingSystem [i]	= True ;

	newlineinfo.m_nCarriageReturn	= 0 ; 
	newlineinfo.m_nLineFeed			= 0 ; 
	newlineinfo.m_nCRLF				= 0 ;
	newlineinfo.m_fPostCR			= False ;

	/*
	 *	Lead Character ¸ߤʤȤƤޤ
	 */
	iBufShiftJisUsage	= 0 ;
	iBufEucJpUsage		= 0 ;
	iBufIso2022JpUsage	= 0 ;

	while (nCheckBytes > 0){
		cc	= *pBytes ;

		/*
		 *	SHIFTJIS ɤ沽ƤȲꤷ硣
		 */
		if (fProbabilityCodingSystem [KCODING_SYSTEM_SHIFTJIS] &&
			!checkShiftJisCodingSystem (bufShiftJisChar, &iBufShiftJisUsage, cc))
			fProbabilityCodingSystem [KCODING_SYSTEM_SHIFTJIS] = False ;

		/*
		 *	EUC-JP 沽ƤȲꤷ硣
		 */
		if (fProbabilityCodingSystem [KCODING_SYSTEM_EUCJP] &&
			!checkEucJpCodingSystem (bufEucJpChar, &iBufEucJpUsage, cc))
			fProbabilityCodingSystem [KCODING_SYSTEM_EUCJP]	= False ;

		/*
		 *	JUNET 沽ƤȲꤷ硣
		 */
		if (fProbabilityCodingSystem [KCODING_SYSTEM_ISO2022JP2]){
			switch (checkIso2022Jp2CodingSystem (bufIso2022JpChar, &iBufIso2022JpUsage, cc)){
			case	KCODING_SYSTEM_CHECK_False:
				/*
				 *	ʬȤǤ뤳ȤϤʤ
				 */
				fProbabilityCodingSystem [KCODING_SYSTEM_ISO2022JP2]	= False ;
				break ;
			case	KCODING_SYSTEM_CHECK_True_EXCLUSIVE:
				/*
				 *	ʬͥ٤ʸ沽ϤꤨʤȤ롣
				 */
				for (i = 0 ; i < MAX_KCODING_SYSTEM ; i ++){
					if (priorityListOfCodingSystem [i] == KCODING_SYSTEM_ISO2022JP2)
						break ;
					fProbabilityCodingSystem [priorityListOfCodingSystem [i]]	= False ;
				}
				break ;
			case	KCODING_SYSTEM_CHECK_True:
			default:
				break ;
			}
		}

		checkNewline (cc, &newlineinfo) ;

		pBytes		++ ;
		nCheckBytes -- ;
	}

	if (newlineinfo.m_nCRLF > 0) {
		iNewlineType	= KNEWLINE_MSDOS ;
	} else if (newlineinfo.m_nLineFeed > 0) {
		iNewlineType	= KNEWLINE_UNIX ;
	} else if (newlineinfo.m_nCarriageReturn > 0) {
		iNewlineType	= KNEWLINE_MAC ;
	} else {
		iNewlineType	= KNEWLINE_UNKNOWN ;
	}
	if (pNewlineType != NULL)
		*pNewlineType	= iNewlineType ;

	/*
	 *	ʸɤ̤ͥ˱ *餷* 沽ˡ֤
	 */
	for (i = 0 ; i < MAX_KCODING_SYSTEM ; i ++){
		if (fProbabilityCodingSystem [priorityListOfCodingSystem [i]])
			return	priorityListOfCodingSystem [i] ;
	}

	/*
	 *	Ǥˤϡɬ Unicode ˤʤ롣Unicode Ȥ
	 *	դ路ʤɤäƤΤ̵Ȼפ顣
	 */
	return	KCODING_SYSTEM_UNKNOWN ;
}

/*========================================================================*
 *	ΥǤΤѤɽŪʴؿ
 *========================================================================*/

/*========================================================================*
 *	ܸ EUC ν˴ؤؿ
 *========================================================================*/
/*
 *	Extended Unix Code ξܵ롣
 */
int
restartEucJpStateMachine (
	register PKANJISTATEMACHINE	pKSM)
{
	register PISO2022STATE	pState ;

	assert (pKSM != NULL) ;

	pState	= &pKSM->m_state.m_iso2022 ;
	pState->m_fEscapeSequence	= False ;
	pState->m_iEscapeState		=  0 ;
	pState->m_iLeadChara		= '\0' ;
	pState->m_fSingleShift		= False ;
	pState->m_iCharset [0]		= KCHARSET_ASCII ;
	pState->m_iCharset [1]		= KCHARSET_JISX0208_1983 ;
	pState->m_iCharset [2]		= KCHARSET_JISX0201_1976 ;
	pState->m_iCharset [3]		= KCHARSET_JISX0212_1990 ;
	pState->m_iGL [0]			=  0 ;
	pState->m_iGL [1]			= -1 ;
	pState->m_iGR [0]			=  1 ;
	pState->m_iGR [1]			= -1 ;
	return	True ;
}

/*
 *	Extended Unix Code ξֵܤ롣
 *----
 *	pState		ֵθ֡
 *	iChara		ϥܥ롣
 *	pOutput		ϥܥ롣
 */
int
transferEucJpStateMachine (
	register PKANJISTATEMACHINE	pKSM,
	register int				iChara,
	register Char*				pOutput)
{
	register PISO2022STATE	pState ;

	assert (pKSM != NULL) ;
	assert (pOutput != NULL) ;

	pState	= &pKSM->m_state.m_iso2022 ;

	if (pState->m_iLeadChara){
		switch (pState->m_iCharset [pState->m_iGR [0]]){
		case	KCHARSET_JISX0208_1983:
			if (pState->m_fSingleShift)
				return	-1 ;
			
			*pOutput				= Char_Make (KCHARSET_JISX0208_1983, ((pState->m_iLeadChara << 8) | iChara) & 0x7F7F) ;
			pState->m_iLeadChara	= '\0' ;
			return	True ;
		case	KCHARSET_JISX0212_1990:
			if (!pState->m_fSingleShift)
				return	-1 ;
			*pOutput				= Char_Make (KCHARSET_JISX0212_1990, ((pState->m_iLeadChara << 8) | iChara) & 0x7F7F) ;
			pState->m_iGR [0]		= 1 ;
			pState->m_iLeadChara	= '\0' ;
			pState->m_fSingleShift	= False ;
			return	True ;
		default:
			pState->m_iGR [0]		= 1 ;
			pState->m_iLeadChara	= '\0' ;
			pState->m_fSingleShift	= False ;
			return	-1 ;
		}
	}
	if (iChara <= (0x80 + 0x20)){
		if (pState->m_fSingleShift || pState->m_iLeadChara){
			pState->m_fSingleShift	= False ;
			pState->m_iLeadChara	= '\0' ;
			return	-1 ;
		}
		if (iChara < 0x80){
			*pOutput	= Char_MakeAscii ((char)iChara) ;
			return	True ;
		} else if (iChara == 0x8E){
			pState->m_iGR [0]		= 2 ;
			pState->m_fSingleShift	= True ;
			return	False ;
		} else if (iChara == 0x8F){
			pState->m_iGR [0]		= 3 ;
			pState->m_fSingleShift	= True ;
			return	False ;
		} else {
			return	-1 ;
		}
	}
	switch (pState->m_iCharset [pState->m_iGR [0]]){
	case	KCHARSET_JISX0201_1976:
		if (!pState->m_fSingleShift)
			return	-1 ;
		*pOutput				= Char_Make (KCHARSET_JISX0201_1976, iChara) ;
		pState->m_iGR [0]		= 1 ;
		pState->m_fSingleShift	= False ;
		return	True ;
	case	KCHARSET_JISX0208_1983:
	case	KCHARSET_JISX0212_1990:
		pState->m_iLeadChara	= iChara & 0x00FF ;
		return	False ;
	default:
		return	-1 ;
	}
}

/*
 *()
 *	ѴȤȤ r(everse)transfer Ȥ̾ˤƤ롣
 */
int
rtransferEucJpStateMachine (
	register PKANJISTATEMACHINE	pKSM,
	register Char				cc,
	register char*				pOutput)
{
	assert (pKSM != NULL) ;
	assert (pOutput != NULL) ;

	switch (Char_Charset (cc)) {
	case	KCHARSET_JISX0208_1978:
	case	KCHARSET_JISX0208_1983:
		*pOutput	++	= (char)(Char_Code (cc) >> 8) ;
		*pOutput	++	= (char)(Char_Code (cc) >> 0) ;
		return	2 ;

	case	KCHARSET_JISX0201_1976:
		if (Char_Code (cc) >= 128) {
			*pOutput	++	= '\x8E' ;
			*pOutput	++	= (char)Char_Code (cc) ;
			return	2 ;
		} else {
			*pOutput	++	= (char)Char_Code (cc) ;
			return	1 ;
		}

	case	KCHARSET_JISX0212_1990:
		*pOutput	++	= '\x8F' ;
		*pOutput	++	= (char)(Char_Code (cc) >> 8) ;
		*pOutput	++	= (char)(Char_Code (cc) >> 0) ;
		return	3 ;

	default:
		if (Char_IsAscii (cc)) {
			*pOutput	++	= (char)Char_Code (cc) ;
			return	1 ;
		}
		break ;
	}
	return	0 ;
}

/*========================================================================
 *	ISO2022-JP-2 ν˴ؤؿ
 *========================================================================*/
/*
 *	iso-2022-jp-2 ܵ롣
 *-----
 *()
 *	iso-2022-jp-2 ιƬǤξ֤ˤ碌롣
 *	ΤĹƬǤϡ
 *		G0 ʸϡ
 *		G1 ʸϡJISX0201-1976 αȾ̡
 *		G2 ʸϡꡣ
 *		G3 ʸϡꡣ
 *		GL  G0 ʸؤ
 *		GR  G1 ʸؤ
 *	ˤʤȦ
 *	JISX0201-1976 αȾ̤ΰˤĤƤɤ狼ʤ
 *	Mule ʤɤɤʤäƤĤȤƤ
 */
int
restartIso2022JpStateMachine (
	register PKANJISTATEMACHINE	pKSM)
{
	register PISO2022STATE	pState ;

	assert (pKSM != NULL) ;

	pState						= &pKSM->m_state.m_iso2022 ;
	pState->m_fEscapeSequence	= False ;
	pState->m_iEscapeState		= 0 ;
	pState->m_iLeadChara		= '\0' ;
	pState->m_fSingleShift		= False ;
	pState->m_iCharset [0]		= KCHARSET_ASCII ;
	pState->m_iCharset [1]		= KCHARSET_JISX0201_1976 ;
	pState->m_iCharset [2]		= KCHARSET_NOTHING ;
	pState->m_iCharset [3]		= KCHARSET_NOTHING ;
	pState->m_iGL [0]			= 0 ;
	pState->m_iGL [1]			= -1 ;
	pState->m_iGR [0]			= 1 ;
	pState->m_iGR [1]			= -1 ;
	return	True ;
}

/*
 *	iso-2022-jp-2 ܵ򥨥ߥ졼Ȥؿ
 *-----
 *()
 *	lpState		iso-2022-jp-2 ܵθߤξ֡
 *	iChara		ϥե٥åȡ
 *	pOutput		ϥե٥å(¸ߤ)֤ХåեΥݥ󥿡
 *(֤)
 *	0			ϥե٥åȤ¸ߤʤ
 *	1			ϥե٥åȤ¸ߤ롣
 *	-1			顼ȯMSɤ Unicode (ɥڡ:) ǰʤʸĤäƤ⥨顼
 */
int
transferIso2022JpStateMachine (
	register PKANJISTATEMACHINE	pKSM,
	register int				iChara,
	register Char*				pOutput)
{
	static	int	(*apIso2022jpEscapeSequenceHandlerTable [])(PISO2022STATE, int)	= {
		iso2022jpEscapeSequenceState0,
		iso2022jpEscapeSequenceState1,
		iso2022jpEscapeSequenceState2,
		iso2022jpEscapeSequenceState3,
		iso2022jpEscapeSequenceState4,
		iso2022jpEscapeSequenceState5,
		iso2022jpEscapeSequenceDefault,
	} ;
	register PISO2022STATE	pState ;

	assert (pKSM != NULL) ;
	assert (pOutput != NULL) ;

	pState	= &pKSM->m_state.m_iso2022 ;

	/*
	 *	ץ󥹤βʤСɤĹޤɤ߹
	 *	ʬ롣
	 */
	if (pState->m_fEscapeSequence){
		assert (0 <= pState->m_iEscapeState && pState->m_iEscapeState <= 6) ;
		return	(apIso2022jpEscapeSequenceHandlerTable [pState->m_iEscapeState]) (pState, iChara) ;
	}

	/*
	 *	ǤʤĲ餫ʸɤ߹硣
	 */
	if (pState->m_fSingleShift){
		pState->m_fSingleShift	= False ;
		pState->m_iLeadChara	= '\0' ;
		pState->m_iGR [0]		= pState->m_iGR [1] ;
		return	-1 ;
	}
	if (iChara <= 0x20){
		if (pState->m_iLeadChara){
			pState->m_iLeadChara	= '\0' ;
			return	-1 ;
		}
		if (iChara == 0x1b){
			pState->m_fEscapeSequence	= True ;
			pState->m_iEscapeState		= ISO2022JPSTATE_ESCAPE ;
			return	False ;
		} else {
			*pOutput	= Char_MakeAscii ((char)iChara) ;
			return	True ;
		}
	} else if (iChara < 0x80){
		if (pState->m_iGL [0] < 0 || pState->m_iGL [0] > 2)
			return	-1 ;

		switch (pState->m_iCharset [pState->m_iGL [0]]){
		case	KCHARSET_ASCII:
			if (pState->m_iLeadChara){
				pState->m_iLeadChara	= '\0' ;
				return	-1 ;
			}
			*pOutput	= Char_Make (pState->m_iCharset [pState->m_iGL [0]], iChara & 0x00FF) ;
			return	True ;

		case	KCHARSET_JISX0201_1976_KATAKANA:
			if (pState->m_iLeadChara){
				pState->m_iLeadChara	= '\0' ;
				return	-1 ;
			}
			*pOutput	= Char_Make (KCHARSET_JISX0201_1976, (iChara & 0x00FF) | 0x80) ;
			return	True ;

		case	KCHARSET_JISX0208_1978:
		case	KCHARSET_JISX0208_1983:
			if (pState->m_iLeadChara){
				*pOutput				= Char_Make (KCHARSET_JISX0208_1983, (pState->m_iLeadChara << 8) | iChara) ;
				pState->m_iLeadChara	= '\0' ;
				return	True ;
			} else {
				pState->m_iLeadChara	= iChara & 0x00FF ;
				return	False ;
			}

		case	KCHARSET_JISX0212_1990:
			if (pState->m_iLeadChara){
				*pOutput				= Char_Make (KCHARSET_JISX0212_1990, (pState->m_iLeadChara << 8) | iChara) ;
				pState->m_iLeadChara	= '\0' ;
				return	True ;
			} else {
				pState->m_iLeadChara	= iChara & 0x00FF ;
				return	False ;
			}

		default:
			return	-1 ;
		}
	}

	if (pState->m_iGR [0] < 0 || pState->m_iGR [0] > 2)
		return	-1 ;

	switch (pState->m_iCharset [pState->m_iGR [0]]){
	case	KCHARSET_JISX0201_1976:
		if (pState->m_iLeadChara){
			pState->m_iLeadChara	= '\0' ;
			return	-1 ;
		}
		*pOutput	= Char_Make (KCHARSET_JISX0201_1976, iChara) ;
		return	True ;
	default:
		break ;
	}

	return	-1 ;
}

/*
 *	ץ󥹤ν
 *()
 *	ޥåʥСäƤ˵ʤȤ٤
 *
 *
 *	[0]	"ESC" ľξ
 *		+--------------+--------------+---------------------+
 *		| ϥܥ | ξ | ¾              |
 *		+--------------+--------------+---------------------+
 *		|     '$'      |  1       |                     |
 *		+--------------+--------------+---------------------+
 *		|     '('      |  3       |                     |
 *		+--------------+--------------+---------------------+
 *		|     'N'      |  5       |                     |
 *		+--------------+--------------+---------------------+
 *		|     '.'      |  4       |                     |
 *		+--------------+--------------+---------------------+
 *		|   ʳ   |          | 顼              |
 *		+--------------+--------------+---------------------+
 *
 *	[1]	"ESC $"
 *		+--------------+--------------+---------------------+
 *		| ϥܥ | ξ | ¾              |
 *		+--------------+--------------+---------------------+
 *		|     '@'      |          | G0  JISC6226-1978 |
 *		|              |              | Ƥ롣      |
 *		+--------------+--------------+---------------------+
 *		|     'B'      |          | G0  JISX0208-1983 |
 *		|              |              | Ƥ롣      |
 *		+--------------+--------------+---------------------+
 *		|     'A'      |          | G0  GB2312-1980   |
 *		|              |              | Ƥ롣      |
 *		+--------------+--------------+---------------------+
 *		|     '('      |  2       |                     |
 *		+--------------+--------------+---------------------+
 *		|   ʳ   |          | 顼              |
 *		+--------------+--------------+---------------------+
 *
 *	[2]	"ESC $ ("
 *		+--------------+--------------+---------------------+
 *		| ϥܥ | ξ | ¾              |
 *		+--------------+--------------+---------------------+
 *		|     '@'      |          | G0  JISC6226-1978 |
 *		|              |              | Ƥ롣      |
 *		+--------------+--------------+---------------------+
 *		|     'B'      |          | G0  JISX0208-1983 |
 *		|              |              | Ƥ롣      |
 *		+--------------+--------------+---------------------+
 *		|     '('      |          | G0  KSC6501-1987  |
 *		|              |              | Ƥ롣      |
 *		+--------------+--------------+---------------------+
 *		|     'D'      |          | G0  JISX0212-1990 |
 *		|              |              | Ƥ롣      |
 *		+--------------+--------------+---------------------+
 *		|   ʳ   |          | 顼              |
 *		+--------------+--------------+---------------------+
 *
 *	[3]	"ESC ("
 *		+--------------+--------------+---------------------+
 *		| ϥܥ | ξ | ¾              |
 *		+--------------+--------------+---------------------+
 *		|     'B'      |          | G0  ASCII   |
 *		|              |              | Ƥ롣            |
 *		+--------------+--------------+---------------------+
 *		|     'J'      |          | G0  JISX0201-1976 |
 *		|              |              | Ƥ롣      |
 *		+--------------+--------------+---------------------+
 *		|   ʳ   |          | 顼              |
 *		+--------------+--------------+---------------------+
 *
 *	[4]	"ESC ."
 *		+--------------+--------------+---------------------+
 *		| ϥܥ | ξ | ¾              |
 *		+--------------+--------------+---------------------+
 *		|     'A'      |          | G2  ISO8859-1   |
 *		|              |              | Ƥ롣        |
 *		+--------------+--------------+---------------------+
 *		|     'F'      |          | G2  ISO8859-7   |
 *		|              |              | Ƥ롣        |
 *		+--------------+--------------+---------------------+
 *		|   ʳ   |          | 顼              |
 *		+--------------+--------------+---------------------+
 *
 *	[5]	"ESC N"
 *		ְäƤ뵤롣
 *		Shingle Shift Sequence?
 */
int
iso2022jpEscapeSequenceState0 (
	register PISO2022STATE	pState,
	register int			iChara)
{
	switch (iChara){
	case '$':
		pState->m_iEscapeState	= ISO2022JPSTATE_G0_94_2_CHARSET1 ;
		break ;
	case '(':
		pState->m_iEscapeState	= ISO2022JPSTATE_G0_94_1_CHARSET ;
		break ;
	case 'N':
		/*	shingle-shift-seq */
		pState->m_iEscapeState	= ISO2022JPSTATE_SINGLE_SHIFT ;
		break ;
	case '.':
		/* 96 character sets ESC sequence */
		pState->m_iEscapeState	= ISO2022JPSTATE_G2_96_1_CHARSET ;
		break ;
	default:
		/*	Τʤץ󥹤褿ν*/
		pState->m_iEscapeState		= ISO2022JPSTATE_ESCAPE ;
		pState->m_fEscapeSequence	= False ;
		break ;
	}
	return	False ;
}

/*
 *	ץ󥹤ν
 *()
 */
int
iso2022jpEscapeSequenceState1 (
	register PISO2022STATE	pState,
	register int			iChara)
{
	switch (iChara){
	case '@':
		/*	G0  JISC6226-1978 (94^2ʸƤ*/
		pState->m_iCharset [0] = KCHARSET_JISX0208_1978 ;
		break ;
	case 'B':
		/*	G0  JISX0208-1983 (94^2ʸƤ) */
		pState->m_iCharset [0] = KCHARSET_JISX0208_1983 ;
		break ;
	case 'A':
		/*	G0  GB2312-1980 (94^2 ʸƤ) */
		pState->m_iCharset [0] = KCHARSET_GB2312_1980 ;
		break ;
	case '(':
		pState->m_iEscapeState = ISO2022JPSTATE_G0_94_2_CHARSET2 ;
		break ;
	default:
		break ;
	}
	return	iso2022jpEscapeSequenceDefault (pState, iChara) ;
}

/*
 *	ץ󥹤ν
 *()
 */
int
iso2022jpEscapeSequenceState2 (
	register PISO2022STATE	pState,
	register int			iChara)
{
	switch (iChara){
	case '@':
		/*	G0  JISC6226-1978 (94^2ʸ)Ƥ */
		pState->m_iCharset [0] = KCHARSET_JISX0208_1978 ;
		break ;
	case 'B':
		/*	G0  JISX0208-1983 (94^2ʸ)Ƥ */
		pState->m_iCharset [0] = KCHARSET_JISX0208_1983 ;
		break ;
	case '(':
		/*	G0  KSC6501-1987 (94^2ʸ)Ƥ */
		pState->m_iCharset [0] = KCHARSET_KSC5601_1987 ;
		break ;
	case 'D':
		/*	G0  JISX0212-1990 (94^2ʸ)Ƥ */
		pState->m_iCharset [0] = KCHARSET_JISX0212_1990 ;
		break ;
	default:
		break ;
	}
	return	iso2022jpEscapeSequenceDefault (pState, iChara) ;
}

/*
 *	ץ󥹤ν
 *()
 */
int
iso2022jpEscapeSequenceState3 (
	register PISO2022STATE		pState,
	register int				iChara)
{
	if (iChara == 'B'){
		/*	G0  ASCII (94^1ʸ) Ƥ롣*/
		pState->m_iCharset [0]	= KCHARSET_ASCII ;
	} else if (iChara == 'J'){
		/*	G0  JISX0201-1976(Roman) (94^1ʸ) Ƥ롣*/
		pState->m_iCharset [0]	= KCHARSET_JISX0201_1976 ;
	} else if (iChara == 'I'){
		/*
		 *	˵ʤɤ⡢96 ʸȾʬ
		 *	ƤӹࡣäơCharset äƤȤ鸫ȱʤ
		 *	ʤ
		 */
		pState->m_iCharset [0]	= KCHARSET_JISX0201_1976_KATAKANA ;
	}
	return	iso2022jpEscapeSequenceDefault (pState, iChara) ;
}

/*
 *	ץ󥹤ν
 *()
 */
int
iso2022jpEscapeSequenceState4 (
	register PISO2022STATE	pState,
	register int			iChara)
{
	if (iChara == 'A'){
		/*	G2  ISO8859-1 Ƥ롣*/
		pState->m_iCharset [2]	= KCHARSET_ISO8859_1 ;
	} else if (iChara == 'F'){
		/*	G2  ISO8859-7 Ƥ롣*/
		pState->m_iCharset [2]	= KCHARSET_ISO8859_7 ;
	}
	return	iso2022jpEscapeSequenceDefault (pState, iChara) ;
}

/*
 *	ץ󥹤ν
 *()
 */
int	
iso2022jpEscapeSequenceState5 (PISO2022STATE pState, int iChara)
{
	/* Single-Shift-Char */
	pState->m_iLeadChara	= '\0' ;
	pState->m_fSingleShift	= True ;
	pState->m_iGR [1]		= pState->m_iGR [0] ;
	pState->m_iGR [0]		= 2 ;
	return	iso2022jpEscapeSequenceDefault (pState, iChara) ;
}

int
iso2022jpEscapeSequenceDefault (PISO2022STATE pState, int iChara)
{
	pState->m_iEscapeState		= ISO2022JPSTATE_ESCAPE ;
	pState->m_fEscapeSequence	= False ;
	return	False ;

	UNREFERENCED_PARAMETER (iChara) ;
}

/*
 *
 */
int
rtransferIso2022JpStateMachine (
	register PKANJISTATEMACHINE	pKSM,
	register Char				cc,
	register char*				pOutput)
{
	register PISO2022STATE	pState ;
	register char*			pOutputTop ;
	register const char*	pEscSeq ;

	assert (pKSM != NULL) ;
	assert (pOutput != NULL) ;

	pState		= &pKSM->m_state.m_iso2022 ;
	pOutputTop	= pOutput ;

	/*
	 *	ɽƤʸϷ빽פǤ롣
	 */
	switch (Char_Charset (cc)) {
	case	KCHARSET_JISX0208_1978:
		pEscSeq	= "\033$@" ;
		goto	state_2byte_char ;

	case	KCHARSET_JISX0208_1983:
		pEscSeq	= "\033$B" ;
		goto	state_2byte_char ;

	case	KCHARSET_JISX0212_1990:
		pEscSeq	= "\033$(D" ;
		goto	state_2byte_char ;

	case	KCHARSET_GB2312_1980:
		pEscSeq	= "\033$A" ;
		goto	state_2byte_char ;

	case	KCHARSET_KSC5601_1987:
		pEscSeq	= "\033$(C" ;

	state_2byte_char:
		/*
		 *	G0 ʸ礬Ĵ٤롣
		 *	GL <= G0 ϸꡣ
		 */
		if (pState->m_iCharset [0] != (int)Char_Charset (cc)) {
			while (*pEscSeq)
				*pOutput ++	= *pEscSeq ++ ;
			pState->m_iCharset [0]	= Char_Charset (cc) ;
		}
		*pOutput ++	= (char)(Char_Code (cc) >> 8) ;
		*pOutput ++	= (char)(Char_Code (cc) >> 0) ;
		break ;

	case	KCHARSET_JISX0201_1976:
		/*
		 *	JISX0201-ROMAN  ASCII Ȥϰ㤦餷
		 *	Ʊ˿äƤޤʤ褦դ롣
		 */
		if (Char_Code (cc) < 0x20) {
			*pOutput ++ = (char)Char_Code (cc) ;
			break ;
		}
		if (Char_Code (cc) < 0x80) {
			if (pState->m_iCharset [0] != KCHARSET_JISX0201_1976) {
				*pOutput ++	= 0x1b ;
				*pOutput ++	= '(' ;
				*pOutput ++	= 'J' ;
				pState->m_iCharset [0] = KCHARSET_JISX0201_1976 ;
			}
		} else {
			if (pState->m_iCharset [0] != KCHARSET_JISX0201_1976_KATAKANA) {
				*pOutput ++	= 0x1b ;
				*pOutput ++	= '(' ;
				*pOutput ++	= 'I' ;
				pState->m_iCharset [0] = KCHARSET_JISX0201_1976_KATAKANA ;
			}
		}
		*pOutput ++ = (char)(Char_Code (cc) & 0x7F) ;
		break ;

	case	KCHARSET_ISO8859_1:
		if (Char_IsAscii (cc))
			goto	state_ascii ;

		pEscSeq	= "\033.A" ;
		goto	state_96charset ;

	case	KCHARSET_ISO8859_7:
		if (Char_IsAscii (cc))
			goto	state_ascii ;

		pEscSeq	= "\033.F" ;

	state_96charset:
		if (pState->m_iCharset [2] != (int)Char_Charset (cc)) {
			while (*pEscSeq)
				*pOutput ++	= *pEscSeq ++ ;
			pState->m_iCharset [2]	= Char_Charset (cc) ;
		}
		*pOutput ++	= 0x1b ;
		*pOutput ++	= 'N' ;
		*pOutput ++ = (char)(Char_Code (cc) & 0x7F) ;
		break ;

	default:
		/*
		 *	ASCII ǤʤФ⤦ɽԲǽ
		 */
		if (!Char_IsAscii (cc)) 
			break ;

	state_ascii:
		if (pState->m_iCharset [0] != KCHARSET_ASCII) {
			*pOutput ++	= 0x1b ;
			*pOutput ++	= '(' ;
			*pOutput ++	= 'B' ;
			pState->m_iCharset [0]	= KCHARSET_ASCII ;
		}
		*pOutput ++ = (char)Char_Code (cc) ;
		break ;
	}
	return	(pOutput - pOutputTop) ;
}

/*========================================================================*
 *	ShiftJis ν˴ؤؿ
 *========================================================================*/

int
restartShiftJisStateMachine (
	register PKANJISTATEMACHINE	pKSM)
{
	register PSHIFTJISSTATE	pState ;

	assert (pKSM != NULL) ;

	pState					= &pKSM->m_state.m_shiftjis ;
	pState->m_iLeadChara	= '\0' ;
	return	True ;
}

int
transferShiftJisStateMachine (
	register PKANJISTATEMACHINE	pKSM,
	register int				iChara,
	register Char*				pOutput)
{
	register PSHIFTJISSTATE	pState ;
	register int			sh, sl, jh, jl ;

	assert (pKSM != NULL) ;
	assert (pOutput != NULL) ;

	pState	= &pKSM->m_state.m_shiftjis ;

	if (pState->m_iLeadChara){
		if (IS_SHIFTJIS_2BYTE_CODE2 (iChara)){
			sh	= pState->m_iLeadChara & 0xFF ;
			sl	= iChara & 0xFF ;
			jh	= ((sh - ((sh >= 0xa0)? 0xc1: 0x81)) << 1)+0x21;
			if (sl >= 0x9f) {
				jh	++;
				jl	= sl - 0x7e ;
			} else {
				jl	= sl - ((sl <= 0x7e) ? 0x1f : 0x20) ;
			}
			*pOutput	= Char_Make (KCHARSET_JISX0208_1983, (jh << 8) | jl) ;
			pState->m_iLeadChara	= '\0' ;
			return	True ;
		} else {
			/*
			 *	ʸ
			 */
			pState->m_iLeadChara	= '\0' ;
			return	-1 ;
		}
	}

	if (!IS_SHIFTJIS_2BYTE_CODE1 (iChara)){
		if (IS_SHIFTJIS_JISX201_KATAKANA (iChara)){
			/*
			 *	JISX0201-1976
			 */
			*pOutput	= Char_Make (KCHARSET_JISX0201_1976, iChara & 0xFF) ;
			return	True ;
		} else {
			/*
			 *	ʸ ASCII ʸȻפ
			 */
			*pOutput	= Char_MakeAscii ((char)(iChara & 0x007F)) ;
			return	True ;
		}
	}
	pState->m_iLeadChara	= iChara ;
	return	False ;
}

/*
 *
 */
int
rtransferShiftJisStateMachine (
	register PKANJISTATEMACHINE	pKSM,
	register Char				cc,
	register char*				pOutput)
{
	register int	jh, jl, sh, sl ;

	assert (pKSM != NULL) ;
	assert (pOutput != NULL) ;

	switch (Char_Charset (cc)) {
	case	KCHARSET_JISX0208_1978:
	case	KCHARSET_JISX0208_1983:
		jh	= (Char_Code (cc) >> 8) & 0x00FF ;
		jl	= (Char_Code (cc) >> 0) & 0x00FF ;
		sh	= ((jh - 0x21) >> 1) + 0x81 ;
		if (sh > 0x9f)
			sh	+= 0x40 ;
		if (jh & 1) {
			sl	= jl + 0x1f ;
			if (jl > 0x5f)
				sl	++ ;
		} else {
			sl	= jl + 0x7e ;
		}
		*pOutput ++	= (char) sh ;
		*pOutput ++	= (char) sl ;
		return	2 ;

	case	KCHARSET_JISX0201_1976:
		*pOutput ++	= (char)Char_Code (cc) ;
		return	1 ;

	default:
		if (Char_IsAscii (cc)) {
			*pOutput ++	= (char)Char_Code (cc) ;
			return	1 ;
		}
		break ;
	}
	return	0 ;
}

/*========================================================================*
 *	CompoundText ν˴ؤؿ
 *========================================================================*/
int
restartCompoundTextStateMachine (
	register PKANJISTATEMACHINE	pKSM)
{
	register PCTEXTSTATE	pState ;

	assert (pKSM != NULL) ;
	pState						= &pKSM->m_state.m_ctext ;
	pState->m_fEscapeSequence	= False ;
	pState->m_iEscapeState		= CTEXTSTATE_ESCAPE ;
	pState->m_iLeadChara		= '\0' ;
	pState->m_fSingleShift		= False ;
	pState->m_nGL 				= KCHARSET_ASCII ;
	pState->m_nGR 				= KCHARSET_ISO8859_1 ;
	return	True ;
}

int
transferCompoundTextStateMachine (
	register PKANJISTATEMACHINE	pKSM,
	register int				nChara,
	register Char*				pOutput)
{
	static	int (*apCTextEscapeSeqHandlerTable [])(PCTEXTSTATE, int)	= {
		ctextEscapeSequenceState0,
		ctextEscapeSequenceState1,
		ctextEscapeSequenceState2,
		ctextEscapeSequenceState3,
		ctextEscapeSequenceState4,
		ctextEscapeSequenceState5,
		ctextEscapeSequenceState6,
	} ;
	register PCTEXTSTATE	pState ;

	assert (pKSM != NULL) ;
	assert (pOutput != NULL) ;

	pState	= &pKSM->m_state.m_ctext ;
	if (pState->m_fEscapeSequence) {
		return	(apCTextEscapeSeqHandlerTable [pState->m_iEscapeState])(pState, nChara) ;
	}
	if (nChara <= 0x20) {
		if (nChara == 0x1b) {
			pState->m_fEscapeSequence	= True ;
			pState->m_iEscapeState		= CTEXTSTATE_ESCAPE ;
			return	False ;
		} else {
			*pOutput	= Char_MakeAscii ((char)nChara) ;
			return	True ;
		}
	} else if (nChara < 0x80) {
		switch (pState->m_nGL) {
		case	KCHARSET_ASCII:
			if (pState->m_iLeadChara){
				pState->m_iLeadChara	= '\0' ;
				return	-1 ;
			}
			*pOutput	= Char_Make (pState->m_nGL, nChara & 0x00FF) ;
			return	True ;

		case	KCHARSET_JISX0201_1976_KATAKANA:
			if (pState->m_iLeadChara){
				pState->m_iLeadChara	= '\0' ;
				return	-1 ;
			}
			*pOutput	= Char_Make (KCHARSET_JISX0201_1976, (nChara & 0x00FF) | 0x80) ;
			return	True ;

		case	KCHARSET_JISX0208_1978:
			if (pState->m_iLeadChara){
				*pOutput				= Char_Make (KCHARSET_JISX0208_1983, (pState->m_iLeadChara << 8) | nChara) ;
				pState->m_iLeadChara	= '\0' ;
				return	True ;
			} else {
				pState->m_iLeadChara	= nChara & 0x00FF ;
				return	False ;
			}

		case	KCHARSET_JISX0212_1990:
		case	KCHARSET_JISX0208_1983:
		case	KCHARSET_GB2312_1980:
		case	KCHARSET_KSC5601_1987:
			if (pState->m_iLeadChara){
				*pOutput				= Char_Make (pState->m_nGL, (pState->m_iLeadChara << 8) | nChara) ;
				pState->m_iLeadChara	= '\0' ;
				return	True ;
			} else {
				pState->m_iLeadChara	= nChara & 0x00FF ;
				return	False ;
			}
		}
	}
	if (pState->m_iLeadChara){
		pState->m_iLeadChara	= '\0' ;
		return	-1 ;
	}
	*pOutput	= Char_Make (pState->m_nGR, nChara) ;
	return	True ;
}

int
ctextEscapeSequenceState0 (
	register PCTEXTSTATE	pState,
	register int			nChara)
{
	switch (nChara){
	case	0x24:	/* 94^N charset? */
		pState->m_iEscapeState	= CTEXTSTATE_94NCHARSET ;
		break ;
	case	0x28:	/* 94 charset */
		pState->m_iEscapeState	= CTEXTSTATE_GL_94CHARSET ;
		break ;
	case	0x29:	/* 94 charset */
		pState->m_iEscapeState	= CTEXTSTATE_GR_94CHARSET ;
		break ;
	case	0x2D:	/* 96 charset */
		pState->m_iEscapeState	= CTEXTSTATE_GR_96CHARSET ;
		break ;
	default:
		/*	Τʤץ󥹤褿ν*/
		pState->m_iEscapeState		= CTEXTSTATE_ESCAPE ;
		pState->m_fEscapeSequence	= False ;
		break ;
	}
	return	False ;
}

int
ctextEscapeSequenceState1 (
	register PCTEXTSTATE	pState,
	register int			nChara)
{
	switch (nChara) {
	case	0x42:	/* 7-bit ASCII graphics */
	case	0x4A:	/* Left half of JISX0201-1976 */
		pState->m_nGL	= KCHARSET_ASCII ;
		break ;
	case	0x49:	/* Right half of JISX0201-1976 */
		pState->m_nGL	= KCHARSET_JISX0201_1976_KATAKANA ;
		break ;
	default:
		break ;
	}
	return	ctextEscapeSequenceDefault (pState, nChara) ;
}

int
ctextEscapeSequenceState2 (
	register PCTEXTSTATE	pState,
	register int			nChara)
{
	switch (nChara) {
	case	0x28:
		pState->m_iEscapeState	= CTEXTSTATE_GL_94NCHARSET ;
		break ;
	case	0x29:
		pState->m_iEscapeState	= CTEXTSTATE_GR_94CHARSET ;
		break ;
	default:
		return	ctextEscapeSequenceDefault (pState, nChara) ;
	}
	return	False ;
}

int
ctextEscapeSequenceState3 (
	register PCTEXTSTATE	pState,
	register int			nChara)
{
	static	int		sr94_2_CharsetTbl []	= {
		KCHARSET_GB2312_1980,	KCHARSET_JISX0208_1983,
		KCHARSET_KSC5601_1987,
	} ;
	if (0x41 <= nChara && nChara <= 0x43) 
		pState->m_nGL	= sr94_2_CharsetTbl [nChara - 0x41] ;
	return	ctextEscapeSequenceDefault (pState, nChara) ;
}

int
ctextEscapeSequenceState4 (
	register PCTEXTSTATE	pState,
	register int			nChara)
{
	static	int		sr94_2_CharsetTbl []	= {
		KCHARSET_GB2312_1980,	KCHARSET_JISX0208_1983,
		KCHARSET_KSC5601_1987,
	} ;
	if (0x41 <= nChara && nChara <= 0x43) 
		pState->m_nGR	= sr94_2_CharsetTbl [nChara - 0x41] ;
	return	ctextEscapeSequenceDefault (pState, nChara) ;
}

int
ctextEscapeSequenceState5 (
	register PCTEXTSTATE	pState,
	register int			nChara)
{
	switch (nChara) {
	case	0x42:	/* 7-bit ASCII graphics */
	case	0x4A:	/* Left half of JISX0201-1976 */
		pState->m_nGR	= KCHARSET_ASCII ;
		break ;
	case	0x49:	/* Right half of JISX0201-1976 */
		pState->m_nGR	= KCHARSET_JISX0201_1976_KATAKANA ;
		break ;
	default:
		break ;
	}
	return	ctextEscapeSequenceDefault (pState, nChara) ;
}

int
ctextEscapeSequenceState6 (
	register PCTEXTSTATE	pState,
	register int			nChara)
{
	static	int		sr96CharsetTbl []	= {
		KCHARSET_ISO8859_1,			KCHARSET_ISO8859_2,
		KCHARSET_ISO8859_3,			KCHARSET_ISO8859_4,
		KCHARSET_NOTHING,			KCHARSET_ISO8859_7,
		KCHARSET_ISO8859_6,			KCHARSET_ISO8859_8,
		KCHARSET_NOTHING,			KCHARSET_NOTHING,
		KCHARSET_NOTHING,			KCHARSET_ISO8859_5,
		KCHARSET_ISO8859_9,
	} ;
	if (0x41 <= nChara && nChara <= 0x4D &&
		sr96CharsetTbl [nChara - 0x41] != KCHARSET_NOTHING) {
		pState->m_nGR	= sr96CharsetTbl [nChara - 0x41] ;
	}
	return	ctextEscapeSequenceDefault (pState, nChara) ;
}

int
ctextEscapeSequenceDefault (
	register PCTEXTSTATE 	pState,
	register int 			nChara)
{
	pState->m_iEscapeState		= CTEXTSTATE_ESCAPE ;
	pState->m_fEscapeSequence	= False ;
	return	False ;

	UNREFERENCED_PARAMETER (nChara) ;
}


int
rtransferCompoundTextStateMachine (
	register PKANJISTATEMACHINE	pKSM,
	register Char				cc,
	register char*				pOutput)
{
	register PCTEXTSTATE	pState ;
	register char*			pOutputTop ;
	register const char*	pEscSeq ;
	static   const char*	prEscSeqTbl []	= {
		"\033(B",	"\033-A",	"\033-B",	"\033-C",	"\033-D",
		"\033-L",	"\033-G",	"\033-F",	"\033-H",	"\033-M",
		NULL,		NULL,		NULL,		"\033$(B",	"\033$(B",
		NULL,		"\033$(A",	NULL,		NULL,		NULL,
		NULL,		NULL,		"\033$(C",	NULL,
	} ;
	register int			nCharset ;

	assert (pKSM != NULL) ;
	assert (pOutput != NULL) ;

	pState		= &pKSM->m_state.m_ctext ;
	pOutputTop	= pOutput ;

	/*	ɽƤʸϷ빽פǤ롣*/
	nCharset	= (int)Char_Charset (cc) ;
	switch (nCharset) {
	case	KCHARSET_JISX0208_1978:
	case	KCHARSET_JISX0208_1983:
	case	KCHARSET_GB2312_1980:
	case	KCHARSET_KSC5601_1987:
		if (pState->m_nGL != nCharset) {
			pEscSeq	= prEscSeqTbl [nCharset] ;
			while (*pEscSeq)
				*pOutput ++	= *pEscSeq ++ ;
			pState->m_nGL	= Char_Charset (cc) ;
		}
		*pOutput ++	= (char)(Char_Code (cc) >> 8) ;
		*pOutput ++	= (char)(Char_Code (cc) >> 0) ;
		break ;

	case	KCHARSET_JISX0201_1976:
		if (Char_Code (cc) < 0x20) {
			*pOutput ++ = (char)Char_Code (cc) ;
			break ;
		}
		pEscSeq	= "" ;
		if (Char_Code (cc) < 0x80) {
			if (pState->m_nGL != KCHARSET_JISX0201_1976) {
				pEscSeq	= "\033(J" ;
				pState->m_nGL = KCHARSET_JISX0201_1976 ;
			}
		} else {
			if (pState->m_nGR != KCHARSET_JISX0201_1976_KATAKANA) {
				pEscSeq	= "\033)I" ;
				pState->m_nGR = KCHARSET_JISX0201_1976_KATAKANA ;
			}
		}
		while (*pEscSeq)
			*pOutput ++	= *pEscSeq ++ ;
		*pOutput ++ = (char)Char_Code (cc) ;
		break ;

	case	KCHARSET_ISO8859_1:
	case	KCHARSET_ISO8859_2:
	case	KCHARSET_ISO8859_3:
	case	KCHARSET_ISO8859_4:
	case	KCHARSET_ISO8859_5:
	case	KCHARSET_ISO8859_6:
	case	KCHARSET_ISO8859_7:
	case	KCHARSET_ISO8859_8:
	case	KCHARSET_ISO8859_9:
		if (Char_IsAscii (cc))
			goto	state_ascii ;

		if (pState->m_nGR != (int)Char_Charset (cc)) {
			pEscSeq	= prEscSeqTbl [nCharset] ;
			while (*pEscSeq)
				*pOutput ++	= *pEscSeq ++ ;
			pState->m_nGR	= Char_Charset (cc) ;
		}
		*pOutput ++ = (char)Char_Code (cc) ;
		break ;

	default:
		/*	ASCII ǤʤФ⤦ɽԲǽ*/
		if (!Char_IsAscii (cc)) 
			break ;

	state_ascii:
		if (pState->m_nGL != KCHARSET_ASCII) {
			pEscSeq	= prEscSeqTbl [KCHARSET_ASCII] ;
			while (*pEscSeq)
				*pOutput ++	= *pEscSeq ++ ;
			pState->m_nGL	= KCHARSET_ASCII ;
		}
		*pOutput ++ = (char)Char_Code (cc) ;
		break ;
	}
	return	(pOutput - pOutputTop) ;
}


int
checkShiftJisCodingSystem (
	register char*		pchBuffer,
	register int*		piUsage,
	register int		iChara)
{
	register int		fRetvalue ;
	register int		iUsage ;

	assert (pchBuffer != NULL) ;
	assert (piUsage != NULL) ;
			
	/*
	 *	괺 ShiftJIS 餷ȤȤˤƤޤ
	 */
	fRetvalue	= True ;

	iUsage		= *piUsage ;
	if (iUsage <= 0){
		if (!IS_SHIFTJIS_2BYTE_CODE1 (iChara)){
			if (!IS_SHIFTJIS_JISX201_KATAKANA (iChara) && iChara >= 0x80)
				fRetvalue	= False ;
		} else {
			*(pchBuffer + iUsage)	= (char) iChara ;
			iUsage	++ ;
		}
	} else {
		if (!IS_SHIFTJIS_2BYTE_CODE2 (iChara))
			fRetvalue	= False ;
		iUsage	= 0 ;
	}
	*piUsage	= iUsage ;

	return	fRetvalue ;
}

int
checkEucJpCodingSystem (
	register char*		pchBuffer, 
	register int*		piUsage,
	register int		iChara)
{
	register int		fRetvalue ;
	register int		iUsage ;

	assert (pchBuffer != NULL) ;
	assert (piUsage != NULL) ;
			
	fRetvalue	= True ;
	iUsage		= *piUsage ;

	if (iUsage <= 0){
		if (iChara == 0x8E ||
			iChara == 0x8F ||
			(0xA1 <= iChara && iChara <= 0xFE)){
			*(pchBuffer + iUsage ++)	= (char) iChara ;
		} else if (iChara >= 0x80){
			fRetvalue	= False ;
		}
	} else {
		if (0xA1 <= iChara && iChara <= 0xFE){
			switch ((unsigned char)*pchBuffer){
			case 0x8F:
				if (iUsage < 2){
					*(pchBuffer + iUsage ++)	= (char) iChara ;
				} else {
					iUsage	= 0 ;
				}
				break ;
			case 0x8E:
			default:
				if (iUsage >= 2)
					fRetvalue	= False ;
				iUsage	= 0 ;
				break ;
			}
		} else {
			fRetvalue	= False ;
			iUsage		= 0 ;
		}
	}
	return	fRetvalue ;
}

int
checkIso2022Jp2CodingSystem (
	register char*	pchBuffer,
	register int*	piUsage,
	register int	iChara)
{
	register int		fRetvalue ;
	register int		i ;
	register int		iUsage ;
	/*
	 *	iso-2022-jp-2 ˵Ƥ륨ץ󥹡
	 */
	const char*				iso2022jpEscapeSequences [] = {
		"\x1b$@",	"\x1b$B",	"\x1b$(@",	"\x1b$(B",	"\x1b(B",	"\x1b(J",
		"\x1b(I",	"\x1b$A",	"\x1b$(C",	"\x1b$(D",	"\x1b.A",	"\x1b.F",
		NULL,
	} ;

	assert (pchBuffer != NULL) ;
	assert (piUsage != NULL) ;

	fRetvalue	= True ;
	iUsage		= *piUsage ;

	if (iUsage <= 0){
		if (iChara == 0x1b){
			*(pchBuffer + iUsage ++)	= (char) iChara ;
		} else if (iChara >= 0x80){
			fRetvalue	= False ;
		}
	} else {
		/*	 Escape Sequence ʤΤ*/
		*(pchBuffer + iUsage ++)	= (char) iChara ;
		for (i = 0 ; iso2022jpEscapeSequences [i] != NULL ; i ++){
			if (strncmp (pchBuffer, iso2022jpEscapeSequences [i], iUsage) == 0){
				if (!(iso2022jpEscapeSequences [i])[iUsage]){
					/*
					 *	ץ󥹤դˤϼʬ
					 *	ͥ٤ι⤤ Coding System ϤʤȤ롣
					 */
					fRetvalue	= KCODING_SYSTEM_CHECK_True_EXCLUSIVE ;
					iUsage		= 0 ;
				}
				break ;
			}
		}
		/*
		 *	Τʤץ󥹤äˤϡ
		 *	ISO2022-JP-2 βǽϤʤ
		 */
		if (!iso2022jpEscapeSequences [i]){
			fRetvalue	= False ;
			iUsage		= 0 ;
		}
	}
	*piUsage	= iUsage ;
	return	fRetvalue ;
}

Boolean
checkNewline (
	register Char			cc,
	register NEWLINEINFO*	pNewlineInfo)
{
	assert (pNewlineInfo != NULL) ;

	switch (cc) {
	case	0x0D:
		pNewlineInfo->m_nCarriageReturn	++ ;
		pNewlineInfo->m_fPostCR	= True ;
		break ;
		
	case	0x0A:
		if (pNewlineInfo->m_fPostCR) {
			pNewlineInfo->m_nCRLF	++ ;
		} else {
			pNewlineInfo->m_nLineFeed	++ ;
		}
		pNewlineInfo->m_fPostCR	= False ;
		break ;
		
	default:
		pNewlineInfo->m_fPostCR	= False ;
		break ;
	}
	return	True ;
}

