#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include "bufstringp.h"

static	TBufStringNode*		tBufString_newNode (void) ;
static	TBufStringLeaf*		tBufString_newLeaf (void) ;
static	void				tBufString_updatePath (TBufStringNode*, int) ;
static	Boolean				tBufStringNode_setLeft (TBufStringNode*, void*, Boolean) ;
static	Boolean				tBufStringNode_setRight (TBufStringNode*, void*, Boolean) ;
static	void				tBufStringNode_setParent (TBufStringNode*, TBufStringNode*) ;
static	Boolean				tBufStringMarker_insertChar (TBufStringMarker*, Char) ;
static	Boolean				tBufStringLeaf_leftp (const TBufStringLeaf*) ;
static	Boolean				tBufStringLeaf_rightp (const TBufStringLeaf*) ;
static	void				tBufStringLeaf_setParent (TBufStringLeaf*, TBufStringNode*) ;
static	Boolean				tBufStringMarker_deleteChar (TBufStringMarker*, int) ;
static	Boolean				tBufStringNode_deleteRecursively (TBufStringNode**, TBufStringNode*) ;
static	Boolean				tBufStringLeaf_delete (TBufString*, TBufStringLeaf*) ;
static	Boolean				tBufString_checkRegion (TBufStringNode*) ;
static	void				tBufString_destroyTree (TBufStringNode*) ;
static	void				tBufStringNode_dump (TBufStringNode*) ;
static	void				tBufStringLeaf_dump (TBufStringLeaf*) ;

#define	INLINE	inline

#define	RIGHT_NODE(pNode)	((pNode)->Right.u.m_pNode)
#define	RIGHT_LEAF(pNode)	((pNode)->Right.u.m_pLeaf)
#define	LEFT_NODE(pNode)	((pNode)->Left.u.m_pNode)
#define	LEFT_LEAF(pNode)	((pNode)->Left.u.m_pLeaf)
#define	SET_RED(pNode)		tBufStringNode_setColor((pNode), TBUSTRINGNODECOLOR_RED)


/*
 *
 */
static	INLINE	TBufStringNode*
tBufString_newNode (void)
{
	register TBufStringNode*	pNode	= MALLOC (sizeof (TBufStringNode)) ;

	if (pNode != NULL) {
		pNode->m_nLength		= 0 ;
		pNode->m_pParent		= NULL ;
		pNode->Left.m_fLeaf		= True ;
		pNode->Left.u.m_pLeaf	= NULL ;
		pNode->Right.m_fLeaf	= True ;
		pNode->Right.u.m_pLeaf	= NULL ;
		pNode->m_nColor			= TBUFSTRINGNODECOLOR_RED ;
	}
	return	 pNode ;
}

static	INLINE	TBufStringLeaf*
tBufString_newLeaf (void)
{
	register TBufStringLeaf*	pLeaf	= MALLOC (sizeof (TBufStringLeaf)) ;

	if (pLeaf != NULL) {
		pLeaf->m_nLength	= 0 ;
		pLeaf->m_pParent	= NULL ;
		pLeaf->m_pPrev		= NULL ;
		pLeaf->m_pNext		= NULL ;
	}
	return	 pLeaf ;
}

/*	ꤵ줿Ρɤ롼Ȥޤ Length ͤ nDiff û
 *	ޤ롣
 */
static	INLINE	void
tBufString_updatePath (
	register TBufStringNode*	pNode,
	register int				nDiff)
{
	if (nDiff == 0)
		return ;

	while (pNode != NULL) {
		pNode->m_nLength	+= nDiff ;
		pNode	= pNode->m_pParent ;
	}
	return ;
}

static	INLINE	Boolean
tBufStringLeaf_leftp (
	register const TBufStringLeaf*	pLeaf)
{
	register const TBufStringNode*	pNode ;
	pNode	= pLeaf->m_pParent ;
	return	TSUCCEEDED (pNode->Left.m_fLeaf) && (pNode->Left.u.m_pLeaf == pLeaf) ;
}

static	INLINE	Boolean
tBufStringLeaf_rightp (
	register const TBufStringLeaf*	pLeaf)
{
	register const TBufStringNode*	pNode ;
	pNode	= pLeaf->m_pParent ;
	return	TSUCCEEDED (pNode->Right.m_fLeaf) && (pNode->Right.u.m_pLeaf == pLeaf) ;
}

static	INLINE	void
tBufStringLeaf_setParent (
	register TBufStringLeaf*	pLeaf,
	register TBufStringNode*	pParentNode)
{
	pLeaf->m_pParent	= pParentNode ;
	return ;
}

static	INLINE	int
tBufStringNode_getLeftLength (
	register TBufStringNode*	pNode)
{
	if (pNode->Left.m_fLeaf) {
		return	(pNode->Left.u.m_pLeaf == NULL)? 0 : pNode->Left.u.m_pLeaf->m_nLength ;
	} else {
		return	(pNode->Left.u.m_pNode == NULL)? 0 : pNode->Left.u.m_pNode->m_nLength ;
	}
}

static	INLINE	int
tBufStringNode_getRightLength (
	register TBufStringNode*	pNode)
{
	if (pNode->Right.m_fLeaf) {
		return	(pNode->Right.u.m_pLeaf == NULL)? 0 : pNode->Right.u.m_pLeaf->m_nLength ;
	} else {
		return	(pNode->Right.u.m_pNode == NULL)? 0 : pNode->Right.u.m_pNode->m_nLength ;
	}
}

static	INLINE	Boolean
tBufStringNode_setLeft (
	register TBufStringNode*	pNode,
	register void*				pPtr,
	register Boolean				fLeaf)
{
	assert (pNode != NULL) ;

	pNode->Left.m_fLeaf	= fLeaf ;
	if (fLeaf) {
		register TBufStringLeaf*	pChild	= pPtr ;
		pNode->Left.u.m_pLeaf	= pChild ;
		if (pChild != NULL)
			pChild->m_pParent	= pNode ;
	} else {
		register TBufStringNode*	pChild	= pPtr ;
		pNode->Left.u.m_pNode	= pChild ;
		if (pChild != NULL)
			pChild->m_pParent	= pNode ;
	}
	return	True ;
}

static	INLINE	Boolean
tBufStringNode_setRight (
	register TBufStringNode*	pNode,
	register void*				pPtr,
	register Boolean			fLeaf)
{
	assert (pNode != NULL) ;

	pNode->Right.m_fLeaf	= fLeaf ;
	if (fLeaf) {
		register TBufStringLeaf*	pChild	= pPtr ;
		pNode->Right.u.m_pLeaf	= pPtr ;
		if (pChild != NULL)
			pChild->m_pParent	= pNode ;
	} else {
		register TBufStringNode*	pChild	= pPtr ;
		pNode->Right.u.m_pNode	= pPtr ;
		if (pChild != NULL)
			pChild->m_pParent	= pNode ;
	}
	return	True ;
}

static	INLINE	void
tBufStringNode_setParent (
	register TBufStringNode*	pNode,
	register TBufStringNode*	pParentNode)
{
	pNode->m_pParent	= pParentNode ;
	return ;
}

static	INLINE	TBufStringNode*
tBufStringNode_successor (
	register TBufStringNode*	pNodeX)
{
	register TBufStringNode*	pNodeY ;

	if (TFAILED (pNodeX->Right.m_fLeaf)) {
		register TBufStringNode*	pNode	= pNodeX->Right.u.m_pNode ;
		assert (pNode != NULL) ;
		while (TFAILED (pNode->Right.m_fLeaf))
			pNode	= pNode->Right.u.m_pNode ;
		assert (pNode != NULL) ;
		return	pNode ;
	}
	pNodeY	= pNodeX->m_pParent ;
	while (pNodeY != NULL && 
		   TFAILED (pNodeY->Right.m_fLeaf) && pNodeX == pNodeY->Right.u.m_pNode) {
		pNodeX	= pNodeY ;
		pNodeY	= pNodeY->m_pParent ;
	}
	return	pNodeY ;
}

static	INLINE	TBufStringNodeColor
tBufStringNode_getColor (
	register TBufStringNode*	pNode)
{
	if (pNode == NULL)
		return	TBUFSTRINGNODECOLOR_BLACK ;
	return	pNode->m_nColor ;
}

static	INLINE	void
tBufStringNode_setColor (
	register TBufStringNode*		pNode,
	register TBufStringNodeColor	nColor)
{
	if (pNode == NULL)
		return ;
	pNode->m_nColor	= nColor ;
	return ;
}

static	INLINE	TBufStringNode*
tBufStringNode_getLeftNode (
	register TBufStringNode*		pNode)
{
	if (pNode == NULL || TSUCCEEDED (pNode->Left.m_fLeaf))
		return	NULL ;
	return	pNode->Left.u.m_pNode ;
}

static	INLINE	TBufStringNode*
tBufStringNode_getRightNode (
	register TBufStringNode*		pNode)
{
	if (pNode == NULL || TSUCCEEDED (pNode->Right.m_fLeaf))
		return	NULL ;
	return	pNode->Right.u.m_pNode ;
}

static	INLINE	TBufStringNode*
tBufStringNode_getParentNode (
	register TBufStringNode*		pNode)
{
	if (pNode == NULL)
		return	NULL ;
	return	pNode->m_pParent ;
}

static	INLINE	Boolean
tBufStringNode_update (
	register TBufStringNode*	pNode)
{
	pNode->m_nLength	= (tBufStringNode_getLeftLength  (pNode) +
						   tBufStringNode_getRightLength (pNode)) ;
	return	True ;
}

static	INLINE	Boolean
tBufStringNode_leftRotate (
	register TBufStringNode**	ppRoot,
	register TBufStringNode*	pNodeX)
{
	register TBufStringNode*	pNodeY ;
	register TBufStringNode*	pNodeParentOfX ;

	assert (ppRoot != NULL && *ppRoot != NULL) ;

#if defined (DEBUG_LV99)
	assert (tBufString_checkRegion (*ppRoot) == True) ;
#endif
	if (pNodeX == NULL)
		return	True ;

	/*	դäѤޤ魯ȤϤǤʤ
	 */
	if (TSUCCEEDED (pNodeX->Right.m_fLeaf))
		return	False ;

	/*	y <- right [x]
	 */
	pNodeParentOfX	= pNodeX->m_pParent ;
	pNodeY			= RIGHT_NODE (pNodeX) ;

	/*	right [x] <- left [y]
	 *	if left [y] != nil 
	 *		then p [left [y]] <- x
	 */
	if (pNodeY->Left.m_fLeaf) {
		tBufStringNode_setRight (pNodeX, LEFT_LEAF (pNodeY), True) ;
	} else {
		tBufStringNode_setRight (pNodeX, LEFT_NODE (pNodeY), False) ;
	}

	/*	p [y] <- p [x]
	 */
	tBufStringNode_setParent (pNodeY, pNodeParentOfX) ;

	/*	if p [x] = nil
	 *		then root [T] <- y
	 *		else if x = left [p [x]]
	 *			then left [p [x]] <- y
	 *			else right [p [x]] <- y
	 */
	if (pNodeParentOfX == NULL) {
		*ppRoot	= pNodeY ;
	} else {
		if (pNodeX == LEFT_NODE (pNodeParentOfX)) {
			LEFT_NODE (pNodeParentOfX)	= pNodeY ;
		} else {
			RIGHT_NODE (pNodeParentOfX)	= pNodeY ;
		}
	}

	/*	left [y] <- x
	 *	p [x] <- y
	 */
	tBufStringNode_setLeft (pNodeY, pNodeX, False) ;

	/*	X  Y  length ƹ롣ƶϰϤϤδ֤ʤΤǡ
	 *	֤ X -> Y Ǥ뤳ȡ
	 */
	tBufStringNode_update (pNodeX) ;
	tBufStringNode_update (pNodeY) ;
#if defined (DEBUG_LV99)
	assert (tBufString_checkRegion (*ppRoot) == True) ;
#endif
	return	True ;
}

static	INLINE	Boolean
tBufStringNode_rightRotate (
	register TBufStringNode**	ppRoot,
	register TBufStringNode*	pNodeX)
{
	register TBufStringNode*	pNodeY ;
	register TBufStringNode*	pNodeParentOfX ;

	assert (ppRoot != NULL && *ppRoot != NULL) ;

#if defined (DEBUG_LV99)
	assert (tBufString_checkRegion (*ppRoot) == True) ;
#endif
	if (pNodeX == NULL)
		return	True ;

	/*	դäѤޤ魯ȤϤǤʤ
	 */
	if (TSUCCEEDED (pNodeX->Left.m_fLeaf))
		return	False ;

	/*	y <- left [x]
	 */
	pNodeParentOfX	= pNodeX->m_pParent ;
	pNodeY			= LEFT_NODE (pNodeX) ;

	/*	left [x] <- right [y]
	 *	p [right [y]] <- x
	 */
	if (pNodeY->Right.m_fLeaf) {
		tBufStringNode_setLeft (pNodeX, RIGHT_LEAF (pNodeY), True) ;
	} else {
		tBufStringNode_setLeft (pNodeX, RIGHT_NODE (pNodeY), False) ;
	}

	/*	p [y] <- p [x]
	 */
	tBufStringNode_setParent (pNodeY, pNodeParentOfX) ;

	/*	if p [x] = NIL
	 *		then root [T] <- y
	 *		else if x = right [p [x]] 
	 *		then right [p [x]] <- y
	 *		else left  [p [x]] <- y
	 */
	if (pNodeParentOfX == NULL) {
		*ppRoot	= pNodeY ;
	} else {
		if (pNodeX == RIGHT_NODE (pNodeParentOfX)) {
			RIGHT_NODE (pNodeParentOfX)	= pNodeY ;
		} else {
			LEFT_NODE  (pNodeParentOfX)	= pNodeY ;
		}
	}
	/*	right [y] <- x
	 *	p [x] <- y
	 */
	tBufStringNode_setRight (pNodeY, pNodeX, False) ;

	/*	X  Y  length ƹ롣ƶϰϤϤδ֤ʤΤǡ
	 *	֤ X -> Y Ǥ뤳ȡ
	 */
	tBufStringNode_update (pNodeX) ;
	tBufStringNode_update (pNodeY) ;
#if defined (DEBUG_LV99)
	assert (tBufString_checkRegion (*ppRoot) == True) ;
#endif
	return	True ;
}

static	INLINE	Boolean
tBufStringNode_rbupdate (
	register TBufStringNode**	ppRootNode,
	register TBufStringNode*	pNodeX)
{
	register TBufStringNode*	pNodeY ;
	register TBufStringNode*	pNodePX ;
	register TBufStringNode*	pNodePPX ;

#if defined (DEBUG_LV99)
	assert (tBufString_checkRegion (*ppRootNode) == True) ;
#endif
	tBufStringNode_setColor (pNodeX, TBUFSTRINGNODECOLOR_RED) ;
	for ( ; ; ) {
		pNodePX = tBufStringNode_getParentNode (pNodeX) ;
		if (pNodePX == NULL || tBufStringNode_getColor (pNodePX) != TBUFSTRINGNODECOLOR_RED) 
			break ;
		pNodePPX	= tBufStringNode_getParentNode (pNodePX) ;
		if (pNodePX == tBufStringNode_getLeftNode (pNodePPX)) {
			pNodeY	= tBufStringNode_getRightNode (pNodePPX) ;
			if (tBufStringNode_getColor (pNodeY) == TBUFSTRINGNODECOLOR_RED) {
				tBufStringNode_setColor (pNodePX,  TBUFSTRINGNODECOLOR_BLACK) ;
				tBufStringNode_setColor (pNodeY,   TBUFSTRINGNODECOLOR_BLACK) ;
				tBufStringNode_setColor (pNodePPX, TBUFSTRINGNODECOLOR_RED) ;
				pNodeX				= pNodePPX ;
			} else {
				if (pNodeX == tBufStringNode_getRightNode (pNodePX)) {
					pNodeX 			= pNodePX ;
					tBufStringNode_leftRotate (ppRootNode, pNodeX) ;
				}
				/*	left-rotate η̡pNodeX  parent ǽ
				 *	ΤǼľ*/
				pNodePX				= tBufStringNode_getParentNode (pNodeX) ;
				pNodePPX			= tBufStringNode_getParentNode (pNodePX) ;
				tBufStringNode_setColor (pNodePX,  TBUFSTRINGNODECOLOR_BLACK) ;
				tBufStringNode_setColor (pNodePPX, TBUFSTRINGNODECOLOR_RED) ;
				tBufStringNode_rightRotate (ppRootNode, pNodePPX) ;
			}
		} else {
			pNodeY	= tBufStringNode_getLeftNode (pNodePPX) ;
			if (tBufStringNode_getColor (pNodeY) == TBUFSTRINGNODECOLOR_RED) {
				tBufStringNode_setColor (pNodePX,  TBUFSTRINGNODECOLOR_BLACK) ;
				tBufStringNode_setColor (pNodeY,   TBUFSTRINGNODECOLOR_BLACK) ;
				tBufStringNode_setColor (pNodePPX, TBUFSTRINGNODECOLOR_RED) ;
				pNodeX				= pNodePPX ;
			} else {
				if (pNodeX == tBufStringNode_getLeftNode (pNodePX)) {
					pNodeX 			= pNodePX ;
					tBufStringNode_rightRotate (ppRootNode, pNodeX) ;
				}
				/*	left-rotate η̡pNodeX  parent ǽ
				 *	ΤǼľ*/
				pNodePX				= tBufStringNode_getParentNode (pNodeX) ;
				pNodePPX			= tBufStringNode_getParentNode (pNodePX) ;
				tBufStringNode_setColor (pNodePX,  TBUFSTRINGNODECOLOR_BLACK) ;
				tBufStringNode_setColor (pNodePPX, TBUFSTRINGNODECOLOR_RED) ;
				tBufStringNode_leftRotate (ppRootNode, pNodePPX) ;
			}
		}
	}
	tBufStringNode_setColor (*ppRootNode, TBUFSTRINGNODECOLOR_BLACK) ;
#if defined (DEBUG_LV99)
	assert (tBufString_checkRegion (*ppRootNode) == True) ;
#endif
	return	True ;
}

static	INLINE	Boolean
tBufStringLeaf_insertLeft (
	register TBufStringNode**	ppRootNode,
	register TBufStringLeaf*	pPrevLeaf,
	register TBufStringLeaf*	pNewLeaf)
{
	register TBufStringNode*	pNode ;
	register TBufStringNode*	pNewNode ;
	register int				nNewLength ;

	pNode				= pPrevLeaf->m_pParent ;
	if (TSUCCEEDED (tBufStringLeaf_leftp (pPrevLeaf))) {
		if (pNode->Right.u.m_pLeaf == NULL) {
			tBufStringNode_setRight (pNode, pNewLeaf, True) ;
			nNewLength	= pNewLeaf->m_nLength + pPrevLeaf->m_nLength ;
			tBufString_updatePath (pNode, nNewLength - pNode->m_nLength) ;
#if defined (DEBUG_LV99)
			assert (tBufString_checkRegion (*ppRootNode) == True) ;
#endif
			return	True ;
		}
		pNewNode			= tBufString_newNode () ;
		if (pNewNode == NULL)
			return	False ;
		tBufStringNode_setLeft   (pNewNode, pPrevLeaf, True) ;
		tBufStringNode_setRight  (pNewNode, pNewLeaf,  True) ;
		tBufStringNode_setLeft   (pNode,    pNewNode,  False) ;
		pNewNode->m_nLength	= pPrevLeaf->m_nLength + pNewLeaf->m_nLength ;
		nNewLength	= pNewNode->m_nLength + tBufStringNode_getRightLength (pNode) ;
		tBufString_updatePath (pNode, nNewLength - pNode->m_nLength) ;
	} else {
		assert (TSUCCEEDED (tBufStringLeaf_rightp (pPrevLeaf))) ;
		pNewNode			= tBufString_newNode () ;
		if (pNewNode == NULL)
			return	False ;
		tBufStringNode_setLeft  (pNewNode, pPrevLeaf, True) ;
		tBufStringNode_setRight (pNewNode, pNewLeaf,  True) ;
		tBufStringNode_setRight (pNode,    pNewNode,  False) ;
		pNewNode->m_nLength	= pPrevLeaf->m_nLength + pNewLeaf->m_nLength ;
		nNewLength	= pNewNode->m_nLength + tBufStringNode_getLeftLength (pNode) ;
		tBufString_updatePath (pNode, nNewLength - pNode->m_nLength) ;
	}
#if defined (DEBUG_LV99)
	assert (tBufString_checkRegion (*ppRootNode) == True) ;
#endif
	return	tBufStringNode_rbupdate (ppRootNode, pNewNode) ;
}

/*	ꤵ줿 Leaf μ˿ Leaf 롣
 *	Null μꤷˤϡƬ뤳Ȥ̣롣
 */
static	INLINE	Boolean
tBufStringLeaf_insertNext (
	register TBufStringNode**	ppRootNode,
	register TBufStringLeaf*	pPrevLeaf,
	register TBufStringLeaf*	pNewLeaf)
{
	register TBufStringLeaf*	pNextLeaf ;

	/*	Leaf ʬꥹȤˤʤäƤΤǡʬ
	 *	ƹԤ
	 */
	assert (pPrevLeaf != NULL) ;
	pNextLeaf			= pPrevLeaf->m_pNext ;
	pNewLeaf->m_pNext	= pNextLeaf ;
	pNewLeaf->m_pPrev	= pPrevLeaf ;

	if (pNextLeaf != NULL)
		pNextLeaf->m_pPrev	= pNewLeaf ;
	pPrevLeaf->m_pNext	= pNewLeaf ;

	/*	Node Tree κƹԤ
	 */
	return	tBufStringLeaf_insertLeft  (ppRootNode, pPrevLeaf, pNewLeaf) ;
}

/*	Хåեʸν
 *
 */
Boolean
TBufString_Initialize (
	register TBufString*		pBufString)
{
	TBufStringNode*		pNode ;
	TBufStringLeaf*		pLeaf ;

	pLeaf	= tBufString_newLeaf () ;
	pNode	= tBufString_newNode () ;
	if (pLeaf == NULL || pNode == NULL) {
		FREE (pLeaf) ;
		FREE (pNode) ;
		return	False ;
	}
	tBufStringNode_setLeft  (pNode, pLeaf, True) ;
	tBufStringNode_setRight (pNode, NULL,  True) ;

	pBufString->m_pRoot	= pNode ;
	pBufString->m_pHead	= pLeaf ;
	return	True ;
}

Boolean
TBufString_Uninitialize (
	register TBufString*		pBufString)
{
	register TBufStringLeaf*		pLeaf ;
	register TBufStringLeaf*		pNextLeaf ;

	pLeaf	= pBufString->m_pHead ;
	while (pLeaf != NULL) {
		pNextLeaf	= pLeaf->m_pNext ;
		FREE (pLeaf) ;
		pLeaf		= pNextLeaf ;
	}
	pBufString->m_pHead	= NULL ;

	tBufString_destroyTree (pBufString->m_pRoot) ;
	pBufString->m_pRoot	= NULL ;

	return	False ;
}

/*	ޡβԤؿ
 *
 *		ʸƬ index Ϳ줿ˡʬǤ줿ʸ
 *		ݥȤޡꤷ֤
 *		ʸγؤƤˤϡFalse 롣
 *
 *	ꡧ
 *		ʸɬ 1 Ĥ Root  Leaf äƤ֤ˤ롣
 *
 *		ޡϤۤȤɻȤΤƾ֤ʤΤǡѤɬδؿ
 *		ȤäƲ褹ɬפ롣
 */
Boolean
TBufString_SolveMarker (
	register TBufString*		pBufString,
	register int				nIndex,
	register TBufStringMarker*	pMarker)
{
	register TBufStringNode*	pNode ;
	register int				nLeft ;

	assert (pBufString != NULL) ;
	assert (pMarker    != NULL) ;

	pNode		= pBufString->m_pRoot ;
	if (nIndex < 0) {
		nIndex	= 0 ;
	} else if (nIndex > pNode->m_nLength) {
		nIndex	= pNode->m_nLength ;
	}
	pMarker->m_pBufString	= pBufString ;
	pMarker->m_nOffset		= nIndex ;

	while (pNode != NULL) {
		/*	λ¦ʸ򸫤롣
		 */
		if (pNode->Left.m_fLeaf) {
			register TBufStringLeaf*	pLeaf = pNode->Left.u.m_pLeaf ;
			if (pLeaf == NULL) {
				nLeft	= 0 ;
			} else {
				nLeft	= pLeaf->m_nLength ;
				if ((nIndex < nLeft)  ||
					(nIndex == nLeft && pLeaf->m_pNext == NULL)) {
					pMarker->m_pLeaf		= pLeaf ;
					pMarker->m_nPosition	= nIndex ;
					return	True ;
				}
			}
		} else {
			register TBufStringNode*	pLeftNode	= pNode->Left.u.m_pNode ;
			assert (pLeftNode != NULL) ;
			nLeft	= pLeftNode->m_nLength ;
			if ((nIndex < nLeft) ||
				(nIndex == nLeft &&
				 ((pNode->Right.m_fLeaf && pNode->Right.u.m_pLeaf == NULL) ||
				  pNode->Right.u.m_pNode == NULL))) {
				pNode	= pLeftNode ;
				continue ;
			}
		}
		/*	λ¦ʸ򸫤롣
		 */
		nIndex	-= nLeft ;
		if (pNode->Right.m_fLeaf) {
			register TBufStringLeaf*	pLeaf	= pNode->Right.u.m_pLeaf ;
			if (pLeaf == NULL) {
#if defined (DEBUG)
				fprintf (stderr, "right leaf == null\n") ;
				tBufStringNode_dump (pNode) ;
				assert (pLeaf != NULL) ;
#endif
				return	False ;
			}
			/*	Ĺ᤮ʸ󤬻ꤵ줿ˤϡǽ֤ݥ
			 *	롣
			 */
			if (nIndex > pLeaf->m_nLength) {
				if (pLeaf->m_pNext != NULL) {
#if defined (DEBUG)
					tBufStringLeaf_dump (pLeaf) ;
					fprintf (stderr, "fatal ... region invalid\n") ;
#endif
					return	False ;
				}
				pMarker->m_nOffset	-= (nIndex - pLeaf->m_nLength) ;
				nIndex				= pLeaf->m_nLength ;
			}
			pMarker->m_pLeaf		= pLeaf ;
			pMarker->m_nPosition	= nIndex ;
			return	True ;
		} else {
			pNode	= pNode->Right.u.m_pNode ;
		}
	}
#if defined (DEBUG)
	fprintf (stderr, "fatal ... not found\n") ;
#endif
	return	False ;
}

int
TBufString_GetLength (
	register TBufString*	pBufString)
{
	assert (pBufString != NULL) ;
	assert (pBufString->m_pRoot != NULL) ;

	return	pBufString->m_pRoot->m_nLength ;
}

Boolean
TBufStringMarker_InsertString (
	register TBufStringMarker*	pMarker,
	register const Char*		pString,
	register int				nString)
{
	assert (pMarker != NULL) ;

	while (nString > 0 && TSUCCEEDED (tBufStringMarker_insertChar (pMarker, *pString))) {
		TBufStringMarker_Forward (pMarker, 1) ;
		pString	++ ;
		nString	-- ;
	}
	return	(nString == 0) ;
}

Boolean
TBufStringMarker_InsertChar (
	register TBufStringMarker*	pMarker,
	register Char				cc,
	register int				nCount)
{
#if defined (DEBUG_LV99)
	register TBufString*	pBufString ;
	register int			nLength, nCountBack ;

	assert (pMarker != NULL) ;
	pBufString	= pMarker->m_pBufString ;
	assert (pBufString != NULL) ;
	assert (pBufString->m_pRoot != NULL) ;
	nLength		= pBufString->m_pRoot->m_nLength ;
	nCountBack	= nCount ;
	assert (tBufString_checkRegion (pBufString->m_pRoot) == True) ;

	while (nCount > 0 && TSUCCEEDED (tBufStringMarker_insertChar (pMarker, cc))) {
		TBufStringMarker_Forward (pMarker, 1) ;
		nCount	-- ;
	}
	assert (tBufString_checkRegion (pBufString->m_pRoot) == True) ;
	assert (pBufString->m_pRoot->m_nLength == (nLength + nCountBack)) ;
	return	(nCount == 0) ;
#else
	assert (pMarker != NULL) ;
	while (nCount > 0 && TSUCCEEDED (tBufStringMarker_insertChar (pMarker, cc))) {
		TBufStringMarker_Forward (pMarker, 1) ;
		nCount	-- ;
	}
	return	(nCount == 0) ;
#endif
}

Boolean
TBufStringMarker_DeleteChar (
	register TBufStringMarker*	pMarker,
	register int				nCount)
{
#if defined (DEBUG_LV99)
	register Boolean			fRetval ;
	TBufString*	pBufString ;
	int			nLength ;

	assert (pMarker != NULL) ;
	assert (pMarker->m_pLeaf != NULL) ;

	pBufString	= pMarker->m_pBufString ;
	nLength		= pBufString->m_pRoot->m_nLength ;
	assert (tBufString_checkRegion (pBufString->m_pRoot) == True) ;
	fRetval		= tBufStringMarker_deleteChar (pMarker, nCount) ;

	assert (tBufString_checkRegion (pBufString->m_pRoot) == True) ;
	assert (pBufString->m_pRoot->m_nLength == (nLength - nCount)) ;
	return	fRetval ;
#else
	assert (pMarker != NULL) ;
	return	tBufStringMarker_deleteChar (pMarker, nCount) ;
#endif
}

Boolean
TBufStringMarker_Forward (
	register TBufStringMarker*	pMarker,
	register int				nCount)
{
	register TBufStringLeaf*	pLeaf ;

	assert (pMarker != NULL) ;

	if (nCount < 0)
		return	TBufStringMarker_Backward (pMarker, - nCount) ;

	pLeaf		= pMarker->m_pLeaf ;
	if (pLeaf == NULL)
		return	False ;

	/*	˵ˤĤƤǤɤäƤߤ롣¿ʬϤ䤤
	 */
	if (nCount < SIZE_TBUFBASESTRING) {
		register int	nPosition	= pMarker->m_nPosition + nCount ;
		register int	nMove		= 0 ;

		while (nPosition >= pLeaf->m_nLength && pLeaf->m_pNext != NULL) {
			nPosition	-= pLeaf->m_nLength ;
			nMove		+= pLeaf->m_nLength ;
			pLeaf		= pLeaf->m_pNext ;
		}
		pMarker->m_pLeaf	= pLeaf ;
		if (nPosition < pLeaf->m_nLength) {
			pMarker->m_nOffset		+= nCount ;
			pMarker->m_nPosition	=  nPosition ;
		} else {
			pMarker->m_nOffset		+= nMove + pLeaf->m_nLength - pMarker->m_nPosition ;
			pMarker->m_nPosition	=  pLeaf->m_nLength ;
		}
		return	True ;
	}

	/*	ˤĤƤ Top Down  log Ĵ٤Ƥߤ롣
	 */
	return	TBufString_SolveMarker (pMarker->m_pBufString, pMarker->m_nOffset + nCount, pMarker) ;
}

Boolean
TBufStringMarker_Backward (
	register TBufStringMarker*	pMarker,
	register int				nCount)
{
	register TBufStringLeaf*	pLeaf ;
	register int				nPosition ;

	assert (pMarker != NULL) ;

	if (nCount < 0)
		return	TBufStringMarker_Forward (pMarker, - nCount) ;

	pLeaf		= pMarker->m_pLeaf ;
	if (pLeaf == NULL)
		return	False ;

	if (nCount < SIZE_TBUFBASESTRING) {
		pLeaf		= pMarker->m_pLeaf ;
		nPosition	= pMarker->m_nPosition - nCount ;
		while (nPosition < 0 && pLeaf->m_pPrev != NULL) {
			pLeaf		= pLeaf->m_pPrev ;
			nPosition	+= pLeaf->m_nLength ;
		}
		pMarker->m_pLeaf	= pLeaf ;
		if (nPosition >= 0) {
			pMarker->m_nPosition	=  nPosition ;
			pMarker->m_nOffset		-= nCount ;
		} else {
			pMarker->m_nPosition	=  0 ;
			pMarker->m_nOffset		+= (nPosition - pMarker->m_nPosition + nCount) ;
		}
		return	True ;	
	}

	return	TBufString_SolveMarker (pMarker->m_pBufString, pMarker->m_nOffset - nCount, pMarker) ;
}

int
TBufStringMarker_GetOffset (
	register TBufStringMarker*	pMarker)
{
	assert (pMarker != NULL) ;
	return	(pMarker->m_pLeaf != NULL)? pMarker->m_nOffset : -1 ;
}

Char
TBufStringMarker_GetChar (
	register TBufStringMarker*	pMarker)
{
	register TBufStringLeaf*	pLeaf ;

	assert (pMarker != NULL) ;
	assert (pMarker->m_pLeaf != NULL) ;
	pLeaf	= pMarker->m_pLeaf ;
	if (pMarker->m_nPosition < 0 ||
		pMarker->m_nPosition >= pLeaf->m_nLength)
		return	(Char) -1 ;
	return	pLeaf->m_achText [pMarker->m_nPosition] ;
}

void
TBufStringMarker_SetVoid (
	register TBufStringMarker*	pMarker)
{
	assert (pMarker != NULL) ;
	pMarker->m_pBufString	= NULL ;
	pMarker->m_pLeaf		= NULL ;
	pMarker->m_nPosition	= -1 ;
	pMarker->m_nOffset		= -1 ;
	return ;
}

Boolean
TBufStringMarker_Voidp (
	register TBufStringMarker*	pMarker)
{
	assert (pMarker != NULL) ;
	return	(pMarker->m_pLeaf == NULL) ;
}

Boolean
tBufStringMarker_insertChar (
	register TBufStringMarker*	pMarker,
	register Char				cc)
{
	register TBufString*		pBufString ;
	register TBufStringLeaf*	pLeaf ;
	register TBufStringLeaf*	pNewLeaf ;
	register int				nPosition ;
	register int				nLength ;

	assert (pMarker != NULL) ;
	pBufString	= pMarker->m_pBufString ;
	pLeaf		= pMarker->m_pLeaf ;
	assert (pLeaf   != NULL) ;
	nLength		= pLeaf->m_nLength + 1 ;
	nPosition	= pMarker->m_nPosition ;
	assert (0 <= 0 && nPosition < nLength) ;

#if defined (DEBUG_LV99)
	assert (tBufString_checkRegion (pBufString->m_pRoot) == True) ;
#endif
	if (nLength <= SIZE_TBUFBASESTRING) {
		/*	ʸ륹ڡ롣
		 */
		memmove (pLeaf->m_achText + nPosition + 1,
				 pLeaf->m_achText + nPosition,
				 sizeof (Char) * (pLeaf->m_nLength - nPosition)) ;
		/*	ʸ񤭹ࡣ
		 */
		pLeaf->m_achText [nPosition]	= cc ;
		pLeaf->m_nLength	++ ;
		/*	Ѥ Tree ξ򹹿롣O (log n) 롣
		 *	¹ԤȤɤɡġ
		 */
		tBufString_updatePath (pLeaf->m_pParent, 1) ;
#if defined (DEBUG_LV99)
		assert (tBufString_checkRegion (pBufString->m_pRoot) == True) ;
#endif
		return	True ;
	}

	/*	ƥȤ濴2Ĥʬ䤹롣ʬ䤵줿˸˱Ӥ
	 *	ȤˤƤ롣Τˡޤ Leaf ݤƤ롣
	 */
	pNewLeaf	= tBufString_newLeaf () ;
	if (pNewLeaf == NULL)
		return	False ;

	/*	ä Leaf ˥ƥȤư롣ưʤΤǡ
	 *	offset  length ⲿѲʤ*/
#if defined (DEBUG_LV99)
	fprintf (stderr, "Leaf = %d\n", pLeaf->m_nLength) ;
#endif
	memcpy (pNewLeaf->m_achText,
			pLeaf->m_achText + SIZE_TBUFBASESTRING / 2,
			sizeof (Char) * (pLeaf->m_nLength - SIZE_TBUFBASESTRING / 2)) ;

	pNewLeaf->m_nLength	= pLeaf->m_nLength - SIZE_TBUFBASESTRING / 2 ;
	pLeaf->m_nLength	= SIZE_TBUFBASESTRING / 2 ;

	if (nPosition >= SIZE_TBUFBASESTRING / 2) {
		pMarker->m_pLeaf		= pNewLeaf ;
		pMarker->m_nPosition	= pMarker->m_nPosition - SIZE_TBUFBASESTRING / 2 ;
	}

	if (TFAILED (tBufStringLeaf_insertNext (&pBufString->m_pRoot, pLeaf, pNewLeaf))) {
		FREE (pNewLeaf) ;
		return	False ;
	}
	/*	ƵȤäƺ١ʸ롣٤1ʸ٤Ǥ
	 *	ƥȤʬäʤȦǤ롣
	 *
	 *	ĽŤʽʤ
	 */
#if defined (DEBUG_LV99)
	assert (tBufString_checkRegion (pBufString->m_pRoot) == True) ;
#endif
	return	tBufStringMarker_insertChar (pMarker, cc) ;
}

Boolean
tBufStringMarker_deleteChar (
	register TBufStringMarker*	pMarker,
	register int				nCount)
{
	register TBufString*		pBufString ;
	register TBufStringLeaf*	pLeaf ;
	register int				nStart, nEnd, nNewCount ;
	register int				fQuit ;

	pLeaf		= pMarker->m_pLeaf ;
	if (pLeaf == NULL)
		return	False ;

	pBufString	= pMarker->m_pBufString ;
	assert (pBufString != NULL) ;

	nStart		= pMarker->m_nPosition ;
	nEnd		= pMarker->m_nPosition + nCount ;
	/*	õϰϤޤ硣
	 */
	if (nEnd <= pLeaf->m_nLength) {
		memmove (pLeaf->m_achText + nStart, pLeaf->m_achText + nEnd, (pLeaf->m_nLength - nEnd) * sizeof (Char)) ;
		pLeaf->m_nLength	-= nCount ;
		tBufString_updatePath (pLeaf->m_pParent, - nCount) ;
		if (pLeaf->m_nLength == nStart && pLeaf->m_pNext != NULL) {
			pMarker->m_pLeaf		= pLeaf->m_pNext ;
			pMarker->m_nPosition	= 0 ;
		}
		if (pLeaf->m_nLength > 0)
			return	True ;
		if (pLeaf->m_pPrev == NULL && pLeaf->m_pNext == NULL)
			return	True ;
		if (pMarker->m_pLeaf == pLeaf) {
			assert (pMarker->m_nPosition == 0) ;
			assert (pLeaf->m_pPrev != NULL) ;
			pMarker->m_pLeaf		= pLeaf->m_pPrev ;
			pMarker->m_nPosition	= pMarker->m_pLeaf->m_nLength ;
		}
		return	tBufStringLeaf_delete (pBufString, pLeaf) ;
	}

	nNewCount	= pLeaf->m_nLength - nStart ;
	/*	ĤʬǤʤ⤷ʤΥ꡼դ˷³꡼դʤ
	 *	ˤϡλưΤǡQuit ʤȤʤ*/
	fQuit		= (pLeaf->m_pNext == NULL) ;
	if (TFAILED (tBufStringMarker_deleteChar (pMarker, nNewCount)))
		return	False ;
	return	fQuit || tBufStringMarker_deleteChar (pMarker, nCount - nNewCount) ;
}

Boolean
tBufStringNode_rbDeleteFixup (
	register TBufStringNode**	ppRootNode,
	register TBufStringNode*	pNodeX)
{
	register TBufStringNode*	pNodePX ;
	register TBufStringNode*	pNodeW ;
	register TBufStringNode*	pNodeWL ;
	register TBufStringNode*	pNodeWR ;

	while (pNodePX = tBufStringNode_getParentNode (pNodeX), pNodePX != NULL &&
		   pNodeX->m_nColor == TBUFSTRINGNODECOLOR_BLACK) {
		if (pNodeX == tBufStringNode_getLeftNode (pNodePX)) {
			pNodeW	= tBufStringNode_getRightNode (pNodePX) ;
			if (tBufStringNode_getColor (pNodeW) == TBUFSTRINGNODECOLOR_RED) {
				tBufStringNode_setColor (pNodeW,  TBUFSTRINGNODECOLOR_BLACK) ;
				tBufStringNode_setColor (pNodePX, TBUFSTRINGNODECOLOR_RED) ;
				tBufStringNode_leftRotate (ppRootNode, pNodePX) ;
				pNodePX	= tBufStringNode_getParentNode (pNodeX) ;
				pNodeW	= tBufStringNode_getRightNode (pNodePX) ;
			}
			pNodeWL	= tBufStringNode_getLeftNode  (pNodeW) ;
			pNodeWR	= tBufStringNode_getRightNode (pNodeW) ;
			if (tBufStringNode_getColor (pNodeWL) == TBUFSTRINGNODECOLOR_BLACK &&
				tBufStringNode_getColor (pNodeWR) == TBUFSTRINGNODECOLOR_BLACK) {
				tBufStringNode_setColor (pNodeW, TBUFSTRINGNODECOLOR_RED) ;
				pNodeX	= tBufStringNode_getParentNode (pNodeX) ;
			} else {
				if (tBufStringNode_getColor (pNodeWR) == TBUFSTRINGNODECOLOR_BLACK) {
					tBufStringNode_setColor (pNodeWL, TBUFSTRINGNODECOLOR_BLACK) ;
					tBufStringNode_setColor (pNodeW,  TBUFSTRINGNODECOLOR_RED) ;
					tBufStringNode_rightRotate (ppRootNode, pNodeW) ;
					pNodePX	= tBufStringNode_getParentNode (pNodeX) ;
					pNodeW	= tBufStringNode_getRightNode (pNodePX) ;
					pNodeWR	= tBufStringNode_getRightNode (pNodeW) ;
				}
				tBufStringNode_setColor (pNodeW,  tBufStringNode_getColor (pNodePX)) ;
				tBufStringNode_setColor (pNodePX, TBUFSTRINGNODECOLOR_BLACK) ;
				tBufStringNode_setColor (pNodeWR, TBUFSTRINGNODECOLOR_BLACK) ;
				tBufStringNode_leftRotate (ppRootNode, pNodePX) ;
				pNodeX		= *ppRootNode ;
			}
		} else {
			pNodeW	= tBufStringNode_getLeftNode (pNodePX) ;
			if (tBufStringNode_getColor (pNodeW) == TBUFSTRINGNODECOLOR_RED) {
				tBufStringNode_setColor (pNodeW,  TBUFSTRINGNODECOLOR_BLACK) ;
				tBufStringNode_setColor (pNodePX, TBUFSTRINGNODECOLOR_RED) ;
				tBufStringNode_leftRotate (ppRootNode, pNodePX) ;
				pNodePX	= tBufStringNode_getParentNode (pNodeX) ;
				pNodeW	= tBufStringNode_getRightNode (pNodePX) ;
			}
			pNodeWL	= tBufStringNode_getLeftNode  (pNodeW) ;
			pNodeWR	= tBufStringNode_getRightNode (pNodeW) ;
			if (tBufStringNode_getColor (pNodeWL) == TBUFSTRINGNODECOLOR_BLACK &&
				tBufStringNode_getColor (pNodeWR) == TBUFSTRINGNODECOLOR_BLACK) {
				tBufStringNode_setColor (pNodeW, TBUFSTRINGNODECOLOR_RED) ;
				pNodeX	= tBufStringNode_getParentNode (pNodeX) ;
			} else {
				if (tBufStringNode_getColor (pNodeWR) == TBUFSTRINGNODECOLOR_BLACK) {
					tBufStringNode_setColor (pNodeWL, TBUFSTRINGNODECOLOR_BLACK) ;
					tBufStringNode_setColor (pNodeW,  TBUFSTRINGNODECOLOR_RED) ;
					tBufStringNode_rightRotate (ppRootNode, pNodeW) ;
					pNodePX	= tBufStringNode_getParentNode (pNodeX) ;
					pNodeW	= tBufStringNode_getRightNode (pNodePX) ;
					pNodeWR	= tBufStringNode_getRightNode (pNodeW) ;
				}
				tBufStringNode_setColor (pNodeW,  tBufStringNode_getColor (pNodePX)) ;
				tBufStringNode_setColor (pNodePX, TBUFSTRINGNODECOLOR_BLACK) ;
				tBufStringNode_setColor (pNodeWR, TBUFSTRINGNODECOLOR_BLACK) ;
				tBufStringNode_leftRotate (ppRootNode, pNodePX) ;
				pNodeX		= *ppRootNode ;
			}
		}
	}
	tBufStringNode_setColor (pNodeX, TBUFSTRINGNODECOLOR_BLACK) ;
	return	True ;
}

/*	õϻҶξȤ nil ˤʤäȸꤹ롣
 */
TBufStringNode*
tBufStringNode_delete (
	register TBufStringNode**	ppRootNode,
	register TBufStringNode*	pNodeZ)
{
	if (tBufStringNode_getRightNode (pNodeZ) != NULL || 
		tBufStringNode_getLeftNode (pNodeZ) != NULL)
		return	False ;

	assert (tBufStringNode_getParentNode (pNodeZ) != NULL) ;
	if (tBufStringNode_getColor (pNodeZ) == TBUFSTRINGNODECOLOR_BLACK) 
		tBufStringNode_rbDeleteFixup (ppRootNode, pNodeZ) ;
	return	pNodeZ ;
}

Boolean
tBufStringNode_deleteRecursively (
	register TBufStringNode**	ppRootNode,
	register TBufStringNode*	pNodeZ)
{
	register TBufStringNode*	pParentNode ;

	while (pNodeZ != NULL) {
		if (tBufStringNode_getColor (pNodeZ) == TBUFSTRINGNODECOLOR_BLACK) 
			tBufStringNode_rbDeleteFixup (ppRootNode, pNodeZ) ;

		if (((pNodeZ->Left.m_fLeaf && pNodeZ->Left.u.m_pLeaf != NULL) ||
			 (pNodeZ->Left.u.m_pNode != NULL)) ||
			((pNodeZ->Right.m_fLeaf && pNodeZ->Right.u.m_pLeaf != NULL) ||
			 (pNodeZ->Right.u.m_pNode != NULL)))
			break ;

		pParentNode	= tBufStringNode_getParentNode (pNodeZ) ;
		if (pParentNode == NULL)
			break ;
		
		if (tBufStringNode_getRightNode (pParentNode) == pNodeZ) {
			tBufStringNode_setRight (pParentNode, NULL, True) ;
		} else {
			assert (tBufStringNode_getLeftNode (pParentNode) == pNodeZ) ;
			tBufStringNode_setLeft  (pParentNode, NULL, True) ;
		}
		FREE (pNodeZ) ;
		pNodeZ	= pParentNode ;
	}
	return	True ;
}

Boolean
tBufStringLeaf_delete (
	register TBufString*		pBufString,
	register TBufStringLeaf*	pLeaf)
{
	register TBufStringLeaf*	pPrevLeaf ;
	register TBufStringLeaf*	pNextLeaf ;
	register TBufStringNode*	pNode ;

	pPrevLeaf	= pLeaf->m_pPrev ;
	pNextLeaf	= pLeaf->m_pNext ;
	if (pPrevLeaf != NULL) {
		pPrevLeaf->m_pNext	= pNextLeaf ;
	} else {
		assert (pNextLeaf != NULL) ;
		pBufString->m_pHead	= pNextLeaf ;
	}
	if (pNextLeaf != NULL)
		pNextLeaf->m_pPrev	= pPrevLeaf ;

	pNode		= pLeaf->m_pParent ;
	if (pNode->Right.m_fLeaf && pNode->Right.u.m_pLeaf == pLeaf) {
		tBufStringNode_setRight (pNode, NULL, True) ;
		if (pNode->Left.m_fLeaf && pNode->Left.u.m_pLeaf == NULL) {
			/*	ΥΡɤפˤʤä*/
			tBufStringNode_deleteRecursively (&pBufString->m_pRoot, pNode) ;
		}
	} else {
		assert (pNode->Left.m_fLeaf && pNode->Left.u.m_pLeaf == pLeaf) ;
		tBufStringNode_setLeft  (pNode, NULL, True) ;
		if (pNode->Right.m_fLeaf && pNode->Right.u.m_pLeaf == NULL) {
			/*	ΥΡɤפˤʤä*/
			tBufStringNode_deleteRecursively (&pBufString->m_pRoot, pNode) ;
		}
	}
	FREE (pLeaf) ;
	return	True ;
}

Boolean
tBufString_checkRegion (
	register TBufStringNode*	pNode)
{
	register int	nLengthL, nLengthR ;

	if (pNode == NULL)
		return	True ;

	if (pNode->Left.m_fLeaf) {
		if (pNode->Left.u.m_pLeaf != NULL) {
			nLengthL	= pNode->Left.u.m_pLeaf->m_nLength ;
		} else {
			nLengthL	= 0 ;
		}
	} else {
		nLengthL	= pNode->Left.u.m_pNode->m_nLength ;
	}
	if (pNode->Right.m_fLeaf) {
		if (pNode->Right.u.m_pLeaf != NULL) {
			nLengthR	= pNode->Right.u.m_pLeaf->m_nLength ;
		} else {
			nLengthR	= 0 ;
		}
	} else {
		if (pNode->Right.u.m_pNode != NULL) {
			nLengthR	= pNode->Right.u.m_pNode->m_nLength ;
		} else {
			nLengthR	= 0 ;
		}
	}
	if (pNode->m_nLength != (nLengthL + nLengthR)) {
		fprintf (stderr, "Node(%p): %d != %d + %d\n",
				 pNode, pNode->m_nLength, nLengthL, nLengthR) ;
		return	False ;
	}
	if (!tBufString_checkRegion (tBufStringNode_getLeftNode (pNode)) ||
		!tBufString_checkRegion (tBufStringNode_getRightNode (pNode)))
		return	False ;
	return	True ;
}

void
tBufString_destroyTree (
	register TBufStringNode*	pRoot)
{
	register TBufStringNode*	pLeftNode ;
	register TBufStringNode*	pRightNode ;

	pLeftNode	= tBufStringNode_getLeftNode (pRoot) ;
	pRightNode	= tBufStringNode_getRightNode (pRoot) ;
	if (pLeftNode != NULL)
		tBufString_destroyTree (pLeftNode) ;
	if (pRightNode != NULL)
		tBufString_destroyTree (pRightNode) ;
	FREE (pRoot) ;
	return ;
}

void
tBufStringNode_dump (
	register TBufStringNode*	pNode)
{
	fprintf (stderr, "TBufStringNode(%p):\n", pNode) ;
	fprintf (stderr, "  length(%d)\n", pNode->m_nLength) ;
	fprintf (stderr, "  Left(Leaf(%d), Pointer(%p))\n",
			 pNode->Right.m_fLeaf, pNode->Right.u.m_pNode) ;
	fprintf (stderr, "  Right(Leaf(%d), Pointer(%p))\n",
			 pNode->Right.m_fLeaf, pNode->Right.u.m_pNode) ;
	fprintf (stderr, "  Parent(%p)\n", pNode->m_pParent) ;
	fprintf (stderr, "  Color(%d)\n", pNode->m_nColor) ;
	return ;
}

void
tBufStringLeaf_dump (
	register TBufStringLeaf*	pLeaf)
{
	fprintf (stderr, "TBufStringLeaf(%p):\n", pLeaf) ;
	fprintf (stderr, "  length(%d)\n", pLeaf->m_nLength) ;
	fprintf (stderr, "  prev(%p)\n", pLeaf->m_pPrev) ;
	fprintf (stderr, "  next(%p)\n", pLeaf->m_pNext) ;
	fprintf (stderr, "  parent(%p)\n", pLeaf->m_pParent) ;
	return ;
}

void
tBufStringNode_walk (
	register TBufStringNode*	pNode,
	register int				nDepth,
	register int*				pnNode,
	register int*				pnDepth,
	register int*				pnLeaf)
{
	if (*pnDepth < nDepth)
		*pnDepth	= nDepth ;
	++ *pnNode ;

	if (!pNode->Left.m_fLeaf) {
		tBufStringNode_walk (pNode->Left.u.m_pNode, nDepth + 1, pnNode, pnDepth, pnLeaf) ;
	} else {
		++ *pnLeaf ;
	}
	if (!pNode->Right.m_fLeaf) {
		tBufStringNode_walk (pNode->Right.u.m_pNode, nDepth + 1, pnNode, pnDepth, pnLeaf) ;
	} else {
		++ *pnLeaf ;
	}
	return ;
}

void
tBufString_checkLog (
	register TBufString*		pBufString)
{
	int	nNode = 0, nDepth = 0, nLeaf = 0 ;

	if (pBufString->m_pRoot != NULL) {
		tBufStringNode_walk (pBufString->m_pRoot, 1, &nNode, &nDepth, &nLeaf) ;
	}
	fprintf (stderr, "BufString(%p): Depth(%d), Node(%d), Leaf(%d)\n",
			 pBufString, nDepth, nNode, nLeaf) ;
	return ;
}

