#include "local.h"
#include <stdio.h>
#include <assert.h>
#include "lmachinep.h"
#include "regex.h"

static	TLMRESULT	lispMachine_search (TLispMachine*, Boolean (*)(TLispMachine*, TLispEntity*, const Char*, int, int, int), int) ;

/*
 *	string-match is a built-in function.
 *	(string-match REGEXP STRING &optional START)
 *	
 *	Return index of start of first match for REGEXP in STRING, or nil.
 *	Case is ignored if `case-fold-search' is non-nil in the current buffer.
 *	If third arg START is non-nil, start search at that index in STRING.
 *	For index of first char beyond the match, do (match-end 0).
 *	`match-end' and `match-beginning' also give indices of substrings
 *	matched by parenthesis constructs in the pattern.
 */
TLMRESULT
lispMachineState_StringMatch (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntREGEXP ;
	TLispEntity*	pEntSTRING ;
	TLispEntity*	pEntSTART ;
	TLispEntity*	pEntRetval ;
	const Char*		pStrREGEXP ;
	int				nStrREGEXP ;
	const Char*		pStrSTRING ;
	int				nStrSTRING ;
	long			lStart ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntREGEXP)) ||
		TFAILED (lispEntity_Stringp (pLispMgr, pEntREGEXP)) ||
		TFAILED (lispEntity_GetCdr  (pLispMgr, pEntArglist, &pEntArglist)) ||
		TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntSTRING)) ||
		TFAILED (lispEntity_Stringp (pLispMgr, pEntSTRING)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntSTART))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispEntity_GetStringValue (pLispMgr, pEntREGEXP, &pStrREGEXP, &nStrREGEXP) ;
	lispEntity_GetStringValue (pLispMgr, pEntSTRING, &pStrSTRING, &nStrSTRING) ;
	if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntSTART, &lStart)))
		lStart	= 0 ;
	if (lStart > nStrSTRING) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	pStrSTRING	+= lStart ;
	nStrSTRING	-= lStart ;
	lispMachineCode_SetRegmatchTarget (pLM, pEntSTRING) ;
	if (TFAILED (lispMachineCode_StringMatch (pLM, pStrREGEXP, nStrREGEXP, pStrSTRING, nStrSTRING))) {
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	} else {
		int		nPosition ;
		lispMachineCode_MatchBeginning (pLM, 0, &nPosition) ;
		lispMgr_CreateInteger (pLispMgr, nPosition, &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*
 *	re-search-forward is an interactive built-in function.
 *	(re-search-forward REGEXP &optional BOUND NOERROR COUNT)
 *
 *	Search forward from point for regular expression REGEXP.
 *	Set point to the end of the occurrence found, and return point.
 *	An optional second argument bounds the search; it is a buffer position.
 *	The match found must not extend after that position.
 *	Optional third argument, if t, means if fail just return nil (no error).
 *	If not nil and not t, move to limit of search and return nil.
 *	Optional fourth argument is repeat count--search for successive occurrences.
 *	See also the functions `match-beginning', `match-end', `match-string',
 *	and `replace-match'.
 */
TLMRESULT
lispMachineState_ReSearchForward (
	register TLispMachine*	pLM)
{
	return	lispMachine_search (pLM, &lispMachineCode_ReSearchForward, 1) ;
}

TLMRESULT
lispMachineState_ReSearchBackward (
	register TLispMachine*	pLM)
{
	return	lispMachine_search (pLM, &lispMachineCode_ReSearchBackward, - 1) ;
}

/*
 *	match-beginning is a built-in function.
 *	(match-beginning SUBEXP)
 *
 *	Ǹ search ˤäƥޥåƥȤκǽΰ֤֤
 *	Return position of start of text matched by last search.
 *	SUBEXP, a number, specifies which parenthesized expression in the last
 *	regexp.
 *	Value is nil if SUBEXPth pair didn't match, or there were less than
 *	SUBEXP pairs.
 *	Zero means the entire text matched by the whole regexp or whole string.
 */
TLMRESULT
lispMachineState_MatchBeginning (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntSUBEXP ;
	TLispEntity*	pEntRetval ;
	long			lNumber ;
	int				nPosition ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntSUBEXP)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntSUBEXP, &lNumber))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispMachineCode_MatchBeginning (pLM, lNumber, &nPosition))) {
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	} else {
		lispMgr_CreateInteger (pLispMgr, nPosition, &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*
 *	match-end is a built-in function.
 *	(match-end SUBEXP)
 *
 *	Return position of end of text matched by last search.
 *	SUBEXP, a number, specifies which parenthesized expression in the last
 *	regexp.
 *	Value is nil if SUBEXPth pair didn't match, or there were less than
 *	SUBEXP pairs.
 *	Zero means the entire text matched by the whole regexp or whole string.
 */
TLMRESULT
lispMachineState_MatchEnd (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntSUBEXP ;
	TLispEntity*	pEntRetval ;
	long			lNumber ;
	int				nPosition ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntSUBEXP)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntSUBEXP, &lNumber))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispMachineCode_MatchEnd (pLM, lNumber, &nPosition))) {
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	} else {
		lispMgr_CreateInteger (pLispMgr, nPosition, &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*
 *	match-data is a built-in function.
 *	(match-data &optional INTEGERS REUSE)
 *
 *	Return a list containing all info on what the last search matched.
 *	Element 2N is `(match-beginning N)'; element 2N + 1 is `(match-end N)'.
 *	All the elements are markers or nil (nil if the Nth pair didn't match)
 *	if the last match was on a buffer; integers or nil if a string was matched.
 *	Use `store-match-data' to reinstate the data in this list.
 *
 *	If INTEGERS (the optional first argument) is non-nil, always use integers
 *	(rather than markers) to represent buffer positions.
 *	If REUSE is a list, reuse it as part of the value.  If REUSE is long enough
 *	to hold all the values, and if INTEGERS is non-nil, no consing is done.
 */
TLMRESULT
lispMachineState_MatchData (
	register TLispMachine*	pLM)
{
	TLispEntity*	pEntRetval ;
	/*	marker ˤΤ񤷤ʤäȥѥ뤫ʡ*/
	lispMachineCode_MatchData (pLM, &pEntRetval) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*	search-forward is an interactive built-in function.
 *	(search-forward STRING &optional BOUND NOERROR COUNT)
 *	
 *	Search forward from point for STRING.
 *	Set point to the end of the occurrence found, and return point.
 *	An optional second argument bounds the search; it is a buffer position.
 *	The match found must not extend after that position.  nil is equivalent
 *	to (point-max).
 *	Optional third argument, if t, means if fail just return nil (no error).
 *	If not nil and not t, move to limit of search and return nil.
 *	Optional fourth argument is repeat count--search for successive occurrences.
 *	
 *	Search case-sensitivity is determined by the value of the variable
 *	`case-fold-search', which see.
 *	
 *	See also the functions `match-beginning', `match-end' and `replace-match'.
 */
TLMRESULT
lispMachineState_SearchForward (
	register TLispMachine*	pLM)
{
	return	lispMachine_search (pLM, &lispMachineCode_SearchForward, 1) ;
}

/*
 */
TLMRESULT
lispMachineState_SearchBackward (
	register TLispMachine*	pLM)
{
	return	lispMachine_search (pLM, &lispMachineCode_SearchBackward, - 1) ;
}

/*	built-in function:
 *		(regexp-quote STRING)
 *
 *	STRING Τ˥ޥåƤ¾ˤϥޥåʤɽʸ֤
 *
 *	Ĥޤϡ[]  *, ., ? ʤɤδʤʸ quote ȤȤʤ
 *	 TCL/TK äơä regex library ʸäƲ
 *	äʡemacs Ȥ quote δطդä褦ʡ
 *
 *	ΤǡTCL/TK ǤΤѤƤط塢"[", "?" ʤɤɬä
 *	ȤȤɬסȡ
 */
TLMRESULT
lispMachineState_RegexpQuote (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr			= pLM->m_pLispMgr ;
	TVarbuffer				vbufSTRING ;
	static const char		rchQuoteChars []	= ".*+?^$()\\[]{}|" ;
	static const Char		chQuote				= '\\' ;
	register TLMRESULT		nResult				= LMR_ERROR ;
	TLispEntity*			pEntArglist ;
	TLispEntity*			pEntSTRING ;
	const Char*				pStrSTRING ;
	int						nStrSTRING ;
	register const Char*	pStrHEAD ;
	register const Char*	pStrResult ;
	register int			nStrResult ;
	TLispEntity*			pEntRetval ;

	assert (pLM != NULL) ;
	assert (pLispMgr != NULL) ;

	if (TFAILED (TVarbuffer_Initialize (&vbufSTRING, sizeof (Char))))
		return	LMR_ERROR ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntSTRING) ;
	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntSTRING, &pStrSTRING, &nStrSTRING))) {
		lispMachineCode_SetError (pLM) ;
		goto	error ;
	}
	/*	".*+?^$()\\[]{}|" ȡ quote 롣*/
	while (nStrSTRING > 0) {
		pStrHEAD	= pStrSTRING ;
		while (nStrSTRING > 0 &&
			   (!Char_IsAscii (*pStrSTRING) ||
				memchr (rchQuoteChars, (char)*pStrSTRING, NELEMENTS(rchQuoteChars)) == NULL)) {
			pStrSTRING	++ ;
			nStrSTRING	-- ;
		}
		if (pStrHEAD < pStrSTRING)
			if (TFAILED (TVarbuffer_Add (&vbufSTRING, pStrHEAD, pStrSTRING - pStrHEAD)))
				goto	error ;
		if (nStrSTRING > 0) {
			if (TFAILED (TVarbuffer_Add (&vbufSTRING, &chQuote, 1)) ||
				TFAILED (TVarbuffer_Add (&vbufSTRING, pStrSTRING, 1)))
				goto	error ;
			pStrSTRING	++ ;
			nStrSTRING	-- ;
		}
	}
	pStrResult	= TVarbuffer_GetBuffer (&vbufSTRING) ;
	nStrResult	= TVarbuffer_GetUsage  (&vbufSTRING) ;
	if (TFAILED (lispMgr_CreateString (pLispMgr, pStrResult, nStrResult, &pEntRetval)))
		goto	error ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	nResult	= LMR_RETURN ;
  error:
	TVarbuffer_Uninitialize (&vbufSTRING) ;
	return	nResult ;
}


TLMRESULT
lispMachine_search (
	register TLispMachine*	pLM,
	register Boolean		(*pCmd)(TLispMachine*, TLispEntity*, const Char*, int, int, int),
	register int			nDir)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntBuffer ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntSTRING ;
	TLispEntity*	pEntBOUND ;
	TLispEntity*	pEntNOERROR ;
	TLispEntity*	pEntCOUNT ;
	const Char*		pStrSTRING ;
	int				nStrSTRING ;
	int				nCOUNT, nBOUND ;
	TLispEntity*	pEntRetval ;

	assert (pLM  != NULL) ;
	assert (pCmd != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;

	lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntSTRING) ;
	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntSTRING, &pStrSTRING, &nStrSTRING))) 
		goto	error ;
	lispEntity_GetCdr  (pLispMgr, pEntArglist, &pEntArglist) ;

	/*	default  bound  point-max/min Ǥ뤬̤˻ꤵƤϤ
	 *	¤ǤϤʤ*/
	lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntBOUND) ;
	if (nDir > 0) {
		lispBuffer_PointMax (pLispMgr, pEntBuffer, &nBOUND) ;
	} else {
		lispBuffer_PointMin (pLispMgr, pEntBuffer, &nBOUND) ;
	}
	if (TFAILED (lispEntity_Nullp (pLispMgr, pEntBOUND))) {
		long	lBOUND ;
		if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntBOUND, &lBOUND)))
			goto	error ;
		nBOUND	= lBOUND ;
	}
	lispEntity_GetCdr  (pLispMgr, pEntArglist, &pEntArglist) ;
	lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntNOERROR) ;

	/*	η֤ default ǤϰǤ롣*/
	nCOUNT	= 1 ;
	lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntCOUNT) ;
	if (TFAILED (lispEntity_Nullp (pLispMgr, pEntCOUNT))) {
		long	lCOUNT ;
		if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntCOUNT, &lCOUNT)))
			goto	error ;
		nCOUNT	= lCOUNT ;
	}
	if (TFAILED ((*pCmd) (pLM, pEntBuffer, pStrSTRING, nStrSTRING, nBOUND, nCOUNT))) {
		if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntNOERROR)))
			goto	error ;
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
		if (TFAILED (lispEntity_Tp (pLispMgr, pEntNOERROR))) {
			/*	ɤ٤ʤΤ ɬפϤΤ */
		}
	} else {
		TLispEntity*	pEntPoint ;
		int				nPosition ;

		(void) lispMachineCode_MatchBeginning (pLM, 0, &nPosition) ;
#if defined (DEBUG)
		fprintf (stderr, "search-%s: result = %d\n",
				 (nDir > 0)? "forward" : "backward", nPosition) ;
#endif
		if (TFAILED (lispMgr_CreateInteger (pLispMgr, nPosition, &pEntRetval)))
			return	LMR_ERROR ;

		/*	backward ξˤϡposition  beginning ΰ֤ˤʤ롣
		 *	forward  ξˤϡposition  end       ΰ֤ˤʤ롣
		 */
		if (nDir > 0)
			(void) lispMachineCode_MatchEnd       (pLM, 0, &nPosition) ;
		lispBuffer_PointMarker (pLispMgr, pEntBuffer, &pEntPoint) ;
		lispMarker_SetBufferPosition (pLispMgr, pEntPoint, pEntBuffer, nPosition) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;

  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

