#include "local.h"
#include <stdio.h>
#include <assert.h>
#if !defined (__linux__)
#include <sys/syslimits.h>
#endif
#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <unistd.h>
#include <pwd.h>
#include "lmachinep.h"
#include "lmstate.h"
#include "filename.h"
#include "kfile.h"
#include "kanji.h"

static	TLMRESULT	lispMachineState_loadStep1		(TLispMachine*) ;
static	TLMRESULT	lispMachineState_loadFinalize	(TLispMachine*) ;
static	Boolean		lispMachine_getFileNameFromEntity	(TLispManager*, TLispEntity*, char*, int) ;

/*
 *	(load-file FILE)
 *
 *	load-file  interactive  lisp function Ǥꡢ`file' 
 *	Ƥ롣
 *	ΤȤinteractive ʬȴƤ롣 buffer 
 *	Ѱդɬפ롣¸Τˤ load-file Τ
 *	ʤΤǡ
 */
TLMRESULT
lispMachineState_LoadFile (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pFile ;
	TLispEntity*	pRetval ;
	const Char*		pFileName ;
	int				nFileNameLen ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pFile)) ||
		TFAILED (lispEntity_Stringp (pLispMgr, pFile))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispEntity_GetStringValue (pLispMgr, pFile, &pFileName, &nFileNameLen) ;
	if (pFileName == NULL || nFileNameLen <= 0) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispMgr_Load (pLispMgr, pFileName, nFileNameLen, &pRetval))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	lispEntity_Release (pLispMgr, pRetval) ;
	/*	ʲ progn ȤưƤ餦*/
	lispMachineCode_SetState (pLM, &lispMachineState_Progn) ;
	return	LMR_CONTINUE ;
}

/*
  load is a built-in function.
  (load FILE &optional NOERROR NOMESSAGE NOSUFFIX MUST-SUFFIX)

  NOSUFFIX  MUST-SUFFIX ưʤäƤʤΤǡ顢
  default  MUST-SUFFIX ˤʤäƤ롣

  Execute a file of Lisp code named FILE.
  First try FILE with `.elc' appended, then try with `.el',
  then try FILE unmodified.  Environment variable references in FILE
  are replaced with their values by calling `substitute-in-file-name'.
  This function searches the directories in `load-path'.
  If optional second arg NOERROR is non-nil,
  report no error if FILE doesn't exist.
  Print messages at start and end of loading unless
  optional third arg NOMESSAGE is non-nil.
  If optional fourth arg NOSUFFIX is non-nil, don't try adding
  suffixes `.elc' or `.el' to the specified name FILE.
  If optional fifth arg MUST-SUFFIX is non-nil, insist on
  the suffix `.elc' or `.el'; don't accept just FILE unless
  it ends in one of those suffixes or includes a directory name.
  Return t if file exists. */
TLMRESULT
lispMachineState_Load (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntArgtop ;
	TLispEntity*	pEntFile ;
	TLispEntity*	pEntNoMsg ;
	TLispEntity*	pEntNoError ;

	assert (pLM != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	pEntArgtop	= pEntArglist ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntFile)) ||
		TFAILED (lispEntity_Stringp (pLispMgr, pEntFile))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntNoError) ;
	lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntNoMsg) ;

	if (TFAILED (lispEntity_Nullp (pLispMgr, pEntNoMsg))) {
		static const char	strFORMAT []	= "Loading ... %q" ;
		TLispEntity*		pEntMsg ;
		lispEntity_FormatA (pLispMgr, strFORMAT, NELEMENTS (strFORMAT) - 1, pEntArgtop, &pEntMsg) ;
		lispMachineCode_SetMinibufferMessage (pLM, pEntMsg) ;
	}
	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_2) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_3) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_4) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_1, pEntFile) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_2, pEntNoError) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_3, pEntNoMsg) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_4, pEntArglist) ;
	lispMachineCode_SetState (pLM, lispMachineState_loadStep1) ;
	return	LMR_CONTINUE ;
}

/*
 *	(expand-file-name NAME &optional DEFAULT-DIRECTORY)
 *
 *	ե̾ NAME ФİդΤѴ롣2
 *	DEFAULT-DIRECTORY Ϥ⤷ NAME Фʤ餽ǤϤޤǥ
 *	ȥǤ롣⤷ DEFAULT-DIRECTORY  nil ޤϷ礱Ƥ
 *	硢current buffer  default-directory ͤȤ롣
 *	ե̾ `.' ޤǤˤϺ롣ޤ`..'
 *	³ե̾Ǥϡ롣
 *	̤Υե̾ե륷ƥˤ뤫ɤΥå
 *	Ϥʤǽ餬 `~/' ǤϤޤˤϡhome directory 
 *	`~USER/' ξˤ USER  home directory Ÿ롣
 */
TLMRESULT
lispMachineState_ExpandFileName (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TVarbuffer		vbuf ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntName ;
	TLispEntity*	pEntDefaultDirectory ;
	TLispEntity*	pEntRetval ;
	const Char*		pName ;
	int				nName ;
	const Char*		pDefault ;
	int				nDefault ;

	assert (pLM != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntName)) ||
		TFAILED (lispEntity_Stringp (pLispMgr, pEntName))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntDefaultDirectory) ;
	if (TFAILED (lispEntity_Stringp (pLispMgr, pEntDefaultDirectory)) &&
		TFAILED (lispEntity_Nullp   (pLispMgr, pEntDefaultDirectory))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (TVarbuffer_Initialize (&vbuf, sizeof (Char)))) 
		return	LMR_ERROR ;
	lispEntity_GetStringValue (pLispMgr, pEntName, &pName, &nName) ;
	if (nName <= 0) 
		goto	expand_end ;
	if (TSUCCEEDED (lispEntity_GetStringValue (pLispMgr, pEntDefaultDirectory, &pDefault, &nDefault))) {
	} else {
		pDefault	= NULL ;
		nDefault	= 0 ;
	}
	if (TFAILED (ExpandFileName (&vbuf, pName, nName, pDefault, nDefault))) {
		TVarbuffer_Uninitialize (&vbuf) ;
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
  expand_end:
	if (TFAILED (lispMgr_CreateString (pLispMgr, TVarbuffer_GetBuffer (&vbuf), TVarbuffer_GetUsage (&vbuf), &pEntRetval))) {
		TVarbuffer_Uninitialize (&vbuf) ;
		return	LMR_ERROR ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	TVarbuffer_Uninitialize (&vbuf) ;
	return	LMR_RETURN ;
}


/*	private functions */
TLMRESULT
lispMachineState_loadStep1 (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*			pEntFile ;
	const Char*				pFileName ;
	int						nFileNameLen ;
	register TLispEntity*	pEntLoadPath ;
	TLispEntity*			pValDirList ;
	TLispEntity*			pEntRetval ;
	register Boolean		fResult ;
	static const Char		chDelimitar	= '/' ;
	TVarbuffer				vbufFileName ;

	lispMachineCode_GetLReg (pLM, LM_LREG_1, &pEntFile) ;
	lispEntity_GetStringValue (pLispMgr, pEntFile, &pFileName, &nFileNameLen) ;
	if (pFileName == NULL || nFileNameLen <= 0) {
		lispMachineCode_SetError (pLM) ;
		lispMachineCode_SetState (pLM, lispMachineState_loadFinalize) ;
		return	LMR_CONTINUE ;
	}

	/*	load-path ֤˸롣*/
	pEntLoadPath	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_LOAD_PATH) ;
	if (TFAILED (lispMachine_GetCurrentSymbolValue (pLM, pEntLoadPath, &pValDirList)))
		lispMgr_CreateNil (pLispMgr, &pValDirList) ;

	if (TFAILED (TVarbuffer_Initialize (&vbufFileName, sizeof (Char))))
		return	LMR_ERROR ;

	fResult	= False ;
	while (!fResult && TFAILED (lispEntity_Nullp (pLispMgr, pValDirList))) {
		TLispEntity*	pValDir ;
		TLispEntity*	pEntCdr ;
		const Char*		strDir ;
		int				nstrDir ;
		const Char*		strPath ;
		int				nstrPath ;
		
		if (TFAILED (lispEntity_GetCar (pLispMgr, pValDirList, &pValDir)) ||
			TFAILED (lispEntity_GetCdr (pLispMgr, pValDirList, &pEntCdr))  ||
			TFAILED (lispEntity_GetStringValue (pLispMgr, pValDir, &strDir, &nstrDir)))
			break ;
		if (TFAILED (TVarbuffer_Add (&vbufFileName, strDir, nstrDir)) ||
			TFAILED (TVarbuffer_Add (&vbufFileName, &chDelimitar, 1)) ||
			TFAILED (TVarbuffer_Add (&vbufFileName, pFileName, nFileNameLen))) 
			return	LMR_ERROR ;
		
		strPath		= TVarbuffer_GetBuffer (&vbufFileName) ;
		nstrPath	= TVarbuffer_GetUsage  (&vbufFileName) ;
		fResult		= lispMgr_Load (pLispMgr, strPath, nstrPath, &pEntRetval) ;
		TVarbuffer_Clear (&vbufFileName) ;
		pValDirList	= pEntCdr ;
	}
	TVarbuffer_Uninitialize (&vbufFileName) ;

	if (TFAILED (fResult)) 
		fResult		= lispMgr_Load (pLispMgr, pFileName, nFileNameLen, &pEntRetval) ;

	if (TFAILED (fResult)) {
		TLispEntity*	pEntNoError ;
		/*	noerror  t ʤե뤬̵ä˥顼ˤʤʤ*/
		lispMachineCode_GetLReg (pLM, LM_LREG_2, &pEntNoError) ;
		if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntNoError))) 
			lispMachineCode_SetError (pLM) ;
		lispMachineCode_SetState (pLM, lispMachineState_loadFinalize) ;
		return	LMR_CONTINUE ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	lispEntity_Release (pLispMgr, pEntRetval) ;
	lispMachineCode_PushState (pLM, lispMachineState_loadFinalize) ;
	lispMachineCode_SetState  (pLM, lispMachineState_Progn) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_loadFinalize (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	static const char	strFORMAT []	= "Loading ... %q ... done" ;	
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntNoMessage ;
	TLispEntity*	pEntMsg ;

	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) 
		goto	exit_finalize ;

	lispMachineCode_GetLReg (pLM, LM_LREG_3, &pEntNoMessage) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntNoMessage))) 
		goto	exit_finalize ;

	lispMachineCode_GetLReg (pLM, LM_LREG_4, &pEntArglist) ;
	lispEntity_FormatA (pLispMgr, strFORMAT, NELEMENTS (strFORMAT) - 1, pEntArglist, &pEntMsg) ;
	lispMachineCode_SetMinibufferMessage (pLM, pEntMsg) ;

exit_finalize:
	lispMachineCode_PopLReg (pLM, LM_LREG_4) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_3) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_2) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
	return	LMR_RETURN ;
}

/*	built-in function:
 *		(insert-file-contents FILENAME &optional VISIT BEG END REPLACE)
 *
 *	ǤϤ뤬FILENAME  VISIT ʳĤϺΤȤʤ
 *	Ť emacs Ǥ FILENAME  VISIT ¾Ȥ뤳ȤϤǤʤä褦
 *
 *	VISIT  non-nil ξˤϡɤ߹ FILE  VISIT ե̾
 *	ơTIMESTAMP 򹹿ơmodified  nil ˤ롣ե뤬
 *	¸ߤʤǤ visiting ϴλ롣
 */
TLMRESULT
lispMachineState_InsertFileContents (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntFileName ;
	TLispEntity*	pEntBuffer ;
	TLispEntity*	apEntity [2] ;
	TLispEntity*	pEntRetval ;
	const Char*		pStrFileName ;
	int				nStrFileName ;
	char			rchPath [PATH_MAX + 1] ;
	KFILE			kfile ;
	int				nReadCount ;
	int				nPoint ;

	assert (pLM != NULL) ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFileName) ;
	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntFileName, &pStrFileName, &nStrFileName))) 
		goto	error ;
	if (TFAILED (ConvertInternalFileName2SystemFileName (rchPath, PATH_MAX + 1, pStrFileName, nStrFileName))) 
		goto	error ;

	lispBuffer_Point (pLispMgr, pEntBuffer, &nPoint) ;
	nReadCount	= 0 ;

	if (!KFile_Open (&kfile, rchPath, KCODING_SYSTEM_UNKNOWN)) {
#if defined (DEBUG) || 1
		fprintf (stderr, "Can't open file: \"%s\"\n", rchPath) ;
#endif
		goto	result ;
	}
	KFile_Rewind (&kfile) ;
	lispBuffer_InsertFileContents (pLispMgr, pEntBuffer, nPoint, &kfile, &nReadCount) ;
	KFile_Close (&kfile) ;

  result:
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, nReadCount, &apEntity [1])))
		return	LMR_ERROR ;
	apEntity [0]	= pEntFileName ;
	if (TFAILED (lispMgr_CreateList (pLispMgr, apEntity, 2, &pEntRetval)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;

  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*	built-in function:
 *		(file-name-nondirectory FILENAME)
 *
 *	̾ FILENAME Υǥ쥯ȥȴ֤ͤñ˺Ǹ slash θ
 *	³ʸ֤ɤ褦
 */
TLMRESULT
lispMachineState_FileNameNondirectory (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntFileName ;
	const Char*		pStrName ;
	int				nStrName ;
	const Char*		pStrResult ;
	int				nStrResult ;
	TLispEntity*	pEntRetval ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFileName) ;
	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntFileName, &pStrName, &nStrName))) 
		goto	error ;
	if (TFAILED (GetFileNameNondirectory (pStrName, nStrName, &pStrResult, &nStrResult)))
		goto	error ;
	if (TFAILED (lispMgr_CreateString (pLispMgr, pStrResult, nStrResult, &pEntRetval)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
	
  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*	(rename-file FILE NEWNAME &optional OK-IF-ALREADY-EXISTS)
 */
TLMRESULT
lispMachineState_RenameFile (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntFile ;
	TLispEntity*	pEntNewName ;
	TLispEntity*	pEntOkIfAExists ;
	TLispEntity*	pEntNil ;
	char			bufOldPath [PATH_MAX + 1] ;
	char			bufNewPath [PATH_MAX + 1] ;
	char			bufCopy [4096] ;
	struct stat		bufstat ;
	register int	nRetval ;
	register FILE*	fpSrc ;
	register FILE*	fpDest ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFile)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntNewName)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntOkIfAExists))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	/*	ѹȤ̾롣*/
	if (TFAILED (lispMachine_getFileNameFromEntity (pLispMgr, pEntFile, bufOldPath, PATH_MAX))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	/*	ѹ̾롣*/
	if (TFAILED (lispMachine_getFileNameFromEntity (pLispMgr, pEntNewName, bufNewPath, PATH_MAX))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}

	nRetval	= lstat (bufNewPath, &bufstat) ;
	if ((nRetval == -1 && errno != ENOENT) ||
		(nRetval == 0  && S_ISREG (bufstat.st_mode) == 0)) {
		/*	ե뤬ʤȤǤϤʤ*/
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	} else if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntOkIfAExists))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}

	/*	rename  copy ʤΤ */
	fpSrc	= fopen (bufOldPath, "rb") ;
	if (fpSrc == NULL) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	fpDest	= fopen (bufNewPath, "wb") ;
	if (fpDest == NULL) {
		lispMachineCode_SetError (pLM) ;
		fclose (fpSrc) ;
		return	LMR_RETURN ;
	}
	while (!feof (fpSrc)) {
		register int nread	= fread (bufCopy, 1, sizeof (bufCopy), fpSrc) ;
		if (nread <= 0)
			break ;
		if (fwrite (bufCopy, 1, nread, fpDest) <= 0)
			break ;
	}
	fclose (fpSrc) ;
	fclose (fpDest) ;

	lispMgr_CreateNil (pLispMgr, &pEntNil) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntNil) ;
	return	LMR_RETURN ;
}

/*	(delete-file)
 */
TLMRESULT
lispMachineState_DeleteFile (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntFile ;
	char			bufFile [PATH_MAX + 1] ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFile))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	/*	ѹȤ̾롣*/
	if (TFAILED (lispMachine_getFileNameFromEntity (pLispMgr, pEntFile, bufFile, PATH_MAX))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}

	if (unlink (bufFile) != 0) {
		lispMachineCode_SetError (pLM) ;
	} else {
		TLispEntity*	pEntNil ;
		lispMgr_CreateNil (pLispMgr, &pEntNil) ;
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntNil) ;
	}
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_FileModes (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntFile ;
	TLispEntity*	pEntMode ;
	char			bufFile [PATH_MAX + 1] ;
	struct stat		bufstat ;
	register int	nMode ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFile))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	/*	ѹȤ̾롣*/
	if (TFAILED (lispMachine_getFileNameFromEntity (pLispMgr, pEntFile, bufFile, PATH_MAX))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	
	if (lstat (bufFile, &bufstat) == -1) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	nMode	= bufstat.st_mode & DEFFILEMODE ;
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, nMode, &pEntMode)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntMode) ;
	return	LMR_RETURN ;
}

/*	(write-region START END FILENAME &optional APPEND VISIT LOCKNAME MUSTBENEW)
 *	
 *	VISIT ʹߤΥݡȤϤʤVISIT  nil  non-nil ʤ
 */
TLMRESULT
lispMachineState_WriteRegion (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntBuffer ;
	TLispEntity*	pEntStart ;
	TLispEntity*	pEntEnd ;
	TLispEntity*	pEntFileName ;
	TLispEntity*	pEntAppend ;
	TLispEntity*	pEntVisit ;
	TLispEntity*	pEntNil ;
	const Char*		strFile ;
	int				nstrFile ;
	char			bufFile [PATH_MAX + 1] ;
	char			buffer [2064] ;
	register char*	ptr ;
	TLispNumber		numStart, numEnd ;
	register int	nUsage, nptr, nctext, nCopy, nStart, nEnd ;
	int				nLength, nPointTop, nPointEnd ;
	register FILE*	fp ;
	register Char	cc ;
	KANJISTATEMACHINE	ksm ;
	TBufStringMarker	mk ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;

	/*	Ф
	 */
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntStart)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntEnd)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFileName)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntAppend)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntVisit))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}

	/*	ηɤå롣
	 */
	if (TFAILED (lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pEntStart, &numStart)) ||
		numStart.m_fFloatp ||
		TFAILED (lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pEntEnd,   &numEnd)) ||
		numEnd.m_fFloatp ||
		numStart.m_Value.m_lLong > numEnd.m_Value.m_lLong ||
		TFAILED (lispEntity_GetStringValue (pLispMgr, pEntFileName, &strFile, &nstrFile))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	nStart	= numStart.m_Value.m_lLong ;
	nEnd	= numEnd.m_Value.m_lLong ;

	/*	ХåեؤƤ뤫ɤå롣
	 */
	lispBuffer_GetString (pLispMgr, pEntBuffer, &mk, &nLength) ;
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointTop) ;
	lispBuffer_PointBufferEnd (pLispMgr, pEntBuffer, &nPointEnd) ;
#if defined (DEBUG)
	fprintf (stdout, "write-region: (%d/%d, %d/%d)\n", nStart, nPointTop, nEnd, nPointEnd) ; 
#endif
	if (nEnd < nPointTop || nStart > nPointEnd) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	TBufStringMarker_Forward (&mk, nStart - nPointTop) ;
	
	/*	ե򳫤
	 */
	nCopy	= (nstrFile < PATH_MAX)? nstrFile : PATH_MAX ;
	cstrtostr (bufFile, strFile, nCopy) ;
	bufFile [nCopy]	= '\0' ;
	if (TFAILED (lispEntity_Nullp (pLispMgr, pEntAppend))) {
		fp	= fopen (bufFile, "ab") ;
	} else {
		fp	= fopen (bufFile, "wb") ;
	}
	/*	no such file or directory 顼֤롣*/
	if (fp == NULL) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}

	InitializeKanjiFiniteStateMachine (&ksm, KCODING_SYSTEM_ISO2022JP2) ;

	ptr		= buffer ;
	nptr	= 0 ;
	nUsage	= nEnd - nStart ;
	/*	ࡢcoding-system ɤ褦 */
	while (nUsage -- > 0) {
		cc		= TBufStringMarker_GetChar (&mk) ;
		TBufStringMarker_Forward (&mk, 1) ;
		nctext	= RtransferKanjiFiniteStateMachine (&ksm, cc, ptr) ;
		ptr		+= nctext ;
		nptr	+= nctext ;
		if (nptr < (sizeof (buffer) - 16)) {
			fwrite (buffer, 1, nptr, fp) ;
#if defined (DEBUG)
			fwrite (buffer, 1, nptr, stdout) ;
#endif
			ptr		= buffer ;
			nptr	= 0 ;
		}
	}
	if (nptr > 0) {
		fwrite (buffer, 1, nptr, fp) ;
#if defined (DEBUG)
		fwrite (buffer, 1, nptr, stdout) ;
#endif
	}

	fclose (fp) ;

	lispMgr_CreateNil (pLispMgr, &pEntNil) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntNil) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_FileExistsp (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntFile ;
	TLispEntity*	pEntRetval ;
	char			bufFile [PATH_MAX + 1] ;
	struct stat		bufstat ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFile))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	/*	ѹȤ̾롣*/
	if (TFAILED (lispMachine_getFileNameFromEntity (pLispMgr, pEntFile, bufFile, PATH_MAX))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	
	if (lstat (bufFile, &bufstat) == -1) {
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	} else {
		lispMgr_CreateT   (pLispMgr, &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*	(set-file-modes FILENAME MODE)
 */
TLMRESULT
lispMachineState_SetFileModes (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntFILENAME ;
	TLispEntity*	pEntMODE ;
	TLispEntity*	pEntRetval ;
	char			bufFile [PATH_MAX + 1] ;
	long			lMode ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntFILENAME)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntMODE))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispMachine_getFileNameFromEntity (pLispMgr, pEntFILENAME, bufFile, PATH_MAX)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntMODE, &lMode))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	
	/*	եΥ⡼ɤѹ롣*/
	if (chmod (bufFile, lMode & 0x0FFF) != 0) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

Boolean
lispMachine_getFileNameFromEntity (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register char*			pDest,
	register int			nDest)
{
	TVarbuffer		vbufPath ;
	const Char*		strFile ;
	int				nstrFile ;
	register int	nCopy ;

	/*	ѹȤ̾롣*/
	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntity, &strFile, &nstrFile))) 
		return	False ;

	if (TFAILED (TVarbuffer_Initialize (&vbufPath, sizeof (Char))))
		return	False ;
	if (TSUCCEEDED (ExpandFileName (&vbufPath, strFile, nstrFile, NULL, 0))) {
		nCopy	= TVarbuffer_GetUsage (&vbufPath) ;
		if (nCopy >= nDest)
			nCopy	= nDest - 1 ;
		assert (nCopy >= 0) ;
		cstrtostr (pDest, TVarbuffer_GetBuffer (&vbufPath), nCopy) ;
		*(pDest + nCopy)	= '\0' ;
	}
	TVarbuffer_Uninitialize (&vbufPath) ;
	return	True ;
}

