#include "sysdep.h"
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "Char.h"
#include "kanji.h"

/*========================================================================
 *	
 *========================================================================*/
enum {
	KCHARSET_JISX0201_1976_KATAKANA		= (MAX_CHARSET + 1),

	KCODING_SYSTEM_CHECK_FALSE			= (0),
	KCODING_SYSTEM_CHECK_TRUE			= (1),
	KCODING_SYSTEM_CHECK_TRUE_EXCLUSIVE	= (2),
} ;


/*========================================================================*
 *	
 *========================================================================*/
typedef struct {
	int		m_nLineFeed ;		/* 0x0A */
	int		m_nCarriageReturn ;	/* 0x0D */
	int		m_nCRLF ;
	BOOL	m_fPostCR ;
}	NEWLINEINFO ;


/*========================================================================*
 *	ץȥ
 *========================================================================*/
static	int		checkShiftJisCodingSystem		(char* pchBuffer, int* piUsage, int iChara) ;
static	int		checkEucJpCodingSystem			(char* pchBuffer, int* piUsage, int iChara) ;
static	int		checkIso2022Jp2CodingSystem		(char* pchBuffer, int* piUsage, int iChara) ;
static	BOOL	checkNewline					(Char cc, NEWLINEINFO* pNewlineInfo) ;
static	int		restartEucJpStateMachine		(PKANJISTATEMACHINE pKSM) ;
static	int		transferEucJpStateMachine		(PKANJISTATEMACHINE pKSM, int iChara, Char* pOutput) ;
static	int		rtransferEucJpStateMachine		(PKANJISTATEMACHINE pKSM, Char cc, char* pOutput) ;
static	int		restartIso2022JpStateMachine	(PKANJISTATEMACHINE pKSM) ;
static	int		transferIso2022JpStateMachine	(PKANJISTATEMACHINE pKSM, int iChara, Char* pOutput) ;
static	int		iso2022jpEscapeSequenceState0	(PISO2022STATE pState, int iChara) ;
static	int		iso2022jpEscapeSequenceState1	(PISO2022STATE pState, int iChara) ;
static	int		iso2022jpEscapeSequenceState2	(PISO2022STATE pState, int iChara) ;
static	int		iso2022jpEscapeSequenceState3	(PISO2022STATE pState, int iChara) ;
static	int		iso2022jpEscapeSequenceState4	(PISO2022STATE pState, int iChara) ;
static	int		iso2022jpEscapeSequenceState5 	(PISO2022STATE pState, int iChara) ;
static	int		iso2022jpEscapeSequenceDefault	(PISO2022STATE pState, int iChara) ;
static	int		rtransferIso2022JpStateMachine	(PKANJISTATEMACHINE pKSM, Char cc, char* pOutput) ;
static	int		restartShiftJisStateMachine		(PKANJISTATEMACHINE pKSM) ;
static	int		transferShiftJisStateMachine	(PKANJISTATEMACHINE pKSM, int iChara, Char* pOutput) ;
static	int		rtransferShiftJisStateMachine	(PKANJISTATEMACHINE pKSM, Char cc, char* pOutput) ;


/*========================================================================
 *	Хؿ
 *========================================================================*/
/*
 *	ɲϾֵνԤ
 *[]
 *	pKSM			ܵ
 *	iCodingSystem	ǥ󥰼ˡ(EUC/SHIFTJIS/...)
 */
int
InitializeKanjiFiniteStateMachine (PKANJISTATEMACHINE pKSM, int iCodingSystem)
{
	static	int		(*pRestartFuncTbl [])(PKANJISTATEMACHINE pKSM)	= {
		restartEucJpStateMachine,
		restartIso2022JpStateMachine,
		restartShiftJisStateMachine,
	} ;
	static	int		(*pTransferFuncTbl [])(PKANJISTATEMACHINE pKSM, int iChara, Char* pOutput)	= {
		transferEucJpStateMachine,
		transferIso2022JpStateMachine,
		transferShiftJisStateMachine,
	} ;
	static	int		(*pRtransferFuncTbl [])(PKANJISTATEMACHINE pKSM, Char cc, char* pOutput)	= {
		rtransferEucJpStateMachine,
		rtransferIso2022JpStateMachine,
		rtransferShiftJisStateMachine,
	} ;

	assert (pKSM != NULL) ;
	assert (KCODING_SYSTEM_EUCJP <= iCodingSystem && iCodingSystem < MAX_KCODING_SYSTEM) ;

	pKSM->m_pRestartFunc	= pRestartFuncTbl   [iCodingSystem] ;
	pKSM->m_pTransferFunc	= pTransferFuncTbl  [iCodingSystem] ;
	pKSM->m_pRtransferFunc	= pRtransferFuncTbl [iCodingSystem] ;
	return	(pKSM->m_pRestartFunc)(pKSM) ;
}

/*
 *	ɾ֤᤹ܵ/ꤹ롣
 *[]
 *	pKSM		ܵ
 */
int
RestartKanjiFiniteStateMachine (PKANJISTATEMACHINE pKSM)
{
	assert (pKSM != NULL) ;
	assert (pKSM->m_pRestartFunc != NULL) ;
	return	(pKSM->m_pRestartFunc)(pKSM) ;
}

/*
 *	ɾܵ1ʸϤͿܤ롣
 *[]
 *	pKSM		ܵ
 *	iChara		ϥܥ
 *	pOutput		ϥܥ
 *[֤]
 *	FALSE		ϥܥ¸ߤʤ
 *	TRUE		ϥܥ뤬¸ߤ롣
 *	-1			顼ȯ
 */
int
TransferKanjiFiniteStateMachine (PKANJISTATEMACHINE pKSM, int iChara, Char* pOutput)
{
	assert (pKSM != NULL) ;
	assert (pKSM->m_pTransferFunc != NULL) ;
	assert (pOutput != NULL) ;
	return	(pKSM->m_pTransferFunc)(pKSM, iChara, pOutput) ;
}

/*
 *
 */
int
RtransferKanjiFiniteStateMachine (PKANJISTATEMACHINE pKSM, Char cc, char* pOutput)
{
	assert (pKSM != NULL) ;
	assert (pKSM->m_pRtransferFunc != NULL) ;
	assert (pOutput != NULL) ;
	return	(pKSM->m_pRtransferFunc)(pKSM, cc, pOutput) ;
}

/*
 *	եʸ沽Ƚ̤ޤ
 *	ºݤˤϷԲǽʾ礬¸ߤΤǤɤˤ⤳ˤ⡣
 *	⡢ߤȽϴְäƤġ
 */
int
DetectKanjiCodingSystem (const char* pBytes, int nCheckBytes, int* pNewlineType)
{
	int			cc ;
	int			i ;
	BOOL		fProbabilityCodingSystem [MAX_KCODING_SYSTEM] ;
	char		bufShiftJisChar [8] ;
	int			iBufShiftJisUsage ;
	char		bufEucJpChar [8] ;
	int			iBufEucJpUsage ;
	char		bufIso2022JpChar [8] ;
	int			iBufIso2022JpUsage ;
	NEWLINEINFO	newlineinfo ;
	int			iNewlineType ;

	/*
	 *	ɤ̥ͥơ֥롣
	 *()
	 *	Windows ʤΤǤ顢EUC  ShiftJIS 
	 *	ͥ٤⤯ʤäƤޤա
	 *	Unicode ϲǤ⤢ʤΤǡäȤͥ٤㤤
	 *	ǡ NUL ʸդäˤϡ¾
	 *	餷Ĥäꤷޤ(^^;;
	 */
	const int				priorityListOfCodingSystem [MAX_KCODING_SYSTEM] = {
		KCODING_SYSTEM_EUCJP,
		KCODING_SYSTEM_SHIFTJIS,
		KCODING_SYSTEM_ISO2022JP2,
	} ;

	assert (pBytes != NULL || nCheckBytes <= 0) ;

	/*
	 *	ɤʸɤƱͤ˳Τ餷פȤȤˤޤ
	 */
	for (i = 0 ; i < MAX_KCODING_SYSTEM ; i ++)
		fProbabilityCodingSystem [i]	= TRUE ;

	newlineinfo.m_nCarriageReturn	= 0 ; 
	newlineinfo.m_nLineFeed			= 0 ; 
	newlineinfo.m_nCRLF				= 0 ;
	newlineinfo.m_fPostCR			= FALSE ;

	/*
	 *	Lead Character ¸ߤʤȤƤޤ
	 */
	iBufShiftJisUsage	= 0 ;
	iBufEucJpUsage		= 0 ;
	iBufIso2022JpUsage	= 0 ;

	while (nCheckBytes > 0){
		cc	= *pBytes ;

		/*
		 *	SHIFTJIS ɤ沽ƤȲꤷ硣
		 */
		if (fProbabilityCodingSystem [KCODING_SYSTEM_SHIFTJIS] &&
			!checkShiftJisCodingSystem (bufShiftJisChar, &iBufShiftJisUsage, cc))
			fProbabilityCodingSystem [KCODING_SYSTEM_SHIFTJIS] = FALSE ;

		/*
		 *	EUC-JP 沽ƤȲꤷ硣
		 */
		if (fProbabilityCodingSystem [KCODING_SYSTEM_EUCJP] &&
			!checkEucJpCodingSystem (bufEucJpChar, &iBufEucJpUsage, cc))
			fProbabilityCodingSystem [KCODING_SYSTEM_EUCJP]	= FALSE ;

		/*
		 *	JUNET 沽ƤȲꤷ硣
		 */
		if (fProbabilityCodingSystem [KCODING_SYSTEM_ISO2022JP2]){
			switch (checkIso2022Jp2CodingSystem (bufIso2022JpChar, &iBufIso2022JpUsage, cc)){
			case	KCODING_SYSTEM_CHECK_FALSE:
				/*
				 *	ʬȤǤ뤳ȤϤʤ
				 */
				fProbabilityCodingSystem [KCODING_SYSTEM_ISO2022JP2]	= FALSE ;
				break ;
			case	KCODING_SYSTEM_CHECK_TRUE_EXCLUSIVE:
				/*
				 *	ʬͥ٤ʸ沽ϤꤨʤȤ롣
				 */
				for (i = 0 ; i < MAX_KCODING_SYSTEM ; i ++){
					if (priorityListOfCodingSystem [i] == KCODING_SYSTEM_ISO2022JP2)
						break ;
					fProbabilityCodingSystem [priorityListOfCodingSystem [i]]	= FALSE ;
				}
				break ;
			case	KCODING_SYSTEM_CHECK_TRUE:
			default:
				break ;
			}
		}

		checkNewline (cc, &newlineinfo) ;

		pBytes		++ ;
		nCheckBytes -- ;
	}

	if (newlineinfo.m_nCRLF > 0) {
		iNewlineType	= KNEWLINE_MSDOS ;
	} else if (newlineinfo.m_nLineFeed > 0) {
		iNewlineType	= KNEWLINE_UNIX ;
	} else if (newlineinfo.m_nCarriageReturn > 0) {
		iNewlineType	= KNEWLINE_MAC ;
	} else {
		iNewlineType	= KNEWLINE_UNKNOWN ;
	}
	if (pNewlineType != NULL)
		*pNewlineType	= iNewlineType ;

	/*
	 *	ʸɤ̤ͥ˱ *餷* 沽ˡ֤
	 */
	for (i = 0 ; i < MAX_KCODING_SYSTEM ; i ++){
		if (fProbabilityCodingSystem [priorityListOfCodingSystem [i]])
			return	priorityListOfCodingSystem [i] ;
	}

	/*
	 *	Ǥˤϡɬ Unicode ˤʤ롣Unicode Ȥ
	 *	դ路ʤɤäƤΤ̵Ȼפ顣
	 */
	return	KCODING_SYSTEM_UNKNOWN ;
}

/*========================================================================*
 *	ΥǤΤѤɽŪʴؿ
 *========================================================================*/

/*========================================================================*
 *	ܸ EUC ν˴ؤؿ
 *========================================================================*/
/*
 *	Extended Unix Code ξܵ롣
 */
int
restartEucJpStateMachine (PKANJISTATEMACHINE pKSM)
{
	PISO2022STATE	pState ;

	assert (pKSM != NULL) ;

	pState	= &pKSM->m_state.m_iso2022 ;
	pState->m_fEscapeSequence	= FALSE ;
	pState->m_iEscapeState		=  0 ;
	pState->m_iLeadChara		= '\0' ;
	pState->m_fSingleShift		= FALSE ;
	pState->m_iCharset [0]		= KCHARSET_ASCII ;
	pState->m_iCharset [1]		= KCHARSET_JISX0208_1983 ;
	pState->m_iCharset [2]		= KCHARSET_JISX0201_1976 ;
	pState->m_iCharset [3]		= KCHARSET_JISX0212_1990 ;
	pState->m_iGL [0]			=  0 ;
	pState->m_iGL [1]			= -1 ;
	pState->m_iGR [0]			=  1 ;
	pState->m_iGR [1]			= -1 ;
	return	TRUE ;
}

/*
 *	Extended Unix Code ξֵܤ롣
 *----
 *	pState		ֵθ֡
 *	iChara		ϥܥ롣
 *	pOutput		ϥܥ롣
 */
int
transferEucJpStateMachine (PKANJISTATEMACHINE pKSM, int iChara, Char* pOutput)
{
	PISO2022STATE	pState ;

	assert (pKSM != NULL) ;
	assert (pOutput != NULL) ;

	pState	= &pKSM->m_state.m_iso2022 ;

	if (pState->m_iLeadChara){
		switch (pState->m_iCharset [pState->m_iGR [0]]){
		case	KCHARSET_JISX0208_1983:
			if (pState->m_fSingleShift)
				return	-1 ;
			
			*pOutput				= Char_Make (KCHARSET_JISX0208_1983, ((pState->m_iLeadChara << 8) | iChara) & 0x7F7F) ;
			pState->m_iLeadChara	= '\0' ;
			return	TRUE ;
		case	KCHARSET_JISX0212_1990:
			if (!pState->m_fSingleShift)
				return	-1 ;
			*pOutput				= Char_Make (KCHARSET_JISX0212_1990, ((pState->m_iLeadChara << 8) | iChara) & 0x7F7F) ;
			pState->m_iGR [0]		= 1 ;
			pState->m_iLeadChara	= '\0' ;
			pState->m_fSingleShift	= FALSE ;
			return	TRUE ;
		default:
			pState->m_iGR [0]		= 1 ;
			pState->m_iLeadChara	= '\0' ;
			pState->m_fSingleShift	= FALSE ;
			return	-1 ;
		}
	}
	if (iChara <= (0x80 + 0x20)){
		if (pState->m_fSingleShift || pState->m_iLeadChara){
			pState->m_fSingleShift	= FALSE ;
			pState->m_iLeadChara	= '\0' ;
			return	-1 ;
		}
		if (iChara < 0x80){
			*pOutput	= Char_MakeAscii ((char)iChara) ;
			return	TRUE ;
		} else if (iChara == 0x8E){
			pState->m_iGR [0]		= 2 ;
			pState->m_fSingleShift	= TRUE ;
			return	FALSE ;
		} else if (iChara == 0x8F){
			pState->m_iGR [0]		= 3 ;
			pState->m_fSingleShift	= TRUE ;
			return	FALSE ;
		} else {
			return	-1 ;
		}
	}
	switch (pState->m_iCharset [pState->m_iGR [0]]){
	case	KCHARSET_JISX0201_1976:
		if (!pState->m_fSingleShift)
			return	-1 ;
		*pOutput				= Char_Make (KCHARSET_JISX0201_1976, iChara) ;
		pState->m_iGR [0]		= 1 ;
		pState->m_fSingleShift	= FALSE ;
		return	TRUE ;
	case	KCHARSET_JISX0208_1983:
	case	KCHARSET_JISX0212_1990:
		pState->m_iLeadChara	= iChara & 0x00FF ;
		return	FALSE ;
	default:
		return	-1 ;
	}
}

/*
 *()
 *	ѴȤȤ r(everse)transfer Ȥ̾ˤƤ롣
 */
int
rtransferEucJpStateMachine (PKANJISTATEMACHINE pKSM, Char cc, char* pOutput)
{
	assert (pKSM != NULL) ;
	assert (pOutput != NULL) ;

	switch (CHAR_CHARSET (cc)) {
	case	KCHARSET_JISX0208_1978:
	case	KCHARSET_JISX0208_1983:
		*pOutput	++	= (char)(CHAR_CODE (cc) >> 8) ;
		*pOutput	++	= (char)(CHAR_CODE (cc) >> 0) ;
		return	2 ;

	case	KCHARSET_JISX0201_1976:
		if (CHAR_CODE (cc) >= 128) {
			*pOutput	++	= '\x8E' ;
			*pOutput	++	= (char)CHAR_CODE (cc) ;
			return	2 ;
		} else {
			*pOutput	++	= (char)CHAR_CODE (cc) ;
			return	1 ;
		}

	case	KCHARSET_JISX0212_1990:
		*pOutput	++	= '\x8F' ;
		*pOutput	++	= (char)(CHAR_CODE (cc) >> 8) ;
		*pOutput	++	= (char)(CHAR_CODE (cc) >> 0) ;
		return	3 ;

	default:
		if (Char_IsAscii (cc)) {
			*pOutput	++	= (char)CHAR_CODE (cc) ;
			return	1 ;
		}
		break ;
	}
	return	0 ;
}

/*========================================================================
 *	ISO2022-JP-2 ν˴ؤؿ
 *========================================================================*/
/*
 *	iso-2022-jp-2 ܵ롣
 *-----
 *()
 *	iso-2022-jp-2 ιƬǤξ֤ˤ碌롣
 *	ΤĹƬǤϡ
 *		G0 ʸϡ
 *		G1 ʸϡJISX0201-1976 αȾ̡
 *		G2 ʸϡꡣ
 *		G3 ʸϡꡣ
 *		GL  G0 ʸؤ
 *		GR  G1 ʸؤ
 *	ˤʤȦ
 *	JISX0201-1976 αȾ̤ΰˤĤƤɤ狼ʤ
 *	Mule ʤɤɤʤäƤĤȤƤ
 */
int
restartIso2022JpStateMachine (PKANJISTATEMACHINE pKSM)
{
	PISO2022STATE	pState ;

	assert (pKSM != NULL) ;

	pState						= &pKSM->m_state.m_iso2022 ;
	pState->m_fEscapeSequence	= FALSE ;
	pState->m_iEscapeState		= 0 ;
	pState->m_iLeadChara		= '\0' ;
	pState->m_fSingleShift		= FALSE ;
	pState->m_iCharset [0]		= KCHARSET_ASCII ;
	pState->m_iCharset [1]		= KCHARSET_JISX0201_1976 ;
	pState->m_iCharset [2]		= KCHARSET_NOTHING ;
	pState->m_iCharset [3]		= KCHARSET_NOTHING ;
	pState->m_iGL [0]			= 0 ;
	pState->m_iGL [1]			= -1 ;
	pState->m_iGR [0]			= 1 ;
	pState->m_iGR [1]			= -1 ;
	return	TRUE ;
}

/*
 *	iso-2022-jp-2 ܵ򥨥ߥ졼Ȥؿ
 *-----
 *()
 *	lpState		iso-2022-jp-2 ܵθߤξ֡
 *	iChara		ϥե٥åȡ
 *	pOutput		ϥե٥å(¸ߤ)֤ХåեΥݥ󥿡
 *(֤)
 *	0			ϥե٥åȤ¸ߤʤ
 *	1			ϥե٥åȤ¸ߤ롣
 *	-1			顼ȯMSɤ Unicode (ɥڡ:) ǰʤʸĤäƤ⥨顼
 */
int
transferIso2022JpStateMachine (PKANJISTATEMACHINE pKSM, int iChara, Char* pOutput)
{
	static	int	(*apIso2022jpEscapeSequenceHandlerTable [])(PISO2022STATE, int)	= {
		iso2022jpEscapeSequenceState0,
		iso2022jpEscapeSequenceState1,
		iso2022jpEscapeSequenceState2,
		iso2022jpEscapeSequenceState3,
		iso2022jpEscapeSequenceState4,
		iso2022jpEscapeSequenceState5,
		iso2022jpEscapeSequenceDefault,
	} ;
	PISO2022STATE	pState ;

	assert (pKSM != NULL) ;
	assert (pOutput != NULL) ;

	pState	= &pKSM->m_state.m_iso2022 ;

	/*
	 *	ץ󥹤βʤСɤĹޤɤ߹
	 *	ʬ롣
	 */
	if (pState->m_fEscapeSequence){
		assert (0 <= pState->m_iEscapeState && pState->m_iEscapeState <= 6) ;
		return	(apIso2022jpEscapeSequenceHandlerTable [pState->m_iEscapeState]) (pState, iChara) ;
	}

	/*
	 *	ǤʤĲ餫ʸɤ߹硣
	 */
	if (pState->m_fSingleShift){
		pState->m_fSingleShift	= FALSE ;
		pState->m_iLeadChara	= '\0' ;
		pState->m_iGR [0]		= pState->m_iGR [1] ;
		return	-1 ;
	}
	if (iChara <= 0x20){
		if (pState->m_iLeadChara){
			pState->m_iLeadChara	= '\0' ;
			return	-1 ;
		}
		if (iChara == 0x1b){
			pState->m_fEscapeSequence	= TRUE ;
			pState->m_iEscapeState		= 0 ;
			return	FALSE ;
		} else {
			*pOutput	= Char_MakeAscii ((char)iChara) ;
			return	TRUE ;
		}
	} else if (iChara < 0x80){
		if (pState->m_iGL [0] < 0 || pState->m_iGL [0] > 2)
			return	-1 ;

		switch (pState->m_iCharset [pState->m_iGL [0]]){
		case	KCHARSET_ASCII:
			if (pState->m_iLeadChara){
				pState->m_iLeadChara	= '\0' ;
				return	-1 ;
			}
			*pOutput	= Char_Make (pState->m_iCharset [pState->m_iGL [0]], iChara & 0x00FF) ;
			return	TRUE ;

		case	KCHARSET_JISX0201_1976_KATAKANA:
			if (pState->m_iLeadChara){
				pState->m_iLeadChara	= '\0' ;
				return	-1 ;
			}
			*pOutput	= Char_Make (KCHARSET_JISX0201_1976, (iChara & 0x00FF) | 0x80) ;
			return	TRUE ;

		case	KCHARSET_JISX0208_1978:
		case	KCHARSET_JISX0208_1983:
			if (pState->m_iLeadChara){
				*pOutput				= Char_Make (KCHARSET_JISX0208_1983, (pState->m_iLeadChara << 8) | iChara) ;
				pState->m_iLeadChara	= '\0' ;
				return	TRUE ;
			} else {
				pState->m_iLeadChara	= iChara & 0x00FF ;
				return	FALSE ;
			}

		case	KCHARSET_JISX0212_1990:
			if (pState->m_iLeadChara){
				*pOutput				= Char_Make (KCHARSET_JISX0212_1990, (pState->m_iLeadChara << 8) | iChara) ;
				pState->m_iLeadChara	= '\0' ;
				return	TRUE ;
			} else {
				pState->m_iLeadChara	= iChara & 0x00FF ;
				return	FALSE ;
			}

		default:
			return	-1 ;
		}
	}

	if (pState->m_iGR [0] < 0 || pState->m_iGR [0] > 2)
		return	-1 ;

	switch (pState->m_iCharset [pState->m_iGR [0]]){
	case	KCHARSET_JISX0201_1976:
		if (pState->m_iLeadChara){
			pState->m_iLeadChara	= '\0' ;
			return	-1 ;
		}
		*pOutput	= Char_Make (KCHARSET_JISX0201_1976, iChara) ;
		return	TRUE ;
	default:
		break ;
	}

	return	-1 ;
}

/*
 *	ץ󥹤ν
 *()
 *	ޥåʥСäƤ˵ʤȤ٤
 *
 *
 *	[0]	"ESC" ľξ
 *		+--------------+--------------+---------------------+
 *		| ϥܥ | ξ | ¾              |
 *		+--------------+--------------+---------------------+
 *		|     '$'      |  1       |                     |
 *		+--------------+--------------+---------------------+
 *		|     '('      |  3       |                     |
 *		+--------------+--------------+---------------------+
 *		|     'N'      |  5       |                     |
 *		+--------------+--------------+---------------------+
 *		|     '.'      |  4       |                     |
 *		+--------------+--------------+---------------------+
 *		|   ʳ   |          | 顼              |
 *		+--------------+--------------+---------------------+
 *
 *	[1]	"ESC $"
 *		+--------------+--------------+---------------------+
 *		| ϥܥ | ξ | ¾              |
 *		+--------------+--------------+---------------------+
 *		|     '@'      |          | G0  JISC6226-1978 |
 *		|              |              | Ƥ롣      |
 *		+--------------+--------------+---------------------+
 *		|     'B'      |          | G0  JISX0208-1983 |
 *		|              |              | Ƥ롣      |
 *		+--------------+--------------+---------------------+
 *		|     'A'      |          | G0  GB2312-1980   |
 *		|              |              | Ƥ롣      |
 *		+--------------+--------------+---------------------+
 *		|     '('      |  2       |                     |
 *		+--------------+--------------+---------------------+
 *		|   ʳ   |          | 顼              |
 *		+--------------+--------------+---------------------+
 *
 *	[2]	"ESC $ ("
 *		+--------------+--------------+---------------------+
 *		| ϥܥ | ξ | ¾              |
 *		+--------------+--------------+---------------------+
 *		|     '@'      |          | G0  JISC6226-1978 |
 *		|              |              | Ƥ롣      |
 *		+--------------+--------------+---------------------+
 *		|     'B'      |          | G0  JISX0208-1983 |
 *		|              |              | Ƥ롣      |
 *		+--------------+--------------+---------------------+
 *		|     '('      |          | G0  KSC6501-1987  |
 *		|              |              | Ƥ롣      |
 *		+--------------+--------------+---------------------+
 *		|     'D'      |          | G0  JISX0212-1990 |
 *		|              |              | Ƥ롣      |
 *		+--------------+--------------+---------------------+
 *		|   ʳ   |          | 顼              |
 *		+--------------+--------------+---------------------+
 *
 *	[3]	"ESC ("
 *		+--------------+--------------+---------------------+
 *		| ϥܥ | ξ | ¾              |
 *		+--------------+--------------+---------------------+
 *		|     'B'      |          | G0  ASCII   |
 *		|              |              | Ƥ롣            |
 *		+--------------+--------------+---------------------+
 *		|     'J'      |          | G0  JISX0201-1976 |
 *		|              |              | Ƥ롣      |
 *		+--------------+--------------+---------------------+
 *		|   ʳ   |          | 顼              |
 *		+--------------+--------------+---------------------+
 *
 *	[4]	"ESC ."
 *		+--------------+--------------+---------------------+
 *		| ϥܥ | ξ | ¾              |
 *		+--------------+--------------+---------------------+
 *		|     'A'      |          | G2  ISO8859-1   |
 *		|              |              | Ƥ롣        |
 *		+--------------+--------------+---------------------+
 *		|     'F'      |          | G2  ISO8859-7   |
 *		|              |              | Ƥ롣        |
 *		+--------------+--------------+---------------------+
 *		|   ʳ   |          | 顼              |
 *		+--------------+--------------+---------------------+
 *
 *	[5]	"ESC N"
 *		ְäƤ뵤롣
 *		Shingle Shift Sequence?
 */
int
iso2022jpEscapeSequenceState0 (PISO2022STATE pState, int iChara)
{
	switch (iChara){
	case '$':
		pState->m_iEscapeState	= 1 ;
		break ;
	case '(':
		pState->m_iEscapeState	= 3 ;
		break ;
	case 'N':
		/*	shingle-shift-seq */
		pState->m_iEscapeState	= 5 ;
		break ;
	case '.':
		/* 96 character sets ESC sequence */
		pState->m_iEscapeState	= 4 ;
		break ;
	default:
		/*	Τʤץ󥹤褿ν*/
		pState->m_iEscapeState		= 0 ;
		pState->m_fEscapeSequence	= FALSE ;
		break ;
	}
	return	FALSE ;
}

/*
 *	ץ󥹤ν
 *()
 */
int
iso2022jpEscapeSequenceState1 (PISO2022STATE pState, int iChara)
{
	switch (iChara){
	case '@':
		/*	G0  JISC6226-1978 (94^2ʸƤ*/
		pState->m_iCharset [0] = KCHARSET_JISX0208_1978 ;
		break ;
	case 'B':
		/*	G0  JISX0208-1983 (94^2ʸƤ) */
		pState->m_iCharset [0] = KCHARSET_JISX0208_1983 ;
		break ;
	case 'A':
		/*	G0  GB2312-1980 (94^2 ʸƤ) */
		pState->m_iCharset [0] = KCHARSET_GB2312_1980 ;
		break ;
	case '(':
		pState->m_iEscapeState = 2 ;
		break ;
	default:
		break ;
	}
	return	iso2022jpEscapeSequenceDefault (pState, iChara) ;
}

/*
 *	ץ󥹤ν
 *()
 */
int
iso2022jpEscapeSequenceState2 (PISO2022STATE pState, int iChara)
{
	switch (iChara){
	case '@':
		/*	G0  JISC6226-1978 (94^2ʸ)Ƥ */
		pState->m_iCharset [0] = KCHARSET_JISX0208_1978 ;
		break ;
	case 'B':
		/*	G0  JISX0208-1983 (94^2ʸ)Ƥ */
		pState->m_iCharset [0] = KCHARSET_JISX0208_1983 ;
		break ;
	case '(':
		/*	G0  KSC6501-1987 (94^2ʸ)Ƥ */
		pState->m_iCharset [0] = KCHARSET_KSC5601_1987 ;
		break ;
	case 'D':
		/*	G0  JISX0212-1990 (94^2ʸ)Ƥ */
		pState->m_iCharset [0] = KCHARSET_JISX0212_1990 ;
		break ;
	default:
		break ;
	}
	return	iso2022jpEscapeSequenceDefault (pState, iChara) ;
}

/*
 *	ץ󥹤ν
 *()
 */
int
iso2022jpEscapeSequenceState3 (PISO2022STATE pState, int iChara)
{
	if (iChara == 'B'){
		/*	G0  ASCII (94^1ʸ) Ƥ롣*/
		pState->m_iCharset [0]	= KCHARSET_ASCII ;
	} else if (iChara == 'J'){
		/*	G0  JISX0201-1976(Roman) (94^1ʸ) Ƥ롣*/
		pState->m_iCharset [0]	= KCHARSET_JISX0201_1976 ;
	} else if (iChara == 'I'){
		/*
		 *	˵ʤɤ⡢96 ʸȾʬ
		 *	ƤӹࡣäơCharset äƤȤ鸫ȱʤ
		 *	ʤ
		 */
		pState->m_iCharset [0]	= KCHARSET_JISX0201_1976_KATAKANA ;
	}
	return	iso2022jpEscapeSequenceDefault (pState, iChara) ;
}

/*
 *	ץ󥹤ν
 *()
 */
int
iso2022jpEscapeSequenceState4 (PISO2022STATE pState, int iChara)
{
	if (iChara == 'A'){
		/*	G2  ISO8859-1 Ƥ롣*/
		pState->m_iCharset [2]	= KCHARSET_ISO8859_1 ;
	} else if (iChara == 'F'){
		/*	G2  ISO8859-7 Ƥ롣*/
		pState->m_iCharset [2]	= KCHARSET_ISO8859_7 ;
	}
	return	iso2022jpEscapeSequenceDefault (pState, iChara) ;
}

/*
 *	ץ󥹤ν
 *()
 */
int	
iso2022jpEscapeSequenceState5 (PISO2022STATE pState, int iChara)
{
	/* Single-Shift-Char */
	pState->m_iLeadChara	= '\0' ;
	pState->m_fSingleShift	= TRUE ;
	pState->m_iGR [1]		= pState->m_iGR [0] ;
	pState->m_iGR [0]		= 2 ;
	return	iso2022jpEscapeSequenceDefault (pState, iChara) ;
}

int
iso2022jpEscapeSequenceDefault (PISO2022STATE pState, int iChara)
{
	pState->m_iEscapeState		= 0 ;
	pState->m_fEscapeSequence	= FALSE ;
	return	FALSE ;

	UNREFERENCED_PARAMETER (iChara) ;
}

/*
 *
 */
int
rtransferIso2022JpStateMachine (PKANJISTATEMACHINE pKSM, Char cc, char* pOutput)
{
	PISO2022STATE	pState ;
	char*			pOutputTop ;
	const char*		pEscSeq ;

	assert (pKSM != NULL) ;
	assert (pOutput != NULL) ;

	pState		= &pKSM->m_state.m_iso2022 ;
	pOutputTop	= pOutput ;

	/*
	 *	ɽƤʸϷ빽פǤ롣
	 */
	switch (CHAR_CHARSET (cc)) {
	case	KCHARSET_JISX0208_1978:
		pEscSeq	= "\033$@" ;
		goto	state_2byte_char ;

	case	KCHARSET_JISX0208_1983:
		pEscSeq	= "\033$B" ;
		goto	state_2byte_char ;

	case	KCHARSET_JISX0212_1990:
		pEscSeq	= "\033$(D" ;
		goto	state_2byte_char ;

	case	KCHARSET_GB2312_1980:
		pEscSeq	= "\033$A" ;
		goto	state_2byte_char ;

	case	KCHARSET_KSC5601_1987:
		pEscSeq	= "\033$(C" ;

	state_2byte_char:
		/*
		 *	G0 ʸ礬Ĵ٤롣
		 *	GL <= G0 ϸꡣ
		 */
		if (pState->m_iCharset [0] != (int)CHAR_CHARSET (cc)) {
			while (*pEscSeq)
				*pOutput ++	= *pEscSeq ++ ;
			pState->m_iCharset [0]	= CHAR_CHARSET (cc) ;
		}
		*pOutput ++	= (char)(CHAR_CODE (cc) >> 8) ;
		*pOutput ++	= (char)(CHAR_CODE (cc) >> 0) ;
		break ;

	case	KCHARSET_JISX0201_1976:
		/*
		 *	JISX0201-ROMAN  ASCII Ȥϰ㤦餷
		 *	Ʊ˿äƤޤʤ褦դ롣
		 */
		if (CHAR_CODE (cc) < 0x20) {
			*pOutput ++ = (char)CHAR_CODE (cc) ;
			break ;
		}
		if (CHAR_CODE (cc) < 0x80) {
			if (pState->m_iCharset [0] != KCHARSET_JISX0201_1976) {
				*pOutput ++	= 0x1b ;
				*pOutput ++	= '(' ;
				*pOutput ++	= 'J' ;
				pState->m_iCharset [0] = KCHARSET_JISX0201_1976 ;
			}
		} else {
			if (pState->m_iCharset [0] != KCHARSET_JISX0201_1976_KATAKANA) {
				*pOutput ++	= 0x1b ;
				*pOutput ++	= '(' ;
				*pOutput ++	= 'I' ;
				pState->m_iCharset [0] = KCHARSET_JISX0201_1976_KATAKANA ;
			}
		}
		*pOutput ++ = (char)(CHAR_CODE (cc) & 0x7F) ;
		break ;

	case	KCHARSET_ISO8859_1:
		if (Char_IsAscii (cc))
			goto	state_ascii ;

		pEscSeq	= "\033.A" ;
		goto	state_96charset ;

	case	KCHARSET_ISO8859_7:
		if (Char_IsAscii (cc))
			goto	state_ascii ;

		pEscSeq	= "\033.F" ;

	state_96charset:
		if (pState->m_iCharset [2] != (int)CHAR_CHARSET (cc)) {
			while (*pEscSeq)
				*pOutput ++	= *pEscSeq ++ ;
			pState->m_iCharset [2]	= CHAR_CHARSET (cc) ;
		}
		*pOutput ++	= 0x1b ;
		*pOutput ++	= 'N' ;
		*pOutput ++ = (char)(CHAR_CODE (cc) & 0x7F) ;
		break ;

	default:
		/*
		 *	ASCII ǤʤФ⤦ɽԲǽ
		 */
		if (!Char_IsAscii (cc)) 
			break ;

	state_ascii:
		if (pState->m_iCharset [0] != KCHARSET_ASCII) {
			*pOutput ++	= 0x1b ;
			*pOutput ++	= '(' ;
			*pOutput ++	= 'B' ;
			pState->m_iCharset [0]	= KCHARSET_ASCII ;
		}
		*pOutput ++ = (char)CHAR_CODE (cc) ;
		break ;
	}
	return	(pOutput - pOutputTop) ;
}

/*========================================================================*
 *	ShiftJis ν˴ؤؿ
 *========================================================================*/

int
restartShiftJisStateMachine (PKANJISTATEMACHINE pKSM)
{
	PSHIFTJISSTATE	pState ;

	assert (pKSM != NULL) ;

	pState					= &pKSM->m_state.m_shiftjis ;
	pState->m_iLeadChara	= '\0' ;
	return	TRUE ;
}

int
transferShiftJisStateMachine (PKANJISTATEMACHINE pKSM, int iChara, Char* pOutput)
{
	PSHIFTJISSTATE	pState ;
	unsigned 		sh, sl, jh, jl ;

	assert (pKSM != NULL) ;
	assert (pOutput != NULL) ;

	pState	= &pKSM->m_state.m_shiftjis ;

	if (pState->m_iLeadChara){
		if (IS_SHIFTJIS_2BYTE_CODE2 (iChara)){
			sh	= pState->m_iLeadChara & 0xFF ;
			sl	= iChara & 0xFF ;
			jh	= ((sh - ((sh >= 0xa0)? 0xc1: 0x81)) << 1) + 0x21;
			if (sl >= 0x9f) {
				jh	++;
				jl	= sl - 0x7e ;
			} else {
				jl	= sl - ((sl <= 0x7e) ? 0x1f : 0x20) ;
			}
#if defined (DEBUG)
			fprintf (stderr, "(%x, %d) -> (%x, %d)\n", sh, sl, jh, jl) ;
#endif
			*pOutput	= Char_Make (KCHARSET_JISX0208_1983, ((jh << 8) & 0xFF00) | (jl & 0x00FF)) ;
			pState->m_iLeadChara	= '\0' ;
			return	TRUE ;
		} else {
			/*
			 *	ʸ
			 */
			pState->m_iLeadChara	= '\0' ;
#if defined (DEBUG) || 1
			printf ("Invalid character (%02X)\n", iChara) ;
#endif
			return	-1 ;
		}
	}

	if (!IS_SHIFTJIS_2BYTE_CODE1 (iChara)){
		if (IS_SHIFTJIS_JISX201_KATAKANA (iChara)){
			/*
			 *	JISX0201-1976
			 */
			*pOutput	= Char_Make (KCHARSET_JISX0201_1976, iChara & 0xFF) ;
			return	TRUE ;
		} else {
			/*
			 *	ʸ ASCII ʸȻפ
			 */
			*pOutput	= Char_MakeAscii ((char)(iChara & 0x007F)) ;
			return	TRUE ;
		}
	}
	pState->m_iLeadChara	= iChara ;
	return	FALSE ;
}

/*
 *
 */
int
rtransferShiftJisStateMachine (PKANJISTATEMACHINE pKSM, Char cc, char* pOutput)
{
	int				jh, jl, sh, sl ;

	assert (pKSM != NULL) ;
	assert (pOutput != NULL) ;

	switch (CHAR_CHARSET (cc)) {
	case	KCHARSET_JISX0208_1978:
	case	KCHARSET_JISX0208_1983:
		jh	= (CHAR_CODE (cc) >> 8) & 0x00FF ;
		jl	= (CHAR_CODE (cc) >> 0) & 0x00FF ;
		sh	= ((jh - 0x21) >> 1) + 0x81 ;
		if (sh > 0x9f)
			sh	+= 0x40 ;
		if (jh & 1) {
			sl	= jl + 0x1f ;
			if (jl > 0x5f)
				sl	++ ;
		} else {
			sl	= jl + 0x7e ;
		}
		*pOutput ++	= (char) sh ;
		*pOutput ++	= (char) sl ;
		return	2 ;

	case	KCHARSET_JISX0201_1976:
		*pOutput ++	= (char)CHAR_CODE (cc) ;
		return	1 ;

	default:
		if (Char_IsAscii (cc)) {
			*pOutput ++	= (char)CHAR_CODE (cc) ;
			return	1 ;
		}
		break ;
	}
	return	0 ;
}

int
checkShiftJisCodingSystem (char* pchBuffer, int* piUsage, int iChara)
{
	int		fRetvalue ;
	int		iUsage ;

	assert (pchBuffer != NULL) ;
	assert (piUsage != NULL) ;
			
	/*
	 *	괺 ShiftJIS 餷ȤȤˤƤޤ
	 */
	fRetvalue	= TRUE ;

	iUsage		= *piUsage ;
	if (iUsage <= 0){
		if (!IS_SHIFTJIS_2BYTE_CODE1 (iChara)){
			if (!IS_SHIFTJIS_JISX201_KATAKANA (iChara) && iChara >= 0x80)
				fRetvalue	= FALSE ;
		} else {
			*(pchBuffer + iUsage)	= (char) iChara ;
			iUsage	++ ;
		}
	} else {
		if (!IS_SHIFTJIS_2BYTE_CODE2 (iChara))
			fRetvalue	= FALSE ;
		iUsage	= 0 ;
	}
	*piUsage	= iUsage ;

	return	fRetvalue ;
}

int
checkEucJpCodingSystem (char* pchBuffer, int* piUsage, int iChara)
{
	int		fRetvalue ;
	int		iUsage ;

	assert (pchBuffer != NULL) ;
	assert (piUsage != NULL) ;
			
	fRetvalue	= TRUE ;
	iUsage		= *piUsage ;

	if (iUsage <= 0){
		if (iChara == 0x8E ||
			iChara == 0x8F ||
			(0xA1 <= iChara && iChara <= 0xFE)){
			*(pchBuffer + iUsage ++)	= (char) iChara ;
		} else if (iChara >= 0x80 || iChara < 0){
			fRetvalue	= FALSE ;
		}
	} else {
		if (0xA1 <= iChara && iChara <= 0xFE){
			switch ((unsigned char)*pchBuffer){
			case 0x8F:
				if (iUsage < 2){
					*(pchBuffer + iUsage ++)	= (char) iChara ;
				} else {
					iUsage	= 0 ;
				}
				break ;
			case 0x8E:
			default:
				if (iUsage >= 2)
					fRetvalue	= FALSE ;
				iUsage	= 0 ;
				break ;
			}
		} else {
			fRetvalue	= FALSE ;
			iUsage		= 0 ;
		}
	}
	return	fRetvalue ;
}

int
checkIso2022Jp2CodingSystem (char* pchBuffer, int* piUsage, int iChara)
{
	int		fRetvalue ;
	int		i ;
	int		iUsage ;
	/*
	 *	iso-2022-jp-2 ˵Ƥ륨ץ󥹡
	 */
	const char*				iso2022jpEscapeSequences [] = {
		"\x1b$@",	"\x1b$B",	"\x1b$(@",	"\x1b$(B",	"\x1b(B",	"\x1b(J",
		"\x1b(I",	"\x1b$A",	"\x1b$(C",	"\x1b$(D",	"\x1b.A",	"\x1b.F",
		NULL,
	} ;

	assert (pchBuffer != NULL) ;
	assert (piUsage != NULL) ;

	fRetvalue	= TRUE ;
	iUsage		= *piUsage ;

	if (iUsage <= 0){
		if (iChara == 0x1b){
			*(pchBuffer + iUsage ++)	= (char) iChara ;
		} else if (iChara >= 0x80){
			fRetvalue	= FALSE ;
		}
	} else {
		/*	 Escape Sequence ʤΤ*/
		*(pchBuffer + iUsage ++)	= (char) iChara ;
		for (i = 0 ; iso2022jpEscapeSequences [i] != NULL ; i ++){
			if (!strncmp (pchBuffer, iso2022jpEscapeSequences [i], iUsage)){
				if (!(iso2022jpEscapeSequences [i])[iUsage]){
					/*
					 *	ץ󥹤դˤϼʬ
					 *	ͥ٤ι⤤ Coding System ϤʤȤ롣
					 */
					fRetvalue	= KCODING_SYSTEM_CHECK_TRUE_EXCLUSIVE ;
					iUsage		= 0 ;
				}
				break ;
			}
		}
		/*
		 *	Τʤץ󥹤äˤϡ
		 *	ISO2022-JP-2 βǽϤʤ
		 */
		if (!iso2022jpEscapeSequences [i]){
			fRetvalue	= FALSE ;
			iUsage		= 0 ;
		}
	}
	*piUsage	= iUsage ;
	return	fRetvalue ;
}

BOOL
checkNewline (Char cc, NEWLINEINFO* pNewlineInfo)
{
	assert (pNewlineInfo != NULL) ;

	switch (cc) {
	case	0x0D:
		pNewlineInfo->m_nCarriageReturn	++ ;
		pNewlineInfo->m_fPostCR	= TRUE ;
		break ;
		
	case	0x0A:
		if (pNewlineInfo->m_fPostCR) {
			pNewlineInfo->m_nCRLF	++ ;
		} else {
			pNewlineInfo->m_nLineFeed	++ ;
		}
		pNewlineInfo->m_fPostCR	= FALSE ;
		break ;
		
	default:
		pNewlineInfo->m_fPostCR	= FALSE ;
		break ;
	}
	return	TRUE ;
}

