/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <assert.h>
#include "lmachinep.h"
#include "lmkeymap.h"

typedef struct {
	TLispEntity*	m_pEntDefinition ;
	TVarbuffer*		m_pvbufKeySeq ;
	TLispConscell*	m_plstResult ;
	Boolean			m_fOnce ;
}	TLispWhereIsInternalArg ;

typedef struct {
	TLispEntity*	m_pEntKey ;
	int				m_nLength ;
	Boolean			m_fNearMatch ;
	Boolean			m_fAcceptDefault ;
	int				m_nMatch ;
	TLispEntity*	m_pEntFunc ;
}	TLispKeyBindingArg ;

static	Boolean	lispMachine_keybindingLookupKey	(TLispMachine*, TLispEntity*, TLispKeyBindingArg*) ;
static	Boolean	lispMachine_keybindingEnum		(TLispMachine*, TLispEntity*, void*, Boolean*) ;
static	Boolean	lispMachine_whereIsInternal1keymap(TLispMachine*, TLispEntity*, TLispEntity*, TVarbuffer*, TLispConscell*) ;
static	Boolean	lispMachine_whereIsInternalVector(TLispMachine*, TLispEntity*, TLispEntity*, TVarbuffer*, TLispConscell*) ;
static	Boolean	lispMachine_whereIsInternalCons	(TLispMachine*, TLispEntity*, TLispEntity*, TVarbuffer*, TLispConscell*) ;
static	Boolean	lispMachine_whereIsInternalEnum	(TLispMachine*, TLispEntity*, void*, Boolean*) ;

/*
 *	(keymap CHARTABLE . ALIST) Ȥοޥåפä֤
 *	CHARTABLE ϼºݤˤ ASCII ʸ줿 char-table ʤΤ
 *	vector ѤƤ롣ALIST  function key  mouse event ʤ
 *	input stream ˤ¾βä assoc-list Ǥ롣
 *	ƤΥȥϺǽ nil ˡcommand undefinedפ̣Ƥ
 *	뤬Ƥ롣
 *
 *	optional  STRING ϥޥåפΥ˥塼̾ͿƤ롣x-popup-menu
 *	ǻȤ餷
 *
 *	(make-keymap &optional STRING)
 */
TLMRESULT
lispMachineState_MakeKeymap (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pString ;
	TLispEntity*	pKeymap ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pArglist, &pString))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispMgr_CreateKeymap (pLispMgr, pString, &pKeymap))) 
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pKeymap) ;
	return	LMR_RETURN ;
}

/*
 *	(make-sparse-keymap &optional STRING)
 */
TLMRESULT
lispMachineState_MakeSparseKeymap (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pString ;
	TLispEntity*	pKeymap ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pArglist, &pString))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispMgr_CreateSparseKeymap (pLispMgr, pString, &pKeymap))) 
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pKeymap) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_CopyKeymap (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntKeymap ;
	TLispEntity*	pEntRetval ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntKeymap) ;
	if (TFAILED (lispMgr_CopyKeymap (pLispMgr, pEntKeymap, &pEntRetval))) {
		lispMachineCode_SetError (pLM) ;
	} else {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	}
	return	LMR_RETURN ;
}

/*
 *	(keymapp OBJECT)
 *
 *	keymapp  built-in function Ǥ롣
 *	⤷ OBJECT ޥåפǤСt ֤
 *	ޥåפ (keymap . ALIST) ޤϡδؿޥåפǤ
 *	ܥǤ(θԤʸϤϰ̣)
 *	ALIST Ǥ (CHAR . DEFN) ޤ (SYMBOL . DEFN) Τ褦ʷ򤷤Ƥ
 *	롣vector λϡġ
 */
TLMRESULT
lispMachineState_Keymapp (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pEntity ;
	TLispEntity*	pRetval ;

	assert (pLM != NULL) ;

	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	/*
	 *	ʳǤϰ Eval ϺѤǤ롣
	 *	ϥꥹȤȤʤäơACC äƤ롣
	 */
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;

	if (TFAILED (lispEntity_GetCar (pLispMgr, pArglist, &pEntity))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_Keymapp (pLispMgr, pEntity))) {
		lispMgr_CreateT   (pLispMgr, &pRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

/*
 *
 */
TLMRESULT
lispMachineState_LookupKey (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pKey ;
	TLispEntity*	pKeymap ;
	TLispEntity*	pFunction ;
	TLispEntity*	pAcceptDefault ;
	Boolean			fAcceptDefault ;
	int				nMatch ;
	int				nLength ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	/*
	 *	ʳǤϰ Eval ϺѤǤ롣
	 *	ϥꥹȤȤʤäơACC äƤ롣
	 */
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	assert (pArglist != NULL) ;

	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pKeymap)) ||
		TFAILED (lispEntity_GetCdr  (pLispMgr, pArglist, &pArglist)) ||
		TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pKey)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pArglist, &pAcceptDefault))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispEntity_Nullp (pLispMgr, pAcceptDefault))) {
		fAcceptDefault	= True ;
	} else {
		fAcceptDefault	= False ;
	}
	/*	ACCEPT-DEFAULT Ϥޤʤ*/
	nMatch	= 0 ;
	if (TFAILED (lispKeymap_Lookup (pLispMgr, pKeymap, pKey, fAcceptDefault, False, &nMatch, &pFunction))) 
		lispMachineCode_SetError (pLM) ;

	(void) lispEntity_GetLength (pLispMgr, pKey, &nLength) ;
	if (nLength == nMatch) {
		/*	100% פˤϡ֤줿 function 򸫤롣*/
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pFunction) ;
	} else {
		TLispEntity*	hInteger ;
		
		/*	פʸ֤*/
		if (TFAILED (lispMgr_CreateInteger (pLispMgr, nMatch, &hInteger))) 
			return	LMR_ERROR ;
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, hInteger) ;
	}
	return	LMR_RETURN ;
}

/*
 *	(define-key KEYMAP KE DEF)
 *
 *	 KEYMAP, KEY, DEFKEYMAP   KEY 
 *	DEF Ȥ롣KEYMAP  keymap Ǥ롣KEY ʸޤ
 *	 symbol ʸΤΥ٥ȥǤ롣
 *	ASCII ʸ (127 Ķ)  vector ȤäƤʤޤ
 *	ȤǤ롣DEF  key Ǥ벿Ǥ롣
 *	nil  keymap ˤ
 *	command
 *	string
 *	keymap
 *	symbol
 *	cons (STRING . DEFN)
 *	... 
 */
TLMRESULT
lispMachineState_DefineKey (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pKEYMAP ;
	TLispEntity*	pKEY ;
	TLispEntity*	pDEF ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	assert (pArglist != NULL) ;

	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pKEYMAP)) ||
		TFAILED (lispEntity_GetCdr  (pLispMgr, pArglist, &pArglist)) ||
		TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pKEY)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pArglist, &pDEF))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	/*	pKEY  VECTOR ξϺϹͤƤʤ*/
	if (TSUCCEEDED (lispEntity_Vectorp (pLispMgr, pKEY))) {
		TLispEntity**	ppEntKeySeq ;
		int				nKeySeq ;

		(void) lispEntity_GetVectorValue (pLispMgr, pKEY, &ppEntKeySeq, &nKeySeq) ;
		if (TFAILED (lispKeymap_DefineKeyWithVector (pLispMgr, pKEYMAP, ppEntKeySeq, nKeySeq, pDEF)))
			lispMachineCode_SetError (pLM) ;
	} else {
		const Char*		pString ;
		int				nString ;

		if (TFAILED (lispEntity_GetStringValue (pLispMgr, pKEY, &pString, &nString))) {
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
		if (TFAILED (lispKeymap_DefineKey (pLispMgr, pKEYMAP, pString, nString, pDEF))) 
			lispMachineCode_SetError (pLM) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pDEF) ;
	return	LMR_RETURN ;
}

/*
 *	(current-minor-mode-maps)
 *
 *	current-buffer  minor-modes ˳ƤƤ keymap Υꥹ
 *	֤
 *
 *	minor-mode-map-alist ˰פΤФơ
 *	(keymap1 keymap2 ...) Ȥ list 롣
 */
TLMRESULT
lispMachineState_CurrentMinorModeMaps (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntKeymaps ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	if (TFAILED (lispMachineCode_CurrentMinorModeMaps (pLM, &pEntKeymaps))) {
		lispMachineCode_SetError (pLM) ;
	} else {
		lispMachineCode_SetLReg  (pLM, LM_LREG_1, pEntKeymaps) ;
	}
	return	LMR_RETURN ;
}

/*
 *	(use-local-map KEYMAP)
 *
 *	local keymap Ȥ KEYMAP 򤹤롣⤷ KEYMAP  nil ʤ
 *	local keymap ̵ˤ뤳Ȥ̣롣
 */
TLMRESULT
lispMachineState_UseLocalMap (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntKeymap ; 
	TLispEntity*	pEntCurBuffer ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntKeymap)) ||
		TFAILED (lispMachineCode_GetCurrentBuffer (pLM, &pEntCurBuffer))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispBuffer_SetKeymap (pLispMgr, pEntCurBuffer, pEntKeymap) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntKeymap) ;
	return	LMR_RETURN ;
}

/*
 *	(current-local-map)
 *
 *	current-buffer  local keymap ֤⤷äƤʤ
 *	nil ֤
 */
TLMRESULT
lispMachineState_CurrentLocalMap (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntCurBuffer ;
	TLispEntity*	pEntKeymap ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	if (TFAILED (lispMachineCode_GetCurrentBuffer (pLM, &pEntCurBuffer))) {
		lispMachineCode_SetError (pLM) ;
	} else {
		if (TFAILED (lispBuffer_GetKeymap (pEntCurBuffer, &pEntKeymap)) ||
			pEntKeymap == NULL) 
			lispMgr_CreateNil (pLispMgr, &pEntKeymap) ;
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntKeymap) ;
	}
	return	LMR_RETURN ;
}

/*
 *	(where-is-internal DEFINITION &optional KEYMAP FIRSTONLY NOINDIRECT)
 *
 *	DEFINITION ƤӽФΥꥹȤ֤
 *	⤷ KEYMAP  nil ǤʤС KEYMAP  global keymap 
 *	롣
 *	⤷ KEYMAP  nil ʤС active  keymap Ƥ򸡺롣
 *	⤷ KEYMAP  keymap ΥꥹȤǤС keymap 򸡺
 *	롣
 *
 *	3ܤΰ FIRSTONLY  nil ǤʤСդäǽΥ
 *	󥹤֤(ƤΥ󥹤Ǥʤ)
 *	⤷ FIRSTONLY ܥ `non-ascii' ʤС줬Ǥ뤫ˤ
 *	餺ǽ˸դäХǥ󥰤֤
 *	FIRSTONLY ̤ non-nil value ǤСASCII ʸΥ󥹤֡
 *	˥塼Хǥ󥰤ϵݤ롣
 *	4ܤΰ NOINDIRECT  non-nil ʤС¾Υޥåפޤ slot 
 *	ؤδܻȤ˽ʤϴŪ򸡺뤳Ȥǽˤ롣
 *	If optional 4th arg NOINDIRECT is non-nil, don't follow indirections
 *	to other keymaps or slots.  This makes it possible to search for an
 *	indirect definition itself.
 *
 *	current-minor-mode-maps, current-local-map, global-map
 */
TLMRESULT
lispMachineState_WhereIsInternal (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntArglist ;
	TLispEntity*			pEntDefinition ;
	TLispEntity*			pEntKeymap ;
	register TLispEntity*	pEntGlobalMap ;
	TLispConscell			lstResult ;
	TVarbuffer				vbufKeySeq ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	/*	ΤȤ34ưʤ*/
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntDefinition)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist))    ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntKeymap))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (TVarbuffer_Initialize (&vbufKeySeq, sizeof (TLispEntity*))))
		return	LMR_ERROR ;

	lstResult.m_pCar	= lstResult.m_pCdr	= NULL ;
	if (TFAILED (lispEntity_Nullp (pLispMgr, pEntKeymap))) {
		if (TSUCCEEDED (lispEntity_Keymapp (pLispMgr, pEntKeymap))) {
			if (TFAILED (lispMachine_whereIsInternal1keymap (pLM, pEntKeymap, pEntDefinition, &vbufKeySeq, &lstResult)))
				goto	error ;
			TVarbuffer_Clear (&vbufKeySeq) ;
		} else {
			TLispEntity*	pEntTarget ;
			TLispEntity*	pEntNext ;

			if (TFAILED (lispEntity_Consp (pLispMgr, pEntKeymap)))
				goto	error ;

			while (TFAILED (lispEntity_Nullp (pLispMgr, pEntKeymap))) {
				if (TFAILED (lispEntity_GetCar (pLispMgr, pEntKeymap, &pEntTarget)) ||
					TFAILED (lispEntity_GetCdr (pLispMgr, pEntKeymap, &pEntNext)))
					goto	error ;
				if (TFAILED (lispMachine_whereIsInternal1keymap (pLM, pEntTarget, pEntDefinition, &vbufKeySeq, &lstResult)))
					goto	error ;
				TVarbuffer_Clear (&vbufKeySeq) ;
				pEntKeymap	= pEntNext ;
			}
		}
	} else {
		TLispWhereIsInternalArg	arg ;
		TLispEntity*			pEntCurBuffer ;

		/*	 active  keymap Ƹ... sigh
		 *	Ĥޤꡢ
		 *		1. minor-mode-map-alist 
		 *		2. local-map 
		 *		3. global-map 
		 *	뤳Ȥˤʤ롣
		 */
		arg.m_pEntDefinition	= pEntDefinition ;
		arg.m_pvbufKeySeq		= &vbufKeySeq ;
		arg.m_plstResult		= &lstResult ;
		lispMachine_EnumMinorModeMaps (pLM, lispMachine_whereIsInternalEnum, &arg) ;

		if (TFAILED (lispMachineCode_GetCurrentBuffer (pLM, &pEntCurBuffer))) 
			goto	error ;
		if (TSUCCEEDED (lispBuffer_GetKeymap (pEntCurBuffer, &pEntKeymap)) ||
			pEntKeymap != NULL) {
			if (TFAILED (lispMachine_whereIsInternal1keymap (pLM, pEntKeymap, pEntDefinition, &vbufKeySeq, &lstResult)))
				goto	error ;
			TVarbuffer_Clear (&vbufKeySeq) ;
		}
	}

	/*	 global-map 򸡺롣
	 */
	pEntGlobalMap	= lispMachine_GetGlobalMap (pLM) ;
	if (TFAILED (lispMachine_whereIsInternal1keymap (pLM, pEntGlobalMap, pEntDefinition, &vbufKeySeq, &lstResult)))
		goto	error ;

	if (lstResult.m_pCar == NULL) {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_NIL)) ;
	} else {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, lstResult.m_pCar) ;
		lispEntity_Release (pLispMgr, lstResult.m_pCar) ;
	}
	return	LMR_RETURN ;

  error:
	TVarbuffer_Uninitialize (&vbufKeySeq) ;
	if (lstResult.m_pCar != NULL)
		lispEntity_Release (pLispMgr, lstResult.m_pCar) ;
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*	built-in function
 *		(key-binding KEY &optional ACCEPT-DEFAULT)
 *
 *	ߤ keymap  KEY Ф륳ޥɤ֤KEY ʸޤ
 *	٥ȥޤϥȥǤ롣Хɤ¿ʬؿ
 *	äܥǤ
 *
 *	key-binding  default ΥХǥ󥰤Ȥư t 
 *	ФХǥ󥰤̵뤹롣
 */
TLMRESULT
lispMachineState_KeyBinding (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*		pEntArglist ;
	TLispEntity*		pEntKey ;
	TLispEntity*		pEntAcceptDefault ;
	TLispEntity*		pEntKeymap ;
	TLispEntity*		pEntFunc ;
	TLispKeyBindingArg	arg ;
	TLispEntity*		pEntCurBuffer ;
	Boolean				fAcceptDefault ;
	int					nLength ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntKey) ;
	lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntAcceptDefault) ;
	fAcceptDefault	= !lispEntity_Nullp (pLispMgr, pEntAcceptDefault) ;

	if (TFAILED (lispEntity_GetLength (pLispMgr, pEntKey, &nLength))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}

	/*		1. minor-mode-map-alist 򸡺롣*/
	arg.m_pEntKey			= pEntKey ;
	arg.m_nLength			= nLength ;
	arg.m_fAcceptDefault	= fAcceptDefault ;
	arg.m_fNearMatch		= False ;
	arg.m_pEntFunc			= NULL ;
	lispMachine_EnumMinorModeMaps (pLM, lispMachine_keybindingEnum, &arg) ;
	if (arg.m_nMatch > 0 && arg.m_pEntFunc != NULL) {
		pEntFunc	= arg.m_pEntFunc ;
		goto	found ;
	}
	/*		2. local-map 򸡺롣*/
	lispMachineCode_GetCurrentBuffer (pLM, &pEntCurBuffer) ;
	assert (pEntCurBuffer != NULL) ;
	(void) lispBuffer_GetKeymap (pEntCurBuffer, &pEntKeymap) ;
	(void) lispMachine_keybindingLookupKey (pLM, pEntKeymap, &arg) ;
	if (arg.m_nMatch > 0 && arg.m_pEntFunc != NULL) {
		pEntFunc	= arg.m_pEntFunc ;
		goto	found ;
	}

	/*		3. global-map 򸡺롣*/
	pEntKeymap	= lispMachine_GetGlobalMap (pLM) ;
	(void) lispMachine_keybindingLookupKey (pLM, pEntKeymap, &arg) ;
	if (arg.m_nMatch > 0 && arg.m_pEntFunc != NULL) {
		pEntFunc	= arg.m_pEntFunc ;
	} else {
		pEntFunc	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_NIL) ;
	}
  found:
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntFunc) ;
	return	LMR_RETURN ;
}

/*	built-in function
 *		(j-key-binding KEY &optional ACCEPT-DEFAULT)
 *
 *	ߤ keymap  KEY Ф륳ޥɤ֤KEY ʸޤ
 *	٥ȥޤϥȥǤ롣Хɤ¿ʬؿ
 *	äܥǤ
 *
 *	key-binding  default ΥХǥ󥰤Ȥư t 
 *	ФХǥ󥰤̵뤹롣
 *
 *
 *	key-binding Ȥưΰ㤤ϡwindow-proc ƱͤΥȽ򤹤
 *	ȤǤ롣顢㤨С\C-\S-t  bind Ĵ٤ˤ 
 *	\C-\S-t  bind μ \C-t  bind 򸫤롣
 */
TLMRESULT
lispMachineState_JKeyBinding (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*		pEntArglist ;
	TLispEntity*		pEntKey ;
	TLispEntity*		pEntAcceptDefault ;
	TLispEntity*		pEntKeymap ;
	TLispEntity*		pEntGlobalKeymap ;
	TLispEntity*		pEntFunc ;
	TLispKeyBindingArg	arg ;
	TLispEntity*		pEntCurBuffer ;
	Boolean				fAcceptDefault ;
	int					nLength ;
	register int		i ;
	static Boolean		afNearMatch []	= { False, True, } ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntKey) ;
	lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntAcceptDefault) ;
	fAcceptDefault	= !lispEntity_Nullp (pLispMgr, pEntAcceptDefault) ;

	if (TFAILED (lispEntity_GetLength (pLispMgr, pEntKey, &nLength))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}

	lispMachineCode_GetCurrentBuffer (pLM, &pEntCurBuffer) ;
	assert (pEntCurBuffer != NULL) ;
	(void) lispBuffer_GetKeymap (pEntCurBuffer, &pEntKeymap) ;
	pEntGlobalKeymap	= lispMachine_GetGlobalMap (pLM) ;

	arg.m_pEntKey			= pEntKey ;
	arg.m_nLength			= nLength ;
	arg.m_fAcceptDefault	= fAcceptDefault ;
	arg.m_pEntFunc			= NULL ;

	for (i = 0 ; i < NELEMENTS (afNearMatch) ; i ++) {
		/*		1. minor-mode-map-alist 򸡺롣*/
		arg.m_fNearMatch		= afNearMatch [i] ;
		lispMachine_EnumMinorModeMaps (pLM, lispMachine_keybindingEnum, &arg) ;
		if (arg.m_nMatch > 0 && arg.m_pEntFunc != NULL) {
			pEntFunc	= arg.m_pEntFunc ;
			goto	found ;
		}
		/*		2. local-map 򸡺롣*/
		(void) lispMachine_keybindingLookupKey (pLM, pEntKeymap, &arg) ;
		if (arg.m_nMatch > 0 && arg.m_pEntFunc != NULL) {
			pEntFunc	= arg.m_pEntFunc ;
			goto	found ;
		}
		/*		3. global-map 򸡺롣*/
		(void) lispMachine_keybindingLookupKey (pLM, pEntGlobalKeymap, &arg) ;
		if (arg.m_nMatch > 0 && arg.m_pEntFunc != NULL) {
			pEntFunc	= arg.m_pEntFunc ;
			goto	found ;
		}
	}
	pEntFunc	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_NIL) ;
  found:
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntFunc) ;
	return	LMR_RETURN ;
}

/*========================================================================*
 *	private functions
 */
Boolean
lispMachine_keybindingLookupKey (
	register TLispMachine*			pLM,
	register TLispEntity*			pEntKeymap,
	register TLispKeyBindingArg*	pArg)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	register TLispEntity*	pEntKey ;
	register int			nLength ;
	register Boolean		fNearMatch ;
	register Boolean		fAcceptDefault ;
	int						nMatch ;
	TLispEntity*			pEntFunc ;

	pEntKey			= pArg->m_pEntKey ;
	nLength			= pArg->m_nLength ;
	fNearMatch		= pArg->m_fNearMatch ;
	fAcceptDefault	= pArg->m_fAcceptDefault ;
	nMatch			= 0 ;
	pEntFunc		= NULL ;
	if (TSUCCEEDED (lispKeymap_Lookup (pLispMgr, pEntKeymap, pEntKey, fAcceptDefault, fNearMatch, &nMatch, &pEntFunc)) &&
		TFAILED (lispEntity_Nullp (pLispMgr, pEntFunc)) &&
		nMatch > 0 && (nMatch == nLength || fAcceptDefault)) {
		pArg->m_nMatch		= nMatch ;
		pArg->m_pEntFunc	= pEntFunc ;
	} else {
		pArg->m_nMatch		= 0 ;
		pArg->m_pEntFunc	= NULL ;
	}
	return	True ;
}

Boolean
lispMachine_keybindingEnum (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntKeymap,
	register void*			pCaller,
	register Boolean*		pfContinue)
{
	register TLispKeyBindingArg*	pArg	= (TLispKeyBindingArg *) pCaller ;

	if (TFAILED (lispMachine_keybindingLookupKey (pLM, pEntKeymap, pArg)))
		return	False ;
	if (pArg->m_nMatch > 0 && pArg->m_pEntFunc != NULL) 
		*pfContinue	= False ;
	return	True ;
}

/*	1 keymap Ф where-is-internal ưԤ
 */
Boolean
lispMachine_whereIsInternal1keymap (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntKeymap,
	register TLispEntity*	pEntSymbol,
	register TVarbuffer*	pvbufKeySeq,
	register TLispConscell*	pEntRetval)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntTarget ;
	TLispEntity*	pEntNext ;
	int				nType ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	if (pEntKeymap == NULL ||
		TFAILED (lispEntity_Keymapp (pLispMgr, pEntKeymap)))	/* null ̵뤹롣*/
		return	True ;

	(void) lispEntity_GetCdr (pLispMgr, pEntKeymap, &pEntNext) ;
	pEntKeymap	= pEntNext ;

	while (TFAILED (lispEntity_Nullp (pLispMgr, pEntKeymap))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pEntKeymap, &pEntTarget)) ||
			TFAILED (lispEntity_GetCdr (pLispMgr, pEntKeymap, &pEntNext)))
			return	False ;
		(void) lispEntity_GetType (pLispMgr, pEntTarget, &nType) ;
		switch (nType) {
		case	LISPENTITY_VECTOR:
			if (TFAILED (lispMachine_whereIsInternalVector (pLM, pEntTarget, pEntSymbol, pvbufKeySeq, pEntRetval)))
				return	False ;
			break ;
		case	LISPENTITY_CONSCELL:
			if (TFAILED (lispMachine_whereIsInternalCons (pLM, pEntTarget, pEntSymbol, pvbufKeySeq, pEntRetval)))
				return	False ;
			break ;
		default:
			/*	¿ʬ˥顼ʡ*/
			return	False ;
		}
		pEntKeymap	= pEntNext ;
	}
	return	True ;
}

Boolean
lispMachine_whereIsInternalVector (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntVector,
	register TLispEntity*	pEntSymbol,
	register TVarbuffer*	pvbufKeySeq,
	register TLispConscell*	pEntRetval)
{
	register TLispManager*	pLispMgr ;
	TLispEntity**			ppElements ;
	int						nElements ;
	register TLispEntity**	ptr ;
	register int			i ;
	TLispEntity*			pEntKey ;
	TLispEntity*			pEntResult ;
	register Boolean		f ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	if (TFAILED (lispEntity_GetVectorValue (pLispMgr, pEntVector, &ppElements, &nElements)))
		return	False ;

	for (i = 0, ptr = ppElements ; i < nElements ; i ++, ptr ++) {
		if (TSUCCEEDED (lispEntity_Eq (pLispMgr, *ptr, pEntSymbol))) {
			register TLispEntity**	pEntKeySeq ;
			register int			nKeySeq ;

			if (TFAILED (lispMgr_CreateInteger (pLispMgr, (long)i, &pEntKey)))
				return	False ;
			if (TFAILED (TVarbuffer_Add (pvbufKeySeq, &pEntKey, 1)))
				return	False ;
			pEntKeySeq	= (TLispEntity **)TVarbuffer_GetBuffer (pvbufKeySeq) ;
			nKeySeq		= TVarbuffer_GetUsage (pvbufKeySeq) ;
			lispEntity_AddRef (pLispMgr, pEntKey) ;
			f	= lispMgr_CreateVector (pLispMgr, pEntKeySeq, nKeySeq, &pEntResult) ;
			lispEntity_Release (pLispMgr, pEntKey) ;
			if (TFAILED (f) || pEntResult == NULL)
				return	False ;
			lispEntity_AddRef (pLispMgr, pEntResult) ;
			f	= lispEntity_Push2List (pLispMgr, pEntRetval, pEntResult) ;
			lispEntity_Release (pLispMgr, pEntResult) ;
			if (TFAILED (f))
				return	False ;
		}
	}
	return	True ;
}

Boolean
lispMachine_whereIsInternalCons (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntCons,
	register TLispEntity*	pEntSymbol,
	register TVarbuffer*	pvbufKeySeq,
	register TLispConscell*	pEntRetval)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntKey ;
	TLispEntity*			pEntFunc ;
	TLispEntity*			pEntResult ;
	register Boolean		f ;
	
	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntCons, &pEntKey)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntCons, &pEntFunc))) 
		return	False ;

	if (TSUCCEEDED (lispEntity_Symbolp (pLispMgr, pEntFunc))) {
		if (TSUCCEEDED (lispEntity_Eq (pLispMgr, pEntFunc, pEntSymbol))) {
			register TLispEntity**	pEntKeySeq ;
			register int			nKeySeq ;

			if (TFAILED (TVarbuffer_Add (pvbufKeySeq, &pEntKey, 1)))
				return	False ;

			pEntKeySeq	= (TLispEntity **)TVarbuffer_GetBuffer (pvbufKeySeq) ;
			nKeySeq		= TVarbuffer_GetUsage (pvbufKeySeq) ;
			lispEntity_AddRef (pLispMgr, pEntKey) ;
			f	= lispMgr_CreateVector (pLispMgr, pEntKeySeq, nKeySeq, &pEntResult) ;
			lispEntity_Release (pLispMgr, pEntKey) ;
			if (TFAILED (f) || pEntResult == NULL)
				return	False ;

			lispEntity_AddRef (pLispMgr, pEntResult) ;
			f	= lispEntity_Push2List (pLispMgr, pEntRetval, pEntResult) ;
			lispEntity_Release (pLispMgr, pEntResult) ;
			if (TFAILED (f))
				return	False ;
		}
		return	True ;
	}
	if (TFAILED (lispEntity_Keymapp (pLispMgr, pEntFunc)))
		return	False ;

	if (TFAILED (TVarbuffer_Add (pvbufKeySeq, &pEntKey, 1)))
		return	False ;

	f	= lispMachine_whereIsInternal1keymap (pLM, pEntFunc, pEntSymbol, pvbufKeySeq, pEntRetval) ;
	(void) TVarbuffer_Sub (pvbufKeySeq, 1) ;
	return	f ;
}

Boolean
lispMachine_whereIsInternalEnum (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntKeymap,
	register void*			pCaller,
	register Boolean*		pfContinue)
{
	register TLispWhereIsInternalArg*	pArg	= (TLispWhereIsInternalArg*) pCaller ;
	register TLispEntity*	pEntDef ;
	register TVarbuffer*	pvbuf ;
	register TLispConscell*	pResult ;

	pEntDef	= pArg->m_pEntDefinition ;
	pvbuf	= pArg->m_pvbufKeySeq ;
	pResult	= pArg->m_plstResult ;
	if (TFAILED (lispMachine_whereIsInternal1keymap (pLM, pEntKeymap, pEntDef, pvbuf, pResult)))
		return	False ;

	TVarbuffer_Clear (pvbuf) ;
	return	True ;
}

