/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <assert.h>
#if !defined (__linux__)
#if defined (__SVR4) && defined (__sun)
#include <limits.h>
#define DEFFILEMODE (S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH)
#else
#include <sys/syslimits.h>
#endif
#endif
#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <pwd.h>
#include <time.h>
#include "lmachinep.h"
#include "lmstate.h"
#include "filename.h"
#include "kfile.h"
#include "kanji.h"

static	TLMRESULT	lispMachineState_loadStep1		(TLispMachine*) ;
static	TLMRESULT	lispMachineState_loadFinalize	(TLispMachine*) ;
static	Boolean		lispMachine_getFileNameFromEntity	(TLispManager*, TLispEntity*, char*, int) ;

/*
 *	(load-file FILE)
 *
 *	load-file  interactive  lisp function Ǥꡢ`file' 
 *	Ƥ롣
 *	ΤȤinteractive ʬȴƤ롣 buffer 
 *	Ѱդɬפ롣¸Τˤ load-file Τ
 *	ʤΤǡ
 */
TLMRESULT
lispMachineState_LoadFile (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pFile ;
	TLispEntity*	pRetval ;
	const Char*		pFileName ;
	int				nFileNameLen ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pFile)) ||
		TFAILED (lispEntity_Stringp (pLispMgr, pFile))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispEntity_GetStringValue (pLispMgr, pFile, &pFileName, &nFileNameLen) ;
	if (pFileName == NULL || nFileNameLen <= 0) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispMgr_Load (pLispMgr, pFileName, nFileNameLen, &pRetval))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	lispEntity_Release (pLispMgr, pRetval) ;
	/*	ʲ progn ȤưƤ餦*/
	lispMachineCode_SetState (pLM, &lispMachineState_Progn) ;
	return	LMR_CONTINUE ;
}

/*
  load is a built-in function.
  (load FILE &optional NOERROR NOMESSAGE NOSUFFIX MUST-SUFFIX)

  NOSUFFIX  MUST-SUFFIX ưʤäƤʤΤǡ顢
  default  MUST-SUFFIX ˤʤäƤ롣

  Execute a file of Lisp code named FILE.
  First try FILE with `.elc' appended, then try with `.el',
  then try FILE unmodified.  Environment variable references in FILE
  are replaced with their values by calling `substitute-in-file-name'.
  This function searches the directories in `load-path'.
  If optional second arg NOERROR is non-nil,
  report no error if FILE doesn't exist.
  Print messages at start and end of loading unless
  optional third arg NOMESSAGE is non-nil.
  If optional fourth arg NOSUFFIX is non-nil, don't try adding
  suffixes `.elc' or `.el' to the specified name FILE.
  If optional fifth arg MUST-SUFFIX is non-nil, insist on
  the suffix `.elc' or `.el'; don't accept just FILE unless
  it ends in one of those suffixes or includes a directory name.
  Return t if file exists. */
TLMRESULT
lispMachineState_Load (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntArgtop ;
	TLispEntity*	pEntFile ;
	TLispEntity*	pEntNoMsg ;
	TLispEntity*	pEntNoError ;

	assert (pLM != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	pEntArgtop	= pEntArglist ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntFile)) ||
		TFAILED (lispEntity_Stringp (pLispMgr, pEntFile))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntNoError) ;
	lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntNoMsg) ;

	if (TFAILED (lispEntity_Nullp (pLispMgr, pEntNoMsg))) {
		static const char	strFORMAT []	= "Loading ... %q" ;
		TLispEntity*		pEntMsg ;
		lispEntity_FormatA (pLispMgr, strFORMAT, NELEMENTS (strFORMAT) - 1, pEntArgtop, &pEntMsg) ;
		lispMachineCode_SetMinibufferMessage (pLM, pEntMsg) ;
	}
	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_2) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_3) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_4) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_1, pEntFile) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_2, pEntNoError) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_3, pEntNoMsg) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_4, pEntArglist) ;
	lispMachineCode_SetState (pLM, lispMachineState_loadStep1) ;
	return	LMR_CONTINUE ;
}

/*
 *	(expand-file-name NAME &optional DEFAULT-DIRECTORY)
 *
 *	ե̾ NAME ФİդΤѴ롣2
 *	DEFAULT-DIRECTORY Ϥ⤷ NAME Фʤ餽ǤϤޤǥ
 *	ȥǤ롣⤷ DEFAULT-DIRECTORY  nil ޤϷ礱Ƥ
 *	硢current buffer  default-directory ͤȤ롣
 *	ե̾ `.' ޤǤˤϺ롣ޤ`..'
 *	³ե̾Ǥϡ롣
 *	̤Υե̾ե륷ƥˤ뤫ɤΥå
 *	Ϥʤǽ餬 `~/' ǤϤޤˤϡhome directory 
 *	`~USER/' ξˤ USER  home directory Ÿ롣
 */
TLMRESULT
lispMachineState_ExpandFileName (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TVarbuffer		vbuf ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntName ;
	TLispEntity*	pEntDefaultDirectory ;
	TLispEntity*	pEntRetval ;
	const Char*		pName ;
	int				nName ;
	const Char*		pDefault ;
	int				nDefault ;

	assert (pLM != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntName)) ||
		TFAILED (lispEntity_Stringp (pLispMgr, pEntName))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntDefaultDirectory) ;
	if (TFAILED (lispEntity_Stringp (pLispMgr, pEntDefaultDirectory)) &&
		TFAILED (lispEntity_Nullp   (pLispMgr, pEntDefaultDirectory))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (TVarbuffer_Initialize (&vbuf, sizeof (Char)))) 
		return	LMR_ERROR ;
	lispEntity_GetStringValue (pLispMgr, pEntName, &pName, &nName) ;
	if (nName <= 0) 
		goto	expand_end ;
	if (TSUCCEEDED (lispEntity_GetStringValue (pLispMgr, pEntDefaultDirectory, &pDefault, &nDefault))) {
	} else {
		pDefault	= NULL ;
		nDefault	= 0 ;
	}
	if (TFAILED (ExpandFileName (&vbuf, pName, nName, pDefault, nDefault))) {
		TVarbuffer_Uninitialize (&vbuf) ;
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
  expand_end:
	if (TFAILED (lispMgr_CreateString (pLispMgr, TVarbuffer_GetBuffer (&vbuf), TVarbuffer_GetUsage (&vbuf), &pEntRetval))) {
		TVarbuffer_Uninitialize (&vbuf) ;
		return	LMR_ERROR ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	TVarbuffer_Uninitialize (&vbuf) ;
	return	LMR_RETURN ;
}


/*	private functions */
TLMRESULT
lispMachineState_loadStep1 (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*			pEntFile ;
	const Char*				pFileName ;
	int						nFileNameLen ;
	register TLispEntity*	pEntLoadPath ;
	TLispEntity*			pValDirList ;
	TLispEntity*			pEntRetval ;
	register Boolean		fResult ;
	static const Char		chDelimitar	= '/' ;
	TVarbuffer				vbufFileName ;

	lispMachineCode_GetLReg (pLM, LM_LREG_1, &pEntFile) ;
	lispEntity_GetStringValue (pLispMgr, pEntFile, &pFileName, &nFileNameLen) ;
	if (pFileName == NULL || nFileNameLen <= 0) {
		lispMachineCode_SetError (pLM) ;
		lispMachineCode_SetState (pLM, lispMachineState_loadFinalize) ;
		return	LMR_CONTINUE ;
	}

	/*	load-path ֤˸롣*/
	pEntLoadPath	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_LOAD_PATH) ;
	if (TFAILED (lispMachine_GetCurrentSymbolValue (pLM, pEntLoadPath, &pValDirList)))
		lispMgr_CreateNil (pLispMgr, &pValDirList) ;

	if (TFAILED (TVarbuffer_Initialize (&vbufFileName, sizeof (Char))))
		return	LMR_ERROR ;

	fResult	= False ;
	while (!fResult && TFAILED (lispEntity_Nullp (pLispMgr, pValDirList))) {
		TLispEntity*	pValDir ;
		TLispEntity*	pEntCdr ;
		const Char*		strDir ;
		int				nstrDir ;
		const Char*		strPath ;
		int				nstrPath ;
		
		if (TFAILED (lispEntity_GetCar (pLispMgr, pValDirList, &pValDir)) ||
			TFAILED (lispEntity_GetCdr (pLispMgr, pValDirList, &pEntCdr))  ||
			TFAILED (lispEntity_GetStringValue (pLispMgr, pValDir, &strDir, &nstrDir)))
			break ;
		if (TFAILED (TVarbuffer_Add (&vbufFileName, strDir, nstrDir)) ||
			TFAILED (TVarbuffer_Add (&vbufFileName, &chDelimitar, 1)) ||
			TFAILED (TVarbuffer_Add (&vbufFileName, pFileName, nFileNameLen))) 
			return	LMR_ERROR ;
		
		strPath		= TVarbuffer_GetBuffer (&vbufFileName) ;
		nstrPath	= TVarbuffer_GetUsage  (&vbufFileName) ;
		fResult		= lispMgr_Load (pLispMgr, strPath, nstrPath, &pEntRetval) ;
		TVarbuffer_Clear (&vbufFileName) ;
		pValDirList	= pEntCdr ;
	}
	TVarbuffer_Uninitialize (&vbufFileName) ;

	if (TFAILED (fResult)) 
		fResult		= lispMgr_Load (pLispMgr, pFileName, nFileNameLen, &pEntRetval) ;

	if (TFAILED (fResult)) {
		TLispEntity*	pEntNoError ;
		/*	noerror  t ʤե뤬̵ä˥顼ˤʤʤ*/
		lispMachineCode_GetLReg (pLM, LM_LREG_2, &pEntNoError) ;
		if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntNoError))) 
			lispMachineCode_SetError (pLM) ;
		lispMachineCode_SetState (pLM, lispMachineState_loadFinalize) ;
		return	LMR_CONTINUE ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	lispEntity_Release (pLispMgr, pEntRetval) ;
	lispMachineCode_PushState (pLM, lispMachineState_loadFinalize) ;
	lispMachineCode_SetState  (pLM, lispMachineState_Progn) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_loadFinalize (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	static const char	strFORMAT []	= "Loading ... %q ... done" ;	
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntNoMessage ;
	TLispEntity*	pEntMsg ;

	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) 
		goto	exit_finalize ;

	lispMachineCode_GetLReg (pLM, LM_LREG_3, &pEntNoMessage) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntNoMessage))) 
		goto	exit_finalize ;

	lispMachineCode_GetLReg (pLM, LM_LREG_4, &pEntArglist) ;
	lispEntity_FormatA (pLispMgr, strFORMAT, NELEMENTS (strFORMAT) - 1, pEntArglist, &pEntMsg) ;
	lispMachineCode_SetMinibufferMessage (pLM, pEntMsg) ;

exit_finalize:
	lispMachineCode_PopLReg (pLM, LM_LREG_4) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_3) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_2) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
	return	LMR_RETURN ;
}

/*	built-in function:
 *		(insert-file-contents FILENAME &optional VISIT BEG END REPLACE)
 *
 *	ǤϤ뤬FILENAME  VISIT ʳĤϺΤȤʤ
 *	Ť emacs Ǥ FILENAME  VISIT ¾Ȥ뤳ȤϤǤʤä褦
 *
 *	VISIT  non-nil ξˤϡɤ߹ FILE  VISIT ե̾
 *	ơTIMESTAMP 򹹿ơmodified  nil ˤ롣ե뤬
 *	¸ߤʤǤ visiting ϴλ롣
 */
TLMRESULT
lispMachineState_InsertFileContents (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntFileName ;
	TLispEntity*	pEntBuffer ;
	TLispEntity*	apEntity [2] ;
	TLispEntity*	pEntRetval ;
	const Char*		pStrFileName ;
	int				nStrFileName ;
	char			rchPath [PATH_MAX + 1] ;
	KFILE			kfile ;
	int				nReadCount ;
	int				nPoint ;
	register int	nCodingSys ;

	assert (pLM != NULL) ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFileName) ;
	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntFileName, &pStrFileName, &nStrFileName))) 
		goto	error ;
	if (TFAILED (ConvertInternalFileName2SystemFileName (rchPath, PATH_MAX + 1, pStrFileName, nStrFileName))) 
		goto	error ;

	lispBuffer_Point (pLispMgr, pEntBuffer, &nPoint) ;
	nReadCount	= 0 ;

	/*	ɤ߹ե coding-system  coding-system-for-read ޤ
	 *	file-coding-system-alist Ƿꤹ롣
	 */
	nCodingSys	= lispMachine_GetCodingSystemForRead (pLM) ;
	if (!KFile_Open (&kfile, rchPath, nCodingSys)) {
#if defined (DEBUG) || 1
		fprintf (stderr, "Can't open file: \"%s\"\n", rchPath) ;
#endif
		goto	result ;
	}
	KFile_Rewind (&kfile) ;
	lispBuffer_InsertFileContents (pLispMgr, pEntBuffer, nPoint, &kfile, &nReadCount) ;
	KFile_Close (&kfile) ;

  result:
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, nReadCount, &apEntity [1])))
		return	LMR_ERROR ;
	apEntity [0]	= pEntFileName ;
	if (TFAILED (lispMgr_CreateList (pLispMgr, apEntity, 2, &pEntRetval)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;

  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*	built-in function:
 *		(file-name-nondirectory FILENAME)
 *
 *	̾ FILENAME Υǥ쥯ȥȴ֤ͤñ˺Ǹ slash θ
 *	³ʸ֤ɤ褦
 */
TLMRESULT
lispMachineState_FileNameNondirectory (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntFileName ;
	const Char*		pStrName ;
	int				nStrName ;
	const Char*		pStrResult ;
	int				nStrResult ;
	TLispEntity*	pEntRetval ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFileName) ;
	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntFileName, &pStrName, &nStrName))) 
		goto	error ;
	if (TFAILED (GetFileNameNondirectory (pStrName, nStrName, &pStrResult, &nStrResult)))
		goto	error ;
	if (TFAILED (lispMgr_CreateString (pLispMgr, pStrResult, nStrResult, &pEntRetval)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
	
  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*	(rename-file FILE NEWNAME &optional OK-IF-ALREADY-EXISTS)
 */
TLMRESULT
lispMachineState_RenameFile (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*		pEntArglist ;
	TLispEntity*		pEntFile ;
	TLispEntity*		pEntNewName ;
	TLispEntity*		pEntOkIfAExists ;
	TLispEntity*		pEntNil ;
	char				bufOldPath [PATH_MAX + 1] ;
	char				bufNewPath [PATH_MAX + 1] ;
	char				bufCopy [4096] ;
	struct stat			bufstat ;
	register int		nRetval ;
	register FILE*		fpSrc ;
	register FILE*		fpDest ;
	register Boolean	fError ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFile)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntNewName)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntOkIfAExists))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	/*	ѹȤ̾롣*/
	if (TFAILED (lispMachine_getFileNameFromEntity (pLispMgr, pEntFile, bufOldPath, PATH_MAX))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	/*	ѹ̾롣*/
	if (TFAILED (lispMachine_getFileNameFromEntity (pLispMgr, pEntNewName, bufNewPath, PATH_MAX))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}

	nRetval	= lstat (bufNewPath, &bufstat) ;
	if (nRetval == 0) {
		if (S_ISREG (bufstat.st_mode) == 0 ||
			TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntOkIfAExists))) {
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
	} else {
		if (errno != ENOENT) {
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
	}

	/*	rename  copy ʤΤ */
	fpSrc	= fopen (bufOldPath, "rb") ;
	if (fpSrc == NULL) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	fpDest	= fopen (bufNewPath, "wb") ;
	if (fpDest == NULL) {
		lispMachineCode_SetError (pLM) ;
		fclose (fpSrc) ;
		return	LMR_RETURN ;
	}

	fError	= False ;
	while (!feof (fpSrc)) {
		register int nread	= fread (bufCopy, 1, sizeof (bufCopy), fpSrc) ;
		if (nread <= 0) {
			fError	= (nread < 0) ;
			break ;
		}
		if (fwrite (bufCopy, 1, nread, fpDest) <= 0) {
			fError	= True ;
			break ;
		}
	}
	if (fclose (fpSrc) != 0)
		fError	= True ;
	if (fclose (fpDest) != 0)
		fError	= True ;

	if (! fError) {
		if (unlink (bufOldPath) != 0)
			fError	= True ;
	}
	if (fError) {
		lispMachineCode_SetError (pLM) ;
	}
	lispMgr_CreateNil (pLispMgr, &pEntNil) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntNil) ;
	return	LMR_RETURN ;
}

/*	(delete-file)
 */
TLMRESULT
lispMachineState_DeleteFile (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntFile ;
	char			bufFile [PATH_MAX + 1] ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFile))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	/*	ѹȤ̾롣*/
	if (TFAILED (lispMachine_getFileNameFromEntity (pLispMgr, pEntFile, bufFile, PATH_MAX))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}

	if (unlink (bufFile) != 0) {
		lispMachineCode_SetError (pLM) ;
	} else {
		TLispEntity*	pEntNil ;
		lispMgr_CreateNil (pLispMgr, &pEntNil) ;
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntNil) ;
	}
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_FileModes (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntFile ;
	TLispEntity*	pEntMode ;
	char			bufFile [PATH_MAX + 1] ;
	struct stat		bufstat ;
	register int	nMode ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFile))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	/*	ѹȤ̾롣*/
	if (TFAILED (lispMachine_getFileNameFromEntity (pLispMgr, pEntFile, bufFile, PATH_MAX))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	
	if (lstat (bufFile, &bufstat) == -1) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	nMode	= bufstat.st_mode & DEFFILEMODE ;
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, nMode, &pEntMode)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntMode) ;
	return	LMR_RETURN ;
}

/*	(write-region START END FILENAME &optional APPEND VISIT LOCKNAME MUSTBENEW)
 *	
 *	VISIT ʹߤΥݡȤϤʤVISIT  nil  non-nil ʤ
 */
TLMRESULT
lispMachineState_WriteRegion (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*		pEntArglist ;
	TLispEntity*		pEntBuffer ;
	TLispEntity*		pEntStart ;
	TLispEntity*		pEntEnd ;
	TLispEntity*		pEntFileName ;
	TLispEntity*		pEntAppend ;
	TLispEntity*		pEntVisit ;
	TLispEntity*		pEntNil ;
	const Char*			strFile ;
	int					nstrFile ;
	char				bufFile [PATH_MAX + 1] ;
	char				buffer [2064] ;
	register char*		ptr ;
	TLispNumber			numStart, numEnd ;
	register int		nUsage, nptr, nctext, nCopy, nStart, nEnd ;
	int					nLength, nPointTop, nPointEnd ;
	register FILE*		fp ;
	register Char		cc ;
	KANJISTATEMACHINE	ksm ;
	TBufStringMarker	mk ;
	register int		nCodingSys ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;

	/*	Ф
	 */
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntStart)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntEnd)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFileName)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntAppend)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntVisit))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}

	/*	ηɤå롣
	 */
	if (TFAILED (lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pEntStart, &numStart)) ||
		numStart.m_fFloatp ||
		TFAILED (lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pEntEnd,   &numEnd)) ||
		numEnd.m_fFloatp ||
		numStart.m_Value.m_lLong > numEnd.m_Value.m_lLong ||
		TFAILED (lispEntity_GetStringValue (pLispMgr, pEntFileName, &strFile, &nstrFile))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	nStart	= numStart.m_Value.m_lLong ;
	nEnd	= numEnd.m_Value.m_lLong ;

	/*	ХåեؤƤ뤫ɤå롣
	 */
	lispBuffer_GetString (pLispMgr, pEntBuffer, &mk, &nLength) ;
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointTop) ;
	lispBuffer_PointBufferEnd (pLispMgr, pEntBuffer, &nPointEnd) ;
#if defined (DEBUG)
	fprintf (stdout, "write-region: (%d/%d, %d/%d)\n", nStart, nPointTop, nEnd, nPointEnd) ; 
#endif
	if (nEnd < nPointTop || nStart > nPointEnd) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	TBufStringMarker_Forward (&mk, nStart - nPointTop) ;
	
	/*	ե򳫤
	 */
	nCopy	= (nstrFile < PATH_MAX)? nstrFile : PATH_MAX ;
	cstrtostr (bufFile, strFile, nCopy) ;
	bufFile [nCopy]	= '\0' ;
	if (TFAILED (lispEntity_Nullp (pLispMgr, pEntAppend))) {
		fp	= fopen (bufFile, "ab") ;
	} else {
		fp	= fopen (bufFile, "wb") ;
	}
	/*	no such file or directory 顼֤롣*/
	if (fp == NULL) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}

	/*	coding-system-for-write,
	 *	buffer-file-coding-system ν֤ǽ񤭹 coding-system 
	 *	ꤹ롣
	 *	file-coding-system-alist ϥե̾ coding-system 
	 *	ꤹΤ˻Ȥ롢
	 */
	nCodingSys	= lispMachine_GetCodingSystemForWrite (pLM) ;
	if (nCodingSys < 0 || nCodingSys >= MAX_KCODING_SYSTEM)
		nCodingSys	= KCODING_SYSTEM_ISO2022JP2 ;

	InitializeKanjiFiniteStateMachine (&ksm, nCodingSys) ;

	ptr		= buffer ;
	nptr	= 0 ;
	nUsage	= nEnd - nStart ;
	/*	ࡢcoding-system ɤ褦 */
	while (nUsage -- > 0) {
		cc		= TBufStringMarker_GetChar (&mk) ;
		TBufStringMarker_Forward (&mk, 1) ;
		nctext	= RtransferKanjiFiniteStateMachine (&ksm, cc, ptr) ;
		ptr		+= nctext ;
		nptr	+= nctext ;
		if (nptr < (sizeof (buffer) - 16)) {
			fwrite (buffer, 1, nptr, fp) ;
#if defined (DEBUG)
			fwrite (buffer, 1, nptr, stdout) ;
#endif
			ptr		= buffer ;
			nptr	= 0 ;
		}
	}
	if (nptr > 0) {
		fwrite (buffer, 1, nptr, fp) ;
#if defined (DEBUG)
		fwrite (buffer, 1, nptr, stdout) ;
#endif
	}

	fclose (fp) ;

	lispMgr_CreateNil (pLispMgr, &pEntNil) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntNil) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_FileExistsp (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntFile ;
	TLispEntity*	pEntRetval ;
	char			bufFile [PATH_MAX + 1] ;
	struct stat		bufstat ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFile))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	/*	ѹȤ̾롣*/
	if (TFAILED (lispMachine_getFileNameFromEntity (pLispMgr, pEntFile, bufFile, PATH_MAX))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	
	if (lstat (bufFile, &bufstat) == -1) {
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	} else {
		lispMgr_CreateT   (pLispMgr, &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*	(set-file-modes FILENAME MODE)
 */
TLMRESULT
lispMachineState_SetFileModes (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntFILENAME ;
	TLispEntity*	pEntMODE ;
	TLispEntity*	pEntRetval ;
	char			bufFile [PATH_MAX + 1] ;
	long			lMode ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntFILENAME)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntMODE))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispMachine_getFileNameFromEntity (pLispMgr, pEntFILENAME, bufFile, PATH_MAX)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntMODE, &lMode))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	
	/*	եΥ⡼ɤѹ롣*/
	if (chmod (bufFile, lMode & 0x0FFF) != 0) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_FileWritablep (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*		pEntArglist ;
	TLispEntity*		pEntFILENAME ;
	TLispEntity*		pEntRetval ;
	char				bufFile [PATH_MAX + 1] ;
	register Boolean	fWritable ;
	struct stat			bufstat ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFILENAME)) ||
		TFAILED (lispMachine_getFileNameFromEntity (pLispMgr, pEntFILENAME, bufFile, PATH_MAX))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (stat (bufFile, &bufstat) >= 0) {
		fWritable	= (access (bufFile, 2) == 0) ;
	} else {
		const char*	pDir ;
		int			nDir, nPath ;

		/* parent directory դʤС*/
		nPath	= strlen (bufFile) ;
		(void) GetFileDirectoryAnsi (bufFile, nPath, &pDir, &nDir) ;
		if (nDir > nPath || nDir > PATH_MAX) {
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
		bufFile [nDir]	= '\0' ;
		fWritable	= (access (bufFile, 2) == 0) ;
	}
	pEntRetval	= lispMgr_GetReservedEntity (pLispMgr, (fWritable)? LISPMGR_INDEX_T : LISPMGR_INDEX_NIL) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_FileReadablep (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*		pEntArglist ;
	TLispEntity*		pEntFILENAME ;
	TLispEntity*		pEntRetval ;
	char				bufFile [PATH_MAX + 1] ;
	register Boolean	fReadable ;
	struct stat			bufstat ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFILENAME)) ||
		TFAILED (lispMachine_getFileNameFromEntity (pLispMgr, pEntFILENAME, bufFile, PATH_MAX))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (stat (bufFile, &bufstat) >= 0) {
		fReadable	= (access (bufFile, R_OK) == 0) ;
	} else {
		fReadable	= False ;
	}
	pEntRetval	= lispMgr_GetReservedEntity (pLispMgr, (fReadable)? LISPMGR_INDEX_T : LISPMGR_INDEX_NIL) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_MakeDirectoryInternal (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntDIRECTORY ;
	TLispEntity*	pEntRetval ;
	char			bufDir [PATH_MAX + 1] ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntDIRECTORY)) ||
		TFAILED (lispMachine_getFileNameFromEntity (pLispMgr, pEntDIRECTORY, bufDir, PATH_MAX))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (mkdir (bufDir, S_IRWXU) != 0) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

static char		make_temp_name_tbl [64] = {
	'A','B','C','D','E','F','G','H',
	'I','J','K','L','M','N','O','P',
	'Q','R','S','T','U','V','W','X',
	'Y','Z','a','b','c','d','e','f',
	'g','h','i','j','k','l','m','n',
	'o','p','q','r','s','t','u','v',
	'w','x','y','z','0','1','2','3',
	'4','5','6','7','8','9','-','_'
};
static unsigned	make_temp_name_count = 0, make_temp_name_count_initialized_p = 0 ;

/*	(make-temp-name PREFIX)
 */
TLMRESULT
lispMachineState_MakeTempName (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*		pEntArglist ;
	TLispEntity*		pEntPREFIX ;
	TLispEntity*		pEntRetval	= NULL ;
	const Char*			strPREFIX ;
	int					nstrPREFIX ;
	register char*		pDest ;
	register char*		strFile ;
	TVarbuffer			vbuf ;
	TVarbuffer			vbufResult ;
	register Char*		strResult ;
	register int		nResult, nfd ;
	register unsigned	num = make_temp_name_count ;
	register pid_t		pid ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	TVarbuffer_Initialize (&vbufResult, sizeof (Char)) ;
	TVarbuffer_Initialize (&vbuf,       sizeof (char)) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntPREFIX)) ||
		TFAILED (lispEntity_GetStringValue (pLispMgr, pEntPREFIX, &strPREFIX, &nstrPREFIX))) 
		goto	error ;

	if (TFAILED (TVarbuffer_Require (&vbuf, nstrPREFIX + 3 + 3 + 1)))
		goto	error ;

	/*	ࡢfile-system  coding-system ߤȤ... */
	strFile	= TVarbuffer_GetBuffer (&vbuf) ;
	pDest	= strFile ;
	while (nstrPREFIX -- > 0) {
		if (! Char_IsAscii (*strPREFIX)) 
			goto	error ;
		*pDest ++	= (char) *strPREFIX ++ ;
	}

	pid			= (int) getpid ();
	pDest [0]	= make_temp_name_tbl [pid & 63], pid >>= 6 ;
	pDest [1]	= make_temp_name_tbl [pid & 63], pid >>= 6 ;
	pDest [2]	= make_temp_name_tbl [pid & 63], pid >>= 6 ;
	pDest		+= 3 ;
	pDest [3]	= '\0' ;
	  
	if (!make_temp_name_count_initialized_p) {
		make_temp_name_count	= (unsigned) time (NULL) ;
		make_temp_name_count_initialized_p	= 1 ;
    }

	pDest [0]	= make_temp_name_tbl[num & 63], num >>= 6 ;
	pDest [1]	= make_temp_name_tbl[num & 63], num >>= 6 ;
	pDest [2]	= make_temp_name_tbl[num & 63], num >>= 6 ;

	/* Poor man's congruential RN generator.  Replace with
	   ++make_temp_name_count for debugging.  */
	make_temp_name_count += 25229 ;
	make_temp_name_count %= 225307 ;
	
	/*	եǤʤä餱NFS  tmpfile ꤹ뤳Ȥ
	 *	ʤȻפΤ... */
	nfd	= open (strFile, O_CREAT | O_SYNC | O_EXCL | O_RDWR, S_IRUSR | S_IWUSR) ;
	if (nfd < 0) {
		fprintf (stderr, "Cannot create temporary name for prefix `") ;
		lispEntity_Print (pLispMgr, pEntPREFIX) ;
		fprintf (stderr, "'\n") ;
		goto	error ;
	}
	close (nfd) ;

	nResult	= strlen (strFile) ;
	if (TFAILED (TVarbuffer_Require    (&vbufResult, nResult)))
		goto	error ;
	
	strResult	= TVarbuffer_GetBuffer (&vbufResult) ;
	strtocstr (strResult, strFile, nResult) ;
	if (TFAILED (lispMgr_CreateString (pLispMgr, strResult, nResult, &pEntRetval)))
		return	LMR_ERROR ;

  error:
	TVarbuffer_Uninitialize (&vbufResult) ;
	TVarbuffer_Uninitialize (&vbuf) ;
	if (pEntRetval != NULL) {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	} else {
		lispMachineCode_SetError (pLM) ;
	}
	return	LMR_RETURN ;
}

/*	(file-attributes FILENAME)
 *
 *	0. t for directory, string (name linked to) for symbolic link, or nil.
 *	1. Number of links to file.
 *	2. File uid.
 *	3. File gid.
 *	4. Last access time, as a list of two integers.
 *		First integer has high-order 16 bits of time, second has low 16 bits.
 *	5. Last modification time, likewise.
 *	6. Last status change time, likewise.
 *	7. Size in bytes.
 *		This is a floating point number if the size is too large for an integer.
 *	8. File modes, as a string of ten letters or dashes as in ls -l.
 *	9. t iff file's gid would change if file were deleted and recreated.
 *	10. inode number.  If inode number is larger than the Emacs integer,
 *	this is a cons cell containing two integers: first the high part,
 *	then the low 16 bits.
 *	11. Device number.  If it is larger than the Emacs integer, this is
 *	a cons cell, similar to the inode number.
 */
TLMRESULT
lispMachineState_FileAttributes (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntArglist ;
	TLispEntity*			pEntFILENAME ;
	TLispEntity*			pEntRetval ;
	char					bufFile [PATH_MAX + 1] ;
	struct stat				bufstat ;
	TLispEntity*			rEntResult [12] ;
	Char					rbufMode [10] ;
	register TLMRESULT		nRetval	= LMR_ERROR ;
	register int			i ;
	register unsigned int	uMask ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFILENAME)) ||
		TFAILED (lispMachine_getFileNameFromEntity (pLispMgr, pEntFILENAME, bufFile, PATH_MAX))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (stat (bufFile, &bufstat) < 0) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	for (i = 0 ; i < NELEMENTS (rEntResult) ; i ++)
		rEntResult [i]	= NULL ;

	if ((bufstat.st_mode & S_IFMT) == S_IFLNK ||
		(bufstat.st_mode & S_IFMT) == S_IFDIR) {
		lispMgr_CreateT   (pLispMgr, &rEntResult [0]) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &rEntResult [0]) ;
	}
	lispEntity_AddRef     (pLispMgr, rEntResult [0]) ;

	if (TFAILED (lispMgr_CreateInteger (pLispMgr, bufstat.st_nlink, &rEntResult [1])))
		goto	exit_func ;
	lispEntity_AddRef     (pLispMgr, rEntResult [1]) ;
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, bufstat.st_uid,   &rEntResult [2])))
		goto	exit_func ;
	lispEntity_AddRef     (pLispMgr, rEntResult [2]) ;
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, bufstat.st_gid,   &rEntResult [3])))
		goto	exit_func ;
	lispEntity_AddRef     (pLispMgr, rEntResult [3]) ;
	if (TFAILED (lispMgr_CreateTime    (pLispMgr, bufstat.st_atime, &rEntResult [4])))
		goto	exit_func ;
	lispEntity_AddRef     (pLispMgr, rEntResult [4]) ;
	if (TFAILED (lispMgr_CreateTime    (pLispMgr, bufstat.st_mtime, &rEntResult [5])))
		goto	exit_func ;
	lispEntity_AddRef     (pLispMgr, rEntResult [5]) ;
	if (TFAILED (lispMgr_CreateTime    (pLispMgr, bufstat.st_ctime, &rEntResult [6])))
		goto	exit_func ;
	lispEntity_AddRef     (pLispMgr, rEntResult [6]) ;

	/*	ե륵Ǥʬˤ float ˤ롣*/
	if (((long) bufstat.st_size) != bufstat.st_size) {
		if (TFAILED (lispMgr_CreateFloat   (pLispMgr, (float) bufstat.st_size,  &rEntResult [7])))
			goto	exit_func ;
	} else {
		if (TFAILED (lispMgr_CreateInteger (pLispMgr, bufstat.st_size,  &rEntResult [7])))
			goto	exit_func ;
	}
	lispEntity_AddRef     (pLispMgr, rEntResult [7]) ;

	uMask	= 1 ;
	for (i = 0 ; i < 9 ; i ++) {
		if ((bufstat.st_mode & uMask) != 0){
			rbufMode [9 - i]	= "xwr" [i % 3] ;
		} else {
			rbufMode [9 - i]	= '-' ;
		}
		uMask	<<= 1 ;
	}
	if ((bufstat.st_mode & S_ISVTX) != 0) {
		rbufMode [0]	= 't' ;
	} else {
		rbufMode [0]	= '-' ;
	}
	if ((bufstat.st_mode & S_ISUID) != 0) 
		rbufMode [9 - 6]	= (bufstat.st_mode & S_IXUSR)? 's' : 'S' ;
	if ((bufstat.st_mode & S_ISGID) != 0)
		rbufMode [9 - 3]	= (bufstat.st_mode & S_IXGRP)? 's' : 'S' ;

	if (TFAILED (lispMgr_CreateString (pLispMgr, rbufMode, 10, &rEntResult [8])))
		goto	exit_func ;
	lispEntity_AddRef     (pLispMgr, rEntResult [8]) ;

	if (bufstat.st_gid != getegid ()) {
		lispMgr_CreateT   (pLispMgr, &rEntResult [9]) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &rEntResult [9]) ;
	}
	lispEntity_AddRef     (pLispMgr, rEntResult [9]) ;

	if (TFAILED (lispMgr_CreateInteger (pLispMgr, bufstat.st_ino,   &rEntResult [10])))
		goto	exit_func ;
	lispEntity_AddRef     (pLispMgr, rEntResult [10]) ;
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, bufstat.st_dev,   &rEntResult [11])))
		goto	exit_func ;
	lispEntity_AddRef     (pLispMgr, rEntResult [11]) ;

	if (TSUCCEEDED (lispMgr_CreateList (pLispMgr, rEntResult, NELEMENTS (rEntResult), &pEntRetval))) {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
		nRetval	= LMR_RETURN ;
	}
  exit_func:
	for (i = 0 ; i < NELEMENTS (rEntResult) ; i ++) {
		if (rEntResult [i] != NULL) {
			lispEntity_Release (pLispMgr, rEntResult [i]) ;
		} else {
			break ;
		}
	}
	return	nRetval ;
}

Boolean
lispMachine_getFileNameFromEntity (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register char*			pDest,
	register int			nDest)
{
	TVarbuffer		vbufPath ;
	const Char*		strFile ;
	int				nstrFile ;
	register int	nCopy ;

	/*	ѹȤ̾롣*/
	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntity, &strFile, &nstrFile))) 
		return	False ;

	if (TFAILED (TVarbuffer_Initialize (&vbufPath, sizeof (Char))))
		return	False ;
	if (TSUCCEEDED (ExpandFileName (&vbufPath, strFile, nstrFile, NULL, 0))) {
		nCopy	= TVarbuffer_GetUsage (&vbufPath) ;
		if (nCopy >= nDest)
			nCopy	= nDest - 1 ;
		assert (nCopy >= 0) ;
		cstrtostr (pDest, TVarbuffer_GetBuffer (&vbufPath), nCopy) ;
		*(pDest + nCopy)	= '\0' ;
	}
	TVarbuffer_Uninitialize (&vbufPath) ;
	return	True ;
}

