/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <assert.h>
#include <X11/Xlib.h>
#include <X11/keysym.h>
#include "lmachinep.h"
#include "cstring.h"

static	Boolean		lispMachine_initialize		(TLispMachine*) ;
static	Boolean		lispMachine_onKeyPress		(TLispMachine*, void*, void*) ;
static	Boolean		lispMachine_onFrameResize	(TLispMachine*, void*, void*) ;
static	Boolean		lispMachine_keyEvent2lispEntity	(TLispMachine*, void*, TLispEntity**) ;
static	void		lispMachine_quitFlagToSignal	(TLispMachine*) ;

static inline	void
lispMachine_quitFlagToSignal (
	register TLispMachine*	pLM)
{
	register TLispManager*		pLispMgr	= pLM->m_pLispMgr ;
	register TLispEntity*		pEntQuitFlag ;
	register TLispEntity*		pEntInhibitQuit ;
	register TLispEntity*		pEntQuit ;
	register TLispEntity*		pEntT ;
	register TLispEntity*		pEntNil ;
	TLispEntity*	pEntValIQ ;
	TLispEntity*	pEntValQF ;

	/*	quit-flag  nil ʤ鲿⤷ʤ*/
	pEntQuitFlag	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_QUIT_FLAG) ;
	if (TFAILED (lispMachine_GetCurrentSymbolValue (pLM, pEntQuitFlag,    &pEntValQF)) ||
		TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntValQF)))
		return ;

	/*	inhibit-quit  non-nil ʤ鲿⤷ʤ*/
	pEntInhibitQuit	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_INHIBIT_QUIT) ;
	if (TFAILED (lispMachine_GetCurrentSymbolValue (pLM, pEntInhibitQuit, &pEntValIQ)) ||
		TFAILED (lispEntity_Nullp (pLispMgr, pEntValIQ)))
		return ;
	
	/*	(signal 'quit nil) ¹Ԥ롣*/
	pEntQuit		= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_QUIT) ;
	pEntT			= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_T) ;
	pEntNil			= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_NIL) ;
	lispMachineCode_SetSignal (pLM, pEntQuit, pEntT) ;
	lispMachine_SetCurrentSymbolValue (pLM, pEntQuitFlag,    pEntNil) ;
	lispMachine_SetCurrentSymbolValue (pLM, pEntInhibitQuit, pEntNil) ;
	return ;
}


/*
 *	
 */
Boolean
TLispMachine_Create (
	register TLispManager*	pLispMgr,
	register TLispMachine*	pMacParent,
	register TLispMachine** const ppLM)
{
	register TLispMachine*	pLM ;
	register int			i ;

	assert (pLispMgr != NULL) ;
	assert (ppLM != NULL) ;

	pLM	= MALLOC (sizeof (TLispMachine)) ;
	if (pLM == NULL)
		return	False ;

	pLM->m_pLispMgr				= pLispMgr ;
	pLM->m_pTarget				= NULL ;
	pLM->m_pEntException		= NULL ;
	pLM->m_pEntExceptionValue	= NULL ;
	pLM->m_pEntSignal			= NULL ;
	pLM->m_pEntSignalValue		= NULL ;
	pLM->m_uStatusFlag			= 0 ;
	if (TFAILED (Vstack_Initialize (&pLM->m_vstLispObj,    sizeof (TLispEntity *))) ||
		TFAILED (Vstack_Initialize (&pLM->m_vstNonLispObj, sizeof (TNotLispValue))) ||
		TFAILED (TVarbuffer_Initialize (&pLM->m_vbufCmdEvents, sizeof (Char)))) {
		FREE (pLM) ;
		return	False ;
	}
	for (i = 0 ; i < MAX_LISPOBJ_REGS ; i ++)
		pLM->m_apLREGS [i]	= NULL ;
	for (i = 0 ; i < MAX_NONLISPOBJ_REGS ; i ++)
		pLM->m_alVREGS [i].m_lValue	= 0 ;
	for (i = 0 ; i < NELEMENTS (pLM->m_apVariableTable) ; i ++) 
		pLM->m_apVariableTable [i]	= NULL ;
 	for (i = 0 ; i < NELEMENTS (pLM->m_apSymbol2SymbolTable) ; i ++) 
 		pLM->m_apSymbol2SymbolTable [i]	= NULL ;
	for (i = 0 ; i < NELEMENTS (pLM->m_apFunctionTable) ; i ++) 
		pLM->m_apFunctionTable [i]	= NULL ;
	for (i = 0 ; i < NELEMENTS (pLM->m_apPropertyTable) ; i ++) 
		pLM->m_apPropertyTable [i]	= NULL ;
	pLM->m_pState		= NULL ;
	pLM->m_lstBuffer	= NULL ;
	pLM->m_lstFrame		= NULL ;
	pLM->m_pCurBuffer	= NULL ;
	pLM->m_pCurWindow	= NULL ;
	pLM->m_pCurFrame	= NULL ;
	pLM->m_pEntRegMatch	= NULL ;
	pLM->m_fInteractive	= False ;
	pLM->m_pMacParent	= pMacParent ;
	*ppLM				= pLM ;
	lispMachineCode_ResetSignal    (pLM) ;
	lispMachineCode_ResetException (pLM) ;
	lispMachineCode_ClearQuitFlag  (pLM) ;
 	lispMachine_initialize (pLM) ;
	return	True ;
}

Boolean
TLispMachine_Destroy (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	register TLispEntity*	pEntity ;
	register int			i ;
	static int				srEntityOffsets []	= {
		offsetof (TLispMachine, m_pEntRegMatch),
		offsetof (TLispMachine, m_pEntSignal),
		offsetof (TLispMachine, m_pEntSignalValue),
		offsetof (TLispMachine, m_pEntException),
		offsetof (TLispMachine, m_pEntExceptionValue),
	} ;

	assert (pLM != NULL) ;

	pLispMgr		= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	for (i = 0 ; i < MAX_LISPOBJ_REGS ; i ++) {
		if (pLM->m_apLREGS [i] != NULL) { 
			lispEntity_Release (pLispMgr, pLM->m_apLREGS [i]) ;
			pLM->m_apLREGS [i]	= NULL ;
		}
	}
	for (i = 0 ; i < NELEMENTS (srEntityOffsets) ; i ++) {
		register TLispEntity**	ppEntity ;
		ppEntity	= (TLispEntity **)((unsigned char*)pLM + srEntityOffsets [i]) ;
		if (*ppEntity != NULL) {
			lispEntity_Release (pLispMgr, *ppEntity) ;
			*ppEntity	= NULL ;
		}
	}
	if (pLM->m_lstBuffer != NULL) {
		register Boolean	fExit	= False ;
		TLispEntity*		pNextBuffer	;

		pEntity	= pLM->m_lstBuffer ;
		do {
			lispBuffer_GetNext (pEntity, &pNextBuffer) ;
			if (pEntity == pNextBuffer)
				fExit	= True ;
#if defined (DEBUG) || 0
			fprintf (stderr, "RemoveBuffer: ") ;
			lispEntity_Print (pLispMgr, pEntity) ;
			fprintf (stderr, "\n") ;
#endif
			lispMachine_RemoveBuffer (pLM, pEntity) ;
			pEntity	= pNextBuffer ;
		}	while (!fExit) ;

		pLM->m_lstBuffer	= NULL ;
	}
	if (pLM->m_lstFrame != NULL) {
		TLispEntity*	pNextFrame	= pLM->m_lstFrame ;
		do {
			pEntity	= pNextFrame ;
			lispFrame_GetNext (pEntity, &pNextFrame) ;
			lispMachine_RemoveFrame (pLM, pEntity) ;
		}	while (pEntity != pNextFrame) ;

		pLM->m_lstFrame	= NULL ;
	}

	/*	Mutex 򰮤äƤǽΤǡξˤ Mutex  owner 
	 *	Ǥʤʤ롣*/
	lispMgr_AbondonMutex (pLispMgr, pLM) ;

	lispBindTable_Destroy (pLispMgr, pLM->m_apVariableTable,      SIZE_LISP_BIND_TABLE) ;
	lispBindTable_Destroy (pLispMgr, pLM->m_apSymbol2SymbolTable, SIZE_LISP_BIND_TABLE) ;
	lispBindTable_Destroy (pLispMgr, pLM->m_apFunctionTable,      SIZE_LISP_BIND_TABLE) ;
	lispBindTable_Destroy (pLispMgr, pLM->m_apPropertyTable,      SIZE_LISP_BIND_TABLE) ;
	Vstack_Uninitialize (&pLM->m_vstLispObj) ;
	Vstack_Uninitialize (&pLM->m_vstNonLispObj) ;
	TVarbuffer_Uninitialize (&pLM->m_vbufCmdEvents) ;
	FREE (pLM) ;
	return	True ;
}

TLMRESULT
TLispMachine_Test (
	register TLispMachine*	pLM,
	register TLispEntity*	pTarget)
{
	register TLMRESULT	(*pState)(TLispMachine*) ;
	register TLMRESULT	res ;

	pState				= pLM->m_pState ;
	pLM->m_uStatusFlag	= 0 ;
	lispMachineCode_Evaln (pLM, pTarget, &lispMachineState_Done) ;
	do {
		res	= lispMachine_ExecuteLoop (pLM) ; 
	}	while (res == LMR_TICK) ;
#if defined (DEBUG) || defined (DEBUG_CONSOLE)
	if (res == LMR_DONE) {
		TLispEntity*	pValue ;

		if (LISPMACHINE_SIGNALP (pLM)) {
			fprintf (stderr, "signal\n") ;
		} else {
			fprintf (stderr, "result = ") ;
			lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pValue) ;
			lispEntity_Print (pLM->m_pLispMgr, pValue) ;
			fprintf (stderr, "\n") ;
		}
	}
#endif
	assert (res == LMR_DONE) ;
	pLM->m_pState		= pState ;
	return	res ;
}

/*
 *	Lisp Machine Ф볰ϡ
 *---
 *	Ǥ X Event ꤷƤ롣¾˲Τ⤷ʤ
 *	
 */
Boolean
TLispMachine_EventProc (
	register TLispMachine*	pLM,
	register int			nEvent,
	register void*			pLPARAM,
	register void*			pRPARAM)
{
	static Boolean	(*srEventHandler [])(TLispMachine*, void*, void*)	= {
		lispMachine_onKeyPress,
		lispMachine_onFrameResize,
	} ;
	if (nEvent < 0 || nEvent >= sizeof (srEventHandler) / sizeof (srEventHandler [0]))
		return	False ;
	return	(srEventHandler [nEvent])(pLM, pLPARAM, pRPARAM) ;
}

TLispManager*
TLispMachine_GetLispManager (
	register TLispMachine*	pLM)
{
	assert (pLM != NULL) ;
	return	pLM->m_pLispMgr ;
}

#define	MAX_TICK	(6000)

/*
 */
TLMRESULT
lispMachine_ExecuteLoop (
	register TLispMachine*	pLM)
{
	register int		nCount	= MAX_TICK ;
	register TLMRESULT	res ;

	assert (pLM != NULL) ;
	assert (pLM->m_pLispMgr != NULL) ;
	assert (pLM->m_pState   != NULL) ;

	while (nCount -- > 0) {
		res	= (pLM->m_pState) (pLM) ;
		switch (res) {
		case	LMR_RETURN:
			lispMachineCode_PopState (pLM) ;
			/*	fall-down */
		case	LMR_CONTINUE:
			break ;
		case	LMR_TICK:
			nCount	= 0 ;
			break ;
		case	LMR_SUSPEND:
		case	LMR_ERROR:
		case	LMR_DONE:
		default:
			return	res ;
		}
		lispMachine_quitFlagToSignal (pLM) ;
	}
	return	LMR_TICK ;
}

Boolean
lispMachine_CheckArgument (
	register TLispMachine*		pLM,
	register TLispEntity*		pArglist,
	register LMCMDINFO const *	pProcInfo,
	register int*				pnArg)
{
	register Boolean	fRetval ;
	int		nArg ;
	
	if (TFAILED (lispEntity_CountArgument (pLM->m_pLispMgr, pArglist, &nArg)))
		return	False ;
	switch (pProcInfo->m_iArgtype) {
		/*	¡¤꤬硣*/
	case	LISPCMD_ARGTYPE_NORMAL:
		fRetval	= (pProcInfo->m_nMinArgNum <= nArg &&
				   nArg <= pProcInfo->m_nMaxArgNum)? True : False ;
		break ;

		/*	¤硣*/
	case	LISPCMD_ARGTYPE_LOWER:
		fRetval	= (0 <= nArg && pProcInfo->m_nMinArgNum <= nArg)? True : False ;
		break ;

		/*	¤硣*/
	case	LISPCMD_ARGTYPE_UPPER:
		fRetval	= (0 <= nArg && nArg <= pProcInfo->m_nMaxArgNum)? True : False ;
		break ;

		/*	ä¤򤫤ʤ硣*/
	case	LISPCMD_ARGTYPE_NOBOUND:
	case	LISPCMD_ARGTYPE_CDR:
		fRetval	= (nArg >= 0)? True : False ;
		break ;

	case	LISPCMD_ARGTYPE_SPECIAL:
	case	LISPCMD_ARGTYPE_LAMBDA:
	case	LISPCMD_ARGTYPE_MACRO:
		fRetval	= True ;
		break ;

	default:
		fRetval	= False ;
		break ;
	}
	if (pnArg != NULL)
		*pnArg	= nArg ;
	
	return	fRetval ;
}

/*
 *	Buffer  lispmachine δɲä롣Buffer ϻȤƤ SYMBOL 
 *	ʤƤ⾡ GC ˤäƾäꤷʤĤޤϡȼ˻Ȥ
 *	櫓ǡġ
 */
Boolean
lispMachine_InsertBuffer (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntBuffer)
{
	assert (pLM != NULL) ;
	assert (pEntBuffer != NULL) ;
	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;

	if (pLM->m_lstBuffer == NULL) {
		pLM->m_lstBuffer	= pEntBuffer ;
		lispBuffer_SetPrevious (pEntBuffer, pEntBuffer) ;
		lispBuffer_SetNext     (pEntBuffer, pEntBuffer) ;
	} else {
		TLispEntity*	pPrevBuffer ;

		lispBuffer_GetPrevious (pLM->m_lstBuffer,	&pPrevBuffer) ;
		assert (pPrevBuffer != NULL) ;
		lispBuffer_SetPrevious (pEntBuffer,			pPrevBuffer) ;
		lispBuffer_SetNext     (pEntBuffer,			pLM->m_lstBuffer) ;
		lispBuffer_SetPrevious (pLM->m_lstBuffer,	pEntBuffer) ;
		lispBuffer_SetNext     (pPrevBuffer,		pEntBuffer) ;
	}
	lispEntity_AddRef (pLM->m_pLispMgr, pEntBuffer) ;
	return	True ;
}	

Boolean
lispMachine_RemoveBuffer (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntBuffer)
{
	TLispEntity*	pPrevBuffer ;
	TLispEntity*	pNextBuffer ;
	assert (pLM != NULL) ;
	assert (pLM->m_lstBuffer != NULL) ;
	assert (pEntBuffer != NULL) ;
	assert (pEntBuffer->m_iType == LISPENTITY_BUFFER) ;

#if defined (DEBUG_LV99)
	if (pLM->m_lstBuffer != NULL) {
		TLispEntity*	pNode	= pLM->m_lstBuffer ;
		do {
			if (pNode == pEntBuffer)
				break ;
			lispBuffer_GetNext (pNode, &pNode) ;
		}	while (pNode != pLM->m_lstBuffer) ;

		assert (pNode == pEntBuffer) ;
	}
#endif
	lispBuffer_GetPrevious (pEntBuffer, &pPrevBuffer) ;
	assert (pPrevBuffer != NULL) ;
	lispBuffer_GetNext     (pEntBuffer, &pNextBuffer) ;
	assert (pNextBuffer != NULL) ;
	if (pPrevBuffer == pEntBuffer && pNextBuffer == pEntBuffer) {
		assert (pLM->m_lstBuffer == pEntBuffer) ;
		pLM->m_lstBuffer	= NULL ;
	} else {
		lispBuffer_SetNext     (pPrevBuffer, pNextBuffer) ;
		lispBuffer_SetPrevious (pNextBuffer, pPrevBuffer) ;
		if (pLM->m_lstBuffer == pEntBuffer) 
			pLM->m_lstBuffer	= pNextBuffer ;
	}
	lispEntity_Release (pLM->m_pLispMgr, pEntBuffer) ;
	return	True ;
}

/*
 *	Frame  lispmachine δɲä롣Buffer Ʊ٤
 *	¸ߡ
 */
Boolean
lispMachine_InsertFrame (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntFrame)
{
	assert (pLM != NULL) ;
	assert (pEntFrame != NULL) ;
	assert (pEntFrame->m_iType == LISPENTITY_FRAME) ;

	if (pLM->m_lstFrame == NULL) {
		pLM->m_lstFrame		= pEntFrame ;
		lispFrame_SetPrevious (pEntFrame, pEntFrame) ;
		lispFrame_SetNext     (pEntFrame, pEntFrame) ;
	} else {
		TLispEntity*	pPrevFrame ;

		lispFrame_GetPrevious (pLM->m_lstFrame, &pPrevFrame) ;
		assert (pPrevFrame != NULL) ;
		lispFrame_SetPrevious (pEntFrame,    pPrevFrame) ;
		lispFrame_SetNext     (pEntFrame,    pLM->m_lstFrame) ;
		lispFrame_SetPrevious (pLM->m_lstFrame, pEntFrame) ;
		lispFrame_SetNext     (pPrevFrame,      pEntFrame) ;
	}
	lispEntity_AddRef (pLM->m_pLispMgr, pEntFrame) ;
	return	True ;
}

Boolean
lispMachine_RemoveFrame (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntFrame)
{
	TLispEntity*	pPrevFrame ;
	TLispEntity*	pNextFrame ;
	assert (pLM != NULL) ;
	assert (pLM->m_lstFrame != NULL) ;
	assert (pEntFrame != NULL) ;
	assert (pEntFrame->m_iType == LISPENTITY_FRAME) ;

#if defined (DEBUG_LV99)
	if (pLM->m_lstFrame != NULL) {
		TLispEntity*	pNode	= pLM->m_lstFrame ;
		do {
			if (pNode == pEntFrame)
				break ;
			lispFrame_GetNext (pNode, &pNode) ;
		}	while (pNode != pLM->m_lstFrame) ;

		assert (pNode == pEntFrame) ;
	}
#endif
	lispFrame_GetPrevious (pEntFrame, &pPrevFrame) ;
	assert (pPrevFrame != NULL) ;
	lispFrame_GetNext     (pEntFrame, &pNextFrame) ;
	assert (pNextFrame != NULL) ;
	if (pPrevFrame == pEntFrame && pNextFrame == pEntFrame) {
		assert (pLM->m_lstFrame == pEntFrame) ;
		pLM->m_lstFrame	= NULL ;
	} else {
		lispFrame_SetNext     (pPrevFrame, pNextFrame) ;
		lispFrame_SetPrevious (pNextFrame, pPrevFrame) ;
		if (pLM->m_lstFrame == pEntFrame) 
			pLM->m_lstFrame	= pNextFrame ;
	}
	lispEntity_Release (pLM->m_pLispMgr, pEntFrame) ;
	return	True ;
}

Boolean
lispMachine_ActivateAllFrame (
	register TLispMachine*	pLM,
	register Boolean		fActivate)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntFrame ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	if (pLM->m_lstFrame == NULL) 
		return	True ;

	pEntFrame	= pLM->m_lstFrame ;
	do {
		lispFrame_Activate (pLispMgr, pEntFrame, fActivate) ;
		lispFrame_GetNext (pEntFrame, &pEntFrame) ;
	}	while (pEntFrame != pLM->m_lstFrame) ;

	return	True ;
}

Boolean
lispMachine_QueueInEvent (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntQueue,
	register TLispEntity*	pEntEvent)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntNil ;
	TLispEntity*	pEntLstEvent ;
	TLispEntity*	pEntQueueValue ;
	Boolean			fRetval ;

	/*	List ˤĤʤˡLisp Entity  List ˤ롣*/
	lispMgr_CreateNil  (pLispMgr, &pEntNil) ;
	lispEntity_AddRef  (pLispMgr, pEntEvent) ;
	fRetval	= lispMgr_CreateConscell (pLispMgr, pEntEvent, pEntNil, &pEntLstEvent) ;
	lispEntity_Release (pLispMgr, pEntEvent) ;
	if (TFAILED (fRetval))
		return	False ;

	if (TFAILED (lispMachine_GetCurrentSymbolValue (pLM, pEntQueue, &pEntQueueValue)) ||
		TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntQueueValue))) {
		lispMachine_SetCurrentSymbolValue (pLM, pEntQueue, pEntLstEvent) ;
	} else {
		register TLispEntity*	pEntNode	= pEntQueueValue ;
		TLispEntity*	pEntNextNode ;

		/*	list ֤ˤɤäƤΤĺʤޤʤ˹
		 *	٤ǸƤФ櫓ǤϤʤȻפ*/
		for ( ; ; ) {
			if (TFAILED (lispEntity_GetCdr (pLispMgr, pEntNode, &pEntNextNode)))
				return	False ;
			if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntNextNode))) {
				lispEntity_SetCdr (pLispMgr, pEntNode, pEntLstEvent) ;
				break ;
			}
			pEntNode	= pEntNextNode ;
		}
	}
	return	True ;
}

/*	Хåե̾ѤƥХåե򸡺롣
 */
Boolean
lispMachine_GetBuffer (
	register TLispMachine*	pLM,
	register const Char*	pStrName,
	register int			nStrName,
	register TLispEntity**	ppEntRetval)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	register TLispEntity*	pEntTopBuffer ;
	register TLispEntity*	pEntBuffer ;
	TLispEntity*	pEntNextBuffer ;
	TLispEntity*	pEntBufName ;
	const Char*		pBufName ;
	int				nBufName ;

	while (pLM != NULL && pLM->m_pCurBuffer != NULL) {
		pEntTopBuffer	= pLM->m_pCurBuffer ;
		pEntBuffer		= pEntTopBuffer ;
		do {
			lispBuffer_GetName (pLispMgr, pEntBuffer, &pEntBufName) ;
#if defined (DEBUG)
			if (pEntBufName != NULL) {
				fprintf (stderr, "lispMachine_GetBuffer () => ") ;
				lispEntity_Print (pLispMgr, pEntBufName) ;
				fprintf (stderr, "\n") ;
			}
#endif
			if (pEntBufName != NULL && 
				TSUCCEEDED (lispEntity_GetStringValue (pLispMgr, pEntBufName, &pBufName, &nBufName)) &&
				nStrName == nBufName &&
				!Cstrncmp (pStrName, pBufName, nStrName)) {
				*ppEntRetval	= pEntBuffer ;
				return	True ;
			}
			lispBuffer_GetNext (pEntBuffer, &pEntNextBuffer) ;
			pEntBuffer	= pEntNextBuffer ;
		}	while (pEntBuffer != pEntTopBuffer) ;

		pLM	= pLM->m_pMacParent ;
	}

	return	False ;
}

/*	Хåե˳Ƥ줿ե̾ѤƥХåե򸡺롣
 *()
 *	˥Хåե˳Ƥ줿ե¾ΥХåե˳Ƥܤʤ
 *	ɤȤΤ꤬뤬ġ
 *	δؿΤ buffer-local-variable ǤȤ(ɬ buffer-local-variable
 *	Ȥ¸ߤ) buffer-file-name ͤͿ줿ե̾ӤƤ
 *	Ǥ롣
 */
Boolean
lispMachine_GetFileBuffer (
	register TLispMachine*	pLM,
	register const Char*	pFileName,
	register int			nFileName,
	register TLispEntity**	ppEntRetval)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	register TLispEntity*	pEntBufFileName ;
	register TLispEntity*	pEntTopBuffer ;
	register TLispEntity*	pEntBuffer ;
	register Boolean			fRetval ;
	TLispEntity*	pEntNextBuffer ;
	TLispEntity*	pEntValue ;
	const Char*		pBufFileName ;
	int				nBufFileName ;

	pEntBufFileName	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_BUFFER_FILE_NAME) ;
	while (pLM != NULL && pLM->m_pCurBuffer != NULL) {
		pEntTopBuffer	= pLM->m_pCurBuffer ;
		pEntBuffer		= pEntTopBuffer ;
		do {
			fRetval	= lispBuffer_GetSymbolValue (pLispMgr, pEntBuffer, pEntBufFileName, &pEntValue) ;
#if defined (DEBUG)
			if (TSUCCEEDED (fRetval)) {
				lispEntity_Print (pLispMgr, pEntBufFileName) ;
				fprintf (stderr, "(%p) = ", pEntBuffer) ;
				lispEntity_Print (pLispMgr, pEntValue) ;
				fprintf (stderr, "\n") ;
			}
#endif
			if (TSUCCEEDED (fRetval) &&
				TSUCCEEDED (lispEntity_GetStringValue (pLispMgr, pEntValue, &pBufFileName, &nBufFileName)) &&
				nBufFileName == nFileName &&
				!Cstrncmp (pFileName, pBufFileName, nFileName)) {
				*ppEntRetval	= pEntBuffer ;
				return	True ;
			}
			lispBuffer_GetNext (pEntBuffer, &pEntNextBuffer) ;
			pEntBuffer	= pEntNextBuffer ;
		}	while (pEntBuffer != pEntTopBuffer) ;

		pLM	= pLM->m_pMacParent ;
	}

	return	False ;
}

Boolean
lispMachine_GenerateNewBufferName (
	register TLispMachine*	pLM,
	register const Char*	pStrNAME,
	register int			nStrNAME,
	register TLispEntity**	ppEntRetval)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TVarbuffer		vbufNAME ;
	TLispEntity*	pEntBuffer ;
	char			achBuffer [35] ;	/* 35  magic-numberŬˤĤ*/
	register Char*	pBufName ;
	register char*	pSrc ;
	register Char*	pDest ;
	register Char*	pModify ;
	unsigned int	nNumber ;
	register Boolean	fRetval ;

	assert (pLM != NULL) ;
	assert (pStrNAME != NULL && nStrNAME > 0) ;
	assert (ppEntRetval != NULL) ;

	if (TFAILED (TVarbuffer_Initialize (&vbufNAME, sizeof (Char))))
		return	False ;
	if (TFAILED (TVarbuffer_Add (&vbufNAME, pStrNAME, nStrNAME)) ||
		TFAILED (TVarbuffer_Require (&vbufNAME, NELEMENTS (achBuffer))))
		return	False ;

	pBufName	= TVarbuffer_GetBuffer (&vbufNAME) ;
	pModify		= pBufName + nStrNAME ;
	nNumber		= 2 ;
	while (TSUCCEEDED (lispMachine_GetBuffer (pLM, pBufName, nStrNAME, &pEntBuffer)) &&
		   nNumber != 0) {
		snprintf (achBuffer, NELEMENTS (achBuffer) - 1, "<%d>", nNumber) ;
		achBuffer [sizeof (achBuffer) - 1]	= '\0' ;
		nNumber		++ ;
		pSrc		= achBuffer ;
		pDest		= pModify ;
		while (*pSrc != '\0') 
			*pDest ++	= *pSrc ++ ;
		nStrNAME	= pDest - pModify ;
	}
	fRetval	= lispMgr_CreateString (pLispMgr, pBufName, nStrNAME, ppEntRetval) ;
	TVarbuffer_Uninitialize (&vbufNAME) ;
	return	fRetval ;
}

Boolean
lispMachine_EnumBuffer (
	register TLispMachine*	pLM,
	register Boolean		fStartCurrentBufferp,
	register Boolean		(*pEnumProc)(TLispMachine*, TLispEntity*, void*, Boolean*),
	register void*			pCaller)
{
	register TLispEntity*	pEntBuffer ;
	register TLispEntity*	pEntTop ;

	assert (pLM       != NULL) ;
	assert (pEnumProc != NULL) ;

	pEntBuffer	= (fStartCurrentBufferp)? pLM->m_pCurBuffer : pLM->m_lstBuffer ;
	pEntTop		= pEntBuffer ;
	if (pEntBuffer != NULL) {
		TLispEntity*	pEntNext ;
		Boolean			fContinue	= True ;

		do {
			if (TFAILED ((pEnumProc)(pLM, pEntBuffer, pCaller, &fContinue)))
				return	False ;
			if (TFAILED (fContinue))
				break ;
			lispBuffer_GetNext (pEntBuffer, &pEntNext) ;
			pEntBuffer	= pEntNext ;
		}	while (pEntBuffer != pEntTop) ;
	}
	return	True ;
}

/*========================================================================*
 *	private functions
 */
Boolean
lispMachine_initialize (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	static struct {
		int		m_nSrc ;
		int		m_nDest ;
	}	srEquivalentSymbols []	= {
		{	LISPMGR_INDEX_LAST_COMMAND_EVENT,	LISPMGR_INDEX_LAST_COMMAND_CHAR, },
	} ;
	register TLispEntity*	pEntSrc ;
	register TLispEntity*	pEntDest ;
	register int			i ;
	register TLispBind**	pBindTable ;
	TLispBind*				pBind ;

	/*	̾ƱưܥϿ롣
	 */
	pLispMgr	= pLM->m_pLispMgr ;
	pBindTable	= pLM->m_apSymbol2SymbolTable ;
	for (i = 0 ; i < NELEMENTS (srEquivalentSymbols) ; i ++) {
		pEntSrc		= lispMgr_GetReservedEntity (pLispMgr, srEquivalentSymbols [i].m_nSrc) ;
		pEntDest	= lispMgr_GetReservedEntity (pLispMgr, srEquivalentSymbols [i].m_nDest) ;
		if (TFAILED (lispBindTable_MakeEntry (pLispMgr, pBindTable, NELEMENTS (pLM->m_apSymbol2SymbolTable), pEntSrc, &pBind)))
			return	False ;
		if (TFAILED (lispBind_SetValue (pLispMgr, pBind, pEntDest)))
			return	False ;
	}
	return	True ;
}

/*	ʸɤ꿶ʤäʸɤϳ꿶줿ɤ⡢Ʊ
 *	˲Ƥ륭(status)ʸɤ˱ƶʤäˤϡüʥ
 *	ˤɬפ롣
 */
Boolean
lispMachine_onKeyPress (
	register TLispMachine*	pLM,
	register void*			pLPARAM,
	register void*			pRPARAM)
{
	register TLispManager*	pLispMgr ;
	register XEvent*		pEvent	= (XEvent *)pLPARAM ;
	TLispEntity*	pEntEvent ;
	TLispEntity*	pEntQueue ;

	assert (pLM     != NULL) ;
	assert (pLPARAM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	if (pEvent->type != KeyPress)
		return	True ;

	/*	XEvent ʸɤѴ롣ϵȤ Window System
	 *	¸ʾǤꡢlispmachine ڤΥä⤦
	 *	ˤʤ
	 *
	 *	Modifier Key ΤߤξˤϼΤƤ롣
	 */
	if (TFAILED (lispMachine_keyEvent2lispEntity (pLM, pEvent, &pEntEvent)))
		return	True ;
#if defined (DEBUG)
	fprintf (stderr, "key-in: ") ;
	lispEntity_Print (pLispMgr, pEntEvent) ;
	fprintf (stderr, "\n") ;
#endif

	/*	 unread-command-events ɲäȤȡ
	 *	Event  Command Loop ߤʤ...Ȥ󤬤롣
	 *	Ǥʤ unread-command-events ν롣
	 */
	pEntQueue	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_UNREAD_COMMAND_EVENTS) ;
	return	lispMachine_QueueInEvent (pLM, pEntQueue, pEntEvent) ;
}

Boolean
lispMachine_onFrameResize (
	register TLispMachine*	pLM,
	register void*			pLPARAM,
	register void*			pRPARAM)
{
	register TLispEntity*		pEntFrame	= (TLispEntity *)pLPARAM ;
	register const XRectangle*	pRect		= (const XRectangle *)pRPARAM ;
	
	assert (pLM != NULL) ;
	assert (pEntFrame != NULL) ;
	return	lispMachine_ResizeFrame (pLM, pEntFrame, pRect) ;
}	

Boolean
lispMachine_keyboardSignal (
	register TLispMachine*	pLM)
{
	register TLispManager*		pLispMgr ;
	register TLispEntity*		pEntQuitFlag ;
	register TLispEntity*		pEntInhibitQuit ;
	register TLispEntity*		pEntT ;
	TLispEntity*	pEntValue ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	/*	command-loop ˤˤϡ⤷ʤ
	 */
	if (pLM->m_pState == &lispMachineState_WindowProc) 
		return	False ;

	pEntInhibitQuit	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_INHIBIT_QUIT) ;
	pEntQuitFlag	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_QUIT_FLAG) ;
	pEntT			= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_T) ;

	lispMachine_SetCurrentSymbolValue (pLM, pEntQuitFlag, pEntT) ;
	if (TFAILED (lispMachine_GetCurrentSymbolValue (pLM, pEntInhibitQuit, &pEntValue)) ||
		TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntValue))) {
		register TLispEntity*	pEntQuit ;
		pEntQuit		= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_QUIT) ;
		lispMachineCode_SetSignal (pLM, pEntQuit, pEntT) ;
	}
	return	True ;
}

Boolean
lispMachine_keyEvent2lispEntity (
	register TLispMachine*	pLM,
	register void*			pvEvent,
	register TLispEntity**	ppEntRetval)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	register XEvent*		pXEvent		= (XEvent *)pvEvent ;
	register XKeyEvent*		pEvent ;
	register Boolean		fRetval ;
	TLispEntity*		pEntKeyValue ;
	char				aBuf [4] ;
	KeySym				keysym ;
	Char				cc ;
	register int		nKey ;

	assert (pXEvent != NULL) ;
	pEvent	= &pXEvent->xkey ;

	/*	ʸɤ꿶ʤäʸɤϳ꿶줿ɤ⡢Ʊ
	 *	˲Ƥ륭(status)ʸɤ˱ƶʤäˤϡüʥ
	 *	ˤɬפ롣
	 */
	nKey	= XLookupString (pEvent, aBuf, NELEMENTS (aBuf), &keysym, NULL) ;
	if (IsModifierKey (keysym))
		return	False ;

	if (nKey <= 0) {
		char					bufKEY [256] ;
		register unsigned int	uState	= pEvent->state ;

		/*	ʸɤ꿶ʤäν
		 *	ľ KeySym  Status  Char 뤷ʤ
		 *
		 *	ǡstatus  8bit  keysym  16bit Ȥ֤
		 *	롣X11  header 򸫤¤ˤƤϡkeysym  24bit ɬ
		 *	ɡġ
		 */
		if (TFAILED (keysym2string (bufKEY, NELEMENTS (bufKEY), keysym, uState))) {
			cc	= Char_Make (KCHARSET_XKEYSYM, (uState << 16) | keysym) ;
		} else {
			register int	nKeyStr	= strlen (bufKEY) ;

			if (TFAILED (lispMgr_InternSymbolA (pLispMgr, bufKEY, nKeyStr, &pEntKeyValue)))
				return	False ;
			goto	exit_func ;
		}
	} else if (pEvent->state != 0) {
		/*	state ⡢caps äƤƤϤʤΤcontrol 
		 * ƤϤʤΤȤХ롣⤷forward 줿
		 *	caps + Υüʰ̣ƤȡޤҤɤ
		 *	ˤʤ
		 */
		register unsigned int	uState		= 0 ;
		register unsigned int	uMask ;
		register char			chOrig		= aBuf [0] ;
		register int			i ;

		/*	signal äΤʤ顢Ϥǽλ*/
		if (aBuf [0] == 0x07 && nKey == 1 &&
			TSUCCEEDED (lispMachine_keyboardSignal (pLM)))
			return	False ;
		
		for (i = ShiftMapIndex ; i <= Mod5MapIndex ; i ++) {
			uMask	= 1 << i ;
			if (pEvent->state & uMask) {
				pEvent->state	&= ~uMask ;
				nKey	= XLookupString (pEvent, aBuf, NELEMENTS (aBuf), &keysym, NULL) ;

				/*	Ƥޤ State ȴФkeysym ξȤƤ
				 *	ޤΤǡȽˤϻȤʤ
				 */
				if (nKey > 0 && aBuf [0] == chOrig) 
					uState	|= uMask ;
				pEvent->state	|= uMask ;
			}
		}
		cc	= Char_Make ((uState != 0)? KCHARSET_XCHAR : KCHARSET_ASCII, (uState << 16) | (unsigned char)chOrig) ;
	} else {
		cc	= Char_MakeAscii (aBuf [0]) ;
	}
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, (long)cc, &pEntKeyValue)))
		return	False ;
  exit_func:
	lispEntity_AddRef (pLispMgr, pEntKeyValue) ;
	fRetval	= lispMgr_CreateXEvent (pLispMgr, pEntKeyValue, pXEvent, ppEntRetval) ;
	lispEntity_Release (pLispMgr, pEntKeyValue) ;
	return	fRetval ;
}

Char
lispMachine_lispKeyEventSymbol2Char (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntKey)
{
	const Char*		pStrSymName ;
	int				nStrSymName ;
	KeySym			keySym ;
	unsigned int	uState ;

	if (TFAILED (lispEntity_GetSymbolName (pLispMgr, pEntKey, &pStrSymName, &nStrSymName)))
		return	(Char) -1 ;

	if (TFAILED (cstring2keysym (pStrSymName, nStrSymName, &keySym, &uState)))
		return	(Char) -1 ;

	return	Char_Make (KCHARSET_XKEYSYM, (uState << 16) | keySym) ;
}

/*	for debugging
 */
#if defined (DEBUG)
Boolean
lispMachine_ShowRegisterValue (TLispMachine* pLM)
{
	const char*	apName [MAX_LISPOBJ_REGS]	= {
		"ACC",  "REG1", "REG2", "REG3", "REG4",
		"REG5", "REG6", "REG7", "REG8",
	} ;
	int		i ;
	TLispEntity*	pReg ;
	
	for (i = LM_LREG_ACC ; i < MAX_LISPOBJ_REGS ; i ++) {
		lispMachineCode_GetLReg (pLM, i, &pReg) ;
		fprintf (stderr, "%s = ", apName [i - LM_LREG_ACC]) ;
		if (pReg != NULL) {
			lispEntity_Print (pLM->m_pLispMgr, pReg) ;
		} else {
			fprintf (stderr, "(null)") ;
		}
		fprintf (stderr, "\n") ;
	}
	return	True ;
}
#endif

