/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <assert.h>
#include "lispmgrp.h"
#include "cstring.h"

#if defined (LBIND_COLORTREE)
enum {
	COLOR_RED	= 0,
	COLOR_BLACK	= 1,
} ;

static	Boolean		lispBind_destroyTree (TLispManager*, TLispBind** const) ;
static	void		lispBindTree_rbInsert (TLispBind** const, TLispBind*) ;
static	TLispBind*	lispBindTree_rbDelete (TLispBind** const, TLispBind*) ;
#if defined (DEBUG)
static	Boolean		lispBindTree_check	  (TLispBind** const) ;
#endif
#endif

static	inline	int		lispBindTable_createHashkey (
	register TLispEntity*		pEntSymbol,
	register int				nTableSize)
{
	return	((unsigned)(pEntSymbol) * 13) % nTableSize ;
//	return	((unsigned)(pEntSymbol) >> 3) % nTableSize ;
}

static	inline TLispBind*
lispBind_Search (
	register TLispBind*			pNode,
	register TLispEntity*		pSymbol)
{
	register TLispBind*	pRetvalue ;
	register int		nCompare ;

	assert (pSymbol != NULL) ;

	pRetvalue	= NULL ;
	while (pNode != NULL) {
		nCompare	= (int)pNode->m_pEntSymbol - (int)pSymbol ;
		if (nCompare == 0) {
			pRetvalue	= pNode ;
			break ;
		} else if (nCompare < 0) {
			pNode		= pNode->m_pRight ;
		} else {
			pNode		= pNode->m_pLeft ;
		}
	}
	return	pRetvalue ;
}

static inline TLispBind*
lispBindTable_searchEntry (
	register TLispManager*		pLispMgr,
	register TLispBind** const	ppTable,
	register int				nTableSize,
	register TLispEntity*		pSymbol)
{
	register int			iHashkey ;

	iHashkey	= lispBindTable_createHashkey (pSymbol, nTableSize) ;
	assert (0 <= iHashkey && iHashkey < nTableSize) ;
	return	lispBind_Search (*(ppTable + iHashkey), pSymbol) ;
}
  
Boolean
lispBind_Create (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntSymbol,
	register TLispBind**	ppRetBind)
{
	register TLispBind*	pBind ;

	assert (pEntSymbol != NULL) ;
	assert (ppRetBind  != NULL);

	pBind		= MALLOC (sizeof (TLispBind)) ;
	if (pBind == NULL)
		return	False ;

	pBind->m_pEntValue	= NULL ;
	pBind->m_pEntSymbol	= pEntSymbol ;
	lispEntity_AddRef (pLispMgr, pEntSymbol) ;
	pBind->m_pLeft		= NULL ;
	pBind->m_pRight		= NULL ;
#if defined (LBIND_COLORTREE)
	pBind->m_pParent	= NULL ;
	pBind->m_nColor		= COLOR_RED ;
#endif
	*ppRetBind			= pBind ;
	return	True ;
}

Boolean
lispBind_Destroy (
	register TLispManager*	pLispMgr,
	register TLispBind*		pBind)
{
	/*	Unregister Ϥʤ*/
	if (pBind->m_pEntSymbol != NULL) {
		lispEntity_Release (pLispMgr, pBind->m_pEntSymbol) ;
		pBind->m_pEntSymbol	= NULL ;
	}
	if (pBind->m_pEntValue != NULL) {
		lispEntity_Release (pLispMgr, pBind->m_pEntValue) ;
		pBind->m_pEntValue	= NULL ;
	}
	pBind->m_pLeft		= NULL ;
	pBind->m_pRight		= NULL ;
#if defined (LBIND_COLORTREE)
	pBind->m_pParent	= NULL ;
#endif
	FREE (pBind) ;
	return	True ;
}

Boolean
lispBind_GetValue (
	register TLispBind*				pBind,
	register TLispEntity** const	ppReturn)
{
	assert (pBind    != NULL) ;
	assert (ppReturn != NULL) ;

	/*	Ǥ reference count 䤵ʤƤӽФ¦Ǥ
	 *	Ĥȡ*/
	*ppReturn	= pBind->m_pEntValue ;
	return	True ;
}

Boolean
lispBind_SetValue (
	register TLispManager*	pLispMgr,
	register TLispBind*		pBind,
	register TLispEntity*	pValue)
{
	assert (pLispMgr != NULL) ;
	assert (pBind    != NULL);
	assert (pValue   != NULL) ;

	if (pBind->m_pEntValue != NULL) 
		lispEntity_Release (pLispMgr, pBind->m_pEntValue) ;

	pBind->m_pEntValue	= pValue ;
	if (pValue != NULL)
		lispEntity_AddRef (pLispMgr, pValue) ;
	return	True ;
}

Boolean
lispBind_Register (
	register TLispBind** const	ppRootOfTree,
	register TLispBind*			pNewBind)
{
	register TLispBind*		pNode ;
	register TLispEntity*	pEntSymbol ;
	register int			nCompare ;

	assert (ppRootOfTree != NULL) ;
	assert (pNewBind     != NULL) ;

	pNode	= *ppRootOfTree ;
	if (pNode == NULL) {
		*ppRootOfTree		= pNewBind ;
#if defined (LBIND_COLORTREE)
		pNewBind->m_pParent	= NULL ;
		lispBindTree_rbInsert (ppRootOfTree, pNewBind) ;
#if defined (DEBUG)
		assert (lispBindTree_check (ppRootOfTree)) ;
#endif
#endif
		return	True ;
	} 
	pEntSymbol	= pNewBind->m_pEntSymbol ;
	while (pNode != NULL) {
		nCompare	= (int)pNode->m_pEntSymbol - (int)pEntSymbol ;
		if (nCompare == 0) {
			return	False ;
		} else if (nCompare < 0) {
			if (pNode->m_pRight == NULL) {
				pNode->m_pRight		= pNewBind ;
#if defined (LBIND_COLORTREE)
				pNewBind->m_pParent	= pNode ;
#if defined (DEBUG)
				assert (lispBindTree_check (ppRootOfTree)) ;
#endif
				lispBindTree_rbInsert (ppRootOfTree, pNewBind) ;
#if defined (DEBUG)
				assert (lispBindTree_check (ppRootOfTree)) ;
#endif
#endif
				return	True ;
			} else {
				pNode			= pNode->m_pRight ;
			}
		} else {
			if (pNode->m_pLeft  == NULL) {
				pNode->m_pLeft		= pNewBind ;
#if defined (LBIND_COLORTREE)
				pNewBind->m_pParent	= pNode ;
#if defined (DEBUG)
				assert (lispBindTree_check (ppRootOfTree)) ;
#endif
				lispBindTree_rbInsert (ppRootOfTree, pNewBind) ;
#if defined (DEBUG)
				assert (lispBindTree_check (ppRootOfTree)) ;
#endif
#endif
				return	True ;
			} else {
				pNode			= pNode->m_pLeft ;
			}
		}
	}
	return	False ;
}

TLispBind*
lispBind_Unregister (
	register TLispBind** const	ppRootOfTree,
	register TLispBind*			pBind)
{
	assert (ppRootOfTree != NULL) ;
	assert (pBind != NULL) ;
#if defined (LBIND_COLOTREE)
	return	lispBindTree_rbDelete (ppRootOfTree, pBind) ;
#else
	return	NULL ;
#endif
}

Boolean
lispBind_destroyTree (
	register TLispManager*		pLispMgr,
	register TLispBind** const	ppTable)
{
	register TLispBind*	pBind ;

	assert (pLispMgr != NULL) ;
	assert (ppTable  != NULL) ;

	pBind	= *ppTable ;
	if (pBind == NULL)
		return	True ;
	if (pBind->m_pLeft != NULL) 
		lispBind_destroyTree (pLispMgr, &pBind->m_pLeft) ;
	assert (pBind->m_pLeft == NULL) ;
	if (pBind->m_pRight != NULL) 
		lispBind_destroyTree (pLispMgr, &pBind->m_pRight) ;
	assert (pBind->m_pRight == NULL) ;

	lispBind_Destroy (pLispMgr, pBind) ;
	*ppTable	= NULL ;
	return	True ;
}

Boolean
lispBindTable_Destroy (
	register TLispManager*		pLispMgr,
	register TLispBind** const	ppTable,
	register int				nTableSize)
{
	register int	i ;
	for (i = 0 ; i < nTableSize ; i ++)
		lispBind_destroyTree (pLispMgr, &ppTable [i]) ;
	return	True ;
}

/*
 *	Symbol - Value  Bind Table ˿ Symbol (Entry) ɲä롣
 */
Boolean
lispBindTable_MakeEntry (
	register TLispManager*		pLispMgr,
	register TLispBind** const	ppTable,
	register int				nTableSize,
	register TLispEntity*		pSymbol,
	register TLispBind** const	ppBindReturn)
{
	TLispBind*		pBind ;
	TLispEntity*	pVoid ;
	register int	iHashkey ;

	assert (pLispMgr != NULL) ;
	assert (ppTable  != NULL && nTableSize > 0) ;
	assert (pSymbol  != NULL) ;
	
	pBind	= lispBindTable_searchEntry (pLispMgr, ppTable, nTableSize, pSymbol) ;
	if (pBind != NULL) {
		*ppBindReturn	= pBind ;
		return	True ;
	}
	if (TFAILED (lispBind_Create (pLispMgr, pSymbol, &pBind)))
		return	False ;

	iHashkey	= lispBindTable_createHashkey (pSymbol, nTableSize) ;
	assert (0 <= iHashkey && iHashkey < nTableSize) ;
	if (TFAILED (lispBind_Register (&ppTable [iHashkey], pBind)))
		return	False ;

	/*	ͤ void ˤ롣*/
	(void) lispMgr_CreateVoid (pLispMgr, &pVoid) ;
	lispBind_SetValue (pLispMgr, pBind, pVoid) ;

	if (ppBindReturn != NULL)
		*ppBindReturn	= pBind ;
	return	True ;
}

Boolean
lispBindTable_SetEntryValue (
	register TLispManager*		pLispMgr,
	register TLispBind** const	ppTable,
	register int				nTableSize,
	register TLispEntity*		pSymbol,
	register TLispEntity*		pValue)
{
	register TLispBind*	pBind ;

	pBind	= lispBindTable_searchEntry (pLispMgr, ppTable, nTableSize, pSymbol) ;
	if (pBind == NULL)
		return	False ;
	(void) lispBind_SetValue (pLispMgr, pBind, pValue) ;
	return	True ;
}

Boolean
lispBindTable_GetEntryValue (
	register TLispManager*			pLispMgr,
	register TLispBind** const		ppTable,
	register int					nTableSize,
	register TLispEntity*			pSymbol,
	register TLispEntity** const	ppValueReturn)
{
	register TLispBind*		pBind ;
	TLispEntity*	pReturn ;

	assert (pLispMgr != NULL) ;
	assert (ppTable  != NULL && nTableSize > 0) ;
	assert (pSymbol  != NULL) ;
	assert (ppValueReturn != NULL) ;
	/*assert (hValue  != NULL) ;*/

	pBind	= lispBindTable_searchEntry (pLispMgr, ppTable, nTableSize, pSymbol) ;
	if (pBind == NULL)
		return	False ;

	if (TFAILED (lispBind_GetValue (pBind, &pReturn)) ||
		pReturn == NULL)
		return	False ;
	
	*ppValueReturn	= pReturn ;
	return	True ;
}

Boolean
lispBindTable_SearchEntry (
	register TLispManager*		pLispMgr,
	register TLispBind** const	ppTable,
	register int				nTableSize,
	register TLispEntity*		pSymbol,
	register TLispBind** const	ppReturn)
{
	register TLispBind*	pBind ;

	pBind	= lispBindTable_searchEntry (pLispMgr, ppTable, nTableSize, pSymbol) ;
	if (ppReturn != NULL)
		*ppReturn	= pBind ;
	return	True ;
}

#if defined (LBIND_COLORTREE)
/*========================================================================
 */
#define	GETLEFT(pNode)			(((pNode) != NULL)? (pNode)->m_pLeft : NULL)
#define	SETLEFT(pNode,pChild)	{ if ((pNode) != NULL) (pNode)->m_pLeft  = (pChild) ; }
#define	GETRIGHT(pNode)			(((pNode) != NULL)? (pNode)->m_pRight : NULL)
#define	SETRIGHT(pNode,pChild)	{ if ((pNode) != NULL) (pNode)->m_pRight = (pChild) ; }
#define	GETCOLOR(pNode)			(((pNode) != NULL)? (pNode)->m_nColor : COLOR_BLACK)
#define	SETCOLOR(pNode,nColor)	{ if ((pNode) != NULL) (pNode)->m_nColor = (nColor) ; }
#define	GETPARENT(pNode)		(((pNode) != NULL)? (pNode)->m_pParent : NULL)
#define	SETPARENT(pNode,pParent) { if ((pNode) != NULL) (pNode)->m_pParent = (pParent) ; }

static	void	lispBindTree_rbDeleteFixup	(TLispBind** const, TLispBind*) ;
static	void	lispBindTree_leftRotate		(TLispBind** const, TLispBind*) ;
static	void	lispBindTree_rightRotate	(TLispBind** const, TLispBind*) ;
#if defined (DEBUG)
static	Boolean	lispBindTree_checkSub		(TLispBind*, int) ;
#endif

void
lispBindTree_leftRotate (
	register TLispBind** const	ppRoot,
	register TLispBind*			pNodeX)
{
	register TLispBind*	pParentOfNodeX ;
	register TLispBind*	pNodeY ;
	register TLispBind*	pNodeYL ;

	assert (pNodeX  != NULL) ;
	pParentOfNodeX		= GETPARENT (pNodeX) ;
	pNodeY				= GETRIGHT  (pNodeX) ;
	assert (pNodeY  != NULL) ;
	pNodeYL				= GETLEFT   (pNodeY) ;
	SETRIGHT  (pNodeX,  pNodeYL) ;
	SETPARENT (pNodeYL, pNodeX) ;
	SETPARENT (pNodeY,  pParentOfNodeX) ;
	if (pParentOfNodeX == NULL) {
		*ppRoot				= pNodeY ;
	} else {
		assert (pParentOfNodeX != NULL) ;
		if (pNodeX == pParentOfNodeX->m_pLeft) {
			pParentOfNodeX->m_pLeft		= pNodeY ;
		} else {
			assert (pNodeX == pParentOfNodeX->m_pRight) ;
			pParentOfNodeX->m_pRight	= pNodeY ;
		}
	}
	SETLEFT   (pNodeY, pNodeX) ;
	SETPARENT (pNodeX, pNodeY) ;
	return ;
}

void
lispBindTree_rightRotate (
	register TLispBind** const	ppRoot,
	register TLispBind*			pNodeY)
{
	register TLispBind*	pParentOfNodeY ;
	register TLispBind*	pNodeX ;
	register TLispBind*	pNodeXR ;

	assert (pNodeY  != NULL) ;
	pParentOfNodeY		= GETPARENT (pNodeY) ;
	pNodeX				= GETLEFT   (pNodeY) ;
	assert (pNodeX  != NULL) ;
	pNodeXR				= GETRIGHT  (pNodeX) ;
	SETLEFT   (pNodeY,	pNodeXR) ;
	SETPARENT (pNodeXR,	pNodeY) ;
	SETPARENT (pNodeX,	pParentOfNodeY) ;
	if (pParentOfNodeY == NULL) {
		*ppRoot			= pNodeX ;
	} else {
		if (pNodeY == pParentOfNodeY->m_pLeft) {
			pParentOfNodeY->m_pLeft		= pNodeX ;
		} else {
			assert (pNodeY == pParentOfNodeY->m_pRight) ;
			pParentOfNodeY->m_pRight	= pNodeX ;
		}
	}
	SETRIGHT  (pNodeX, pNodeY) ;
	SETPARENT (pNodeY, pNodeX) ;
	return ;
}

void
lispBindTree_rbInsert (
	register TLispBind** const		ppRoot,
	register TLispBind*				pNodeX)
{
	register TLispBind*	pPX ;
	register TLispBind*	pPPX ;
	register TLispBind*	pNodeY ;

	SETCOLOR (pNodeX, COLOR_RED) ;
	for ( ; ; ) {
		pPX 	= GETPARENT (pNodeX) ;
		if (pPX == NULL || GETCOLOR (pPX) != COLOR_RED) 
			break ;
		pPPX	= GETPARENT (pPX) ;
		if (pPPX == NULL)
			break ;
		if (pPX == GETLEFT (pPPX)) {
			pNodeY	= GETRIGHT (pPPX) ;
			if (GETCOLOR (pNodeY) == COLOR_RED) {
				SETCOLOR (pPX,    COLOR_BLACK) ;
				SETCOLOR (pNodeY, COLOR_BLACK) ;
				SETCOLOR (pPPX,   COLOR_RED) ;
				pNodeX				= pPPX ;
			} else {
				if (pNodeX == GETRIGHT (pPX)) {
					pNodeX			= pPX ;
					lispBindTree_leftRotate (ppRoot, pNodeX) ;
					pPX				= GETPARENT (pNodeX) ;
					pPPX			= GETPARENT (pPX) ;
				}
				SETCOLOR (pPX,  COLOR_BLACK) ;
				SETCOLOR (pPPX, COLOR_RED) ;
				lispBindTree_rightRotate (ppRoot, pPPX) ;
			}
		} else {
			assert (pPX == GETRIGHT (pPPX)) ;
			pNodeY	= GETLEFT (pPPX) ;
			if (GETCOLOR (pNodeY) == COLOR_RED) {
				SETCOLOR (pPX,    COLOR_BLACK) ;
				SETCOLOR (pNodeY, COLOR_BLACK) ;
				SETCOLOR (pPPX,   COLOR_RED) ;
				pNodeX				= pPPX ;
			} else {
				if (pNodeX == GETLEFT (pPX)) {
					pNodeX			= pPX ;
					lispBindTree_rightRotate (ppRoot, pNodeX) ;
					pPX				= GETPARENT (pNodeX) ;
					pPPX			= GETPARENT (pPX) ;
				}
				SETCOLOR (pPX,  COLOR_BLACK) ;
				SETCOLOR (pPPX, COLOR_RED) ;
				lispBindTree_leftRotate (ppRoot, pPPX) ;
			}
		}
	}
	SETCOLOR (*ppRoot, COLOR_BLACK) ;
	return ;
}

TLispBind*
lispBindTree_rbDelete (
	register TLispBind** const	ppRoot,
	register TLispBind*			pNodeZ)
{
	register TLispBind*	pNodeX ;
	register TLispBind*	pNodeY ;

	if (GETLEFT (pNodeZ) == NULL || GETRIGHT (pNodeZ) == NULL) {
		pNodeY	= pNodeZ ;
	} else {
		pNodeY	= GETRIGHT (pNodeZ) ;
		if (pNodeY != NULL) {
			while (pNodeY->m_pLeft != NULL)
				pNodeY	= pNodeY->m_pLeft ;
		} else {
			register TLispBind*	pNode	= pNodeZ ;

			pNodeY	= GETPARENT (pNode) ;
			while (pNodeY != NULL && pNode == GETRIGHT (pNodeY)) {
				pNode	= pNodeY ;
				pNodeY	= GETPARENT (pNodeY) ;
			}
		}
	}
	pNodeX	= GETLEFT (pNodeY) ;
	if (pNodeX == NULL) 
		pNodeX	= GETRIGHT (pNodeY) ;
	SETPARENT (pNodeX, GETPARENT (pNodeY)) ;
	if (GETPARENT (pNodeY) == NULL) {
		*ppRoot	= pNodeX ;
	} else {
		if (pNodeY == GETLEFT (GETPARENT (pNodeY))) {
			SETLEFT  (GETPARENT (pNodeY), pNodeX) ;
		} else {
			SETRIGHT (GETPARENT (pNodeY), pNodeX) ;
		}
	}
	if (pNodeY != pNodeZ) {
		pNodeZ->m_pEntSymbol	= pNodeY->m_pEntSymbol ;
		pNodeZ->m_pEntValue		= pNodeY->m_pEntValue ;
		pNodeY->m_pEntSymbol	= NULL ;
		pNodeY->m_pEntValue		= NULL ;
	}
	if (GETCOLOR (pNodeY) == COLOR_BLACK)
		lispBindTree_rbDeleteFixup (ppRoot, pNodeX) ;
	return	pNodeY ;
}

void
lispBindTree_rbDeleteFixup (
	register TLispBind** const	ppRoot,
	register TLispBind*			pNodeX)
{
	register TLispBind*	pPX ;
	register TLispBind*	pNodeW ;
	register TLispBind*	pNodeWL ;
	register TLispBind*	pNodeWR ;

	while (pNodeX != *ppRoot && GETCOLOR (pNodeX) == COLOR_BLACK) {
		pPX	= GETPARENT (pNodeX) ;
		if (pNodeX == GETLEFT (pPX)) {
			pNodeW	= GETRIGHT (pPX) ;
			if (GETCOLOR (pNodeW) == COLOR_RED) {
				SETCOLOR (pNodeW, COLOR_BLACK) ;
				SETCOLOR (pPX,    COLOR_RED) ;
				lispBindTree_leftRotate (ppRoot, pPX) ;
				pPX		= GETPARENT (pNodeX) ;
				pNodeW	= GETRIGHT  (pPX) ;
			}
			pNodeWL	= GETLEFT  (pNodeW) ;
			pNodeWR	= GETRIGHT (pNodeW) ;
			if (GETCOLOR (pNodeWL) == COLOR_BLACK &&
				GETCOLOR (pNodeWR) == COLOR_BLACK) {
				SETCOLOR (pNodeW, COLOR_RED) ;
				pNodeX	= GETPARENT (pNodeX) ;
			} else {
				if (GETCOLOR (pNodeWR) == COLOR_BLACK) {
					SETCOLOR (pNodeWL,  COLOR_BLACK) ;
					SETCOLOR (pNodeW,   COLOR_RED) ;
					lispBindTree_rightRotate (ppRoot, pNodeW) ;
					pPX		= GETPARENT (pNodeX) ;
					pNodeW	= GETRIGHT  (pPX) ;
					pNodeWR	= GETRIGHT  (pNodeW) ;
				}
				SETCOLOR (pNodeW,  GETCOLOR (pPX)) ;
				SETCOLOR (pPX,     COLOR_BLACK) ;
				SETCOLOR (pNodeWR, COLOR_BLACK) ;
				lispBindTree_leftRotate (ppRoot, pPX) ;
				pNodeX		= *ppRoot ;
			}
		} else {
			pNodeW	= GETLEFT (pPX) ;
			if (GETCOLOR (pNodeW) == COLOR_RED) {
				SETCOLOR (pNodeW, COLOR_BLACK) ;
				SETCOLOR (pPX,    COLOR_RED) ;
				lispBindTree_leftRotate (ppRoot, pPX) ;
				pPX		= GETPARENT (pNodeX) ;
				pNodeW	= GETRIGHT  (pPX) ;
			}
			pNodeWL		= GETLEFT  (pNodeW) ;
			pNodeWR		= GETRIGHT (pNodeW) ;
			if (GETCOLOR (pNodeWL) == COLOR_BLACK &&
				GETCOLOR (pNodeWR) == COLOR_BLACK) {
				SETCOLOR (pNodeW, COLOR_RED) ;
				pNodeX	= GETPARENT (pNodeX) ;
			} else {
				if (GETCOLOR (pNodeWR) == COLOR_BLACK) {
					SETCOLOR (pNodeWL, COLOR_BLACK) ;
					SETCOLOR (pNodeW,  COLOR_RED) ;
					lispBindTree_rightRotate (ppRoot, pNodeW) ;
					pPX		= GETPARENT (pNodeX) ;
					pNodeW	= GETRIGHT (pPX) ;
					pNodeWR	= GETRIGHT (pNodeW) ;
				}
				SETCOLOR (pNodeW,  GETCOLOR (pPX)) ;
				SETCOLOR (pPX,     COLOR_BLACK) ;
				SETCOLOR (pNodeWR, COLOR_BLACK) ;
				lispBindTree_leftRotate (ppRoot, pPX) ;
				pNodeX	= *ppRoot ;
			}
		}
	}
	SETCOLOR (pNodeX, COLOR_BLACK) ;
	return ;
}

#if defined (DEBUG)
/*	ڤξ֤ݤƤ뤫å롣
 */
Boolean
lispBindTree_check (
	register TLispBind** const		ppRootOfTree)
{
	register TLispBind*	pNode		= *ppRootOfTree ;
	register int		nRootKey ;

	if (pNode == NULL)
		return	True ;
	nRootKey	= (((unsigned)pNode->m_pEntSymbol) * 13) % 37 ;
	return	lispBindTree_checkSub (pNode, nRootKey) ;
}

Boolean
lispBindTree_checkSub (
	register TLispBind*		pNode,
	register int			nRootKey)
{
	register int	nKey ;

	assert (pNode != NULL); 

	if (pNode->m_pLeft != NULL) {
		if ((int)pNode->m_pLeft->m_pEntSymbol  >= (int)pNode->m_pEntSymbol)
			return	False ;
		if (pNode->m_pLeft->m_pParent != pNode)
			return	False ;
		if (! lispBindTree_checkSub (pNode->m_pLeft, nRootKey))
			return	False ;
	}
	if (pNode->m_pRight != NULL) {
		if ((int)pNode->m_pRight->m_pEntSymbol <= (int)pNode->m_pEntSymbol)
			return	False ;
		if (pNode->m_pRight->m_pParent != pNode)
			return	False ;
		if (! lispBindTree_checkSub (pNode->m_pRight, nRootKey))
			return	False ;
	}
	nKey	= (((unsigned) pNode->m_pEntSymbol) * 13) % 37 ;
	if (nKey != nRootKey)
		return	False ;
	return	True ;
}
#endif
#endif

