/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <assert.h>
#include "lmachinep.h"
#include "filename.h"

typedef struct {
	TLispEntity*	m_pEntBufferToKill ;
	TLispEntity*	m_pEntNextBuffer ;
}	TKillBufferEnumProcArg ;

static	TLMRESULT	lispMachineState_saveCurrentBufferFinalize (TLispMachine*) ;

static	TLMRESULT	lispMachineState_point	(TLispMachine*, Boolean (*)(TLispManager*, TLispEntity*, int*)) ;
static	TLMRESULT	lispMachineState_pointMarker (TLispMachine*, Boolean (*)(TLispManager*, TLispEntity*, int*)) ;
static	Boolean		lispMachine_killBufferFrameProc	(TLispMachine*, TLispEntity*, void*, Boolean*) ;
static	Boolean		lispMachine_killBufferWindowProc	(TLispMachine*, TLispEntity*, void*, Boolean*) ;
static	Boolean		lispMachine_enumBufferList	(TLispMachine*, TLispEntity*, void*, Boolean*) ;

/*
 *	(current-buffer)
 */
TLMRESULT
lispMachineState_CurrentBuffer (
	register TLispMachine*	pLM)
{
	TLispEntity*	pEntCurBuffer ;

	lispMachineCode_GetCurrentBuffer (pLM, &pEntCurBuffer) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntCurBuffer) ;
	return	LMR_RETURN ;
}

/*
 *	buffer-string is a built-in function.
 *
 *	current buffer ƤʸȤ֤⤷ narrowing Ƥ顢
 *	Хåեθʬ֤롣
 *
 *	(buffer-string)
 */
TLMRESULT
lispMachineState_BufferString (
	register TLispMachine*	pLM)
{
	TLispManager*		pLispMgr ;
	TLispEntity*		pEntBuffer ;
	int					nBufferString ;
	int					nPointMin, nPointMax, nPointBTop ;
	TLispEntity*		pRetval ;
	TVarbuffer			vbufSTRING ;
	TBufStringMarker	mkBuffer ;
	register Char*		pTop ;
	register Char*		pString ;
	register int		nString ;
	register Boolean	fRetval ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	if (TFAILED (lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer)) ||
		TFAILED (lispBuffer_GetString (pLispMgr, pEntBuffer, &mkBuffer, &nBufferString))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointBTop) ;
	lispBuffer_PointMin (pLispMgr, pEntBuffer, &nPointMin) ;
	lispBuffer_PointMax (pLispMgr, pEntBuffer, &nPointMax) ;
	assert (0 <= (nPointMin - nPointBTop) && (nPointMin - nPointBTop) <= nBufferString) ;
	assert (0 <= (nPointMax - nPointBTop) && (nPointMax - nPointBTop) <= nBufferString) ;
	assert (nPointMin <= nPointMax) ;

	if (TFAILED (TVarbuffer_Initialize (&vbufSTRING, sizeof (Char))) ||
		TFAILED (TVarbuffer_Require (&vbufSTRING, nPointMax - nPointMin)))
		return	LMR_ERROR ;
	TBufStringMarker_Forward (&mkBuffer, nPointMin - nPointBTop) ;
	pTop	= TVarbuffer_GetBuffer (&vbufSTRING) ;
	pString	= pTop ;
	nString	= nPointMax - nPointMin ;
	while (nString -- > 0) {
		*pString ++	= TBufStringMarker_GetChar (&mkBuffer) ;
		TBufStringMarker_Forward (&mkBuffer, 1) ;
	}

	fRetval	= lispMgr_CreateString (pLispMgr, pTop, nPointMax - nPointMin, &pRetval) ;
	TVarbuffer_Uninitialize (&vbufSTRING) ;

	if (TFAILED (fRetval))
		return	LMR_ERROR ;

	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

/*
 *	buffer-substring is a built-in function.
 *
 *	ȥХåեƤΰʸȤ֤2Ĥΰ START, END 
 *	饯ΰ֤Ǥ롣֤Ϥɤ餬Ǥ⤤⤷Хåեޥ
 *	Ȥʤ顢֤ʸޥХȤǤ롣
 *
 *	(buffer-substring START END)
 *
 *	START, END  integer ޤ marker
 */
TLMRESULT
lispMachineState_BufferSubstring (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*		pEntBuffer ;
	int					nBufString ;
	TLispEntity*		pArglist ;
	TLispEntity*		pStart ;
	TLispEntity*		pEnd ;
	TLispNumber			lnStart, lnEnd ;
	int					nStart, nEnd ;
	TLispEntity*		pRetval ;
	int					nPointMin, nPointMax, nPointBTop ;
	TBufStringMarker	mkBuffer ;
	TVarbuffer			vbufSTRING ;
	register Char*		pTop ;
	register Char*		pString ;
	register int		nString ;
	register Boolean	fRetval ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;

	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pStart)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pArglist, &pEnd)) ||
		TFAILED (lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer)) ||
		TFAILED (lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pStart, &lnStart)) ||
		TFAILED (lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pEnd,   &lnEnd)) ||
		lnStart.m_fFloatp || lnEnd.m_fFloatp ) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	/*	narrowing αƶ롣*/
	lispBuffer_PointMin (pLispMgr, pEntBuffer, &nPointMin) ;
	lispBuffer_PointMax (pLispMgr, pEntBuffer, &nPointMax) ;
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointBTop) ;
	assert (nPointMin > 0 && nPointMax > 0) ;

	nStart	= lnStart.m_Value.m_lLong ;
	nEnd	= lnEnd.m_Value.m_lLong ;
	if (nStart > nEnd) {
		int	nTmp	= nStart ;
		nStart	= nEnd ;
		nEnd	= nTmp ;
	}
	/*	Out of range Υå*/
	if (nStart < nPointMin || nPointMax < nEnd) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	/*	ХåեƬ֤ 1 顣GetString  buffer-top  buffer-end ޤǤ
	 *	ȴФ*/
	lispBuffer_GetString (pLispMgr, pEntBuffer, &mkBuffer, &nBufString) ;
	if (TFAILED (TVarbuffer_Initialize (&vbufSTRING, sizeof (Char))) ||
		TFAILED (TVarbuffer_Require (&vbufSTRING, nEnd - nStart)))
		return	LMR_ERROR ;
	TBufStringMarker_Forward (&mkBuffer, nStart - nPointBTop) ;
	pTop	= TVarbuffer_GetBuffer (&vbufSTRING) ;
	pString	= pTop ;
	nString	= nEnd - nStart ;
	while (nString -- > 0) {
		*pString ++	= TBufStringMarker_GetChar (&mkBuffer) ;
		TBufStringMarker_Forward (&mkBuffer, 1) ;
	}

	fRetval	= lispMgr_CreateString (pLispMgr, pTop, nEnd - nStart, &pRetval) ;
	TVarbuffer_Uninitialize (&vbufSTRING) ;

	if (TFAILED (fRetval))
		return	LMR_ERROR ;

	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

/*
 *	(point)
 */
TLMRESULT
lispMachineState_Point (
	register TLispMachine*	pLM)
{
	return	lispMachineState_point (pLM, &lispBuffer_Point) ;
}

/*
 *	(point-min)
 */
TLMRESULT
lispMachineState_PointMin (
	register TLispMachine*	pLM)
{
	return	lispMachineState_point (pLM, &lispBuffer_PointMin) ;
}

/*
 *	(point-max)
 */
TLMRESULT
lispMachineState_PointMax (
	register TLispMachine*	pLM)
{
	return	lispMachineState_point (pLM, &lispBuffer_PointMax) ;
}

/*
 *	(point-marker)
 */
TLMRESULT
lispMachineState_PointMarker (
	register TLispMachine*	pLM)
{
	return	lispMachineState_pointMarker (pLM, &lispBuffer_Point) ;
}

/*
 *	(point-min-marker)
 */
TLMRESULT
lispMachineState_PointMinMarker (
	register TLispMachine*	pLM)
{
	return	lispMachineState_pointMarker (pLM, &lispBuffer_PointMin) ;
}

/*
 *	(point-max-marker)
 */
TLMRESULT
lispMachineState_PointMaxMarker (
	register TLispMachine*	pLM)
{
	return	lispMachineState_pointMarker (pLM, &lispBuffer_PointMax) ;
}

/*	built-in function:
 *		(bobp)
 *
 *	⤷ point ХåեƬˤ t ֤narrowing Ƥ
 *	ˤϤαƶ롣
 */
TLMRESULT
lispMachineState_Bobp (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntBuffer ;
	TLispEntity*	pEntRetval ;
	int				nPoint, nPointMin ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	if (TFAILED (lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer)) ||
		TFAILED (lispBuffer_Point    (pLispMgr, pEntBuffer, &nPoint)) ||
		TFAILED (lispBuffer_PointMin (pLispMgr, pEntBuffer, &nPointMin))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (nPoint == nPointMin) {
		(void) lispMgr_CreateT   (pLispMgr, &pEntRetval) ;
	} else {
		(void) lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*
 *	(eobp)
 *
 *	⤷ point ХåեκǸˤ t ֤⤷Хåե narrow
 *	ƤСαƶ롣
 */
TLMRESULT
lispMachineState_Eobp (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntBuffer ;
	TLispEntity*	pEntRetval ;
	int				nPoint, nPointMax ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	if (TFAILED (lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer)) ||
		TFAILED (lispBuffer_Point    (pLispMgr, pEntBuffer, &nPoint)) ||
		TFAILED (lispBuffer_PointMax (pLispMgr, pEntBuffer, &nPointMax))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (nPoint == nPointMax) {
		(void) lispMgr_CreateT   (pLispMgr, &pEntRetval) ;
	} else {
		(void) lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*	built-in function:
 *		(eolp)
 *
 *	⤷ݥȤԤκǸΰ֤ˤСt ֹ֤ԤκǸפˤ
 *	ХåեκǸޤǤ롣
 */
TLMRESULT
lispMachineState_Eolp (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*		pEntBuffer ;
	TLispEntity*		pEntRetval ;
	TBufStringMarker	mkBuffer ;
	int					nPoint, nPointMin, nPointMax, nLength ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	if (TFAILED (lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer)) ||
		TFAILED (lispBuffer_Point    (pLispMgr, pEntBuffer, &nPoint)) ||
		TFAILED (lispBuffer_PointMin (pLispMgr, pEntBuffer, &nPointMin)) ||
		TFAILED (lispBuffer_PointMax (pLispMgr, pEntBuffer, &nPointMax)) ||
		TFAILED (lispBuffer_GetString (pLispMgr, pEntBuffer, &mkBuffer, &nLength))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	TBufStringMarker_Forward (&mkBuffer, nPoint - nPointMin) ;
	if (TBufStringMarker_GetChar (&mkBuffer) == '\n' ||
		nPoint == nPointMax) {
		(void) lispMgr_CreateT   (pLispMgr, &pEntRetval) ;
	} else {
		(void) lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*
 *	(insert-char CHARACTER COUNT &optional INHERIT)
 *
 *	INHERIT  text property ηѾ˴ؤʤΤǡǤ̵Ǥ롣
 */
TLMRESULT
lispMachineState_InsertChar (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pCharacter ;
	TLispEntity*	pCount ;
	TLispEntity*	pNil ;
	TLispEntity*	pEntBuffer ;
	long			lChar, lCount ;
	register Char	cc ;
	int				nPos ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;

	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pCharacter)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pArglist, &pCount)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pCharacter, &lChar)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pCount, &lCount)) ||
		TFAILED (lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer)) ||
		TFAILED (lispBuffer_Point (pLispMgr, pEntBuffer, &nPos))) 
		goto	error ;

	cc	= (Char) lChar ;
	if (Char_Charset (cc) < KCHARSET_ASCII || Char_Charset (cc) >= MAX_CHARSET) {
		if (Char_Charset (cc) != KCHARSET_XCHAR)
			goto	error ;
		cc	= Char_MakeAscii (Char_Code (cc) & 0x7F) ;
	}
	if (TFAILED (lispBuffer_InsertChar (pLispMgr, pEntBuffer, nPos, cc, lCount)))
		goto	error ;
	
	lispMgr_CreateNil (pLispMgr, &pNil) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pNil) ;
	return	LMR_RETURN ;

  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*
 *	(insert &rest ARG)
 */
TLMRESULT
lispMachineState_Insert (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pEntChar ;
	TLispEntity*	pNil ;
	TLispEntity*	pEntBuffer ;
	const Char*		pString ;
	Char			ch ;
	long			lChar ;
	int				nPos, nString ;

	assert (pLM != NULL) ;

	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	if (TFAILED (lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;

	while (TFAILED (lispEntity_Nullp (pLispMgr, pArglist))) {
		lispEntity_GetCar (pLispMgr, pArglist, &pEntChar) ;
		if (TSUCCEEDED (lispEntity_Stringp (pLispMgr, pEntChar))) {
			lispEntity_GetStringValue (pLispMgr, pEntChar, &pString, &nString) ;
#if defined (DEBUG)
			{
				int	nPoint ;
				lispBuffer_Point (pLispMgr, pEntBuffer, &nPoint) ;
				fprintf (stderr, "insert-char(%d): ", nPoint) ;
				lispEntity_Print (pLispMgr, pEntChar) ;
				fprintf (stderr, ", %d\n", nString) ;
			}
#endif
		} else if (TSUCCEEDED (lispEntity_Integerp (pLispMgr, pEntChar))) {
			lispEntity_GetIntegerValue (pLispMgr, pEntChar, &lChar) ;
			ch		= (Char) lChar ;
			if (Char_Charset (ch) < KCHARSET_ASCII || Char_Charset (ch) >= MAX_CHARSET) {
				if (Char_Charset (ch) != KCHARSET_XCHAR) {
					lispMachineCode_SetError (pLM) ;
					break ;
				}
				ch	= Char_MakeAscii (Char_Code (ch) & 0x7F) ;
			}
			pString	= &ch ;
			nString	= 1 ;
		} else {
			lispMachineCode_SetError (pLM) ;
			break ;
		}
		if (TFAILED (lispBuffer_Point (pLispMgr, pEntBuffer, &nPos)) ||
			TFAILED (lispBuffer_InsertString (pLispMgr, pEntBuffer, nPos, pString, nString))) {
			lispMachineCode_SetError (pLM) ;
			break ;
		}
		lispEntity_GetCdr (pLispMgr, pArglist, &pArglist) ;
	}
	lispMgr_CreateNil (pLispMgr, &pNil) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pNil) ;
	return	LMR_RETURN ;
}

/*
 *	(following-char)
 */
TLMRESULT
lispMachineState_FollowingChar (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*		pEntBuffer ;
	int					nBufString ;
	Char				ch ;
	int					nPointMin, nPointMax, nPointBTop, nPoint ;
	TLispEntity*		pEntRetval ;
	TBufStringMarker	mkBuffer ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	if (TFAILED (lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer)) ||
		TFAILED (lispBuffer_GetString (pLispMgr, pEntBuffer, &mkBuffer, &nBufString))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispBuffer_PointMin (pLispMgr, pEntBuffer, &nPointBTop) ;
	lispBuffer_PointMin (pLispMgr, pEntBuffer, &nPointMin) ;
	lispBuffer_PointMax (pLispMgr, pEntBuffer, &nPointMax) ;
	lispBuffer_Point    (pLispMgr, pEntBuffer, &nPoint) ;
	if (nPoint < nPointMin || nPoint >= nPointMax) {
		ch	= 0 ;
	} else {
		TBufStringMarker_Forward (&mkBuffer, nPoint - nPointBTop) ;
		ch	= TBufStringMarker_GetChar (&mkBuffer) ;
	}
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, (long) ch, &pEntRetval)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
	
}

/*
 *
 */
TLMRESULT
lispMachineState_PrecedingChar (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntBuffer ;
	int				nStrBuffer ;
	int				nPoint, nPointMin, nPointBTop ;
	register Char	ch ;
	TLispEntity*	pEntRetval ;
	TBufStringMarker	mkBuffer ;

	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	lispBuffer_Point     (pLispMgr, pEntBuffer, &nPoint) ;
	lispBuffer_PointMin  (pLispMgr, pEntBuffer, &nPointMin) ;
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointBTop) ;
	lispBuffer_GetString (pLispMgr, pEntBuffer, &mkBuffer, &nStrBuffer) ;
	if (nPoint <= nPointMin) {
		ch	= 0 ;
	} else {
		assert ((nPoint - nPointMin) <= nStrBuffer) ;
		TBufStringMarker_Forward (&mkBuffer, nPoint - nPointBTop - 1) ;
		ch	= TBufStringMarker_GetChar (&mkBuffer) ;
	}
#if defined (DEBUG)
	fprintf (stderr, "preceding-char: %lx (point: %d/min: %d)\n", ch, nPoint, nPointMin) ;
#endif
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, ch, &pEntRetval)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*
 *	(set-buffer BUFFER)
 *
 *	buffer BUFFER Խ current ꤹ롣BUFFER 
 *	buffer ⤷¸ߤХåե̾Ǥ롣
 *	(ʤ顢ΤȤ skkime  lisp ƤΥХåե
 *	̵̾ʤΤ̾ԲǽǤ)
 */
TLMRESULT
lispMachineState_SetBuffer (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntArglist ;
	TLispEntity*			pEntTarget ;
	TLispEntity*			pEntBuffer ;
	const Char*				strBuffer ;
	int						nstrBuffer ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntTarget) ;

	if (TSUCCEEDED (lispEntity_Bufferp (pLispMgr, pEntTarget))) {
		lispMachineCode_SetCurrentBuffer (pLM, pEntTarget) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntTarget, &strBuffer, &nstrBuffer)) ||
		TFAILED (lispMachine_GetBuffer (pLM, strBuffer, nstrBuffer, &pEntBuffer)) ||
		TFAILED (lispEntity_Bufferp (pLispMgr, pEntBuffer))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_SetCurrentBuffer (pLM, pEntBuffer) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_SaveCurrentBuffer (
	register TLispMachine*	pLM)
{
	TLispEntity*	pEntBuffer ;

	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_1, pEntBuffer) ;
	lispMachineCode_SetState (pLM, &lispMachineState_Progn) ;
	lispMachineCode_PushState (pLM, &lispMachineState_saveCurrentBufferFinalize) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_saveCurrentBufferFinalize (
	register TLispMachine*	pLM)
{
	TLispEntity*	pEntBuffer ;

	lispMachineCode_GetLReg (pLM, LM_LREG_1, &pEntBuffer) ;
	lispMachineCode_SetCurrentBuffer (pLM, pEntBuffer) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
	return	LMR_RETURN ;
}

/*
 *	(make-local-variable VARIABLE)
 *
 *	make-local-variable  interactive built-in function
 *	VARIABLE  current buffer ̤ͤƤ褦ˤ롣¾
 *	buffer ϶̤ default value ͭ³롣buffer-local 
 *	ͤ VARIABLE ˻äƤΤƱͤϤޤ롣
 *	⤷ VARIABLE  void ʤ void Τޤޡ
 */
TLMRESULT
lispMachineState_MakeLocalVariable (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntBuffer ;
	TLispEntity*			pEntArglist ;
	TLispEntity*			pEntSymbol ;
	TLispEntity*			pDummy ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntSymbol)) ||
		TFAILED (lispEntity_Symbolp (pLispMgr, pEntSymbol)) ||
		TFAILED (lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}

	if (TFAILED (lispBuffer_GetSymbolValue (pLispMgr, pEntBuffer, pEntSymbol, &pDummy))) {
		TLispEntity*	pEntValue ;

		/*	default-value ¸ߤʤСvoid Τޤޡ*/
		if (TFAILED (lispMachine_GetCurrentSymbolValue (pLM, pEntSymbol, &pEntValue)))
			lispMgr_CreateVoid (pLispMgr, &pEntValue) ;

		if (TFAILED (lispBuffer_MakeSymbolValue (pLispMgr, pEntBuffer, pEntSymbol)))
			return	LMR_ERROR ;
		lispBuffer_SetSymbolValue (pLispMgr, pEntBuffer, pEntSymbol, pEntValue) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntSymbol) ;
	return	LMR_RETURN ;
}

/*
 *	(make-variable-buffer-local VARIABLE)
 */
TLMRESULT
lispMachineState_MakeVariableBufferLocal (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntBuffer ;
	TLispEntity*			pEntArglist ;
	TLispEntity*			pEntSymbol ;
	TLispEntity*			pDummy ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntSymbol)) ||
		TFAILED (lispEntity_Symbolp (pLispMgr, pEntSymbol)) ||
		TFAILED (lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}

	if (TFAILED (lispBuffer_GetSymbolValue (pLispMgr, pEntBuffer, pEntSymbol, &pDummy))) {
		TLispEntity*	pEntValue ;

		/*	default-value ¸ߤʤСnil ꤹ롣*/
		if (TFAILED (lispMachine_GetCurrentSymbolValue (pLM, pEntSymbol, &pEntValue)) ||
			TSUCCEEDED (lispEntity_Voidp (pLispMgr, pEntValue))) {
			lispMgr_CreateNil (pLispMgr, &pEntValue) ;
			lispMachine_SetCurrentSymbolValue (pLM, pEntSymbol, pEntValue) ;
		}

		if (TFAILED (lispBuffer_MakeSymbolValue (pLispMgr, pEntBuffer, pEntSymbol)))
			return	LMR_ERROR ;
		lispBuffer_SetSymbolValue (pLispMgr, pEntBuffer, pEntSymbol, pEntValue) ;

		/*	ʹ buffer ٤ȿǤ褦˾ manager Ϳ롣*/
		lispMgr_AddSymbolToLocalSymbols (pLispMgr, pEntSymbol) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntSymbol) ;
	return	LMR_RETURN ;
}


/*
 *	(looking-at REGEXP)
 *
 *	⤷ point ³ƥȤ REGEXP ɽŬ礷Ƥ
 *	 t ֤δؿ match-beginning, match-end, 
 *	match-data Υ򤹤Τǡmatch data ѹƤ
 *	ޤ⤷¸ƤΤʤ顢match data 
 *	֡ʤФʤʤ
 *
 *	ࡣREGEXP μݤʤġ
 */
TLMRESULT
lispMachineState_LookingAt (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntCurBuffer ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntREGEXP ;
	const Char*		pString ;
	int				nString ;
	TLispEntity*	pEntRetval ;
	int				nPoint, nPointMax ;

	lispMachineCode_GetCurrentBuffer (pLM, &pEntCurBuffer) ;
	assert (pEntCurBuffer != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntREGEXP) ;
	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntREGEXP, &pString, &nString))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	lispBuffer_Point    (pLispMgr, pEntCurBuffer, &nPoint) ;
	lispBuffer_PointMax (pLispMgr, pEntCurBuffer, &nPointMax) ;

	if (TSUCCEEDED (lispMachineCode_ReSearchForward (pLM, pEntCurBuffer, pString, nString, nPointMax, 1))) {
		int		nPosition ;
		lispMachineCode_MatchBeginning (pLM, 0, &nPosition) ;
#if defined (DEBUG)
		fprintf (stderr, "looking-at: (hit/cur) = (%d/%d)\n", nPosition, nPoint) ;
#endif
		if (nPosition == nPoint) 
			lispMgr_CreateT (pLispMgr, &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*
 *	(char-after &optional POS)
 *
 *	Return character in current buffer at position POS.
 *	POS is an integer or a marker.
 *	If POS is out of range, the value is nil.
 */
TLMRESULT
lispMachineState_CharAfter (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntBuffer ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntPOS ;
	TLispEntity*	pEntRetval ;
	int				nBufString ;
	int				nPoint, nPointMin, nPointMax, nPointBTop ;
	TBufStringMarker	mkBuffer ;
	
	if (TFAILED (lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer)) ||
		TFAILED (lispBuffer_GetString (pLispMgr, pEntBuffer, &mkBuffer, &nBufString))) 
		goto	error ;
	assert (pEntBuffer != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntPOS) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntPOS))) {
		lispBuffer_Point (pLispMgr, pEntBuffer, &nPoint) ;
	} else {
		TLispNumber	num ;

		if (TFAILED (lispEntity_GetNumberOrMarkerValue (pLispMgr, pEntPOS, &num)) ||
			TSUCCEEDED (num.m_fFloatp)) 
			goto	error ;
		nPoint	= num.m_Value.m_lLong ;
	}
	lispBuffer_PointMin (pLispMgr, pEntBuffer, &nPointMin) ;
	lispBuffer_PointMax (pLispMgr, pEntBuffer, &nPointMax) ;
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointBTop) ;
	if (nPoint < nPointMin || nPoint >= nPointMax) {
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	} else {
		TBufStringMarker_Forward (&mkBuffer, nPoint - nPointBTop) ;
		lispMgr_CreateInteger (pLispMgr, TBufStringMarker_GetChar (&mkBuffer), &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;

  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*	(char-before &optional POS)
 *
 *	Return character in current buffer preceding position POS.
 *	POS is an integer or a marker.
 *	If POS is out of range, the value is nil.
 */
TLMRESULT
lispMachineState_CharBefore (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntBuffer ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntPOS ;
	TLispEntity*	pEntRetval ;
	int				nBufString ;
	int				nPoint, nPointMin, nPointMax, nPointBTop ;
	TBufStringMarker	mkBuffer ;
	
	if (TFAILED (lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer)) ||
		TFAILED (lispBuffer_GetString (pLispMgr, pEntBuffer, &mkBuffer, &nBufString))) 
		goto	error ;
	assert (pEntBuffer != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntPOS) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntPOS))) {
		lispBuffer_Point (pLispMgr, pEntBuffer, &nPoint) ;
	} else {
		TLispNumber	num ;

		if (TFAILED (lispEntity_GetNumberOrMarkerValue (pLispMgr, pEntPOS, &num)) ||
			TSUCCEEDED (num.m_fFloatp)) 
			goto	error ;
		nPoint	= num.m_Value.m_lLong ;
	}
	/*	preceding ʤΤ 1 롣*/
	nPoint	-- ;

	lispBuffer_PointMin (pLispMgr, pEntBuffer, &nPointMin) ;
	lispBuffer_PointMax (pLispMgr, pEntBuffer, &nPointMax) ;
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointBTop) ;
	if (nPoint < nPointMin || nPoint >= nPointMax) {
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	} else {
		TBufStringMarker_Forward (&mkBuffer, nPoint - nPointBTop) ;
		lispMgr_CreateInteger (pLispMgr, TBufStringMarker_GetChar (&mkBuffer), &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;

  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*
 *	interactive built-in function:
 *		(narrow-to-region START END)
 *
 *	ߤΥХåեΥȤΥ꡼ԽΰȤ¤롣Ĥ
 *	ΥƥȤϰŪ˸ʤʤäƿʤʤ롣뤳Ȥ
 *	Ǥʤ⤷Хåեե˥֤硢ʤƥ
 *	ϥե˴ޤޤ롣save-restriction ⸫뤳ȡ
 *
 *	START, END  integer-or-marker-p  t ֤ɬפ롣
 */
TLMRESULT
lispMachineState_NarrowToRegion (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntBuffer ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntSTART ;
	TLispEntity*	pEntEND ;
	int				nStart, nEnd ;
	long			lStart, lEnd ;
	TLispEntity*	pEntRetval ;

	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntSTART) ;
	lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntEND) ;
	if (TFAILED (lispEntity_IntegerOrMarkerp (pLispMgr, pEntSTART)) ||
		TFAILED (lispEntity_IntegerOrMarkerp (pLispMgr, pEntEND))) 
		goto	error ;
	if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntSTART, &lStart))) {
		lispMarker_GetBufferPosition (pLispMgr, pEntSTART, NULL, &nStart) ;
	} else {
		nStart	= lStart ;
	}
	if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntEND, &lEnd))) {
		lispMarker_GetBufferPosition (pLispMgr, pEntEND, NULL, &nEnd) ;
	} else {
		nEnd	= lEnd ;
	}
	if (TFAILED (lispBuffer_Narrow (pLispMgr, pEntBuffer, nStart, nEnd)))
		goto	error ;
	lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;

  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*
 *
 */
TLMRESULT
lispMachineState_Widen (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntBuffer ;
	int				nBufferTop, nBufferEnd ;
	TLispEntity*	pEntRetval ;

	assert (pLM      != NULL) ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;

	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nBufferTop) ;
	lispBuffer_PointBufferEnd (pLispMgr, pEntBuffer, &nBufferEnd) ;
	if (TFAILED (lispBuffer_Narrow (pLispMgr, pEntBuffer, nBufferTop, nBufferEnd))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*	private */
TLMRESULT
lispMachineState_point (
	register TLispMachine*	pLM,
	register Boolean		(*pPointFunc)(TLispManager*, TLispEntity*, int*))
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntBuffer ;
	TLispEntity*	pEntRetval ;
	int				nPos ;

	assert (pLM != NULL) ;

	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	if (TFAILED (lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer)) ||
		TFAILED ((*pPointFunc) (pLispMgr, pEntBuffer, &nPos))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	/*	ХåեƬ1˽롣ϰŪ˹ԤΤա*/
	/*	ν marker ľ˽Ф롣*/
	/*lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nBufferTop) ;*/
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, nPos /*- nBufferTop + 1*/, &pEntRetval)))
		return	LMR_ERROR ;

	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_pointMarker (
	register TLispMachine*	pLM,
	register Boolean		(*pPointFunc)(TLispManager*, TLispEntity*, int*))
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntBuffer ;
	TLispEntity*	pEntRetval ;
	int				nPoint ;

	assert (pLM != NULL) ;

	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	if (TFAILED (lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer)) ||
		TFAILED ((*pPointFunc) (pLispMgr, pEntBuffer, &nPoint))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispMgr_CreateMarker (pLispMgr, &pEntRetval)))
		return	LMR_ERROR ;
	lispBuffer_AddMarker (pLispMgr, pEntBuffer, pEntRetval) ;
	lispMarker_SetBufferPosition (pLispMgr, pEntRetval, pEntBuffer, nPoint) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*	built-in function:
 *		(get-buffer-create NAME)
 *
 *	NAME ̾դ줿Хåե֤ޤϤʥХåեꡢ
 *	֤⤷ NAME ̾դ줿ХåեʤпХåե
 *	롣NAME ڡǤϤޤʤ顢Хåե UNDO 
 *	ݻʤNAME ʸ˥Хåեʤ顢줬֤
 *	ͤǤ롣ͤϷ褷 nil ǤϤʤ
 */
TLMRESULT
lispMachineState_GetBufferCreate (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntNAME ;
	TLispEntity*	pEntRetval ;
	const Char*		pStrName ;
	int				nStrName ;
	
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	(void) lispEntity_GetCar (pLispMgr, pEntArglist, &pEntNAME) ;
	if (TSUCCEEDED (lispEntity_GetStringValue (pLispMgr, pEntNAME, &pStrName, &nStrName))) {
		if (TFAILED (lispMachine_GetBuffer (pLM, pStrName, nStrName, &pEntRetval))) {
			if (TFAILED (lispMgr_CreateBuffer (pLispMgr, &pEntRetval)))
				return	LMR_ERROR ;
			lispBuffer_SetName (pLispMgr, pEntRetval, pEntNAME) ;
			lispMachine_InsertBuffer (pLM, pEntRetval) ;
		}
	} else if (TSUCCEEDED (lispEntity_Bufferp (pLispMgr, pEntNAME))) {
		pEntRetval	= pEntNAME ;
	} else {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*
 *	built-in function:
 *		(get-buffer NAME)
 */
TLMRESULT
lispMachineState_GetBuffer (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntNAME ;
	TLispEntity*	pEntRetval ;
	const Char*		pStrName ;
	int				nStrName ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	(void) lispEntity_GetCar (pLispMgr, pEntArglist, &pEntNAME) ;
	if (TSUCCEEDED (lispEntity_GetStringValue (pLispMgr, pEntNAME, &pStrName, &nStrName))) {
		if (TFAILED (lispMachine_GetBuffer (pLM, pStrName, nStrName, &pEntRetval)))
			pEntRetval	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_NIL) ;
	} else if (TSUCCEEDED (lispEntity_Bufferp (pLispMgr, pEntNAME))) {
		pEntRetval	= pEntNAME ;
	} else {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*	interactive built-in function:
 *		(erase-buffer)
 *
 *	ȥХåեΤõ롣narrowing ¤ä롣
 *	ХåեϤθ˶ˤʤ롣
 */
TLMRESULT
lispMachineState_EraseBuffer (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntBuffer ;
	int				nPointBTop, nPointBEnd ;

	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	lispBuffer_Widen (pLispMgr, pEntBuffer) ;
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointBTop) ;
	lispBuffer_PointBufferEnd (pLispMgr, pEntBuffer, &nPointBEnd) ;
	lispBuffer_DeleteChar (pLispMgr, pEntBuffer, nPointBTop, nPointBEnd - nPointBTop) ;
	return	LMR_RETURN ;
}

/*	built-in function:
 *		(generate-new-buffer-name NAME &optional IGNORE)
 *
 *	NAME ١ˤƥХåեȤƤ̾¸ߤƤʤʸ֤
 *	⤷ NAME ̾դ줿Хåե¸ߤʤΤʤ顢NAME ֤
 *	ǤʤС`<NUMBER>' ä̾롣ȤƤʤ̾
 *	դޤ NUMBER äƤIGNORE ϤȤ̾ΥХåե
 *	¸ߤƤȤäɤ̾ꤹ롣
 *
 *	ߡIGNORE ̵롣ʤ
 */
TLMRESULT
lispMachineState_GenerateNewBufferName (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntNAME ;
	const Char*		pStrNAME ;
	int				nStrNAME ;
	TLispEntity*	pEntRetval ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntNAME) ;
	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntNAME, &pStrNAME, &nStrNAME))) 
		goto	error ;
	if (TFAILED (lispMachine_GenerateNewBufferName (pLM, pStrNAME, nStrNAME, &pEntRetval)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;

  error:	
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*	(generate-new-buffer NAME)
 *
 *	NAME ˴𤤤̾ǤäƥХåե֤Хåե̾
 *	generate-new-buffer-name Ȥä򤷤Ƥ롣
 */
TLMRESULT
lispMachineState_GenerateNewBuffer (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntNAME ;
	const Char*		pStrNAME ;
	int				nStrNAME ;
	TLispEntity*	pEntName ;
	TLispEntity*	pEntBuffer ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntNAME) ;
	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntNAME, &pStrNAME, &nStrNAME))) 
		goto	error ;
	if (TFAILED (lispMgr_CreateBuffer (pLispMgr, &pEntBuffer)))
		return	LMR_ERROR ;
	lispEntity_AddRef (pLispMgr, pEntBuffer) ;
	if (TFAILED (lispMachine_GenerateNewBufferName (pLM, pStrNAME, nStrNAME, &pEntName)))
		return	LMR_ERROR ;
	lispBuffer_SetName (pLispMgr, pEntBuffer, pEntName) ;
	lispMachine_InsertBuffer (pLM, pEntBuffer) ;
	lispEntity_Release (pLispMgr, pEntBuffer) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntBuffer) ;
	return	LMR_RETURN ;

  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*	built-in function:
 *		(get-file-buffer FILENAME)
 *
 *	ե FILENAME ˬƤХåե֤ΥХåե 
 *	`buffer-file-name' ΤFILENAME  expansion ȰפƤʤ
 *	Фʤʤ⤷ʤСnil ֤롣
 */
TLMRESULT
lispMachineState_GetFileBuffer (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntNAME ;
	TVarbuffer		vbufNAME ;
	const Char*		pStrNAME ;
	int				nStrNAME ;
	TLispEntity*	pEntRetval ;

	if (TFAILED (TVarbuffer_Initialize (&vbufNAME, sizeof (Char))))
		return	LMR_ERROR ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntNAME) ;
	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntNAME, &pStrNAME, &nStrNAME))) 
		goto	error ;
	/*	expansion ȤΤ expand-file-name ȤȤ*/
	if (TFAILED (ExpandFileName (&vbufNAME, pStrNAME, nStrNAME, NULL, 0)))
		goto	error ;
	/*	ե̾򸰤ˤơbuffer 򸡺롣*/
	pStrNAME	= TVarbuffer_GetBuffer (&vbufNAME) ;
	nStrNAME	= TVarbuffer_GetUsage  (&vbufNAME) ;
	if (TFAILED (lispMachine_GetFileBuffer (pLM, pStrNAME, nStrNAME, &pEntRetval))) 
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	TVarbuffer_Uninitialize (&vbufNAME) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;

  error:
	TVarbuffer_Uninitialize (&vbufNAME) ;
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*	built-in function:
 *		(set-buffer-modified-p FLAG)
 *
 *	FLAG ˽äƥȥХåեƤ뤫Ƥʤޡ
 *	դԤFLAG  non-nil ʤХåեϽ줿˥ޡդ롣
 */
TLMRESULT
lispMachineState_SetBufferModifiedp (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntBuffer ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntFLAG ;

	assert (pLM      != NULL) ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer  != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFLAG) ;
	assert (pEntFLAG    != NULL) ;
	lispBuffer_SetModifiedp (pLispMgr, pEntBuffer, TFAILED (lispEntity_Nullp (pLispMgr, pEntFLAG))) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntFLAG) ;
	return	LMR_RETURN ;
}

/*	built-in function:
 *		(buffer-modified-p &optional BUFFER)
 *
 *	⤷ BUFFER Ǹ˥ե뤬ɤޤơޤϥ֤ƤѹƤ
 *	Сt ֤ʤޤ nil äˤ BUFFER Ȥ
 *	ȥХåեȤȤ̣롣
 */
TLMRESULT
lispMachineState_BufferModifiedp (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntBuffer ;
	TLispEntity*	pEntRetval ;

	assert (pLM      != NULL) ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntBuffer) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntBuffer))) 
		lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	if (TFAILED (lispEntity_Bufferp (pLispMgr, pEntBuffer))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispBuffer_GetModifiedp (pLispMgr, pEntBuffer))) {
		lispMgr_CreateT   (pLispMgr, &pEntRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*	built-in function:
 *		(buffer-size &optional BUFFER)
 */
TLMRESULT
lispMachineState_BufferSize (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*		pEntArglist ;
	TLispEntity*		pEntBuffer ;
	TLispEntity*		pEntRetval ;
	TBufStringMarker	mk ;
	int					nLength ;

	assert (pLM      != NULL) ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntBuffer) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntBuffer))) 
		lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	if (TFAILED (lispEntity_Bufferp (pLispMgr, pEntBuffer))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispBuffer_GetFullString (pLispMgr, pEntBuffer, &mk, &nLength))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, nLength, &pEntRetval)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*	(local-variable-p VARIABLE &optional BUFFER)
 */
TLMRESULT
lispMachineState_LocalVariablep (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*		pEntArglist ;
	TLispEntity*		pEntBuffer ;
	TLispEntity*		pEntVAR ;
	TLispEntity*		pEntRetval ;
	TLispEntity*		pEntTmp ;

	assert (pLM      != NULL) ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntVAR)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntBuffer))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntBuffer))) 
		lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;

	if (TFAILED (lispEntity_Bufferp (pLispMgr, pEntBuffer))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispMachine_GetBufferLocalSymbolValue (pLM, pEntBuffer, pEntVAR, &pEntTmp))) {
		pEntRetval	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_NIL) ;
	} else {
		pEntRetval	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_T) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*	(kill-buffer BUFFER)
 *
 *	kill-buffer is an interactive built-in function.
 *
 *	Kill the buffer BUFFER.
 *	The argument may be a buffer or may be the name of a buffer.
 *	An argument of nil means kill the current buffer.
 *
 *	Value is t if the buffer is actually killed, nil if user says no.
 */
TLMRESULT
lispMachineState_KillBuffer (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*		pEntArglist ;
	TLispEntity*		pEntBUFFER ;
	TLispEntity*		pEntBufferToKill ;
	TLispEntity*		pEntNextBuffer ;
	TLispEntity*		pEntRetval ;
	const Char*			strBuffer ;
	int					nstrBuffer ;

	assert (pLM      != NULL) ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntBUFFER))) {
		goto	error ;
	}

	/*	 nil ξˤϡcurrent-buffer 
	 *	 string ξˤϡ̾ä buffer 
	 *	 buffer ʤ顢 buffer 
	 */
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntBUFFER))) {
		lispMachineCode_GetCurrentBuffer (pLM, &pEntBufferToKill) ;
	} else if (TSUCCEEDED (lispEntity_GetStringValue (pLispMgr, pEntBUFFER, &strBuffer, &nstrBuffer))) {
		if (TFAILED (lispMachine_GetBuffer (pLM, strBuffer, nstrBuffer, &pEntBufferToKill)))
			goto	error ;
	} else if (TSUCCEEDED (lispEntity_Bufferp (pLispMgr, pEntBUFFER))) {
		pEntBufferToKill	= pEntBUFFER ;
	} else {
		goto	error ;
	}
	if (pEntBufferToKill == NULL || 
		TFAILED (lispEntity_Bufferp (pLispMgr, pEntBufferToKill))) 
		goto	error ;

	if (TFAILED (lispBuffer_GetNext (pEntBufferToKill, &pEntNextBuffer)) || 
		pEntNextBuffer == pEntBufferToKill) {
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	} else {
		TKillBufferEnumProcArg	arg ;

		arg.m_pEntBufferToKill	= pEntBufferToKill ;
		arg.m_pEntNextBuffer	= pEntNextBuffer ;

		/*	buffer դƤ Window ¸ߤȦʤΤǡν
		 *	򤷤ʤФʤʤ
		 *
		 *	ȡ줬Ǹ buffer ȺǤʤ褦ˤʤСġ
		 */
		/*	Ƥ frame Ƥ window Ф...
		 */
		if (TFAILED (lispMachine_EnumFrame (pLM, lispMachine_killBufferFrameProc, &arg)))
			goto	error ;
		if (TFAILED (lispMachine_RemoveBuffer (pLM, pEntBufferToKill))) 
			goto	error ;
		lispMgr_CreateT (pLispMgr, &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;

  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*	(buffer-name &optional BUFFER)
 */
TLMRESULT
lispMachineState_BufferName (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*		pEntArglist ;
	TLispEntity*		pEntBUFFER ;
	TLispEntity*		pEntRetval ;

	assert (pLM      != NULL) ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntBUFFER))) 
		goto	error ;

	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntBUFFER))) 
		lispMachineCode_GetCurrentBuffer (pLM, &pEntBUFFER) ;
	if (TFAILED (lispEntity_Bufferp (pLispMgr, pEntBUFFER))) 
		goto	error ;
	if (TFAILED (lispBuffer_GetName (pLispMgr, pEntBUFFER, &pEntRetval))) 
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;

  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*	(buffer-list &optional FRAME)
 *
 *	¸ߤƤХåեƤʤꥹȤ֤롣
 *	⤷FRAME  frame ʤ顢Υե졼 buffer list
 *	롣
 *	ࡢFRAME դưɤʬʤʡġޤ路
 */
TLMRESULT
lispMachineState_BufferList (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*		pEntArglist ;
	TLispEntity*		pEntFRAME ;
	TLispConscell		lstBuffer ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFRAME)) ||
		TFAILED (lispEntity_Nullp  (pLispMgr, pEntFRAME))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lstBuffer.m_pCar = lstBuffer.m_pCdr = NULL ;
	lispMachine_EnumBuffer (pLM, True, lispMachine_enumBufferList, &lstBuffer) ;
	if (lstBuffer.m_pCar != NULL) {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, lstBuffer.m_pCar) ;
		lispEntity_Release (pLispMgr, lstBuffer.m_pCar) ;
	} else {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_NIL)) ;
	}
	return	LMR_RETURN ;
}

Boolean
lispMachine_killBufferFrameProc (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntFrame,
	register void*			pCaller,
	register Boolean*		pfContinue)
{
	return	lispMachine_EnumWindow (pLM, pEntFrame, lispMachine_killBufferWindowProc, pCaller) ;
}

Boolean
lispMachine_killBufferWindowProc (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntWindow,
	register void*			pCaller,
	register Boolean*		pfContinue)
{
	register TLispManager*				pLispMgr ;
	register TKillBufferEnumProcArg*	pArg ;
	TLispEntity*	pEntWindowBuffer ;

	assert (pLM != NULL) ;
	assert (pEntWindow != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	pArg		= (TKillBufferEnumProcArg*) pCaller ;
	assert (pArg != NULL &&
			pArg->m_pEntBufferToKill != NULL && 
			pArg->m_pEntNextBuffer != NULL) ;
	
	if (TFAILED (lispWindow_GetBuffer (pEntWindow, &pEntWindowBuffer)))
		return	False ;
	if (pEntWindowBuffer == pArg->m_pEntBufferToKill) 
		lispWindow_SetBuffer (pLispMgr, pEntWindow, pArg->m_pEntNextBuffer) ;
	return	True ;
}

Boolean
lispMachine_enumBufferList (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntBuffer,
	register void*			pCaller,
	register Boolean*		pfContinue)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	register TLispConscell*	pList		= (TLispConscell*) pCaller ;

	assert (pLispMgr != NULL) ;
	assert (pList != NULL) ;

	return	lispEntity_Push2List (pLispMgr, pList, pEntBuffer) ;
}

