/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <assert.h>
#include "lmachinep.h"
#include "varbuffer.h"
#include "cstring.h"
#include "kanji.h"
#include "regex.h"

static	Boolean	lispMachine_addTail (TLispManager*, TLispEntity*, TLispEntity**, TLispEntity**) ;
#if defined (DEBUG)
static	void	cputstr (FILE*, const Char*, int) ;
#endif

Boolean
lispMachineCode_PushVReg (
	register TLispMachine*	pLM,	
	register int			nReg)
{
	assert (pLM != NULL) ;
	assert (0 <= nReg && nReg < MAX_NONLISPOBJ_REGS) ;
	return	Vstack_Push (&pLM->m_vstNonLispObj, &pLM->m_alVREGS [nReg]) ;
}

Boolean
lispMachineCode_PopVReg (
	register TLispMachine*	pLM,
	register int			nReg)
{
	assert (pLM != NULL) ;
	assert (0 <= nReg && nReg < MAX_NONLISPOBJ_REGS) ;

	return	Vstack_Pop (&pLM->m_vstNonLispObj, &pLM->m_alVREGS [nReg]) ;
}

Boolean
lispMachineCode_SetVRegI (
	register TLispMachine*	pLM,
	register int			nReg,
	register long			lValue)
{
	assert (pLM != NULL) ;
	assert (0 <= nReg && nReg < MAX_NONLISPOBJ_REGS) ;

	pLM->m_alVREGS [nReg].m_lValue	= lValue ;
	return	True ;
}

Boolean
lispMachineCode_SetVRegP (
	register TLispMachine*	pLM,
	register int			nReg,
	register void*			pvValue)
{
	assert (pLM != NULL) ;
	assert (0 <= nReg && nReg < MAX_NONLISPOBJ_REGS) ;

	pLM->m_alVREGS [nReg].m_pvValue	= pvValue ;
	return	True ;
}

Boolean
lispMachineCode_GetVRegI (
	register TLispMachine*	pLM,
	register int			nReg,
	register long*			plValue)
{
	assert (pLM != NULL) ;
	assert (0 <= nReg && nReg < MAX_NONLISPOBJ_REGS) ;
	assert (plValue != NULL) ;

	*plValue	= pLM->m_alVREGS [nReg].m_lValue ;
	return	True ;
}

Boolean
lispMachineCode_GetVRegP (
	register TLispMachine*	pLM,
	register int			nReg,
	register void**			ppvValue)
{
	assert (pLM != NULL) ;
	assert (0 <= nReg && nReg < MAX_NONLISPOBJ_REGS) ;
	assert (ppvValue != NULL) ;

	*ppvValue	= pLM->m_alVREGS [nReg].m_pvValue ;
	return	True ;
}

Boolean
lispMachineCode_MoveVReg (
	register TLispMachine*	pLM,
	register int			nDestReg,
	register int			nSrcReg)
{
	assert (pLM != NULL) ;
	assert (0 <= nDestReg && nDestReg < MAX_NONLISPOBJ_REGS) ;
	assert (0 <= nSrcReg  && nSrcReg  < MAX_NONLISPOBJ_REGS) ;

	memcpy (&pLM->m_alVREGS [nDestReg], &pLM->m_alVREGS [nSrcReg], sizeof (TNotLispValue)) ;
	return	True ;
}

/*
 *	֤ͤ ACC 롣
 */
Boolean
lispMachineCode_Evaln (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntTarget,
	register TLMRESULT		(*pNextState)(TLispMachine*))
{
	int			iType ;

	assert (pLM != NULL) ;
	assert (pEntTarget != NULL) ;
	assert (pLM->m_pLispMgr != NULL) ;

	lispMachineCode_SetLReg   (pLM, LM_LREG_ACC, pEntTarget) ;

	(void )lispEntity_GetType (pLM->m_pLispMgr, pEntTarget, &iType) ;
	switch (iType) {
	case	LISPENTITY_SYMBOL:
		lispMachineCode_PushState (pLM, pNextState) ;
		pLM->m_pState	= &lispMachineState_EvalSymbol ;
		break ;

	case	LISPENTITY_CONSCELL:
		lispMachineCode_PushState (pLM, pNextState) ;
		pLM->m_pState	= &lispMachineState_EvalCons ;
		break ;

	case	LISPENTITY_INTEGER:
	case	LISPENTITY_FLOAT:
	case	LISPENTITY_VECTOR:
	case	LISPENTITY_MARKER:
	case	LISPENTITY_BUFFER:
	case	LISPENTITY_STRING:
	default:
		pLM->m_pState	= pNextState ;
		break ;
	}
	return	True ;
}

Boolean
lispMachineCode_Car (
	register TLispMachine*	pLM,
	register int			nDestReg,
	register int			nSrcReg)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pCAR ;

	assert (pLM != NULL) ;
	assert (0 <= nDestReg && nDestReg < MAX_LISPOBJ_REGS) ;
	assert (0 <= nSrcReg  && nSrcReg  < MAX_LISPOBJ_REGS) ;

	pLispMgr	= pLM->m_pLispMgr ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pLM->m_apLREGS [nSrcReg], &pCAR)) ||
		pCAR == NULL)
		return	False ;
	lispEntity_AddRef (pLispMgr, pCAR) ;
	if (pLM->m_apLREGS [nDestReg] != NULL)
		lispEntity_Release (pLispMgr, pLM->m_apLREGS [nDestReg]) ;
	pLM->m_apLREGS [nDestReg]	= pCAR ;
	return	True ;
}

Boolean
lispMachineCode_SetCurrentFrame (
	register TLispMachine*	pLM,
	register TLispEntity*	pSelFrame)
{
	assert (pLM != NULL) ;
	assert (pSelFrame != NULL) ;
	assert (pSelFrame->m_iType == LISPENTITY_FRAME) ;

	pLM->m_pCurFrame	= pSelFrame ;
	return	True ;
}

Boolean
lispMachineCode_GetCurrentFrame (
	register TLispMachine*	pLM,
	register TLispEntity**	ppSelFrame)
{
	register TLispEntity*	pEntFrame ;

	assert (pLM != NULL) ;
	assert (ppSelFrame != NULL) ;

	pEntFrame	= pLM->m_pCurFrame ;
	*ppSelFrame	= pEntFrame ;
	return	(pEntFrame != NULL)? True : False ;
}

Boolean
lispMachineCode_SetCurrentWindow (
	register TLispMachine*	pLM,
	register TLispEntity*	pCurWindow)
{
	assert (pLM != NULL) ;
	assert (pCurWindow != NULL) ;
	assert (pCurWindow->m_iType == LISPENTITY_WINDOW) ;

	pLM->m_pCurWindow	= pCurWindow ;
	return	True ;
}

Boolean
lispMachineCode_GetCurrentWindow (
	register TLispMachine*	pLM,
	register TLispEntity**	ppCurWindow)
{
	assert (pLM != NULL) ;
	assert (ppCurWindow != NULL) ;

	*ppCurWindow	= pLM->m_pCurWindow ;
	return	True ;
}

Boolean
lispMachineCode_SetCurrentBuffer (
	register TLispMachine*	pLM,
	register TLispEntity*	pCurBuffer)
{
	assert (pLM != NULL) ;
	assert (pCurBuffer != NULL) ;
	assert (pCurBuffer->m_iType == LISPENTITY_BUFFER) ;

	pLM->m_pCurBuffer = pCurBuffer ;
	return	True ;
}

Boolean
lispMachineCode_GetCurrentBuffer (
	register TLispMachine*	pLM,
	register TLispEntity**	ppCurBuffer)
{
	assert (pLM != NULL) ;
	assert (ppCurBuffer != NULL) ;
	assert (pLM->m_pCurBuffer != NULL) ;

	*ppCurBuffer	= pLM->m_pCurBuffer ;
	return	True ;
}

Boolean
lispMachineCode_SetException (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntException,
	register TLispEntity*	pEntValue)
{
	register TLispManager*	pLispMgr ;

	assert (pLM           != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr   != NULL) ;

	assert (pEntException != NULL) ;
	assert (pEntValue     != NULL) ;
	lispEntity_AddRef (pLispMgr, pEntException) ;
	lispEntity_AddRef (pLispMgr, pEntValue) ;

	if (pLM->m_pEntException != NULL) 
		lispEntity_Release (pLispMgr, pLM->m_pEntException) ;
	if (pLM->m_pEntExceptionValue != NULL)
		lispEntity_Release (pLispMgr, pLM->m_pEntExceptionValue) ;

	pLM->m_pEntException		=  pEntException ;
	pLM->m_pEntExceptionValue	=  pEntValue ;

	/*	 exception ꤵ줿Τʤ顢ե饰򤿤ƤƤ*/
	if (TFAILED (lispEntity_Nullp (pLispMgr, pEntException))) {
		pLM->m_uStatusFlag		|= LMSTATE_EXCEPTION ;
	} else {
		pLM->m_uStatusFlag		&= ~LMSTATE_EXCEPTION ;
	}
	return	True ;
}

Boolean
lispMachineCode_GetException (
	register TLispMachine*	pLM,
	register TLispEntity**	ppEntExceptionRet,
	register TLispEntity**	ppEntExceptionValueRet)
{
	assert (pLM != NULL) ;
	*ppEntExceptionRet		= pLM->m_pEntException ;
	*ppEntExceptionValueRet	= pLM->m_pEntExceptionValue ;
	return	True ;
}

Boolean
lispMachineCode_ResetException (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	register TLispEntity*	pEntNil ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr   != NULL) ;
	pEntNil	= pLispMgr->m_apEntReserved [LISPMGR_INDEX_NIL] ;
	return	lispMachineCode_SetException (pLM, pEntNil, pEntNil) ;
}

Boolean
lispMachineCode_SetSignal (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntSymbol,
	register TLispEntity*	pEntValue)
{
	register TLispManager*	pLispMgr ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;

	assert (pLispMgr   != NULL) ;
	assert (pEntSymbol != NULL) ;
	assert (pEntValue  != NULL) ;

	lispEntity_AddRef (pLispMgr, pEntSymbol) ;
	lispEntity_AddRef (pLispMgr, pEntValue) ;

	if (pLM->m_pEntSignal != NULL)
		lispEntity_Release (pLispMgr, pLM->m_pEntSignal) ;
	if (pLM->m_pEntSignalValue != NULL) 
		lispEntity_Release (pLispMgr, pLM->m_pEntSignalValue) ;

	pLM->m_pEntSignal		= pEntSymbol ;
	pLM->m_pEntSignalValue	= pEntValue ;
	/*	 signal ꤵ줿Τʤ顢ե饰򤿤ƤƤ*/
	if (TFAILED (lispEntity_Nullp (pLispMgr, pEntSymbol))) {
		pLM->m_uStatusFlag	|= LMSTATE_SIGNAL ;
	} else {
		pLM->m_uStatusFlag	&= ~LMSTATE_SIGNAL ;
	}
	return	True ;
}

Boolean
lispMachineCode_SetError (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	register TLispEntity*	pEntError ;
	TLispEntity*			pEntNil ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr   != NULL) ;
	lispMgr_CreateNil    (pLispMgr, &pEntNil) ;
	pEntError	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_ERROR) ;
	assert (pEntError  != NULL) ;
	return	lispMachineCode_SetSignal (pLM, pEntError, pEntNil) ;
}

Boolean
lispMachineCode_SetErrorEx (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntErrorValue)
{
	register TLispManager*	pLispMgr ;
	register TLispEntity*	pEntError ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr   != NULL) ;
	pEntError	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_ERROR) ;
	assert (pEntError      != NULL) ;
	assert (pEntErrorValue != NULL) ;
	return	lispMachineCode_SetSignal (pLM, pEntError, pEntErrorValue) ;
}

Boolean
lispMachineCode_ResetSignal (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	register TLispEntity*	pEntNil ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr   != NULL) ;
	pEntNil	= pLispMgr->m_apEntReserved [LISPMGR_INDEX_NIL] ;
	return	lispMachineCode_SetSignal (pLM, pEntNil, pEntNil) ;
}

Boolean
lispMachineCode_GetSignal (
	register TLispMachine*	pLM,
	register TLispEntity**	ppEntSignalRet,
	register TLispEntity**	ppEntSignalValueRet)
{
	assert (pLM != NULL) ;
	*ppEntSignalRet			= pLM->m_pEntSignal ;
	*ppEntSignalValueRet	= pLM->m_pEntSignalValue ;
	return	True ;
}

Boolean
lispMachineCode_SetMessage (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntString)
{
	TLispEntity*	pMinWnd ;

	assert (pLM != NULL) ;
	assert (pLM->m_pCurFrame != NULL) ;

	if (pLM->m_pCurFrame == NULL) {
#if defined (DEBUG) || 1
		if (pEntString != NULL) {
			lispEntity_Print (pLM->m_pLispMgr, pEntString) ;
			fprintf (stderr, "\n") ;
		}
#endif
		return	False ;
	}
	if (! lispFrame_GetMinibufferWindow (pLM->m_pCurFrame, &pMinWnd))
		return	True ;
	return	lispWindow_SetMessage (pLM->m_pLispMgr, pMinWnd, pEntString) ;
}

/*[ư]
 *	KEYMAP-ALIST: ((SYMBOL1 KEYMAP1) (SYMBOL2 KEYMAP2) ...)
 *
 *	η SYMBOL  VALUE 򸫤 KEYMAP ȴФƹԤ
 */
Boolean
lispMachineCode_CurrentMinorModeMaps (
	register TLispMachine*	pLM,
	register TLispEntity**	ppEntRetval)
{
	register TLispManager*	pLispMgr ;
	register TLispEntity*	pEntMinorModeMapAlist ;
	TLispEntity*	pEntNil ;
	TLispEntity*	pEntAlist ;
	TLispEntity*	pEntSymbolKeymap ;
	TLispEntity*	pEntSymbol ;
	TLispEntity*	pEntKeymap ;
	TLispEntity*	pEntSymVal ;
	TLispEntity*	pEntTop ;
	TLispEntity*	pEntTail ;

	assert (pLM != NULL) ;
	assert (ppEntRetval != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMgr_CreateNil (pLispMgr, &pEntNil) ;
	pEntTop	= pEntTail = NULL ;

	pEntMinorModeMapAlist	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_MINOR_MODE_MAP_ALIST) ;
	if (TFAILED (lispMachine_GetCurrentSymbolValue (pLM, pEntMinorModeMapAlist, &pEntAlist)) ||
		TSUCCEEDED (lispEntity_Voidp (pLispMgr, pEntAlist))) {
		*ppEntRetval	= pEntNil ;
		return	True ;
	}
	while (TFAILED (lispEntity_Nullp (pLispMgr, pEntAlist))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pEntAlist, &pEntSymbolKeymap)) ||
			TFAILED (lispEntity_GetCar (pLispMgr, pEntSymbolKeymap, &pEntSymbol)) ||
			TFAILED (lispEntity_GetCdr (pLispMgr, pEntSymbolKeymap, &pEntKeymap)))
			break ;
		if (TSUCCEEDED (lispMachine_GetCurrentSymbolValue (pLM, pEntSymbol, &pEntSymVal)) &&
			TFAILED (lispEntity_Voidp (pLispMgr, pEntSymVal))) 
			if (TFAILED (lispMachine_addTail (pLispMgr, pEntKeymap, &pEntTop, &pEntTail)))
				break ;
		lispEntity_GetCdr (pLispMgr, pEntAlist, &pEntAlist) ;
	}
	if (pEntTop == NULL) {
		*ppEntRetval	= pEntNil ;
	} else {
		lispEntity_Release (pLispMgr, pEntTop) ;
		*ppEntRetval	= pEntTop ;
	}
	return	True ;
}

Boolean
lispMachineCode_PushThisCommandKeys (
	register TLispMachine*	pLM,
	const Char				cc)
{
	return	TVarbuffer_Add (&pLM->m_vbufCmdEvents, &cc, 1) ;
}

Boolean
lispMachineCode_GetThisCommandKeys (
	register TLispMachine*	pLM,
	register Char const **	ppString,
	register int* const		pnString)
{
	*ppString	= TVarbuffer_GetBuffer (&pLM->m_vbufCmdEvents) ;
	*pnString	= TVarbuffer_GetUsage  (&pLM->m_vbufCmdEvents) ;
	return	True ;
}

Boolean
lispMachineCode_ClearThisCommandKeys (
	register TLispMachine*	pLM)
{
	TVarbuffer_Clear (&pLM->m_vbufCmdEvents) ;
	return	True ;
}

Boolean
lispMachineCode_SetRegmatchTarget (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntTarget)
{
	register TLispManager*	pLispMgr ;

	assert (pLM        != NULL) ;

	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr   != NULL) ;

	if (pLM->m_pEntRegMatch != NULL)
		lispEntity_Release (pLispMgr, pLM->m_pEntRegMatch) ;
	pLM->m_pEntRegMatch	= pEntTarget ;
	if (pEntTarget != NULL)
		lispEntity_AddRef (pLispMgr, pEntTarget) ;
	return	True ;
}

Boolean
lispMachineCode_StringMatch (
	register TLispMachine*	pLM,
	register const Char*	pStrREGEXP,
	register int			nStrREGEXP,
	register const Char*	pStrSTRING,
	register int			nStrSTRING,
	register int			nStart)
{
	regex_t					re ;
	Boolean					fRetval ;
	rm_detail_t				rd ;
	register regmatch_t*	pMatch ;
	register int			i ;

	assert (pLM != NULL) ;
#if defined (DEBUG)
	fprintf (stderr, "(code) string-match: regexp= \"") ;
	cputstr (stderr, pStrREGEXP, nStrREGEXP) ;
	fprintf (stderr, "\"\n") ;
	fprintf (stderr, "(code) string-match: target= \"") ;
	cputstr (stderr, pStrSTRING, nStrSTRING) ;
	fprintf (stderr, "\"\n") ;
#endif
	if (ReComp (&re, pStrREGEXP, nStrREGEXP, REG_EXTENDED | REG_ADVF | REG_NEWLINE) != 0)
		return	False ;
	pLM->m_aRegMatch [0].rm_so	= -1 ;
	pLM->m_aRegMatch [0].rm_eo	= -1 ;
	fRetval	= !ReExec (&re, pStrSTRING + nStart, nStrSTRING - nStart, &rd, MAX_REGEXP_MATCHES, pLM->m_aRegMatch, 0) ;
	ReFree (&re) ;

	pMatch	= pLM->m_aRegMatch ;
	for (i = 0 ; i < MAX_REGEXP_MATCHES && pMatch->rm_eo >= 0 ; i ++, pMatch ++) {
		pMatch->rm_so	+= nStart ;
		pMatch->rm_eo	+= nStart ;
	}
	return	fRetval ;
}

/*
 *
 */
Boolean
lispMachineCode_SearchForward (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntBuffer,
	register const Char*	pStrPATTERN,
	register int			nStrPATTERN,
	register int			nBOUND,
	register int			nCOUNT)
{
	register TLispManager*	pLispMgr ;
	register Boolean			fFound ;
	TBufStringMarker		mkBuffer, mkBufferBack ;
	register int			nCheck, nOffset ;
	int						nStrSTRING ;
	int						nPoint, nPointMin, nPointMax, nPointBTop ;
	register const Char*	pPtr ;
	register int			nPtr ;

	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	assert (nStrPATTERN >= 0) ;
	lispMachineCode_SetRegmatchTarget (pLM, pEntBuffer) ;
	lispBuffer_GetString      (pLispMgr, pEntBuffer, &mkBuffer, &nStrSTRING) ;
	lispBuffer_Point          (pLispMgr, pEntBuffer, &nPoint) ;
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointBTop) ;
	lispBuffer_PointMin       (pLispMgr, pEntBuffer, &nPointMin) ;
	lispBuffer_PointMax       (pLispMgr, pEntBuffer, &nPointMax) ;

#if defined (DEBUG)
	fprintf (stderr, "(code) search-forward: pattern(%d) = \"", nStrPATTERN) ;
	cputstr (stderr, pStrPATTERN, nStrPATTERN) ;
	fprintf (stderr, "\", nCount(%d)\n", nCOUNT) ;
#endif
	TBufStringMarker_Forward (&mkBuffer, nPoint - nPointBTop) ;
	if (nPointMax > nBOUND)
		nPointMax	= nBOUND ;
	nPtr	= nPointMax - nPoint ;
	nOffset	= 0 ;
	assert (nPtr <= nStrSTRING) ;

	fFound						= False ;
	while (nCOUNT -- > 0) {
		pLM->m_aRegMatch [0].rm_so	= -1 ;
		pLM->m_aRegMatch [0].rm_eo	= -1 ;
		fFound	= False ;
		
		while (nPtr >= nStrPATTERN && TFAILED (fFound)) {
			mkBufferBack	= mkBuffer ;
			pPtr			= pStrPATTERN ;
			nCheck			= nStrPATTERN ;
			while (nCheck > 0 && *pPtr == TBufStringMarker_GetChar (&mkBuffer)) {
				TBufStringMarker_Forward (&mkBuffer, 1) ;
				pPtr	++ ;
				nCheck -- ;
			}
			if (nCheck == 0) {
				pLM->m_aRegMatch [0].rm_so	= nOffset + nPoint ;
				pLM->m_aRegMatch [0].rm_eo	= nOffset + nPoint + nStrPATTERN ;
#if defined (DEBUG)
				fprintf (stderr, "(search-forward) [%d ... %d]\n", 
						 (int)pLM->m_aRegMatch [0].rm_so, (int)pLM->m_aRegMatch [0].rm_eo) ;
#endif
				fFound	= True ;
			}
			mkBuffer		= mkBufferBack ;
			TBufStringMarker_Forward (&mkBuffer, 1) ;
			nOffset	++ ;
			nPtr	-- ;
		}
		TBufStringMarker_Forward (&mkBuffer, nStrPATTERN) ;
		nOffset	+= nStrPATTERN ;
		nPtr	-= nStrPATTERN ;
	}
	return	fFound ;
}

/*
 *
 */
Boolean
lispMachineCode_SearchBackward (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntBuffer,
	register const Char*	pStrPATTERN,
	register int			nStrPATTERN,
	register int			nBOUND,
	register int			nCOUNT)
{
	register TLispManager*	pLispMgr ;
	register Boolean			fFound ;
	TBufStringMarker		mkBuffer, mkBufferBack ;
	register int			nCheck, nOffset ;
	int						nStrSTRING ;
	int						nPoint, nPointMin, nPointMax, nPointBTop ;
	register const Char*	pPtr ;
	register int			nPtr ;

	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	assert (nStrPATTERN >= 0) ;
	lispMachineCode_SetRegmatchTarget (pLM, pEntBuffer) ;
	lispBuffer_GetString      (pLispMgr, pEntBuffer, &mkBuffer, &nStrSTRING) ;
	lispBuffer_Point          (pLispMgr, pEntBuffer, &nPoint) ;
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointBTop) ;
	lispBuffer_PointMin       (pLispMgr, pEntBuffer, &nPointMin) ;
	lispBuffer_PointMax       (pLispMgr, pEntBuffer, &nPointMax) ;

#if defined (DEBUG)
	fprintf (stderr, "(code) search-backward: pattern(%d) = \"", nStrPATTERN) ;
	cputstr (stderr, pStrPATTERN, nStrPATTERN) ;
	fprintf (stderr, "\"\n") ;
	fprintf (stderr, "(code) search-backward: target(%d) = \"", nStrSTRING) ;
	fprintf (stderr, "\"\n") ;
	fprintf (stderr, "(code) search-backward: (min/-/max) = (%d/%d/%d)\n", 
			 nPointMin, nPoint, nPointMax) ;
#endif
	nPoint	-- ;
	TBufStringMarker_Forward (&mkBuffer, nPoint - nPointBTop) ;
	if (nPointMin < nBOUND)
		nBOUND	= nPointMin ;
	nPtr	= nPoint - nBOUND ;
	nOffset	= 0 ;
	assert (nPtr <= nStrSTRING) ;

	fFound						= False ;
	while (nCOUNT -- > 0 && nPtr > 0) {
		pLM->m_aRegMatch [0].rm_so	= -1 ;
		pLM->m_aRegMatch [0].rm_eo	= -1 ;
		fFound	= False ;
		
		while (nPtr >= nStrPATTERN) {
			mkBufferBack	= mkBuffer ;
			pPtr			= pStrPATTERN + nStrPATTERN - 1 ;
			nCheck			= nStrPATTERN ;
			while (nCheck > 0 && *pPtr == TBufStringMarker_GetChar (&mkBuffer)) {
				TBufStringMarker_Backward (&mkBuffer, 1) ;
				pPtr	-- ;
				nCheck	-- ;
			}
			if (nCheck == 0) {
				pLM->m_aRegMatch [0].rm_so	= nOffset + nPoint - nStrPATTERN ;
				pLM->m_aRegMatch [0].rm_eo	= nOffset + nPoint ;
				fFound	= True ;
#if defined (DEBUG)
				fprintf (stderr, "(search-backward) [%d ... %d]\n", 
						 (int) pLM->m_aRegMatch [0].rm_so, (int) pLM->m_aRegMatch [0].rm_eo) ;
#endif
				break ;
			}
			mkBuffer		= mkBufferBack ;
			TBufStringMarker_Backward (&mkBuffer, 1) ;
			nOffset	-- ;
			nPtr	-- ;
		}
		TBufStringMarker_Backward (&mkBuffer, nStrPATTERN) ;
		nOffset	-= nStrPATTERN ;
		nPtr	-= nStrPATTERN ;
	}
	return	fFound ;
}

Boolean
lispMachineCode_ReSearchForward (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntBuffer,
	register const Char*	pStrREGEXP,
	register int			nStrREGEXP,
	register int			nBOUND,
	register int			nCOUNT)
{
	register TLispManager*	pLispMgr ;
	register Boolean			fRetval	= False ;
	register int			i ;
	register regmatch_t*	pMatch ;
	regex_t					re ;
	rm_detail_t				rd ;
	TBufStringMarker		mkStart, mkStop ;
	int						nStrSTRING ;
	int						nPoint, nPointMin, nPointMax, nPointBTop ;
	register int			nBase, nBaseAdd ;

	assert (pLM != NULL) ;
#if defined (DEBUG)
	fprintf (stderr, "(code) re-search-forward: regexp= \"") ;
	cputstr (stderr, pStrREGEXP, nStrREGEXP) ;
	fprintf (stderr, "\"\n") ;
#endif
	if (ReComp (&re, pStrREGEXP, nStrREGEXP, REG_EXTENDED | REG_NEWLINE) != 0)
		return	False ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_SetRegmatchTarget (pLM, pEntBuffer) ;
	lispBuffer_GetString      (pLispMgr, pEntBuffer, &mkStart, &nStrSTRING) ;
	lispBuffer_Point          (pLispMgr, pEntBuffer, &nPoint) ;
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointBTop) ;
	lispBuffer_PointMin       (pLispMgr, pEntBuffer, &nPointMin) ;
	lispBuffer_PointMax       (pLispMgr, pEntBuffer, &nPointMax) ;
#if defined (DEBUG)
	fprintf (stderr, "(code) re-search-forward: (min/-/max) = (%d/%d/%d)\n", 
			 nPointMin, nPoint, nPointMax) ;
#endif
	TBufStringMarker_Forward (&mkStart, nPoint - nPointBTop) ;
	if (nPointMax > nBOUND)
		nPointMax	= nBOUND ;
	assert ((nPointMax - nPointBTop) <= nStrSTRING) ;
	nStrSTRING	= nPointMax - nPoint ;
	if (nStrSTRING <= 0)
		nStrSTRING	= 0 ;
	mkStop	= mkStart ;
	TBufStringMarker_Forward (&mkStop, nStrSTRING) ;

	nBase		= 0 ;
	nBaseAdd	= 0 ;
	while (nCOUNT -- > 0) {
		nBase		+= nBaseAdd ;
		pLM->m_aRegMatch [0].rm_so	= -1 ;
		pLM->m_aRegMatch [0].rm_eo	= -1 ;
		fRetval	= !ReExec_2 (&re, mkStart, mkStop, &rd, MAX_REGEXP_MATCHES, pLM->m_aRegMatch, 0) ;
		if (TFAILED (fRetval))
			break ;
		nBaseAdd	=  pLM->m_aRegMatch [0].rm_eo ;
		TBufStringMarker_Forward (&mkStart, nBaseAdd) ;
	}

	/*	pStrSTRING  buffer-string  point ΤǡҥåȤ
	 *	֤ˤϤɬפˤʤ롣
	 */
	pMatch	= pLM->m_aRegMatch ;
	for (i = 0 ; i < MAX_REGEXP_MATCHES && pMatch->rm_eo >= 0 ; i ++) {
		pMatch->rm_so	++ ;
		pMatch->rm_eo	++ ;
#if defined (DEBUG)
		fprintf (stderr, "(%d) [rm_so ... rm_eo] = [%ld ... %ld]\n",
				 i, pMatch->rm_so, pMatch->rm_eo) ;
#endif
		pMatch	++ ;
	}
	ReFree (&re) ;
	return	fRetval ;
}

/*	re-search-backward ¸뤿δؿ
 *
 *	re-search-backward  re-search-forward (ǤϤʤ)δطˤ롣
 *	point ƬäȤᤤ REGEXP ˰פ֤ɽõϵ
 *	ˤɤʤ(ΥȤʧΤΤʤ)ΤǡƬᤤפȤ
 *	ˤʤ롣
 *	ǡդˤɤʤȤʤ limit  point ޤǽ֤õ뤷꤬ʤ
 *	ơ O(n) 롣
 */
Boolean
lispMachineCode_ReSearchBackward (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntBuffer,
	register const Char*	pStrREGEXP,
	register int			nStrREGEXP,
	register int			nBOUND,
	register int			nCOUNT)
{
	register TLispManager*	pLispMgr ;
	register Boolean			fRetval	= False ;
	register regmatch_t*	pMatch ;
	register int			nForward, nHit, n, i ;
	regmatch_t				match ;
	TVarbuffer				vbufMatch ;
	regex_t					re ;
	rm_detail_t				rd ;
	TBufStringMarker		mkStart, mkStop ;
	int						nStrSTRING ;
	int						nPoint, nPointMin, nPointMax, nPointBTop ;

	assert (pLM != NULL) ;

#if defined (DEBUG)
	fprintf (stderr, "(code) re-search-backward: regexp= \"") ;
	cputstr (stderr, pStrREGEXP, nStrREGEXP) ;
	fprintf (stderr, "\"\n") ;
#endif

	if (TFAILED (TVarbuffer_Initialize (&vbufMatch, sizeof (regmatch_t))) ||
		TFAILED (TVarbuffer_Require    (&vbufMatch, nCOUNT)))
		return	False ;
	if (ReComp (&re, pStrREGEXP, nStrREGEXP, REG_EXTENDED | REG_NEWLINE) != 0) {
		TVarbuffer_Uninitialize (&vbufMatch) ;
		return	False ;
	}
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_SetRegmatchTarget (pLM, pEntBuffer) ;
	lispBuffer_GetString      (pLispMgr, pEntBuffer, &mkStart, &nStrSTRING) ;
	lispBuffer_Point          (pLispMgr, pEntBuffer, &nPoint) ;
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointBTop) ;
	lispBuffer_PointMin       (pLispMgr, pEntBuffer, &nPointMin) ;
	lispBuffer_PointMax       (pLispMgr, pEntBuffer, &nPointMax) ;
#if defined (DEBUG)
	fprintf (stderr, "(code) re-search-backward: (min/-/max) = (%d/%d/%d)\n", 
			 nPointMin, nPoint, nPointMax) ;
#endif

	/*	õγϰ֤ nBOUND ǻꤵ롣
	 */
	if (nBOUND < nPointMin)
		nBOUND		= nPointMin ;
	TBufStringMarker_Forward (&mkStart, nBOUND - nPointBTop) ;

	assert (nStrSTRING > (nPoint - nBOUND)) ;
	nStrSTRING	= nPoint - nBOUND ;
	if (nStrSTRING <= 0)
		nStrSTRING	= 0 ;
	mkStop	= mkStart ;
	TBufStringMarker_Forward (&mkStop,  nStrSTRING) ;

	pMatch		= TVarbuffer_GetBuffer (&vbufMatch) ;
	for (nHit = 0 ; ; nHit ++) {
		n			= nHit % nCOUNT ;
		match.rm_so	= -1 ;
		match.rm_eo	= -1 ;
		fRetval	= !ReExec_2 (&re, mkStart, mkStop, &rd, 1, &match, 0) ;
		if (TFAILED (fRetval) || match.rm_so < 0)
			break ;
		memcpy (pMatch + n, &match, sizeof (regmatch_t)) ;
		nForward	= match.rm_eo - TBufStringMarker_GetOffset (&mkStart) ;
		assert (nForward > 0) ;
		TBufStringMarker_Forward (&mkStart, nForward) ;
	}
	/*	٤ҥåȤʤСNOT_MATCH ֤
	 */
	if (nHit <= 0) 
		goto	not_hit ;

	if (nHit < nCOUNT) {
		nHit	= 0 ;
	} else {
		nHit	= (nHit - nCOUNT) % nCOUNT ;
	}
	assert (0 <= nHit && nHit < nCOUNT) ;
	nForward	= (pMatch + nHit)->rm_so - TBufStringMarker_GetOffset (&mkStart) ;
	TBufStringMarker_Forward (&mkStart, nForward) ;
	nForward	= (pMatch + nHit)->rm_eo - TBufStringMarker_GetOffset (&mkStop) ;
	TBufStringMarker_Forward (&mkStop,  nForward) ;
	/*	˥ޥå롣Submatch ɤ줯餤Τʬʤ
	 *	ʤᤷ⡣
	 */
	fRetval	= !exec_2 (&re, mkStart, mkStop, &rd, MAX_REGEXP_MATCHES, pLM->m_aRegMatch, 0) ;
	assert (TSUCCEEDED (fRetval)) ;

	/*	pStrSTRING  buffer-string  point ΤǡҥåȤ
	 *	֤ˤϤɬפˤʤ롣
	 */
	pMatch	= pLM->m_aRegMatch ;
	for (i = 0 ; i < MAX_REGEXP_MATCHES && pMatch->rm_eo >= 0 ; i ++) {
		pMatch->rm_so	++ ;
		pMatch->rm_eo	++ ;
#if defined (DEBUG)
		fprintf (stderr, "(%d) [rm_so ... rm_eo] = [%ld ... %ld]\n",
				 i, pMatch->rm_so, pMatch->rm_eo) ;
#endif
		pMatch	++ ;
	}
  not_hit:
	ReFree (&re) ;
	TVarbuffer_Uninitialize (&vbufMatch) ;
	return	fRetval ;
}

Boolean
lispMachineCode_MatchBeginning (
	register TLispMachine*	pLM,
	register int			nSubexp,
	register int*			pnRetval)
{
	assert (pLM      != NULL) ;
	assert (pnRetval != NULL) ;
	if (nSubexp < 0 || nSubexp >= MAX_REGEXP_MATCHES ||
		pLM->m_aRegMatch [nSubexp].rm_so < 0) 
		return	False ;
	*pnRetval	= pLM->m_aRegMatch [nSubexp].rm_so ;
	return	True ;
}

Boolean
lispMachineCode_MatchEnd (
	register TLispMachine*	pLM,
	register int			nSubexp,
	register int*			pnRetval)
{
	assert (pLM      != NULL) ;
	assert (pnRetval != NULL) ;
	if (nSubexp < 0 || nSubexp >= MAX_REGEXP_MATCHES ||
		pLM->m_aRegMatch [nSubexp].rm_eo < 0) 
		return	False ;
	*pnRetval	= pLM->m_aRegMatch [nSubexp].rm_eo ;
	return	True ;
}

Boolean
lispMachineCode_MatchData (
	register TLispMachine*	pLM,
	register TLispEntity**	ppEntRetval)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	register regmatch_t*	pMatch ;
	register int			i ;
	TLispEntity*	pEntTop ;
	TLispEntity*	pEntTail ;
	TLispEntity*	pEntPosition ;

	pEntTop		= NULL ;
	pEntTail	= NULL ;
	pMatch		= pLM->m_aRegMatch ;
	for ( i = 0 ; i < MAX_REGEXP_MATCHES ; i ++, pMatch ++) {
		if (pMatch->rm_so < 0)
			break ;
		if (TFAILED (lispMgr_CreateInteger (pLispMgr, pMatch->rm_so, &pEntPosition)))
			return	False ;
		lispMachine_addTail (pLispMgr, pEntPosition, &pEntTop, &pEntTail) ;
		if (TFAILED (lispMgr_CreateInteger (pLispMgr, pMatch->rm_eo, &pEntPosition)))
			return	False ;
		lispMachine_addTail (pLispMgr, pEntPosition, &pEntTop, &pEntTail) ;
	}
	if (pEntTop == NULL) {
		lispMgr_CreateNil (pLispMgr, ppEntRetval) ;
	} else {
		lispEntity_Release (pLispMgr, pEntTop) ;
		*ppEntRetval	= pEntTop ;
	}
	return	True ;
}

Boolean
lispMachineCode_CreateMarker (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntBuffer,
	register int			nPosition,
	register TLispEntity**	ppEntRetval)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntMarker ;

	assert (pLM         != NULL) ;
	assert (pLispMgr    != NULL) ;
	assert (ppEntRetval != NULL) ;

	if (TFAILED (lispMgr_CreateMarker (pLispMgr, &pEntMarker))) 
		return	False ;
	if (pEntBuffer != NULL) {
		lispBuffer_AddMarker (pLispMgr, pEntBuffer, pEntMarker) ;
		lispMarker_SetBufferPosition (pLispMgr, pEntMarker, pEntBuffer, nPosition) ;
	}
	*ppEntRetval	= pEntMarker ;
	return	True ;
}

Boolean
lispMachineCode_Featurep (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntFeature)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	register TLispEntity*	pEntFeatures ;
	TLispEntity*	pEntCar ;
	TLispEntity*	pValFeatures ;

	pEntFeatures	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_FEATURES) ;
	if (TFAILED (lispMachine_GetCurrentSymbolValue (pLM, pEntFeatures, &pValFeatures)) ||
		TSUCCEEDED (lispEntity_Voidp (pLispMgr, pValFeatures))) 
		return	False ;

	while (TFAILED (lispEntity_Nullp (pLispMgr, pValFeatures))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pValFeatures, &pEntCar)) ||
			TFAILED (lispEntity_GetCdr (pLispMgr, pValFeatures, &pValFeatures)))
			return	False ;
		if (TSUCCEEDED (lispEntity_Eq (pLispMgr, pEntCar, pEntFeature))) {
			return	True ;
		}
	}
	return	False ;
}

Boolean
lispMachineCode_Provide (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntFeature)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	register TLispEntity*	pEntFeatures ;
	TLispEntity*	pValFeatures ;
	TLispEntity*	pEntity ;

	if (TSUCCEEDED (lispMachineCode_Featurep (pLM, pEntFeature)))
		return	True ;

	pEntFeatures	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_FEATURES) ;
	if (TFAILED (lispMachine_GetCurrentSymbolValue (pLM, pEntFeatures, &pValFeatures)) ||
		TSUCCEEDED (lispEntity_Voidp (pLispMgr, pValFeatures))) 
		lispMgr_CreateNil (pLispMgr, &pValFeatures) ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pEntFeature, pValFeatures, &pEntity)))
		return	False ;
	return	lispMachine_SetCurrentSymbolValue (pLM, pEntFeatures, pEntity) ;
}

/*	private functions */
Boolean
lispMachine_addTail (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntElement,
	register TLispEntity**	ppEntTop,
	register TLispEntity**	ppEntTail)
{
	register TLispEntity*	pEntTail ;
	TLispEntity*			pEntNil ;
	TLispEntity*			pEntNewTail ;

	assert (pLispMgr    != NULL) ;
	assert (pEntElement != NULL) ;
	assert (ppEntTop    != NULL) ;
	assert (ppEntTail   != NULL) ;

	lispMgr_CreateNil (pLispMgr, &pEntNil) ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pEntElement, pEntNil, &pEntNewTail)))
		return	False ;
	pEntTail	= *ppEntTail ;
	if (pEntTail == NULL) {
		assert (*ppEntTop == NULL) ;
		*ppEntTop	= pEntNewTail ;
		lispEntity_AddRef (pLispMgr, pEntNewTail) ;
	} else {
		if (TFAILED (lispEntity_SetCdr (pLispMgr, pEntTail, pEntNewTail)))
			return	False ;
	}
	*ppEntTail	= pEntNewTail ;
	return	True ;
}

#if defined (DEBUG)
void
cputstr (
	register FILE*			pFile,
	register const Char*	pString,
	register int			nLength)
{
	KANJISTATEMACHINE	ksm ;
	int					n ;
	char				achBuf [16] ;

	InitializeKanjiFiniteStateMachine (&ksm, KCODING_SYSTEM_SHIFTJIS) ;
	while (nLength > 0) {
		n	= RtransferKanjiFiniteStateMachine (&ksm, *pString ++, achBuf) ;
		achBuf [n]	= '\0' ;
		fprintf (pFile, "%s", achBuf) ;
		nLength	-- ;
	}
	return ;
}
#endif


