/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <assert.h>
#include "lmachinep.h"

static	Boolean	lispMachine_backwardCharN	(TLispMachine*, int) ;
static	Boolean	lispMachine_forwardCharN	(TLispMachine*, int) ;
static	Boolean	lispMachine_beginningOfLineN(TLispMachine*, int) ;
static	Boolean	lispMachine_endOfLineN		(TLispMachine*, int) ;
static	Boolean	lispMachine_deleteCharN		(TLispMachine*, int) ;
static	Boolean	lispMachine_forwardWordN	(TLispMachine*, int, Boolean*) ;
static	Boolean	lispMachine_backwardWordN	(TLispMachine*, int, Boolean*) ;
static	Boolean	lispMachine_forwardLineN	(TLispManager*, TLispEntity*, int, int*) ;
static	Boolean	lispMachine_backwardLineN	(TLispManager*, TLispEntity*, int, int*) ;

/*
 *	(self-insert-command)
 *
 *	last-command-event (last-command-char) ʤФʤʤ
 *	ȡ command Ǥ롣 function ǤϤʤ
 */
TLMRESULT
lispMachineState_SelfInsertCommand (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	register TLispEntity*	pEntLastCmdEvent ;
	TLispEntity*			pArglist ;
	TLispEntity*			pEntBuffer ;
	TLispEntity*			pLastKey ;
	TLispEntity*			pNum ;
	int						nPosition ;
	long					lValue, lNum ;
	register Char			cc ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	/*
	 *	"P" Ƿ֤ͿƤɬפ롣
	 */
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pArglist, &pNum)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pNum, &lNum))) 
		goto	error ;

	/*	current buffer ꤷơ*/
	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;

	/*	ǸϤ줿ʸ򤽤롣*/
	pEntLastCmdEvent	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_LAST_COMMAND_EVENT) ;
	if (TFAILED (lispMachine_GetCurrentSymbolValue (pLM, pEntLastCmdEvent, &pLastKey)) ||
		pLastKey == NULL ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pLastKey, &lValue))) {
		return	LMR_RETURN ;
	}
	cc	= (Char) lValue ;
	if (Char_Charset (cc) >= MAX_CHARSET) {
		if (Char_Charset (cc) != KCHARSET_XCHAR) 
			goto	error ;
		cc	= Char_MakeAscii (Char_Code (cc) & 0x7F) ; 
	}
	lispBuffer_Point (pLispMgr, pEntBuffer, &nPosition) ;
	lispBuffer_InsertChar (pLM->m_pLispMgr, pEntBuffer, nPosition, cc, lNum) ;
	return	LMR_RETURN ;

  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*
 *	(delete-char N &optional KILLFLAG)
 *
 *	³ N ʸ롣⤷ N ʤ N ʸ롣
 *	ץʥ2ܤΰ KILLFLAG  non-nil ʤСkill 
 *	롣
 */
TLMRESULT
lispMachineState_DeleteChar (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pArglist ;
	TLispEntity*			pNum ;
	long					lNum ;
	
	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	assert (pArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pArglist, &pNum)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pNum, &lNum)) ||
		TFAILED (lispMachine_deleteCharN (pLM, lNum)))
		lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*
 *	(backward-char &optional N)
 */
TLMRESULT
lispMachineState_BackwardChar (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pArglist ;
	TLispEntity*			pNum ;
	long					lNum ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	assert (pArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pArglist, &pNum)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pNum, &lNum))) 
		lNum	= 1 ;

	if (TFAILED (lispMachine_backwardCharN (pLM, lNum)))
		lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*
 *	(forward-char &optional N)
 */
TLMRESULT
lispMachineState_ForwardChar (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pArglist ;
	TLispEntity*			pNum ;
	TLispEntity*			pEntRetval ;
	long					lNum ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	assert (pArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pArglist, &pNum)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pNum, &lNum))) 
		lNum	= 1 ;

	if (TFAILED (lispMachine_forwardCharN (pLM, lNum)))
		lispMachineCode_SetError (pLM) ;

	lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*
 *	(forward-word COUNT)
 */
TLMRESULT
lispMachineState_ForwardWord (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pArglist ;
	TLispEntity*			pNum ;
	TLispEntity*			pRetEntity ;
	long					lNum ;
	Boolean					fRetval ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	assert (pArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pArglist, &pNum)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pNum, &lNum))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
#if defined (DEBUG)
	fprintf (stderr, "(forward-word ") ;
	lispEntity_Print (pLispMgr, pArglist) ;
	fprintf (stderr, ")\n") ;
#endif

	if (TFAILED (lispMachine_forwardWordN (pLM, lNum, &fRetval)))
		lispMachineCode_SetError (pLM) ;
	if (TSUCCEEDED (fRetval)) {
		lispMgr_CreateT   (pLispMgr, &pRetEntity) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pRetEntity) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetEntity) ;
	return	LMR_RETURN ;
}

/*	interactive built-in function:
 *	(beginning-of-line &optional N)
 */
TLMRESULT
lispMachineState_BeginningOfLine (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pArglist ;
	TLispEntity*			pNum ;
	long					lNum ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	assert (pArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pArglist, &pNum)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pNum, &lNum))) 
		lNum	= 1 ;

	if (TFAILED (lispMachine_beginningOfLineN (pLM, lNum)))
		lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*	interactive built-in function:
 *		(end-of-line &optional N)
 */
TLMRESULT
lispMachineState_EndOfLine (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pArglist ;
	TLispEntity*			pNum ;
	long					lNum ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	assert (pArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pArglist, &pNum)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pNum, &lNum))) 
		lNum	= 1 ;

	if (TFAILED (lispMachine_endOfLineN (pLM, lNum)))
		lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*	interactive built-in function:
 *		(forward-line &optional N)
 *
 *	N ˰ư (N ʤ N )⤷ point  I ˤ顢Τ
 *	I + N ԤƬ˰ư롣⤷Ĥ꤬ʤˤʤäʤ¤ʤࡣ
 *	(顼ˤϤʤʤ)
 *	ưΤ˻ĤäԤο֤⤷˿ʤǤΤʤ顢N - ưԿ
 *	ˤʤ⤷ä顢N + ưԿˤʤ롣N ʤ顢Ǥʤ
 *	ԤκǸǤäưԤȤƥȤ롣
 */
TLMRESULT
lispMachineState_ForwardLine (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*		pEntBuffer ;
	TLispEntity*		pEntArglist ;
	TLispEntity*		pEntN ;
	long				lN ;
	int					nRetval ;
	TLispEntity*		pEntRetval ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntN) ;
	if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntN, &lN))) {
		if (TFAILED (lispEntity_Nullp (pLispMgr, pEntN))) {
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
		lN	= 1 ;
	}
	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;
	lispMachine_forwardLineN (pLispMgr, pEntBuffer, lN, &nRetval) ;
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, nRetval, &pEntRetval)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*
 *	backward-delete-char  delete-backward-char  alias
 */
TLMRESULT
lispMachineState_DeleteBackwardChar (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pArglist ;
	TLispEntity*			pNum ;
	long					lNum ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	assert (pArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pArglist, &pNum)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pNum, &lNum)) ||
		TFAILED (lispMachine_deleteCharN (pLM, - lNum)))
		lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*
 *	(delete-region START END)
 *
 *	point  mark δ֤ΥƥȤõ롣ץफ
 *	ƤФ줿ˤϰ2Ĥ뤳ȤԤ롣
 *	(integer ޤ marker)
 */
TLMRESULT
lispMachineState_DeleteRegion (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntCar ;
	TLispEntity*	pEntBuffer ;
	TLispNumber		numStart, numEnd ;
	int				nPointBTop ;
	register int	nPosition, nCount ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntCar)) ||
		TFAILED (lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pEntCar, &numStart)) ||
		TSUCCEEDED (numStart.m_fFloatp) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntCar)) ||
		TFAILED (lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pEntCar, &numEnd)) ||
		TSUCCEEDED (numEnd.m_fFloatp)) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointBTop) ;
	if (numStart.m_Value.m_lLong < numEnd.m_Value.m_lLong) {
		nPosition	= numStart.m_Value.m_lLong ;
		nCount		= numEnd.m_Value.m_lLong - numStart.m_Value.m_lLong ;
	} else {
		nPosition	= numEnd.m_Value.m_lLong ;
		nCount		= numStart.m_Value.m_lLong - numEnd.m_Value.m_lLong ;
	}
#if defined (DEBUG)
	{
		int	nPointMin, nPointMax, nPoint ;
		lispBuffer_Point    (pLispMgr, pEntBuffer, &nPoint) ;
		lispBuffer_PointMin (pLispMgr, pEntBuffer, &nPointMin) ;
		lispBuffer_PointMax (pLispMgr, pEntBuffer, &nPointMax) ;
		fprintf (stderr, "buffer: (btop/min/-/max) = (%d/%d/%d/%d)\n",
				 nPointBTop, nPointMin, nPoint, nPointMax) ;
		fprintf (stderr, "delete-region: %d - %d\n", nPosition, nPosition + nCount) ;
	}
#endif
	if (TFAILED (lispBuffer_DeleteChar (pLispMgr, pEntBuffer, nPosition, nCount))) {
		lispMachineCode_SetError (pLM) ;
	} else {
		TLispEntity*	pEntNil ;
		lispMgr_CreateNil (pLispMgr, &pEntNil) ;
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntNil) ;
	}
	return	LMR_RETURN ;
}

/*	interactive built-in function
 *	(goto-char POSITION)
 *
 *	POSITION, number ޤ marker  point 򥻥åȤ롣
 *	ХåեƬ (point-min)Ǹ (point-max) Ǥ롣
 *()
 *	ȡϰϤʡĤ⤷ narrowing Ƥɤ
 *	ʤΤ
 */
TLMRESULT
lispMachineState_GotoChar (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntPOSITION ;
	TLispNumber		num ;
	TLispEntity*	pEntCurBuffer ;
	TLispEntity*	pMkPoint ;
	int				nPointMax, nPointMin ;
	register int	nPoint ;
	
	lispMachineCode_GetCurrentBuffer (pLM, &pEntCurBuffer) ;
	assert (pEntCurBuffer != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntPOSITION) ;
	if (TFAILED (lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pEntPOSITION, &num)) ||
		TSUCCEEDED (num.m_fFloatp)) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	nPoint	= num.m_Value.m_lLong ;
	lispBuffer_PointMin (pLispMgr, pEntCurBuffer, &nPointMin) ;
	lispBuffer_PointMax (pLispMgr, pEntCurBuffer, &nPointMax) ;
	if (nPoint < nPointMin)
		nPoint	= nPointMin ;
	if (nPoint > nPointMax)
		nPoint	= nPointMax ;

	lispBuffer_PointMarker (pLispMgr, pEntCurBuffer, &pMkPoint) ;
	lispMarker_SetBufferPosition (pLispMgr, pMkPoint, pEntCurBuffer, nPoint) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntPOSITION) ;
	return	LMR_RETURN ;
}

/*	private functions */
Boolean
lispMachine_backwardCharN (
	register TLispMachine*	pLM,
	register int			nNum)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntBuffer ;
	TLispEntity*			pMkPoint ;
	int						nPointMin, nPoint ;
	register Boolean		fRetval	= True ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	if (nNum < 0)
		return	lispMachine_forwardCharN (pLM, - nNum) ;

	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;
	lispBuffer_PointMarker (pLispMgr, pEntBuffer, &pMkPoint) ;
	lispBuffer_PointMin (pLispMgr, pEntBuffer, &nPointMin) ;
	lispMarker_GetBufferPosition (pLispMgr, pMkPoint, &pEntBuffer, &nPoint) ;
	nPoint	-= nNum ;
	if (nPoint < nPointMin) {
		nPoint	= nPointMin ;
		/*	message ("Beginning of buffer") */
		fRetval	= False ;
	}
	lispMarker_SetBufferPosition (pLispMgr, pMkPoint, pEntBuffer, nPoint) ;
	return	fRetval ;
}

Boolean
lispMachine_forwardCharN (
	register TLispMachine*	pLM,
	register int			nNum)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntBuffer ;
	TLispEntity*			pMkPoint ;
	int						nPointMax, nPoint ;
	register Boolean		fRetval	= True ;

	if (nNum < 0)
		return	lispMachine_backwardCharN (pLM, - nNum) ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;
	lispBuffer_PointMarker (pLispMgr, pEntBuffer, &pMkPoint) ;
	lispBuffer_PointMax (pLispMgr, pEntBuffer, &nPointMax) ;
	lispMarker_GetBufferPosition (pLispMgr, pMkPoint, &pEntBuffer, &nPoint) ;

	nPoint	+= nNum ;
	if (nPoint > nPointMax) {
		nPoint	= nPointMax ;
		/*	message ("End of buffer") */
		fRetval	= False ;
	}
	lispMarker_SetBufferPosition (pLispMgr, pMkPoint, pEntBuffer, nPoint) ;
	return	fRetval ;
}

Boolean
lispMachine_beginningOfLineN (
	register TLispMachine*	pLM,
	register int			nNum)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntBuffer ;
	TLispEntity*			pMkPoint ;
	int						nPointMin, nPointMax, nPoint, nPointBTop, nLength ;
	TBufStringMarker		mkBuffer ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;
	lispBuffer_GetString   (pLispMgr, pEntBuffer, &mkBuffer, &nLength) ;
	lispBuffer_PointMarker (pLispMgr, pEntBuffer, &pMkPoint) ;
	lispBuffer_PointMax    (pLispMgr, pEntBuffer, &nPointMax) ;
	lispBuffer_PointMin    (pLispMgr, pEntBuffer, &nPointMin) ;
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointBTop) ;
	lispMarker_GetBufferPosition (pLispMgr, pMkPoint, &pEntBuffer, &nPoint) ;

	TBufStringMarker_Forward (&mkBuffer, nPoint - nPointMin) ;
	nNum	-- ;
	if (nNum > 0) {
		while (nPoint < nPointMax) {
			if (TBufStringMarker_GetChar (&mkBuffer) == '\n') {
				nNum	-- ;
				if (nNum <= 0) {
					TBufStringMarker_Forward (&mkBuffer, 1) ;
					nPoint	++ ;
					break ;
				}
			}
			TBufStringMarker_Forward (&mkBuffer, 1) ;
			nPoint	++ ;
		}
	} else {
		nNum	= - nNum ;
		if (nPointMin < nPoint) {
			nPoint	-- ;
			TBufStringMarker_Backward (&mkBuffer, 1) ;

			while (nPointMin < nPoint) {
				if (TBufStringMarker_GetChar (&mkBuffer) == '\n') {
					nNum	-- ;
					if (nNum <= 0) {
						TBufStringMarker_Forward (&mkBuffer, 1) ;
						nPoint	++ ;
						break ;
					}
				}
				TBufStringMarker_Backward (&mkBuffer, 1) ;
				nPoint	-- ;
			}
		}
	}
	lispMarker_SetBufferPosition (pLispMgr, pMkPoint, pEntBuffer, nPoint) ;
	return	(nNum <= 0)? True : False ;
}

Boolean
lispMachine_endOfLineN (
	register TLispMachine*	pLM,
	register int			nNum)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntBuffer ;
	TLispEntity*			pMkPoint ;
	int						nPointMin, nPointMax, nPoint, nLength, nPointBTop ;
	TBufStringMarker		mkBuffer ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;
	lispBuffer_GetString   (pLispMgr, pEntBuffer, &mkBuffer, &nLength) ;
	lispBuffer_PointMarker (pLispMgr, pEntBuffer, &pMkPoint) ;
	lispBuffer_PointMax    (pLispMgr, pEntBuffer, &nPointMax) ;
	lispBuffer_PointMin    (pLispMgr, pEntBuffer, &nPointMin) ;
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointBTop) ;
	lispMarker_GetBufferPosition (pLispMgr, pMkPoint, &pEntBuffer, &nPoint) ;

	TBufStringMarker_Forward (&mkBuffer, nPoint - nPointMin) ;
	nNum	-- ;
	if (nNum >= 0) {
		while (nPoint < nPointMax) {
			if (TBufStringMarker_GetChar (&mkBuffer) == '\n') {
				nNum	-- ;
				if (nNum <= 0) 
					break ;
			}
			TBufStringMarker_Forward (&mkBuffer, 1) ;
			nPoint	++ ;
		}
	} else {
		nNum	= - nNum ;
		while (nPointMin < nPoint) {
			if (TBufStringMarker_GetChar (&mkBuffer) == '\n') {
				nNum	-- ;
				if (nNum <= 0) 
					break ;
			}
			TBufStringMarker_Backward (&mkBuffer, 1) ;
			nPoint	-- ;
		}
	}
	lispMarker_SetBufferPosition (pLispMgr, pMkPoint, pEntBuffer, nPoint) ;
	return	(nNum <= 0)? True : False ;
}

Boolean
lispMachine_deleteCharN (
	register TLispMachine*	pLM,
	register int			nNum)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntBuffer ;
	TLispEntity*			pNil ;
	int						nPoint, nPointMin, nPointMax ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;
	lispBuffer_Point    (pLispMgr, pEntBuffer, &nPoint) ;
	lispBuffer_PointMin (pLispMgr, pEntBuffer, &nPointMin) ;
	lispBuffer_PointMax (pLispMgr, pEntBuffer, &nPointMax) ;
	if (nNum < 0) {
		if ((nPoint + nNum) < nPointMin) {
			/*	beginning-of-buffer */
			return	False ;
		}
		nPoint	+= nNum ;
		nNum	= -nNum ;
	}
	if ((nPoint + nNum) > nPointMax) {
		/*	end-of-buffer */
		return	False ;
	}
	lispBuffer_DeleteChar (pLispMgr, pEntBuffer, nPoint, nNum) ;
	lispMgr_CreateNil (pLispMgr, &pNil) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pNil) ;
	return	True ;
}


Boolean
lispMachine_forwardWordN (
	register TLispMachine*	pLM,
	register int			nNum,
	register Boolean*		pfRetval)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntBuffer ;
	TLispEntity*			pMkPoint ;
	int						nPointMax, nPoint, nLength, nPointBTop ;
	TBufStringMarker		mkBuffer ;
	register Char			cc ;

	assert (pLM != NULL) ;
	assert (pfRetval != NULL) ;

	if (nNum < 0)
		return	lispMachine_backwardWordN (pLM, - nNum, pfRetval) ;

	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;
	lispBuffer_GetString   (pLispMgr, pEntBuffer, &mkBuffer, &nLength) ;
	lispBuffer_PointMarker (pLispMgr, pEntBuffer, &pMkPoint) ;
	lispBuffer_PointMax    (pLispMgr, pEntBuffer, &nPointMax) ;
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointBTop) ;
	lispMarker_GetBufferPosition (pLispMgr, pMkPoint, &pEntBuffer, &nPoint) ;

	TBufStringMarker_Forward (&mkBuffer, nPoint - nPointBTop) ;
	while (nNum > 0) {
		while (nPoint < nPointMax) {
			cc	= TBufStringMarker_GetChar (&mkBuffer) ;
			if ((((Char)0x61 <= cc && cc <= (Char)0x7a) ||
				 ((Char)0x41 <= cc && cc <= (Char)0x5a) ||
				 ((Char)0x30 <= cc && cc <= (Char)0x39)))
				break ;
			TBufStringMarker_Forward (&mkBuffer, 1) ;
			nPoint	++ ;
		}
		if (nPoint < nPointMax) {
			while (nPoint < nPointMax) {
				cc	= TBufStringMarker_GetChar (&mkBuffer) ;
				if (!(((Char)0x61 <= cc && cc <= (Char)0x7a) ||
					  ((Char)0x41 <= cc && cc <= (Char)0x5a) ||
					  ((Char)0x30 <= cc && cc <= (Char)0x39)))
					break ;
				TBufStringMarker_Forward (&mkBuffer, 1) ;
				nPoint	++ ;
			}
			nNum	-- ;
		} else {
			break ;
		}
	}
	lispMarker_SetBufferPosition (pLispMgr, pMkPoint, pEntBuffer, nPoint) ;
	*pfRetval	= (nNum == 0)? True : False ;
	return	True ;
}

Boolean
lispMachine_backwardWordN (
	register TLispMachine*	pLM,
	register int			nNum,
	register Boolean*		pfRetval)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntBuffer ;
	TLispEntity*			pMkPoint ;
	int						nPointMin, nPoint, nLength, nPointBTop ;
	TBufStringMarker		mkBuffer ;
	register Char			cc ;

	assert (pLM != NULL) ;
	assert (pfRetval != NULL) ;

	if (nNum < 0)
		return	lispMachine_forwardWordN (pLM, - nNum, pfRetval) ;

	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;
	lispBuffer_GetString   (pLispMgr, pEntBuffer, &mkBuffer, &nLength) ;
	lispBuffer_PointMarker (pLispMgr, pEntBuffer, &pMkPoint) ;
	lispBuffer_PointMin    (pLispMgr, pEntBuffer, &nPointMin) ;
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointBTop) ;
	lispMarker_GetBufferPosition (pLispMgr, pMkPoint, &pEntBuffer, &nPoint) ;

	TBufStringMarker_Forward (&mkBuffer, nPoint - nPointBTop) ;
	/*	ߥΤ֤ˤɤʸäƤ⡢ʸ̵뤹롣*/
	if (nPointMin < nPoint) {
		TBufStringMarker_Backward (&mkBuffer, 1) ;
		nPoint	-- ;
	}
	while (nNum > 0) {
		/*	Word ʤʸɤФ*/
		while (nPointMin < nPoint) {
			cc	= TBufStringMarker_GetChar (&mkBuffer) ;
			if (((Char)0x61 <= cc && cc <= (Char)0x7a) ||
				((Char)0x41 <= cc && cc <= (Char)0x5a) ||
				((Char)0x30 <= cc && cc <= (Char)0x39))
				break ;
			TBufStringMarker_Backward (&mkBuffer, 1) ;
			nPoint	-- ;
		}
		/*	Word ʸɤФ*/
		if (nPointMin < nPoint) {
			while (nPointMin < nPoint) {
				cc	= TBufStringMarker_GetChar (&mkBuffer) ;
				if (!(((Char)0x61 <= cc && cc <= (Char)0x7a) ||
					  ((Char)0x41 <= cc && cc <= (Char)0x5a) ||
					  ((Char)0x30 <= cc && cc <= (Char)0x39)))
					break ;
				TBufStringMarker_Backward (&mkBuffer, 1) ;
				nPoint	-- ;
			}
			nNum	-- ;
		} else {
			break ;
		}
	}
	/*	Word ʸƬ᤹*/
	if (nNum == 0) {
		cc	= TBufStringMarker_GetChar (&mkBuffer) ;
		if (!(((Char)0x61 <= cc && cc <= (Char)0x7a) ||
			  ((Char)0x41 <= cc && cc <= (Char)0x5a) ||
			  ((Char)0x30 <= cc && cc <= (Char)0x39))) 
			nPoint	++ ;
		*pfRetval	= True ;
	} else {
		*pfRetval	= False ;
	}
	lispMarker_SetBufferPosition (pLispMgr, pMkPoint, pEntBuffer, nPoint) ;
	return	True ;
}

Boolean
lispMachine_forwardLineN (
	register TLispManager*	pLispMgr, 
	register TLispEntity*	pEntBuffer,
	register int			nCount,
	register int*			pnLeftCount)
{
	TLispEntity*		pMkPoint ;
	int					nPointMin, nPointMax, nPoint, nLength ;
	register int		nLineTop ;
	TBufStringMarker	mkBuffer ;
	register Char		cc ;

	if (nCount < 0)
		return	lispMachine_backwardLineN (pLispMgr, pEntBuffer, nCount, pnLeftCount) ;

	lispBuffer_GetString   (pLispMgr, pEntBuffer, &mkBuffer, &nLength) ;
	lispBuffer_PointMarker (pLispMgr, pEntBuffer, &pMkPoint) ;
	lispBuffer_Point       (pLispMgr, pEntBuffer, &nPoint) ;
	lispBuffer_PointMin    (pLispMgr, pEntBuffer, &nPointMin) ;
	lispBuffer_PointMax    (pLispMgr, pEntBuffer, &nPointMax) ;

	TBufStringMarker_Forward (&mkBuffer, nPoint - nPointMin) ;
	while (nCount > 0) {
		nLineTop	= nPoint ;
		while (nPoint < nPointMax) {
			cc	= TBufStringMarker_GetChar (&mkBuffer) ;
			if (cc == '\n')
				break ;
			TBufStringMarker_Forward (&mkBuffer, 1) ;
			nPoint	++ ;
		}
		/*	ХåեκǸޤƤΤ
		 */
		if (nPoint >= nPointMax) {
			if (nLineTop < nPoint)
				nCount	-- ;
			break ;
		}
		/*	ιԤƬؤȥư롣
		 */
		TBufStringMarker_Forward (&mkBuffer, 1) ;
		nPoint	++ ;
		nCount	-- ;
	}
	lispMarker_SetBufferPosition (pLispMgr, pMkPoint, pEntBuffer, nPoint) ;
	*pnLeftCount	= nCount ;
	return	True ;
}

Boolean
lispMachine_backwardLineN (
	register TLispManager*	pLispMgr, 
	register TLispEntity*	pEntBuffer,
	register int			nCount,
	register int*			pnLeftCount)
{
	TLispEntity*		pMkPoint ;
	int					nPointMin, nPoint, nLength ;
	TBufStringMarker	mkBuffer ;
	register Char		cc ;

	if (nCount > 0)
		return	lispMachine_forwardLineN (pLispMgr, pEntBuffer, nCount, pnLeftCount) ;

	lispBuffer_GetString   (pLispMgr, pEntBuffer, &mkBuffer, &nLength) ;
	lispBuffer_PointMarker (pLispMgr, pEntBuffer, &pMkPoint) ;
	lispBuffer_Point       (pLispMgr, pEntBuffer, &nPoint) ;
	lispBuffer_PointMin    (pLispMgr, pEntBuffer, &nPointMin) ;

	TBufStringMarker_Forward (&mkBuffer, nPoint - nPointMin) ;
	while (nCount < 0) {
		while (nPoint > nPointMin) {
			cc	= TBufStringMarker_GetChar (&mkBuffer) ;
			if (cc == '\n')
				break ;
			TBufStringMarker_Backward (&mkBuffer, 1) ;
			nPoint	-- ;
		}
		if (nPoint <= nPointMin) 
			break ;
		/*	ΰ֤ȡǥ뤬ߤƤޤΤǡ
		 *	롼פȴ塢Ƭ˰ưɬפ롣
		 */
		TBufStringMarker_Backward (&mkBuffer, 1) ;
		nPoint	-- ;
		nCount	++ ;
	}
	/*	ߤƤ륫ƬؤȰư롣Хåե
	 *	ƬƤʤ餽θ¤ǤϤʤ
	 */
	while (nPoint > nPointMin) {
		cc	= TBufStringMarker_GetChar (&mkBuffer) ;
		/*	ԥɤդ顢1᤹ǹƬˤȦ
		 */
		if (cc == '\n') {
			TBufStringMarker_Forward (&mkBuffer, 1) ;
			nPoint	++ ;
			break ;
		}
		TBufStringMarker_Backward (&mkBuffer, 1) ;
		nPoint	-- ;
	}
	lispMarker_SetBufferPosition (pLispMgr, pMkPoint, pEntBuffer, nPoint) ;
	*pnLeftCount	= nCount ;
	return	True ;
}

