/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <assert.h>
#include "lmachinep.h"
#include "cstring.h"

/*
 *	
 */
TLMRESULT
lispMachineState_Car (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pLIST ;
	TLispEntity*	pCar ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;

	if (TFAILED (lispEntity_GetCar (pLispMgr, pArglist, &pLIST)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pLIST,    &pCar))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pCar) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Cdr (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pLIST ;
	TLispEntity*	pCdr ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;

	if (TFAILED (lispEntity_GetCar (pLispMgr, pArglist, &pLIST)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pLIST,    &pCdr))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pCdr) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Cons (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pCAR ;
	TLispEntity*	pCDR ;
	TLispEntity*	pCONS ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pCAR)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pArglist, &pCDR))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	/*	CONS κ˼ԤΤ FATAL ERROR Ǥ롣*/
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pCAR, pCDR, &pCONS)) ||
		pCONS == NULL)
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pCONS) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_CarSafe (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntLIST ;
	TLispEntity*	pEntRetval ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntLIST)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pEntLIST,    &pEntRetval))) 
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Set (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pSymbol ;
	TLispEntity*	pValue ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pSymbol)) ||
		TFAILED (lispEntity_Symbolp (pLispMgr, pSymbol))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispEntity_GetCadr (pLispMgr, pArglist, &pValue)) ||
		TFAILED (lispMachine_SetCurrentSymbolValue (pLM, pSymbol, pValue))) 
		lispMachineCode_SetError (pLM) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pValue) ;
	return	LMR_RETURN ;
}

/*
 *	(fset SYMBOL DEFINITION)
 */
TLMRESULT
lispMachineState_Fset (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntSymbol ;
	TLispEntity*	pEntDefinition ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntSymbol)) ||
		TFAILED (lispEntity_Symbolp (pLispMgr, pEntSymbol)) ||
		TFAILED (lispEntity_GetCdr  (pLispMgr, pEntArglist, &pEntArglist)) ||
		TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntDefinition)) ||
		TFAILED (lispMachine_SetSymbolFunctionValue (pLM, pEntSymbol, pEntDefinition))) {
		lispMachineCode_SetError (pLM) ;
	} else {
#if defined (DEBUG)
		fprintf (stderr, "(fset ") ;
		lispEntity_Print (pLispMgr, pEntSymbol) ;
		fprintf (stderr, " ") ;
		lispEntity_Print (pLispMgr, pEntDefinition) ;
		fprintf (stderr, "\n") ;
#endif
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntSymbol) ;
	}
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Eq (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pObj1 ;
	TLispEntity*	pObj2 ;
	TLispEntity*	pRetval ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pObj1)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pArglist, &pObj2))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_Eq (pLispMgr, pObj1, pObj2))) {
		lispMgr_CreateT   (pLispMgr, &pRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Equal (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pObj1 ;
	TLispEntity*	pObj2 ;
	TLispEntity*	pRetval ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pObj1)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pArglist, &pObj2))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_Equal (pLispMgr, pObj1, pObj2))) {
		lispMgr_CreateT   (pLispMgr, &pRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Null (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pObject ;
	TLispEntity*	pRetval ;

	assert (pLM       != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr  != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pObject))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pObject))) {
		lispMgr_CreateT   (pLispMgr, &pRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Numberp (TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pObject ;
	TLispEntity*	pRetval ;

	assert (pLM       != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr  != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pObject))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_Numberp (pLispMgr, pObject))) {
		lispMgr_CreateT   (pLispMgr, &pRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Integerp (TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pObject ;
	TLispEntity*	pRetval ;

	assert (pLM       != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr  != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pObject))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_Integerp (pLispMgr, pObject))) {
		lispMgr_CreateT   (pLispMgr, &pRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Floatp (TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pObject ;
	TLispEntity*	pRetval ;

	assert (pLM       != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr  != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pObject))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_Floatp (pLispMgr, pObject))) {
		lispMgr_CreateT   (pLispMgr, &pRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Stringp (TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pObject ;
	TLispEntity*	pRetval ;

	assert (pLM       != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr  != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pObject))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_Stringp (pLispMgr, pObject))) {
		lispMgr_CreateT   (pLispMgr, &pRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Consp (TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pObject ;
	TLispEntity*	pRetval ;

	assert (pLM       != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr  != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pObject))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_Consp (pLispMgr, pObject))) {
		lispMgr_CreateT   (pLispMgr, &pRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Listp (TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pObject ;
	TLispEntity*	pRetval ;

	assert (pLM       != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr  != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pObject))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_Listp (pLispMgr, pObject))) {
		lispMgr_CreateT   (pLispMgr, &pRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Vectorp (TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pObject ;
	TLispEntity*	pRetval ;

	assert (pLM       != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr  != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pObject))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_Vectorp (pLispMgr, pObject))) {
		lispMgr_CreateT   (pLispMgr, &pRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Arrayp (TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pObject ;
	TLispEntity*	pRetval ;

	assert (pLM       != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr  != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pObject))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_Arrayp (pLispMgr, pObject))) {
		lispMgr_CreateT   (pLispMgr, &pRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Sequencep (TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pObject ;
	TLispEntity*	pRetval ;

	assert (pLM       != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr  != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pObject))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_Sequencep (pLispMgr, pObject))) {
		lispMgr_CreateT   (pLispMgr, &pRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Markerp (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pObject ;
	TLispEntity*	pRetval ;

	assert (pLM       != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr  != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pObject))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_Markerp (pLispMgr, pObject))) {
		lispMgr_CreateT   (pLispMgr, &pRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_IntegerOrMarkerp (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pObject ;
	TLispEntity*	pRetval ;

	assert (pLM       != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr  != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pObject))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_IntegerOrMarkerp (pLispMgr, pObject))) {
		lispMgr_CreateT   (pLispMgr, &pRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Subrp (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pObject ;
	TLispEntity*	pRetval ;

	assert (pLM       != NULL) ;
	assert (pLispMgr  != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pObject))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_Subrp (pLispMgr, pObject))) {
		lispMgr_CreateT   (pLispMgr, &pRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

/*
 *	byte-code ưȤ¿ʬʤΤǡɬ nil ֤
 */
TLMRESULT
lispMachineState_ByteCodeFunctionp (
	register TLispMachine*	pLM)
{
	TLispEntity*	pEntNil ;
	assert (pLM != NULL) ;
	assert (pLM->m_pLispMgr != NULL) ;
	lispMgr_CreateNil (pLM->m_pLispMgr, &pEntNil) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntNil) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Assoc (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pCar ;
	TLispEntity*	pList ;
	TLispEntity*	pKey ;
	TLispEntity*	pRetval ;
	
	pLispMgr	= pLM->m_pLispMgr ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pKey)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pArglist, &pList)) ||
		TFAILED (lispEntity_Listp (pLispMgr, pList))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	(void) lispMgr_CreateNil (pLispMgr, &pRetval) ;
	
	while (TFAILED (lispEntity_Nullp (pLispMgr, pList))) {
		if (TSUCCEEDED (lispEntity_GetCaar (pLispMgr, pList, &pCar))) {
			if (TSUCCEEDED (lispEntity_Equal (pLispMgr, pKey, pCar))) {
				lispEntity_GetCar (pLispMgr, pList, &pRetval) ;
				break ;
			}
		}
		(void) lispEntity_GetCdr (pLispMgr, pList, &pList) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Assq (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pCar ;
	TLispEntity*	pList ;
	TLispEntity*	pKey ;
	TLispEntity*	pRetval ;
	
	pLispMgr	= pLM->m_pLispMgr ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;

	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pKey)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pArglist, &pList)) ||
		TFAILED (lispEntity_Listp (pLispMgr, pList))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMgr_CreateNil (pLispMgr, &pRetval) ;
	
	while (TFAILED (lispEntity_Nullp (pLispMgr, pList))) {
		if (TSUCCEEDED (lispEntity_GetCaar (pLispMgr, pList, &pCar)) &&
			TSUCCEEDED (lispEntity_Eq (pLispMgr, pKey, pCar))) {
			lispEntity_GetCar (pLispMgr, pList, &pRetval) ;
			break ;
		}
		(void )lispEntity_GetCdr (pLispMgr, pList, &pList) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Memq (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntList ;
	TLispEntity*	pEntElt ;
	TLispEntity*	pEntRetval ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntElt)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntList)) ||
		TFAILED (lispEntity_Memq    (pLispMgr, pEntElt, pEntList, &pEntRetval))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Member (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntList ;
	TLispEntity*	pEntElt ;
	TLispEntity*	pEntRetval ;
	
	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntElt)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntList)) ||
		TFAILED (lispEntity_Member  (pLispMgr, pEntElt, pEntList, &pEntRetval))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Nthcdr (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pN ;
	TLispEntity*	pList ;
	long			lN ;
	
	pLispMgr	= pLM->m_pLispMgr ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pN)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pArglist, &pList)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pN, &lN)) ||
		TFAILED (lispEntity_Listp    (pLispMgr, pList))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	while (lN > 0 && TFAILED (lispEntity_Nullp (pLispMgr, pList))) {
		lispEntity_GetCdr (pLispMgr, pList, &pList) ;
		lN	-- ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pList) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Rassoc (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntList ;
	TLispEntity*	pEntKey ;
	TLispEntity*	pEntRetval ;
	
	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntKey)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntList)) ||
		TFAILED (lispEntity_Rassoc  (pLispMgr, pEntKey, pEntList, &pEntRetval))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*
 *	(nconc &rest LISTS)
 */
TLMRESULT
lispMachineState_Nconc (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntRetval ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_Nconc (pLispMgr, pEntArglist, &pEntRetval))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Append (register TLispMachine* pLM)
{
	TLispEntity*	pArglist ;
	TLispEntity*	pRetval ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispMgr_Append (pLM->m_pLispMgr, pArglist, &pRetval))) {
		lispMachineCode_SetError (pLM) ;
	} else {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	}
	return	LMR_RETURN ;
}

/*
 *	(throw TAG VALUE)
 */
TLMRESULT
lispMachineState_Throw (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*		pArglist ;
	TLispEntity*		pEntTag ;
	TLispEntity*		pEntValue ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	lispEntity_GetCar  (pLispMgr, pArglist, &pEntTag) ;
	lispEntity_GetCadr (pLispMgr, pArglist, &pEntValue) ;
	lispMachineCode_SetException (pLM, pEntTag, pEntValue) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Vconcat (register TLispMachine* pLM)
{
	TLispEntity*	pResult ;
	TLispEntity*	pArglist ;
	
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispMgr_Vconcat (pLM->m_pLispMgr, pArglist, &pResult))) {
		lispMachineCode_SetError (pLM) ;
	} else {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pResult) ;
	}
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Nreverse (register TLispMachine* pLM)
{
	TLispEntity*	pResult ;
	TLispEntity*	pList ;
	TLispEntity*	pArglist ;
	
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar (pLM->m_pLispMgr, pArglist, &pList)) ||
		TFAILED (lispMgr_Nreverse (pLM->m_pLispMgr, pList, &pResult))) {
		lispMachineCode_SetError (pLM) ;
	} else {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pResult) ;
	}
	return	LMR_RETURN ;
}

/*
 *	(delete ELT LIST)
 *
 *	built-in function 
 *	LIST ΥФ ELT ¸ߤ LIST ѤȤƾõ롣
 *	LIST ֤롣Ӥ `equal' ǰ٤롣⤷LIST 
 *	κǽΥФ ELT ʤ顢ѤǤꤷʤ
 *	ñ̤ΥꥹȤȤǤ롣(cdr ֤Ȼפ)
 */
TLMRESULT
lispMachineState_Delete (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pList ;
	TLispEntity*	pElt ;
	TLispEntity*	pRetval ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	assert (pArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pElt)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pArglist, &pList)) ||
		TFAILED (lispMgr_Delete (pLispMgr, pElt, pList, &pRetval))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Delq (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pList ;
	TLispEntity*	pElt ;
	TLispEntity*	pRetval ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	assert (pArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pElt)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pArglist, &pList)) ||
		TFAILED (lispMgr_Delq (pLispMgr, pElt, pList, &pRetval))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

/*
 *	(signal ERROR-SYMBOL DATA)
 *
 *	顼 signal 롣 ERROR-SYMBOL Ȥ˴Ϣդ
 *	ǡǤ롣δؿ֤ͤʤ
 *
 *	顼ܥ `error-conditions' ץѥƥäܥ
 *	롣
 *	ϥɥ
 */
TLMRESULT
lispMachineState_Signal (register TLispMachine* pLM)
{
	TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pErrorSymbol ;
	TLispEntity*	pData ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	assert (pArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pErrorSymbol)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pArglist, &pData)) ||
		TFAILED (lispEntity_Symbolp (pLispMgr, pErrorSymbol))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_SetSignal (pLM, pErrorSymbol, pData) ;
	return	LMR_RETURN ;
}

/*
 *	(setcar CELL NEWCAR)
 *
 *	CELL  car  NEWCAR ꤹ롣NEWCAR ֤
 */
TLMRESULT
lispMachineState_Setcar (TLispMachine* pLM)
{
	TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pCell ;
	TLispEntity*	pNewcar ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	assert (pArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pCell)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pArglist, &pNewcar)) ||
		TFAILED (lispEntity_SetCar  (pLispMgr, pCell, pNewcar))) {
		lispMachineCode_SetError (pLM) ;
	} else {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pNewcar) ;
	}
	return	LMR_RETURN ;
}

/*
 *	(setcdr CELL NEWCDR)
 *
 *	CELL  cdr  NEWCDR ꤹ롣NEWCDR ֤
 */
TLMRESULT
lispMachineState_Setcdr (TLispMachine* pLM)
{
	TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pCell ;
	TLispEntity*	pNewcdr ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	assert (pArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pCell)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pArglist, &pNewcdr)) ||
		TFAILED (lispEntity_SetCdr  (pLispMgr, pCell,    pNewcdr))) {
		lispMachineCode_SetError (pLM) ;
	} else {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pNewcdr) ;
	}
	return	LMR_RETURN ;
}

/*
 *	(boundp SYMBOL)
 *
 *	SYMBOL ͤ void ǤʤСt ֤
 */
TLMRESULT
lispMachineState_Boundp (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntSymbol ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntSymbol)) ||
		TFAILED (lispEntity_Symbolp (pLispMgr, pEntSymbol))) {
		lispMachineCode_SetError (pLM) ;
	} else {
		TLispEntity*	pEntValue ;
		TLispEntity*	pEntReturn ;

		if (TFAILED (lispMachine_GetCurrentSymbolValue (pLM, pEntSymbol, &pEntValue)) ||
			TSUCCEEDED (lispEntity_Voidp (pLispMgr, pEntValue))) {
			lispMgr_CreateNil (pLispMgr, &pEntReturn) ;
		} else {
			lispMgr_CreateT (pLispMgr, &pEntReturn) ;
		}
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntReturn) ;
	}
	return	LMR_RETURN ;
}

/*
 *	(default-value SYMBOL)
 *
 *	SYMBOL  default value 롣default-value Ȥ global value
 *	Τȡ顢local variable ¸ߤˤΤ̣߰Ĵؿ
 *	Ǥ롣
 */
TLMRESULT
lispMachineState_DefaultValue (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntSymbol ;
	TLispEntity*	pEntValue ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	/*	ꥹȤġΰȴФ*/
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntSymbol)) ||
		TFAILED (lispEntity_Symbolp (pLispMgr, pEntSymbol))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispMachine_GetGlobalSymbolValue (pLM, pEntSymbol, &pEntValue)) ||
		TSUCCEEDED (lispEntity_Voidp (pLispMgr, pEntValue))) {
		if (TFAILED (lispEntity_Nullp (pLispMgr, pEntSymbol)) &&
			TFAILED (lispEntity_Tp    (pLispMgr, pEntSymbol))) {
#if defined (DEBUG)
			fprintf (stderr, "Symbol's value is void: ") ;
			lispEntity_Print (pLispMgr, pEntSymbol) ;
			fprintf (stderr, "\n") ;
#endif
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntSymbol) ;
	} else {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntValue) ;
	}
	return	LMR_RETURN ;
}

/*
 *	(symbol-value SYMBOL)
 *
 *	SYMBOL ֤ͤvoid ʤ error ˤʤ롣
 */
TLMRESULT
lispMachineState_SymbolValue (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntSymbol ;
	TLispEntity*	pEntValue ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	/*	ꥹȤġΰȴФ*/
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntSymbol)) ||
		TFAILED (lispEntity_Symbolp (pLispMgr, pEntSymbol))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispMachine_GetCurrentSymbolValue (pLM, pEntSymbol, &pEntValue)) ||
		TSUCCEEDED (lispEntity_Voidp (pLispMgr, pEntValue))) {
		if (TFAILED (lispEntity_Nullp (pLispMgr, pEntSymbol)) &&
			TFAILED (lispEntity_Tp    (pLispMgr, pEntSymbol))) {
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntSymbol) ;
	} else {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntValue) ;
	}
	return	LMR_RETURN ;
}

/*
 *	(makunbound SYMBOL)
 */
TLMRESULT
lispMachineState_Makunbound (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntSymbol ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntSymbol)) ||
		TFAILED (lispEntity_Symbolp (pLispMgr, pEntSymbol))) {
		lispMachineCode_SetError (pLM) ;
	} else {
		TLispEntity*	pEntVoid ;
		lispMgr_CreateVoid (pLispMgr, &pEntVoid) ;
		lispMachine_SetCurrentSymbolValue (pLM, pEntSymbol, pEntVoid) ;
	}
	return	LMR_RETURN ;
}

/*
 *	(fmakunbound SYMBOL)
 */
TLMRESULT
lispMachineState_Fmakunbound (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntSymbol ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntSymbol)) ||
		TFAILED (lispEntity_Symbolp (pLispMgr, pEntSymbol))) {
		lispMachineCode_SetError (pLM) ;
	} else {
		TLispEntity*	pEntVoid ;
		lispMgr_CreateVoid (pLispMgr, &pEntVoid) ;
		lispMachine_SetSymbolFunctionValue (pLM, pEntSymbol, pEntVoid) ;
	}
	return	LMR_RETURN ;
}

/*
 *	(aref ARRAY IDX)
 *
 *	ARRAY  index IDX Ǥ֤ARRAY  vector ޤʸ
 *	ޤ char-table ޤ bool-vector ޤ byte-code object
 *	Ǥ롣IDX  0 Ϥޤ롣
 */
TLMRESULT
lispMachineState_Aref (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntArray ;
	TLispEntity*	pEntIdx ;
	TLispEntity*	pEntRetval ;
	long			nIdx ;
	Boolean			fRetval ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	/*	ꥹȤġΰȴФ*/
	if (TFAILED (lispEntity_GetCar   (pLispMgr, pEntArglist, &pEntArray)) ||
		TFAILED (lispEntity_Arrayp   (pLispMgr, pEntArray)) ||
		TFAILED (lispEntity_GetCadr  (pLispMgr, pEntArglist, &pEntIdx)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntIdx, &nIdx))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_Vectorp (pLispMgr, pEntArray))) {
		fRetval	= lispEntity_GetVectorElement (pLispMgr, pEntArray, nIdx, &pEntRetval) ;
	} else {
		Char	ch ;
		fRetval	= lispEntity_GetStringElement (pLispMgr, pEntArray, nIdx, &ch) ;
		if (TSUCCEEDED (fRetval)) 
			lispMgr_CreateInteger (pLispMgr, ch, &pEntRetval) ;
	}
	if (TSUCCEEDED (fRetval)) {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	} else {
		lispMachineCode_SetError (pLM) ;
	}
	return	LMR_RETURN ;
}

/*
 *	(aset ARRAY IDX NEWELT)
 *
 *	ARRAY  index IDX Ǥ NEWELT  store 롣ARRAY 
 *	vector, string, char-table, bool-vector Ǥ롣IDX  0 
 *	Ϥޤ롣
 */
TLMRESULT
lispMachineState_Aset (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntArray ;
	TLispEntity*	pEntIdx ;
	TLispEntity*	pEntNewelt ;
	long			nIdx ;
	Boolean			fRetval ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	/*	ꥹȤġΰȴФ*/
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntArray)) ||
		TFAILED (lispEntity_Arrayp  (pLispMgr, pEntArray))) {
#if defined (DEBUG)
		fprintf (stderr, "#<aset> wrong type argument arrayp: ") ;
		lispEntity_Print (pLispMgr, pEntArray) ;
		fprintf (stderr, "\n") ;
#endif
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispEntity_GetCdr  (pLispMgr, pEntArglist, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntIdx)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntIdx, &nIdx))) {
#if defined (DEBUG)
		fprintf (stderr, "#<aset> wrong type argument integerp: ") ;
		lispEntity_Print (pLispMgr, pEntIdx) ;
		fprintf (stderr, "\n") ;
#endif
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntNewelt) ;

	if (TSUCCEEDED (lispEntity_Vectorp (pLispMgr, pEntArray))) {
		fRetval	= lispEntity_SetVectorElement (pLispMgr, pEntArray, nIdx, pEntNewelt) ;
	} else {
		Char	ch ;
		fRetval	= lispEntity_GetIntegerValue (pLispMgr, pEntNewelt, &ch) &&
			lispEntity_SetStringElement (pLispMgr, pEntArray, nIdx, ch) ;
	}
	if (TFAILED (fRetval)) {
		lispMachineCode_SetError (pLM) ;
	} else {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntNewelt) ;
	}
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Vector (register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntElement ;
	TLispEntity*	pEntRetval ;
	TVarbuffer		vbuf ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;

	TVarbuffer_Initialize (&vbuf, sizeof (TLispEntity*)) ;
	while (TFAILED (lispEntity_Nullp (pLispMgr, pEntArglist))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntElement)) ||
			TFAILED (TVarbuffer_Add (&vbuf, &pEntElement, 1)) ||
			TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pEntArglist)))
			return	LMR_ERROR ;
	}
	if (TFAILED (lispMgr_CreateVector (pLispMgr, TVarbuffer_GetBuffer (&vbuf), TVarbuffer_GetUsage (&vbuf), &pEntRetval)))
		return	LMR_ERROR ;
	TVarbuffer_Uninitialize (&vbuf) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Princ (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntElement ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntElement))) {
		lispMachineCode_SetError (pLM) ;
	} else {
		lispEntity_Print (pLispMgr, pEntElement) ;
		fprintf (stderr, "\n") ;
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntElement) ;
	}
	return	LMR_RETURN ;
}

/*
 *	(natnump OBJECT)
 *
 *	Return t if OBJECT is a nonnegative integer.
 */
TLMRESULT
lispMachineState_Natnump (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntObject ;
	TLispEntity*	pEntRetval ;
	long			lValue ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntObject))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntObject, &lValue)) ||
		lValue >= 0) {
		lispMgr_CreateT (pLispMgr, &pEntRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*	built-in function:
 *		(featurep FEATURE)
 *
 *	ѿ `features' ͤ򸫤롣
 */
TLMRESULT
lispMachineState_Featurep (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntFEATURE ;
	TLispEntity*	pEntRetval ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFEATURE) ;
	if (TSUCCEEDED (lispMachineCode_Featurep (pLM, pEntFEATURE))) {
		lispMgr_CreateT   (pLispMgr, &pEntRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*	built-in function:
 *		(provide FEATURE)
 */
TLMRESULT
lispMachineState_Provide (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntFEATURE ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntFEATURE) ;
	if (TFAILED (lispMachineCode_Provide (pLM, pEntFEATURE)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntFEATURE) ;
	return	LMR_RETURN ;
}

