/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <assert.h>
#include "lispmgrp.h"

#define	lispEntity_GetVectorPtr(ptr)	((TLispVector *)((TLispEntity *)(ptr) + 1))

static	Boolean	lispMgr_vconcatConscell	(TLispManager*, TLispEntity*, TVarbuffer*) ;
static	Boolean	lispMgr_vconcatString	(TLispManager*, TLispEntity*, TVarbuffer*) ;
static	Boolean	lispMgr_vconcatVector	(TLispManager*, TLispEntity*, TVarbuffer*) ;
static	Boolean	lispMgr_vconcatInteger	(TLispManager*, TLispEntity*, TVarbuffer*) ;

Boolean
lispMgr_CreateVector (
	register TLispManager*			pLispMgr,
	register TLispEntity** const	ppElement,
	register int					nElement,
	register TLispEntity** const	ppReturn)
{
	TLispEntity*			pEntity ;
	register TLispVector*	pVector ;
	register size_t			nSize ;

	assert (pLispMgr != NULL) ;
	assert (nElement == 0 || (nElement > 0 && ppElement != NULL)) ;
	assert (ppReturn != NULL) ;

#if defined (DEBUG_LV99)
	fprintf (stderr, "Garbage collecting ...") ;
	fflush (stderr) ;
	lispMgr_CollectGarbage (pLispMgr) ;
	fprintf (stderr, "done.\n") ;
#endif	
	if (nElement > 1) {
		nSize	= sizeof (TLispVector) + sizeof (TLispEntity *) * nElement ;
	} else {
		nSize	= sizeof (TLispVector) ;
	}
	if (TFAILED (lispMgr_AllocateEntity (pLispMgr, nSize, &pEntity)))
		return	False ;
	
	pEntity->m_iType	= LISPENTITY_VECTOR ;
	pVector				= lispEntity_GetVectorPtr (pEntity) ;
	pVector->m_nElement	= nElement ;
	if (nElement > 0) 
		memcpy (pVector->m_apElement, ppElement, sizeof (TLispEntity *) * nElement) ;
	lispMgr_RegisterMisc (pLispMgr, pEntity) ;

	*ppReturn			= pEntity ;
	return	True ;
}

Boolean
lispMgr_CreateVectorFill (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pEntFill, 
	register int					nElement,
	register TLispEntity** const	ppReturn)
{
	TLispEntity*			pEntity ;
	register TLispVector*	pVector ;
	register TLispEntity**	ppEntPtr ;
	register size_t			nSize ;

	assert (pLispMgr != NULL) ;
	assert (nElement >= 0) ;
	assert (ppReturn != NULL) ;

#if defined (DEBUG_LV99)
	fprintf (stderr, "Garbage collecting ...") ;
	fflush (stderr) ;
	lispMgr_CollectGarbage (pLispMgr) ;
	fprintf (stderr, "done.\n") ;
#endif	
	if (nElement > 1) {
		nSize	= sizeof (TLispVector) + sizeof (TLispEntity *) * nElement ;
	} else {
		nSize	= sizeof (TLispVector) ;
	}
	if (TFAILED (lispMgr_AllocateEntity (pLispMgr, nSize, &pEntity)))
		return	False ;
	
	pEntity->m_iType	= LISPENTITY_VECTOR ;
	pVector				= lispEntity_GetVectorPtr (pEntity) ;
	pVector->m_nElement	= nElement ;

	ppEntPtr	= pVector->m_apElement ;
	while (nElement -- > 0)
		*ppEntPtr ++	= pEntFill ;
	lispMgr_RegisterMisc (pLispMgr, pEntity) ;

	*ppReturn			= pEntity ;
	return	True ;
}


Boolean
lispMgr_Vconcat (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pArglist,
	register TLispEntity** const	ppReturn)
{
	TVarbuffer		vbufElm ;
	TLispEntity*	pCar ;
	TLispEntity*	pNextArglist ;
	TLispEntity**	ppElement ;
	int				iType ;
	int				nSize ;
	Boolean			fRetval	= False ;

	if (TFAILED (TVarbuffer_Initialize (&vbufElm, sizeof (TLispEntity *))))
		return	False ;

	while (TFAILED (lispEntity_Nullp (pLispMgr, pArglist))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pArglist, &pCar)) ||
			pCar == NULL)
			break ;
		(void) lispEntity_GetType (pLispMgr, pCar, &iType) ;
		switch (iType) {
		case	LISPENTITY_INTEGER:
			if (TFAILED (lispMgr_vconcatInteger (pLispMgr, pCar, &vbufElm)))
				goto	exit_loop ;
			break ;
		case	LISPENTITY_CONSCELL:
			if (TFAILED (lispMgr_vconcatConscell (pLispMgr, pCar, &vbufElm)))
				goto	exit_loop ;
			break ;
		case	LISPENTITY_STRING:
			if (TFAILED (lispMgr_vconcatString (pLispMgr, pCar, &vbufElm)))
				goto	exit_loop ;
			break ;
		case	LISPENTITY_VECTOR:
			if (TFAILED (lispMgr_vconcatVector (pLispMgr, pCar, &vbufElm)))
				goto	exit_loop ;
			break ;
		default:
			goto	exit_loop ;
		}
		if (TFAILED (lispEntity_GetCdr (pLispMgr, pArglist, &pNextArglist)) ||
			pNextArglist == NULL)
			break ;
		pArglist	= pNextArglist ;
	}
 exit_loop:
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pArglist))) {
		ppElement	= TVarbuffer_GetBuffer (&vbufElm) ;
		nSize		= TVarbuffer_GetUsage  (&vbufElm) ;
		if (TSUCCEEDED (lispMgr_CreateVector (pLispMgr, ppElement, nSize, ppReturn))) 
			fRetval		= True ;
	}
	ppElement	= TVarbuffer_GetBuffer (&vbufElm) ;
	nSize		= TVarbuffer_GetUsage  (&vbufElm) ;
	while (nSize > 0) {
		lispEntity_Release (pLispMgr, *ppElement ++) ;
		nSize	-- ;
	}
	TVarbuffer_Uninitialize (&vbufElm) ;
	return	fRetval ;
}

Boolean
lispMgr_vconcatConscell (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbufElm)
{
	TLispEntity*	pNextEntity ;
	TLispEntity*	pElement ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (pvbufElm != NULL) ;

	while (TFAILED (lispEntity_Nullp (pLispMgr, pEntity))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pEntity, &pElement)) ||
			pElement == NULL)
			return	False ;
		if (TFAILED (TVarbuffer_Add (pvbufElm, &pElement, 1))) {
			return	False ;
		} else {
			lispEntity_AddRef (pLispMgr, pElement) ;
		}
		if (TFAILED (lispEntity_GetCdr (pLispMgr, pEntity, &pNextEntity)))
			return	False ;
		pEntity	= pNextEntity ;
	}
	return	True ;
}

Boolean
lispMgr_vconcatString (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbufElm)
{
	const Char*	pString ;
	int			nLength ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (pvbufElm != NULL) ;

	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntity, &pString, &nLength)))
		return	False ;

	if (pString != NULL && nLength > 0) {
		TLispEntity*	pChar ;

		while (nLength > 0) {
			if (TFAILED (lispMgr_CreateInteger (pLispMgr, *pString, &pChar)))
				return	False ;
			if (TFAILED (TVarbuffer_Add (pvbufElm, &pChar, 1))) 
				return	False ;
			lispEntity_AddRef (pLispMgr, pChar) ;
			pString	++ ;
			nLength	-- ;
		}
	}
	return	True ;
}

Boolean
lispMgr_vconcatVector (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbufElm)
{
	TLispEntity**	ppElement ;
	int				nElement ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (pvbufElm != NULL) ;

	if (TFAILED (lispEntity_GetVectorValue (pLispMgr, pEntity, &ppElement, &nElement)))
		return	False ;

	if (ppElement != NULL && nElement > 0) {
		while (nElement > 0) {
			/*	Ǥλȥ󥿤 +1 ƤޤʤФʤʤΤǡ
			 *	ޤȤ add Ǥʤġ*/
			if (TFAILED (TVarbuffer_Add (pvbufElm, ppElement, 1))) 
				return	False ;
			lispEntity_AddRef (pLispMgr, *ppElement) ;
			ppElement	++ ;
			nElement	-- ;
		}
	}
	return	True ;
}

Boolean
lispMgr_vconcatInteger (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbufElm)
{
	TLispEntity*	pChar ;
	long			lValue ;
	char			achBuf [256] ;
	char*			ptr ;
	int				nLength ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (pvbufElm != NULL) ;

	if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntity, &lValue)))
		return	False ;
	snprintf (achBuf, NELEMENTS (achBuf) - 1, "%ld", lValue) ;

	nLength	= strlen (achBuf) ;
	ptr		= achBuf ;
	while (nLength > 0) {
		if (TFAILED (lispMgr_CreateInteger (pLispMgr, (Char) *ptr, &pChar)))
			return	False ;
		if (TFAILED (TVarbuffer_Add (pvbufElm, &pChar, 1))) 
			return	False ;
		lispEntity_AddRef (pLispMgr, pChar) ;
		ptr		++ ;
		nLength	-- ;
	}
	return	True ;
}

