/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "lispmgrp.h"
#include "cstring.h"

#define	lispEntity_GetStringPtr(ptr)	((TLispString *)((TLispEntity *)(ptr) + 1))

static	Boolean	stringToCharListString	(TLispManager* pLispMgr, TLispEntity* pSequence, TLispEntity** ppRetval) ;
static	Boolean	stringToCharListVector	(TLispManager* pLispMgr, TLispEntity* pSequence, TLispEntity** ppRetval) ;
static	Boolean	stringToCharListList	(TLispManager* pLispMgr, TLispEntity* pSequence, TLispEntity** ppRetval) ;
static	Boolean	concatEntity			(TLispManager* pLispMgr, TLispEntity* pArg, TVarbuffer* pvbufStr) ;
static	Boolean	concatString			(TLispManager* pLispMgr, TLispEntity* pArg, TVarbuffer* pvbufStr) ;
static	Boolean	concatVector			(TLispManager* pLispMgr, TLispEntity* pArg, TVarbuffer* pvbufStr) ;
static	Boolean	concatList				(TLispManager* pLispMgr, TLispEntity* pArg, TVarbuffer* pvbufStr) ;
static	Boolean	concatInteger			(TLispManager* pLispMgr, TLispEntity* pArg, TVarbuffer* pvbufStr) ;
static	Boolean	substringString			(TLispManager* pLispMgr, TLispEntity* pEntity, long lFrom, long lTo, TLispEntity** ppReturn) ;
static	Boolean	substringVector			(TLispManager* pLispMgr, TLispEntity* pEntity, long lFrom, long lTo, TLispEntity** ppReturn) ;


/*	LISP ʸ롣ʸξ硢ȤƱʸϿƤƤ
 *	ٺ롣*/
Boolean
lispMgr_CreateString (
	register TLispManager* 			pLispMgr,
	register const Char*			pString,
	register int					nString,
	register TLispEntity** const	ppEntReturn)
{
	TLispEntity*	pEntity ;
	register TLispString*	pLsString ;
	register size_t			nSize ;

	assert (pLispMgr    != NULL) ;
	assert (pString     != NULL || nString == 0) ;
	assert (ppEntReturn != NULL) ;

#if defined (DEBUG_LV99)
	fprintf (stderr, "Garbage collecting ...") ;
	fflush (stderr) ;
	lispMgr_CollectGarbage (pLispMgr) ;
	fprintf (stderr, "done.\n") ;
#endif	
	nSize	= sizeof (TLispString) + sizeof (Char) * ((nString > 1)? nString - 1 : 0) ;
	if (TFAILED (lispMgr_AllocateEntity (pLispMgr, nSize, &pEntity)))
		return	False ;
		
	pEntity->m_iType		= LISPENTITY_STRING ;
	pEntity->m_lReferCount	= 0 ;	/* ľϻȥ󥿤 0*/
	
	pLsString				= lispEntity_GetStringPtr (pEntity) ;
	if (nString > 0) {
		memcpy (pLsString->m_achString, pString, sizeof (Char) * nString) ;
	}
	pLsString->m_nLength	= nString ;
	lispMgr_RegisterMisc (pLispMgr, pEntity) ;

	*ppEntReturn	= pEntity ;
	return	True ;
}

Boolean
lispMgr_StringToCharList (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pSequence,
	register TLispEntity** const	ppReturn)
{
	int		iType ;

	assert (pLispMgr  != NULL) ;
	assert (pSequence != NULL) ;
	assert (ppReturn  != NULL) ;

	if (TFAILED (lispEntity_GetType (pLispMgr, pSequence, &iType)))
		return	False ;
	switch (iType) {
	case	LISPENTITY_SYMBOL:
		if (TFAILED (lispEntity_Nullp (pLispMgr, pSequence)))
			return	False ;
		lispMgr_CreateNil (pLispMgr, ppReturn) ;
		return	True ;
	case	LISPENTITY_STRING:
		return	stringToCharListString (pLispMgr, pSequence, ppReturn) ;
	case	LISPENTITY_VECTOR:
		return	stringToCharListVector (pLispMgr, pSequence, ppReturn) ;
	case	LISPENTITY_CONSCELL:
		return	stringToCharListList   (pLispMgr, pSequence, ppReturn) ;
	default:
		return	False ;
	}
}

Boolean
lispMgr_Concat (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pArglist,
	register TLispEntity*			pSeparator,
	register TLispEntity** const	ppReturn)
{
	TVarbuffer		vbufStr ;
	TLispEntity*	pArg ;
	TLispEntity*	pNextArglist ;
	const Char*		pString ;
	int				nLen ;
	TLispEntity*	pReturn ;

	assert (pLispMgr != NULL) ;
	assert (pArglist != NULL) ;
	assert (ppReturn != NULL) ;

	if (TFAILED (TVarbuffer_Initialize (&vbufStr, sizeof (Char)))) 
		return	False ;

	if (TFAILED (lispEntity_Nullp (pLispMgr, pArglist))) {
		for ( ; ; ) {
			if (TFAILED (lispEntity_GetCar (pLispMgr, pArglist, &pArg)))
				break ;
			if (TFAILED (concatEntity (pLispMgr, pArg, &vbufStr)))
				break ;
			
			if (TFAILED (lispEntity_GetCdr (pLispMgr, pArglist, &pNextArglist)))
				break ;
			pArglist	= pNextArglist ;
			if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pArglist)))
				break ;
			if (pSeparator != NULL &&
				TFAILED (concatEntity (pLispMgr, pSeparator, &vbufStr)))
				break ;
		}
	}

	if (TFAILED (lispEntity_Nullp (pLispMgr, pArglist))) {
		TVarbuffer_Uninitialize (&vbufStr) ;
		return	False ;
	}
	pString	= TVarbuffer_GetBuffer (&vbufStr) ;
	nLen	= TVarbuffer_GetUsage  (&vbufStr) ;
	if (TFAILED (lispMgr_CreateString (pLispMgr, pString, nLen, &pReturn))) {
		TVarbuffer_Uninitialize (&vbufStr) ;
		return	False ;
	}
	*ppReturn	= pReturn ;
	return	True ;
}

Boolean
lispMgr_Substring (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pArray,
	register long					lFrom,
	register long					lTo,
	register TLispEntity** const	ppReturn)
{
	int		iType ;
	
	assert (pLispMgr != NULL) ;
	assert (pArray   != NULL) ;

	if (TFAILED (lispEntity_GetType (pLispMgr, pArray, &iType)))
		return	False ;
	switch (iType) {
	case	LISPENTITY_STRING:
		return	substringString (pLispMgr, pArray, lFrom, lTo, ppReturn) ;
	case	LISPENTITY_VECTOR:
		return	substringVector (pLispMgr, pArray, lFrom, lTo, ppReturn) ;
	default:
		return	False ;
	}
}

Boolean
lispEntity_Upcase (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pObj,
	register TLispEntity**	ppReturn)
{
	int			iType ;
	Char		cc ;
	const Char*	pString ;
	int			nLength ;
	Boolean		fResult ;

	assert (pLispMgr  != NULL) ;
	assert (pObj      != NULL) ;
	assert (ppReturn  != NULL) ;

	if (TFAILED (lispEntity_GetType (pLispMgr, pObj, &iType)) ||
		(iType != LISPENTITY_INTEGER && iType != LISPENTITY_STRING))
		return	False ;
	
	if (iType == LISPENTITY_INTEGER) {
		long	lValue ;

		lispEntity_GetIntegerValue (pLispMgr, pObj, &lValue) ;
		cc		= Char_ToUpper ((Char)lValue) ;
		fResult	= lispMgr_CreateInteger (pLispMgr, cc, ppReturn) ;
	} else {
		TVarbuffer	vbufStr ;

		if (TFAILED (TVarbuffer_Initialize (&vbufStr, sizeof (Char))))
			return	False ;

		lispEntity_GetStringValue (pLispMgr, pObj, &pString, &nLength) ;
		if (pString != NULL && nLength > 0) {
			while (nLength > 0) {
				cc	= Char_ToUpper (*pString) ;
				if (TFAILED (TVarbuffer_Add (&vbufStr, &cc, 1))) 
					return	False ;
				pString	++ ;
				nLength	-- ;
			}
		}
		pString	= TVarbuffer_GetBuffer (&vbufStr) ;
		nLength	= TVarbuffer_GetUsage  (&vbufStr) ;
		fResult	= lispMgr_CreateString (pLispMgr, pString, nLength, ppReturn) ;
		TVarbuffer_Uninitialize (&vbufStr) ;
	}
	return	fResult ;
}

Boolean
lispEntity_Downcase (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pObj,
	register TLispEntity**	ppReturn)
{
	int			iType ;
	Char		cc ;
	Boolean		fResult ;

	assert (pLispMgr  != NULL) ;
	assert (pObj      != NULL) ;
	assert (ppReturn  != NULL) ;

	if (TFAILED (lispEntity_GetType (pLispMgr, pObj, &iType)) ||
		(iType != LISPENTITY_INTEGER && iType != LISPENTITY_STRING))
		return	False ;

	if (iType == LISPENTITY_INTEGER) {
		long	lValue ;

		lispEntity_GetIntegerValue (pLispMgr, pObj, &lValue) ;
		cc		= Char_ToLower ((Char)lValue) ;
		fResult	= lispMgr_CreateInteger (pLispMgr, cc, ppReturn) ;
	} else {
		TVarbuffer	vbufStr ;
		const Char*	pString ;
		int			nLength ;

		if (TFAILED (TVarbuffer_Initialize (&vbufStr, sizeof (Char))))
			return	False ;
		lispEntity_GetStringValue (pLispMgr, pObj, &pString, &nLength) ;
		if (pString != NULL && nLength > 0) {
			while (nLength > 0) {
				cc	= Char_ToLower (*pString) ;
				if (TFAILED (TVarbuffer_Add (&vbufStr, &cc, 1))) 
					return	False ;
				pString	++ ;
				nLength	-- ;
			}
		}
		pString	= TVarbuffer_GetBuffer (&vbufStr) ;
		nLength	= TVarbuffer_GetUsage  (&vbufStr) ;
		fResult	= lispMgr_CreateString (pLispMgr, pString, nLength, ppReturn) ;
		TVarbuffer_Uninitialize (&vbufStr) ;
	}
	return	fResult ;
}

/*	private functions */
Boolean
stringToCharListString (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pSequence,
	register TLispEntity**	ppRetval)
{
	TLispEntity*	pRetval ;
	TLispEntity*	pNil ;
	const Char*		pString ;
	int				nLength ;

	lispMgr_CreateNil (pLispMgr, &pNil) ;
	lispEntity_GetStringValue (pLispMgr, pSequence, &pString, &nLength) ;

	if (pString != NULL && nLength > 0) {
		TLispEntity*	pTail ;
		TLispEntity*	pNewTail ;
		TLispEntity*	pChar ;

		if (TFAILED (lispMgr_CreateConscell (pLispMgr, pNil, pNil, &pRetval))) 
			return	False ;
		lispEntity_AddRef (pLispMgr, pRetval) ;

		pTail	= pRetval ;
		for ( ; ; ) {
			if (TFAILED (lispMgr_CreateInteger  (pLispMgr, *pString, &pChar))) 
				break ;
			lispEntity_SetCar (pLispMgr, pTail, pChar) ;
			pString	++ ;
			nLength	-- ;
			if (nLength <= 0)
				break ;
			if (TFAILED (lispMgr_CreateConscell (pLispMgr, pNil, pNil, &pNewTail)))
				break ;
			lispEntity_SetCdr (pLispMgr, pTail, pNewTail) ;
			pTail	= pNewTail ;
		}
		/*	ï⻲Ȥʤ֤ѹ롣GC Υߥ󥰤ϡݤλ
		 *	ʤΤǡ⤽⡢ lisp engine  entity  create ʳ
		 *	ϻȤƤʤ֤ˤʤäƤ뤷*/
		lispEntity_Release (pLispMgr, pRetval) ;
		if (nLength > 0) 
			return	False ;
	} else {
		pRetval	= pNil ;
	}
	*ppRetval	= pRetval ;
	return	True ;
}

Boolean
stringToCharListVector (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pSequence,
	register TLispEntity**	ppRetval)
{
	TLispEntity*	pRetval ;
	TLispEntity*	pNil ;
	TLispEntity**	ppElement ;
	int				nElement ;

	lispMgr_CreateNil (pLispMgr, &pNil) ;
	lispEntity_GetVectorValue (pLispMgr, pSequence, &ppElement, &nElement) ;

	if (ppElement != NULL && nElement > 0) {
		TLispEntity*	pTail ;
		TLispEntity*	pNewTail ;

		if (TFAILED (lispMgr_CreateConscell (pLispMgr, *ppElement, pNil, &pRetval))) 
			return	False ;

		lispEntity_AddRef (pLispMgr, pRetval) ;
		ppElement	++ ;
		nElement	-- ;
		pTail		= pRetval ;
		while (nElement > 0) {
			if (TFAILED (lispMgr_CreateConscell (pLispMgr, *ppElement, pNil, &pNewTail)))
				break ;
			lispEntity_SetCdr (pLispMgr, pTail, pNewTail) ;
			pTail	= pNewTail ;
			ppElement	++ ;
			nElement	-- ;
		}
		lispEntity_Release (pLispMgr, pRetval) ;
		if (nElement > 0)
			return	False ;
	} else {
		pRetval	= pNil ;
	}
	*ppRetval	= pRetval ;
	return	True ;
}

Boolean
stringToCharListList (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pSequence,
	register TLispEntity**	ppRetval)
{
	TLispEntity*	pHead ;
	TLispEntity*	pTail ;
	TLispEntity*	pNewTail ;
	TLispEntity*	pCar ;
	TLispEntity*	pNil ;
	TLispEntity*	pNextSequence ;

	lispMgr_CreateNil (pLispMgr, &pNil) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pSequence))) {
		*ppRetval	= pNil ;
		return	True ;
	}

	if (TFAILED (lispEntity_GetCar (pLispMgr, pSequence, &pCar)) ||
		TFAILED (lispMgr_CreateConscell (pLispMgr, pCar, pNil, &pTail)))
		return	False ;

	pHead	= pTail ;
	lispEntity_AddRef (pLispMgr, pHead) ;
	if (TFAILED (lispEntity_GetCdr (pLispMgr, pSequence, &pNextSequence)))
		return	False ;
	pSequence	= pNextSequence ;
	
	while (TFAILED (lispEntity_Nullp (pLispMgr, pSequence))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pSequence, &pCar)) ||
			TFAILED (lispMgr_CreateConscell (pLispMgr, pCar, pNil, &pNewTail)))
			break ;
		lispEntity_SetCdr (pLispMgr, pTail, pNewTail) ;
		pTail	= pNewTail ;
		if (TFAILED (lispEntity_GetCdr (pLispMgr, pSequence, &pNextSequence)))
			break ;
		pSequence	= pNextSequence ;
	}
	lispEntity_Release (pLispMgr, pHead) ;
	if (TFAILED (lispEntity_Nullp (pLispMgr, pSequence))) {
		*ppRetval	= NULL ;
		return	False ;
	} else {
		*ppRetval	= pHead ;
		return	True ;
	}
}

Boolean
concatEntity (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pArg,
	register TVarbuffer*	pvbufStr)
{
	Boolean	fRetval	= False ;
	int		iType ;
	
	if (TFAILED (lispEntity_GetType (pLispMgr, pArg, &iType)))
		return	False ;
	switch (iType) {
	case	LISPENTITY_STRING:
		fRetval	= concatString (pLispMgr, pArg, pvbufStr) ;
		break ;
	case	LISPENTITY_CONSCELL:
		fRetval	= concatList (pLispMgr, pArg, pvbufStr) ;
		break ;
	case	LISPENTITY_VECTOR:
		fRetval	= concatVector (pLispMgr, pArg, pvbufStr) ;
		break ;
	case	LISPENTITY_INTEGER:
		fRetval	= concatInteger (pLispMgr, pArg, pvbufStr) ;
		break ;
	default:
		break ;
	}
	return	fRetval ;
}

Boolean
concatString (TLispManager* pLispMgr, TLispEntity* pArg, TVarbuffer* pvbufStr)
{
	const Char*	pString ;
	int			nLength ;

	assert (pLispMgr != NULL) ;
	assert (pArg     != NULL) ;
	assert (pvbufStr != NULL) ;

	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pArg, &pString, &nLength)))
		return	False ;
	if (pString != NULL && nLength > 0) 
		return	TVarbuffer_Add (pvbufStr, pString, nLength) ;
	return	True ;
}

Boolean
concatVector (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pArg,
	register TVarbuffer*	pvbufStr)
{
	TLispEntity*	pElement ;
	long			lValue ;
	Char			cc ;
	int				nLength ;
	int				nIndex ;
	
	assert (pLispMgr != NULL) ;
	assert (pArg     != NULL) ;
	assert (pvbufStr != NULL) ;

	if (TFAILED (lispEntity_GetLength (pLispMgr, pArg, &nLength)))
		return	False ;
	
	for (nIndex = 0 ; nIndex < nLength ; nIndex ++) {
		if (TFAILED (lispEntity_GetVectorElement (pLispMgr, pArg, nIndex, &pElement)) ||
			TFAILED (lispEntity_GetIntegerValue  (pLispMgr, pElement, &lValue)))
			return	False ;
		cc	= (Char) lValue ;
		if (TFAILED (TVarbuffer_Add (pvbufStr, &cc, 1)))
			return	False ;
	}
	return	True ;
}

Boolean
concatList (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pArg,
	register TVarbuffer*	pvbufStr)
{
	TLispEntity*	pElement ;
	TLispEntity*	pNextArg ;
	long			lValue ;
	Char			cc ;
	
	assert (pLispMgr != NULL) ;
	assert (pArg     != NULL) ;
	assert (pvbufStr != NULL) ;

	while (TFAILED (lispEntity_Nullp (pLispMgr, pArg))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pArg, &pElement)) ||
			TFAILED (lispEntity_GetIntegerValue (pLispMgr, pElement, &lValue)))
			return	False ;
		cc	= (Char) lValue ;
		if (TFAILED (TVarbuffer_Add (pvbufStr, &cc, 1)))
			return	False ;
		if (TFAILED (lispEntity_GetCdr (pLispMgr, pArg, &pNextArg)))
			return	False ;
		pArg	= pNextArg ;
	}
	return	True ;
}

Boolean
concatInteger (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pArg,
	register TVarbuffer*	pvbufStr)
{
	Char	achBuf [128] ;
	Char*	ptr ;
	long	lValue, lRemainder ;
	long	lUse ;
	
	assert (pLispMgr != NULL) ;
	assert (pArg     != NULL) ;
	assert (pvbufStr != NULL) ;

	if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pArg, &lValue)))
		return	False ;

	if (lValue == 0) {
		Char	cc	= Char_MakeAscii ('0') ;
		return	TVarbuffer_Add (pvbufStr, &cc, 1) ;
	}
#if defined (DEBUG)
	fprintf (stderr, "concat-integer %ld, ", lValue) ;
#endif
	ptr		= achBuf + NELEMENTS (achBuf) - 1 ;
	lUse	= 0 ;
	if (lValue < 0) {
		while (lUse < NELEMENTS(achBuf) && lValue != 0) {
			lRemainder	=  - (lValue % 10) ;
			assert (0 <= lRemainder && lRemainder < 10) ;
			*ptr --		= Char_MakeAscii ('0' + lRemainder) ;
			lValue		=  lValue / 10 ;
			lUse		++ ;
		}
		if (lUse >= NELEMENTS(achBuf) || lValue != 0)
			return	False ;
		*ptr	= '-' ;
	} else {
		while (lUse < NELEMENTS(achBuf) && lValue != 0) {
			lRemainder	= lValue % 10 ;
			assert (0 <= lRemainder && lRemainder < 10) ;
			*ptr --		= Char_MakeAscii ('0' + lRemainder) ;
			lValue		= lValue / 10 ;
			lUse		++ ;
		}
		if (lValue != 0)
			return	False ;
		ptr	++ ;
	}
#if defined (DEBUG)
	fprintf (stderr, "%ld\n", lUse) ;
#endif
	return	TVarbuffer_Add (pvbufStr, ptr, lUse) ;
}

Boolean
substringString (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register long			lFrom,
	register long			lTo,
	register TLispEntity**	ppReturn)
{
	const Char*		pString ;
	int				nLength ;
	TLispEntity*	pRetval ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	
	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntity, &pString, &nLength)))
		return	False ;
	if (lFrom < 0) 
		lFrom	+= nLength ;
	if (lTo < 0)
		lTo		+= nLength ;
	/* out of range Υå򤹤롣*/
	if (lFrom < 0 || lTo < 0 || lFrom > nLength || lTo > nLength)
		return	False ;

	if (TFAILED (lispMgr_CreateString (pLispMgr, pString + lFrom, lTo - lFrom, &pRetval)))
		return	False ;
	*ppReturn	= pRetval ;
	return	True ;
}

Boolean
substringVector (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register long			lFrom,
	register long			lTo,
	register TLispEntity**	ppReturn)
{
	TLispEntity**	ppElement ;
	int				nElement ;
	TLispEntity*	pRetval ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	
	if (TFAILED (lispEntity_GetVectorValue (pLispMgr, pEntity, &ppElement, &nElement)))
		return	False ;
	if (lFrom < 0) 
		lFrom	+= nElement ;
	if (lTo < 0)
		lTo		+= nElement ;
	/* out of range Υå򤹤롣*/
	if (lFrom < 0 || lTo < 0 || lFrom > nElement || lTo > nElement)
		return	False ;
	if (TFAILED (lispMgr_CreateVector (pLispMgr, ppElement + lFrom, lTo - lFrom, &pRetval)))
		return	False ;
	*ppReturn	= pRetval ;
	return	True ;
}

