/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include "lispmgrp.h"
#include "varbuffer.h"
#include "kfile.h"
#undef	DEBUG

/*	*/
enum {
	MAX_UNGETC_BUF		= (16),
} ;

struct tagTLispParser ;
struct tagTLispEntity ;
struct tagTLispManager ;

/*	*/
typedef TLispEntity* (*LISPPARSEFUNC)(TLispManager*, struct tagTLispParser*) ;

typedef struct tagTLispParser {
	union {
		struct {
			KFILE*		m_pFile ;
			Char		m_achBuf [MAX_UNGETC_BUF] ;
			int			m_iBufUsage ;
		}	m_file ;
		struct {
			const Char*	m_pStringTop ;
			const Char*	m_pHead ;
			int			m_iRest ;
		}	m_string ;
		struct {
			const char*	m_pStringTop ;
			const char*	m_pHead ;
			int			m_iRest ;
		}	m_stringA ;
	} m_src ;

	LISPPARSEFUNC	m_pNextStateFunc ;
	Char			(*m_pGetChar)(struct tagTLispParser*) ;
	int				(*m_pUngetChar)(struct tagTLispParser*, Char) ;
}	TLispParser ;

typedef struct tagTLispParserTbl {
	Char			m_cc ;
	TLispEntity*	(*m_pFunc)(TLispManager*, struct tagTLispParser*) ;
}	TLispParserTbl ;

/*	ץȥ*/
static	Boolean			lispParser_checkNextState (TLispParser*) ;
static	TLispEntity*	lispParser_parseSymbol    (TLispManager*, TLispParser*) ;
static	TLispEntity*	lispParser_parseString    (TLispManager*, TLispParser*) ;
static	TLispEntity*	lispParser_parseQuote     (TLispManager*, TLispParser*) ;
/*static	TLispEntity*	lispParser_parseBackquote (TLispManager*, TLispParser*) ;*/
static	TLispEntity*	lispParser_parseList      (TLispManager*, TLispParser*) ;
static	TLispEntity*	lispParser_parseInteger   (TLispManager*, TLispParser*) ;
static	TLispEntity*	lispParser_parseChar	  (TLispManager*, TLispParser*) ;
static	TLispEntity*	lispParser_parseComment   (TLispManager*, TLispParser*) ;
static	TLispEntity*	lispParser_parseArray     (TLispManager*, TLispParser*) ;

static	TLispEntity*	lispParser_parseQuoteCommon    (TLispManager*, TLispParser*, const Char*, int) ;
static	Char			lispParser_SkipSpace           (TLispParser*) ;
static	Char			lispParser_GetStringChar       (TLispParser*) ;
static	int				lispParser_UngetStringChar     (TLispParser*, Char) ;
static	Char			lispParser_GetStringCharA      (TLispParser*) ;
static	int				lispParser_UngetStringCharA    (TLispParser*, Char) ;
static	Char			lispParser_GetFileChar         (TLispParser*) ;
static	int				lispParser_UngetFileChar       (TLispParser*, Char) ;
static	Boolean			lispParser_readBackslashedChar (TLispParser*, Char*) ;

/*	Хؿ*/

TLispEntity*
lispMgr_ParseString (
	register TLispManager*	pLispMgr,
	register const Char*	pString,
	register int			iLength,
	register int*			pnLastIndex)
{
	TLispParser		parser ;
	TLispEntity*	pRetvalue ;
	LISPPARSEFUNC	pNextStateFunc ;

	/*
	 *	ǥեȤ֤͡
	 */
	pRetvalue		= NULL ;

	/*
	 *	ƤӽФ¦˱ƶ̵褦ˡݥ󥿵ڤӿͤ򥳥ԡ롣
	 */
	parser.m_src.m_string.m_pStringTop	= pString ;
	parser.m_src.m_string.m_pHead		= pString ;
	parser.m_src.m_string.m_iRest		= iLength ;
	parser.m_pGetChar		= lispParser_GetStringChar ;
	parser.m_pUngetChar		= lispParser_UngetStringChar ;
	parser.m_pNextStateFunc	= NULL ;

	for ( ; ; ) {
		if (!lispParser_checkNextState (&parser))
			break ;
		pNextStateFunc	= parser.m_pNextStateFunc ;
		if (parser.m_pNextStateFunc == lispParser_parseComment) {
			(void)(pNextStateFunc)(pLispMgr, &parser) ;
		} else {
			pRetvalue	= (pNextStateFunc)(pLispMgr, &parser) ;
			if (pRetvalue == NULL)
				break ;
		}
	}
	/*	ޤɤ߹֤*/
	if (pnLastIndex != NULL)
		*pnLastIndex	= parser.m_src.m_string.m_pHead - parser.m_src.m_string.m_pStringTop ;
	return	pRetvalue ;
}

TLispEntity*
lispMgr_ParseStringA (
	register TLispManager*	pLispMgr,
	register const char*	pString,
	register int			iLength,
	register int*			pnLastIndex)
{
	TLispParser		parser ;
	TLispEntity*	pRetvalue ;
	LISPPARSEFUNC	pNextStateFunc ;

	/*
	 *	ǥեȤ֤͡
	 */
	pRetvalue		= NULL ;

	/*
	 *	ƤӽФ¦˱ƶ̵褦ˡݥ󥿵ڤӿͤ򥳥ԡ롣
	 */
	parser.m_src.m_stringA.m_pStringTop	= pString ;
	parser.m_src.m_stringA.m_pHead		= pString ;
	parser.m_src.m_stringA.m_iRest		= iLength ;
	parser.m_pGetChar		= lispParser_GetStringCharA ;
	parser.m_pUngetChar		= lispParser_UngetStringCharA ;
	parser.m_pNextStateFunc	= NULL ;

	for ( ; ; ) {
		if (!lispParser_checkNextState (&parser))
			break ;
		pNextStateFunc	= parser.m_pNextStateFunc ;
		if (parser.m_pNextStateFunc == lispParser_parseComment) {
			(void)(pNextStateFunc)(pLispMgr, &parser) ;
		} else {
			pRetvalue	= (pNextStateFunc)(pLispMgr, &parser) ;
			if (pRetvalue == NULL)
				break ;
		}
	}
	/*	ޤɤ߹֤*/
	if (pnLastIndex != NULL)
		*pnLastIndex	= parser.m_src.m_stringA.m_pHead - parser.m_src.m_stringA.m_pStringTop ;
	return	pRetvalue ;
}

/*
 *[ǽ]
 *	ե뤫 lisp ̿ɤ߹ entity Ѵ롣
 */
#if defined (WIN32)
TLispEntity*
lispMgr_ParseFile (
	register TLispManager*	pLispMgr, 
	register LPCWSTR		pFileName)
#else
TLispEntity*
lispMgr_ParseFile (
	register TLispManager*	pLispMgr,
	register const char*	pFileName)
#endif
{
	KFILE			kfile ;
	TLispParser		parser ;
	TLispEntity*	pEntity ;
	TLispEntity*	pRetvalue ;
	LISPPARSEFUNC	pNextStateFunc ;
	TVarbuffer		vbEntity ;
	int				nEntity ;

	assert (pLispMgr  != NULL) ;
	assert (pFileName != NULL) ;

	if (TFAILED (TVarbuffer_Initialize (&vbEntity, sizeof (TLispEntity *))))
		return	NULL ;

	if (!KFile_Open (&kfile, pFileName, KCODING_SYSTEM_UNKNOWN/*KCODING_SYSTEM_EUCJP*/))
		return	NULL ;
	KFile_Rewind (&kfile) ;

	pRetvalue				= NULL ;

	parser.m_src.m_file.m_pFile		= &kfile ;
	parser.m_src.m_file.m_iBufUsage	= 0 ;
	parser.m_pGetChar		= lispParser_GetFileChar ;
	parser.m_pUngetChar		= lispParser_UngetFileChar ;
	parser.m_pNextStateFunc	= NULL ;

	for ( ; ; ) {
		if (!lispParser_checkNextState (&parser)) 
			break ;
		pNextStateFunc	= parser.m_pNextStateFunc ;
		if (pNextStateFunc == lispParser_parseComment) {
			(void)(pNextStateFunc)(pLispMgr, &parser) ;
		} else {
			pEntity		= (pNextStateFunc)(pLispMgr, &parser) ;
			if (pEntity == NULL)
				break ;
			TVarbuffer_Add (&vbEntity, &pEntity, 1) ;
		}
	}
	KFile_Close (&kfile) ;

	nEntity		= TVarbuffer_GetUsage (&vbEntity) ;
	if (nEntity > 0) {
		TLispEntity**	ppTop ;
		TLispEntity**	ppEntity ;
		TLispEntity*	pNil ;
		TLispEntity*	pLastElm ;
		TLispEntity*	pListTop ;
		int				i ;

		ppTop		= (TLispEntity **)TVarbuffer_GetBuffer (&vbEntity) ;
		assert (ppTop != NULL) ;
		(void) lispMgr_CreateNil (pLispMgr, &pNil) ;
		if (TFAILED (lispMgr_CreateConscell (pLispMgr, *ppTop, pNil, &pLastElm))) {
			ppEntity	= ppTop ;
			for (i = 0 ; i < nEntity ; i ++) 
				lispEntity_Release (pLispMgr, *ppEntity ++) ;
		} else {
			TLispEntity*	pCdr ;
			
			pListTop	= pLastElm ;
			lispEntity_AddRef  (pLispMgr, pListTop) ;
			lispEntity_Release (pLispMgr, *ppTop) ;
		
			ppEntity	= ppTop + 1 ;
			for (i = 1 ; i < nEntity ; i ++) {
				if (TFAILED (lispMgr_CreateConscell (pLispMgr, *ppEntity, pNil, &pCdr))) 
					break ;
				lispEntity_SetCdr  (pLispMgr, pLastElm, pCdr) ;
				lispEntity_Release (pLispMgr, *ppEntity) ;
				ppEntity	++ ;
				pLastElm	= pCdr ;
			}
			if (i < nEntity) {
				while (i < nEntity) {
					lispEntity_Release (pLispMgr, *ppEntity ++) ;
					i	++ ;
				}
				lispEntity_Release (pLispMgr, pListTop) ;
				pRetvalue	= NULL;
			} else {
				pRetvalue	= pListTop ;
			}
		}
	} else {
		pRetvalue	= NULL ;
	}
	TVarbuffer_Uninitialize (&vbEntity) ;

	return	pRetvalue ;
}

/*========================================================================
 *	ɽؿ
 *========================================================================*/

Boolean
lispParser_checkNextState (
	register TLispParser*	pParser)
{
	static TLispParserTbl	saTable []	= {
		{ 0x22,		lispParser_parseString, },
		{ 0x27,		lispParser_parseQuote, },
		{ 0x28,		lispParser_parseList, },
		{ 0x3B,		lispParser_parseComment, },
		{ 0x5B,		lispParser_parseArray, },
		/*{ 0x60,		lispParser_parseBackquote, },*/
		{ '.',		NULL, /*lispParser_parseInteger,*/ },
		{ ']',		NULL, },
		{ ')',		NULL, },
	} ;
	register TLispParserTbl*	ptr ;
	register int				i ;
	register Char	cc ; 

	assert (pParser != NULL) ;

	/*	ɤФ*/
	if ((cc = lispParser_SkipSpace (pParser)) == EOF)
		return	False ;

	/*	Ƚ̤˻Ȥäʸ򲡤᤹ɬפ롣*/
	(pParser->m_pUngetChar)(pParser, cc) ;

	ptr	= saTable ;
	for (i = 0 ; i < NELEMENTS (saTable) ; i ++, ptr ++)
		if (cc == ptr->m_cc)
			goto	found ;

	if (!Char_DifferenceAscii (cc, '?')) {
		pParser->m_pNextStateFunc	= lispParser_parseChar ;
	} else if (Char_IsDigitNum (cc)) {
		pParser->m_pNextStateFunc	= lispParser_parseInteger ;
	} else {
		pParser->m_pNextStateFunc	= lispParser_parseSymbol ;
	}
	return	True ;

  found:
	pParser->m_pNextStateFunc	= ptr->m_pFunc ;
	return	(ptr->m_pFunc != NULL)? True : False ;
}

/*
 *	³ʸ SYMBOL Ǥ롣
 */
TLispEntity*
lispParser_parseSymbol (
	register TLispManager*	pLispMgr,
	register TLispParser*	pParser)
{
	TVarbuffer		vbAtomName ;
	Char			cc ; 
	TLispEntity*	pRetvalue ;
	const Char*		pSymName ;
	int				nSymName ;

	assert (pLispMgr != NULL) ;
	assert (pParser  != NULL) ;

	pRetvalue	= NULL ;

#if defined (DEBUG)
	fprintf (stderr, "[Enter] _parseSymbol ()\n") ;
#endif
	TVarbuffer_Initialize (&vbAtomName, sizeof (Char)) ;

	for ( ; ; ) {
		cc	= (pParser->m_pGetChar)(pParser) ;
		if (cc == EOF)
			break ;
		if (!Char_DifferenceAscii (cc, ' ')		||
			!Char_DifferenceAscii (cc, '\t')	||
			!Char_DifferenceAscii (cc, '\n')	||
			!Char_DifferenceAscii (cc, '\r')	||
			!Char_DifferenceAscii (cc, ';')		||
			!Char_DifferenceAscii (cc, ')')		||
			!Char_DifferenceAscii (cc, ']')		||
			!Char_DifferenceAscii (cc, '(')		||
			!Char_DifferenceAscii (cc, '['))
			break ;
		TVarbuffer_Add (&vbAtomName, &cc, 1) ;
	}
	if (cc != EOF)
		(pParser->m_pUngetChar)(pParser, cc) ;

	pSymName	= TVarbuffer_GetBuffer (&vbAtomName) ;
	nSymName	= TVarbuffer_GetUsage  (&vbAtomName) ;
	if (TFAILED (lispMgr_InternSymbol (pLispMgr, pSymName, nSymName, &pRetvalue))) {
#if defined (DEBUG)
		fprintf (stderr, "[FAILE] InternSymbol ()\n") ;
#endif
		TVarbuffer_Uninitialize (&vbAtomName) ;
		return	NULL ;
	}
	lispEntity_AddRef (pLispMgr, pRetvalue) ;
	TVarbuffer_Uninitialize (&vbAtomName) ;

#if defined (DEBUG)
	fprintf (stderr, "[Leave] _parseSymbol ()\n") ;
	lispEntity_Print (pLispMgr, pRetvalue) ;
	fprintf (stderr, "\n") ;
	fflush (stderr) ;
#endif
	return	pRetvalue ;
}

/*
 *	³ʸ STRING Ǥ롣
 */
TLispEntity*
lispParser_parseString (
	register TLispManager*	pLispMgr,
	register TLispParser*	pParser)
{
	TVarbuffer		vbString ;
	Char			cc ;
	TLispEntity*	pRetvalue ;

	assert (pLispMgr != NULL) ;
	assert (pParser  != NULL) ;

	pRetvalue	= NULL ;

	/*
	 *	ʸʤΤ
	 */
	cc	= (pParser->m_pGetChar)(pParser) ;
	if (cc != 0x22) {
		(pParser->m_pUngetChar)(pParser, cc) ;
		return	NULL ;
	}

#if defined (DEBUG)
	fprintf (stderr, "[Enter] _parseString ()\n") ;
#endif
	TVarbuffer_Initialize (&vbString, sizeof (Char)) ;

	/*
	 *	ʸ󳫻ϡ
	 */
	for ( ; ; ) {
		cc	= (pParser->m_pGetChar)(pParser) ;
		switch (cc) {
		case	EOF:
		case	0x22:
			goto	exit_loop ;
		case	'\\':	/* backquote */
			if (TFAILED (lispParser_readBackslashedChar (pParser, &cc)))
				goto	exit_error ;
			break ;
		default:
			break ;
		}
		TVarbuffer_Add (&vbString, &cc, 1) ;
	}
 exit_loop:

	/*
	 *	ʸȤƽüƤΤ
	 */
	if (cc == 0x22) {
		const Char*	pString ;
		int			nString ;

		pString	= TVarbuffer_GetBuffer (&vbString) ;
		nString	= TVarbuffer_GetUsage  (&vbString) ;
		if (TFAILED (lispMgr_CreateString (pLispMgr, pString, nString, &pRetvalue))) {
			pRetvalue	= NULL ;
		} else {
			lispEntity_AddRef (pLispMgr, pRetvalue) ;
		}
	}
 exit_error:
	TVarbuffer_Uninitialize (&vbString) ;

	return	pRetvalue ;
}

Boolean
lispParser_readBackslashedChar (
	register TLispParser*	pParser,
	register Char*			pchRetval)
{
	Char			cc ;
	Char			chRet ;
	unsigned long	uAndMask, uOrMask ;

	/*	λ '\\' ɤ߽ξ֤ˤ롣*/
	cc	= (pParser->m_pGetChar)(pParser) ;
	switch (cc) {
	case	'C':
		uAndMask	= 0x1F ;
		uOrMask		= 0 ;
		goto	meta_control_common ;
	case	'M':
		uAndMask	= 0x7F ;
		uOrMask		= 0x80 ;
	meta_control_common:
		cc		= (pParser->m_pGetChar)(pParser) ;
		if (cc != '-')
			return	False ;
		cc	= (pParser->m_pGetChar)(pParser) ;
		if (cc == '\\') {
			if (TFAILED (lispParser_readBackslashedChar (pParser, &cc)))
				return	False ;
		}
		*pchRetval	= (cc & uAndMask) | uOrMask ;
		break ;

	case	't':
		*pchRetval	= Char_Make (KCHARSET_ASCII, '\t') ;
		break ;

	case	'n':
		*pchRetval	= Char_Make (KCHARSET_ASCII, '\n') ;
		break ;

	case	'r':
		*pchRetval	= Char_Make (KCHARSET_ASCII, '\r') ;
		break ;

	case	'x':
		chRet	= 0 ;
		for ( ; ; ) {
			cc	= (pParser->m_pGetChar)(pParser) ;
			if ('0' <= cc && cc <= '9') {
				chRet	= chRet * 0x10 + (cc - '0') ;
			} else if ('a' <= cc && cc <= 'f') {
				chRet	= chRet * 0x10 + (cc - 'a' + 10) ;
			} else if ('A' <= cc && cc <= 'F') {
				chRet	= chRet * 0x10 + (cc - 'A' + 10) ;
			} else {
				(pParser->m_pUngetChar)(pParser, cc) ;
				break ;
			}
		}
		*pchRetval	= Char_Make (KCHARSET_ASCII, chRet) ;
		break ;

	default:
		if ('0' <= cc && cc <= '7') {
			register int	i ;
			chRet	= cc - '0' ;
			for (i = 0 ; i < 2 ; i ++) {
				cc		= (pParser->m_pGetChar)(pParser) ;
				if (cc < '0' || cc > '7') {
					(pParser->m_pUngetChar)(pParser, cc) ;
					break ;
				}
				chRet	= chRet * 8 + (cc - '0') ;				
			}
			*pchRetval	= Char_Make (KCHARSET_ASCII, chRet) ;
			break ;
		}
		*pchRetval	= cc ;
		break ;
	}
	return	True ;
}

/*
 *
 */
TLispEntity*
lispParser_parseQuote (
	register TLispManager*	pLispMgr,
	register TLispParser*	pParser)
{
	static const Char	achQuote []	= { 'q', 'u', 'o', 't', 'e', } ;
	register Char		cc ;

	assert (pLispMgr != NULL) ;
	assert (pParser  != NULL) ;

#if defined (DEBUG)
	fprintf (stderr, "[Enter] _parseQuote ()\n") ;
#endif
	cc	= (pParser->m_pGetChar)(pParser) ;
	if (cc == EOF)
		return	NULL ;
	if (cc != 0x27) {
		(pParser->m_pUngetChar)(pParser, cc) ;
		return	NULL ;
	}
	return	lispParser_parseQuoteCommon (pLispMgr, pParser, achQuote, NELEMENTS (achQuote)) ;
}

#if 0
/*
 *	ϼԡǤΥɤפǤ(;_;)
 */
TLispEntity*
lispParser_parseBackquote (
	register TLispManager*	pLispMgr,
	register TLispParser*	pParser)
{
	static const Char	achQuote []	= { 'b', 'a', 'c', 'k', 'q', 'u', 'o', 't', 'e', } ;
	register Char	cc ;

	assert (pLispMgr != NULL) ;
	assert (pParser  != NULL) ;

	cc	= (pParser->m_pGetChar)(pParser) ;
	if (cc == EOF)
		return	NULL ;
	if (cc != 0x60) {
		(pParser->m_pUngetChar)(pParser, cc) ;
		return	NULL ;
	}
	return	lispParser_parseQuoteCommon (pLispMgr, pParser, achQuote, NELEMENTS (achQuote)) ;
}
#endif

TLispEntity*
lispParser_parseQuoteCommon (
	register TLispManager*	pLispMgr,
	register TLispParser*	pParser,
	register const Char*	strSymbol,
	register int			nSymbol)
{
	register LISPPARSEFUNC	pNextStateFunc ;
	register LISPPARSEFUNC	pOrgNextStateFunc ;
	TLispEntity*			pEntQuote ;
	TLispEntity*			pEntArg ;
	TLispEntity*			pRetvalue ;
	TLispEntity*			apEntities [2] ;

	if (TFAILED (lispMgr_InternSymbol (pLispMgr, strSymbol, nSymbol, &pEntQuote))) 
		return	NULL ;
	
	lispEntity_AddRef (pLispMgr, pEntQuote) ;
	apEntities [0]	= pEntQuote ;
	pEntArg			= NULL ;

	pOrgNextStateFunc	= pParser->m_pNextStateFunc ;
	for ( ; ; ) {
		if (!lispParser_checkNextState (pParser))
			break ;
		pNextStateFunc	= pParser->m_pNextStateFunc ;
		pEntArg			= (pNextStateFunc)(pLispMgr, pParser) ;
		if (pNextStateFunc != lispParser_parseComment)
			break ;
	}
	pParser->m_pNextStateFunc	= pOrgNextStateFunc ;

	if (pEntArg == NULL) {
		lispEntity_Release (pLispMgr, pEntQuote) ;
		return	NULL ;
	}
	apEntities [1]	= pEntArg ;
	if (TFAILED (lispMgr_CreateList (pLispMgr, apEntities, 2, &pRetvalue))) {
		lispEntity_Release (pLispMgr, pEntQuote) ;
		lispEntity_Release (pLispMgr, pEntArg) ;
		return	NULL ;
	}
	lispEntity_AddRef  (pLispMgr, pRetvalue) ;
	lispEntity_Release (pLispMgr, pEntQuote) ;
	lispEntity_Release (pLispMgr, pEntArg) ;
	return		pRetvalue ;
}

TLispEntity*
lispParser_parseInteger (
	register TLispManager*	pLispMgr,
	register TLispParser*	pParser)
{
	TVarbuffer		vbufSymbol ;	/* integer  symbol ޤʬʤ*/
	TLispEntity*	pRetvalue ;
	Char			cc ;
	int				iNumber		= 0 ;
	int				iSign		= +1 ;
	Boolean			fNotInteger	= False ;

	assert (pLispMgr != NULL) ;
	assert (pParser  != NULL) ;
	
#if defined (DEBUG)
	fprintf (stderr, "[Enter] _parseInteger ()\n") ;
#endif
	pRetvalue	= NULL ;
	cc	= (pParser->m_pGetChar)(pParser) ;
	if (cc == EOF)
		return	NULL ;
	if (TFAILED (TVarbuffer_Initialize (&vbufSymbol, sizeof (Char))))
		return	NULL ;

	if (cc != '.') {
		if (!Char_IsDigitNum (cc) && cc != '-' && cc != '+') {
			(pParser->m_pUngetChar) (pParser, cc) ;
			return	NULL ;
		}
		TVarbuffer_Add (&vbufSymbol, &cc, 1) ;
		if (!Char_DifferenceAscii (cc, '+') || !Char_DifferenceAscii (cc, '-')) {
			iSign	= (!Char_DifferenceAscii (cc, '-'))? -1 : +1 ;
			cc		= (pParser->m_pGetChar)(pParser) ;
			if (Char_DifferenceAscii (cc, '0') < 0 ||
				Char_DifferenceAscii (cc, '9') > 0) {
				if (cc != EOF)
					(pParser->m_pUngetChar) (pParser, cc) ;
				fNotInteger	= True ;
				goto	not_integer ;
			}
			TVarbuffer_Add (&vbufSymbol, &cc, 1) ;
		}
		while (cc != EOF &&
			   Char_DifferenceAscii (cc, '0') >= 0 &&
			   Char_DifferenceAscii (cc, '9') <= 0) {
			iNumber	= iNumber * 10 + Char_DifferenceAscii (cc, '0') ;
			cc		= (pParser->m_pGetChar)(pParser) ;
			TVarbuffer_Add (&vbufSymbol, &cc, 1) ;
		}
	}
	/*	'.' ƤȤȤư*/
	if (!Char_DifferenceAscii (cc, '.')) {
		double	dShift	= 0.1 ;
		double	dNumber	= (double) iNumber ;
		
		cc		= (pParser->m_pGetChar)(pParser) ;
		TVarbuffer_Add (&vbufSymbol, &cc, 1) ;

		while (cc != EOF &&
			   Char_DifferenceAscii (cc, '0') >= 0 &&
			   Char_DifferenceAscii (cc, '9') <= 0) {
			dNumber	= dNumber + (double)(Char_DifferenceAscii (cc, '0')) * dShift ;
			dShift	= dShift / 10.0 ;
			cc		= (pParser->m_pGetChar)(pParser) ;
			TVarbuffer_Add (&vbufSymbol, &cc, 1) ;
		}
		if (cc != EOF &&
			Char_DifferenceAscii (cc, ' ')		&&
			Char_DifferenceAscii (cc, ']')		&&
			Char_DifferenceAscii (cc, '"')		&&
			Char_DifferenceAscii (cc, '\t')		&&
			Char_DifferenceAscii (cc, '\x39')	&&
			Char_DifferenceAscii (cc, '\x29')	&&
			Char_DifferenceAscii (cc, '\n')		&&
			Char_DifferenceAscii (cc, '\r')) {
			fNotInteger	= True ;
		} else {
			if (cc != EOF)
				(pParser->m_pUngetChar) (pParser, cc) ;
			if (TFAILED (lispMgr_CreateFloat (pLispMgr, (float) (iSign * dNumber), &pRetvalue))) 
				pRetvalue = NULL ;
		}
	} else if (cc != EOF &&
			   Char_DifferenceAscii (cc, ' ')		&&
			   Char_DifferenceAscii (cc, ']')		&&
			   Char_DifferenceAscii (cc, '"')		&&
			   Char_DifferenceAscii (cc, '\t')		&&
			   Char_DifferenceAscii (cc, '\x39')	&&
			   Char_DifferenceAscii (cc, '(')		&&
			   Char_DifferenceAscii (cc, ')')		&&
			   Char_DifferenceAscii (cc, '[')		&&
			   Char_DifferenceAscii (cc, '\n')		&&
			   Char_DifferenceAscii (cc, '\r')) {
		fNotInteger	= True ;
	} else {
		if (cc != EOF)
			(pParser->m_pUngetChar) (pParser, cc) ;
		
		if (TFAILED (lispMgr_CreateInteger (pLispMgr, iSign * iNumber, &pRetvalue))) 
			pRetvalue	= NULL ;
	}

 not_integer:
	if (fNotInteger) {
		const Char*	pString ;
		int			nLength ;

		for ( ; ; ) {
			cc		= (pParser->m_pGetChar)(pParser) ;
			if (cc == EOF ||
				!Char_DifferenceAscii (cc, ' ')		||
				!Char_DifferenceAscii (cc, '[')		||
				!Char_DifferenceAscii (cc, ']')		||
				!Char_DifferenceAscii (cc, '"')		||
				!Char_DifferenceAscii (cc, '\t')	||
				!Char_DifferenceAscii (cc, '\x39')	||
				!Char_DifferenceAscii (cc, '(')		||
				!Char_DifferenceAscii (cc, ')')		||
				!Char_DifferenceAscii (cc, '\n')	||
				!Char_DifferenceAscii (cc, '\r'))
				break ;
			TVarbuffer_Add (&vbufSymbol, &cc, 1) ;
		}
		if (cc != EOF)
			(pParser->m_pUngetChar) (pParser, cc) ;
		
		pString	= TVarbuffer_GetBuffer (&vbufSymbol) ;
		nLength	= TVarbuffer_GetUsage  (&vbufSymbol) ;
		if (TFAILED (lispMgr_InternSymbol (pLispMgr, pString, nLength, &pRetvalue))) {
			pRetvalue	= NULL ;
		}
	}
	if (pRetvalue != NULL)
		lispEntity_AddRef (pLispMgr, pRetvalue) ;
	TVarbuffer_Uninitialize (&vbufSymbol) ;
	return	pRetvalue ;
}

TLispEntity*
lispParser_parseChar (
	register TLispManager*	pLispMgr,
	register TLispParser*	pParser)
{
	TLispEntity*	pRetvalue ;
	Char			cc ;

	assert (pLispMgr != NULL) ;
	assert (pParser  != NULL) ;
	
	pRetvalue	= NULL ;

#if defined (DEBUG)
	fprintf (stderr, "[Enter] _parseChar ()\n") ;
#endif
	cc	= (pParser->m_pGetChar)(pParser) ;
	if (cc == EOF)
		return	NULL ;

	if (Char_DifferenceAscii (cc, '?')) {
		(pParser->m_pUngetChar) (pParser, cc) ;
		return	NULL ;
	}
	cc	= (pParser->m_pGetChar)(pParser) ;
	if (!Char_DifferenceAscii (cc, '\\')) 
		lispParser_readBackslashedChar (pParser, &cc) ;
	if (cc == EOF)
		return	NULL ;
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, cc, &pRetvalue)))
		return	NULL ;
	lispEntity_AddRef (pLispMgr, pRetvalue) ;

	return	pRetvalue ;
}

/*
 *
 */
TLispEntity*
lispParser_parseList (
	register TLispManager*	pLispMgr,
	register TLispParser*	pParser)
{
	TLispEntity*	pEntity ;
	TLispEntity*	pCdr ;
	TLispEntity*	pRetvalue ;
	TVarbuffer		vbufEntity ;
	int				iEntity ;
	Char			cc ;
	LISPPARSEFUNC	pNextStateFunc ;
	LISPPARSEFUNC	pOrgNextStateFunc ;

	assert (pLispMgr != NULL) ;
	assert (pParser  != NULL) ;

#if defined (DEBUG)
	fprintf (stderr, "[Enter] _parseList ()\n") ;
#endif
	pRetvalue	= NULL ;

	cc	= (pParser->m_pGetChar)(pParser) ;
	if (cc == EOF)
		return	NULL ;

	if (Char_DifferenceAscii (cc, '(')) {
		(pParser->m_pUngetChar)(pParser, cc) ;
#if defined (DEBUG)
		fprintf (stderr, "[ParseList] cc != '('\n") ;
#endif
		return	NULL ;
	}

	if (!TVarbuffer_Initialize (&vbufEntity, sizeof (TLispEntity*)))
		return	NULL ;

	pOrgNextStateFunc	= pParser->m_pNextStateFunc ;

	for ( ; ; ) {
		if (!lispParser_checkNextState (pParser))
			break ;
		pNextStateFunc	= pParser->m_pNextStateFunc ;
		pEntity			= (pNextStateFunc)(pLispMgr, pParser) ;
		if (pNextStateFunc == lispParser_parseComment)
			continue ;
		if (pEntity == NULL) {
#if defined (DEBUG)
			fprintf (stderr, "(1) [ParseList] pEntity == NULL\n") ;
#endif
			goto	exit_parse_list ;
		}
		TVarbuffer_Add (&vbufEntity, &pEntity, 1) ;
	}

	/*cc		= (pParser->m_pGetChar)(pParser) ;*/
	cc	= lispParser_SkipSpace (pParser) ;
	if (cc != EOF && !Char_DifferenceAscii (cc, '.')) {
		pEntity	= NULL ;
		for ( ; ; ) {
			if (!lispParser_checkNextState (pParser))
				break ;
			pNextStateFunc	= pParser->m_pNextStateFunc ;
			pEntity			= (pNextStateFunc)(pLispMgr, pParser) ;
			if (pNextStateFunc != lispParser_parseComment)
				break ;
		}
		if (pEntity == NULL) {
#if defined (DEBUG)
			fprintf (stderr, "(2) [ParseList] pEntity == NULL\n") ;
#endif
			goto	exit_parse_list ;
		}
		pCdr	= pEntity ;
		/*cc		= (pParser->m_pGetChar)(pParser) ;*/
		cc		= lispParser_SkipSpace (pParser) ;
	} else {
		(void) lispMgr_CreateNil (pLispMgr, &pCdr) ;
		assert (pCdr != NULL) ;
		lispEntity_AddRef (pLispMgr, pCdr) ;
	}

	if (cc == EOF || Char_DifferenceAscii (cc, ')')) {
#if defined (DEBUG)
		fprintf (stderr, "[ParseList] cc != ')' %lx\n", cc) ;
#endif
		goto	exit_parse_list ;
	}

	iEntity	= TVarbuffer_GetUsage (&vbufEntity) ;
	if (iEntity < 0) {
		(void) lispMgr_CreateNil (pLispMgr, &pRetvalue) ;
		lispEntity_AddRef (pLispMgr, pRetvalue) ;
	} else if (iEntity == 0) {
		pRetvalue	= pCdr ;
	} else {
		TLispEntity**	ppTop ;
		TLispEntity**	ppEntity ;
		TLispEntity*	pListTop ;
		TLispEntity*	pNil ;
		int				i ;

		(void) lispMgr_CreateNil (pLispMgr, &pNil) ;
		ppTop		= (TLispEntity **) TVarbuffer_GetBuffer (&vbufEntity) ;
		if (TFAILED (lispMgr_CreateConscell (pLispMgr, *ppTop, pNil, &pListTop))) {
			ppEntity	= ppTop ;
			for (i = 0 ; i < iEntity ; i ++) 
				lispEntity_Release (pLispMgr, *ppEntity ++) ;
			pRetvalue	= NULL ;
		} else {
			TLispEntity*	pLast ;
			TLispEntity*	pElm ;
			
			lispEntity_AddRef  (pLispMgr, pListTop) ;
			lispEntity_Release (pLispMgr, *ppTop) ;
			pLast		= pListTop ;
		
			ppEntity	= ppTop + 1 ;
			for (i = 1 ; i < iEntity ; i ++) {
				if (TFAILED (lispMgr_CreateConscell (pLispMgr, *ppEntity, pNil, &pElm)))
					break ;
				lispEntity_SetCdr  (pLispMgr, pLast, pElm) ;
				lispEntity_Release (pLispMgr, *ppEntity) ;
				ppEntity	++ ;
				pLast	= pElm ;
			}
			lispEntity_SetCdr  (pLispMgr, pLast, pCdr) ;
			if (i < iEntity) {
				while (i < iEntity) {
					lispEntity_Release (pLispMgr, *ppEntity ++) ;
					i	++ ;
				}
				lispEntity_Release (pLispMgr, pListTop) ;
				pRetvalue	= NULL ;
			} else {
				pRetvalue	= pListTop ;
			}
		}
		lispEntity_Release (pLispMgr, pCdr) ;
	}

 exit_parse_list:
	TVarbuffer_Uninitialize (&vbufEntity) ;

	return	pRetvalue ;
}

/*
 *
 */
TLispEntity*
lispParser_parseComment (
	register TLispManager*	pLispMgr,
	register TLispParser*	pParser)
{
	register Char	cc ;

	assert (pLispMgr != NULL) ;
	assert (pParser  != NULL) ;

	cc	= (pParser->m_pGetChar)(pParser) ;
	if (cc == EOF)
		return	NULL ;

	if (Char_DifferenceAscii (cc, ';')) {
		(pParser->m_pUngetChar)(pParser, cc) ;
		return	NULL ;
	}
#if defined (DEBUG)
	fprintf (stderr, "[Enter] _parseComment ()\n") ;
#endif

	while (cc != EOF && Char_DifferenceAscii (cc, '\n'))
		cc	= (pParser->m_pGetChar)(pParser) ;

	if (cc != EOF)
		(pParser->m_pUngetChar)(pParser, cc) ;

	return	NULL ;
}

/*
 *
 */
TLispEntity*
lispParser_parseArray (
	register TLispManager*	pLispMgr,
	register TLispParser*	pParser)
{
	TVarbuffer		vbufVector ;
	TLispEntity*	pEntity ;
	TLispEntity*	pRetvalue ;
	LISPPARSEFUNC	pNextStateFunc ;
	LISPPARSEFUNC	pOrgNextStateFunc ;
	Char			cc ;
	int				nElements ;

	assert (pLispMgr != NULL) ;
	assert (pParser  != NULL) ;

	pRetvalue	= NULL ;

#if defined (DEBUG)
	fprintf (stderr, "[Enter] _parseArray ()\n") ;
#endif
	/*	󤬤ȳƤ뤫å롣*/
	cc	= (pParser->m_pGetChar)(pParser) ;
	if (cc == EOF)
		return	NULL ;
	if (Char_DifferenceAscii (cc, '[')) {
		(pParser->m_pUngetChar)(pParser, cc) ;
		return	NULL ;
	}

	if (!TVarbuffer_Initialize (&vbufVector, sizeof (TLispEntity*)))
		return	NULL ;

	pOrgNextStateFunc	= pParser->m_pNextStateFunc ;

	/*	ǤФ*/
	for ( ; ; ) {
		/*
		 *	֤Τ˼Ԥ顢ȴ롣
		 *	쳵˥顼ȤϸʤĤȤˤ false 
		 *	äƤΤǡ
		 */
		if (!lispParser_checkNextState (pParser))
			break ;
		pNextStateFunc	= pParser->m_pNextStateFunc ;
		pEntity			= (pNextStateFunc)(pLispMgr, pParser) ;
		if (pEntity != NULL) {
			if (!TVarbuffer_Add (&vbufVector, &pEntity, 1))
				goto	exit_parse_array ;
		} else {
			if (pNextStateFunc != lispParser_parseComment)
				goto	exit_parse_array ;
		}
	}

	/*	󤬤ĤƤ뤫ɤå롣*/
	cc	= (pParser->m_pGetChar)(pParser) ;
	if (cc == EOF || Char_DifferenceAscii (cc, ']'))
		goto	exit_parse_array ;
	
	nElements	= TVarbuffer_GetUsage (&vbufVector) ;
	if (nElements <= 0) {
		if (TFAILED (lispMgr_CreateVector (pLispMgr, NULL, 0, &pRetvalue)))
			pRetvalue	= NULL ;
	} else {
		TLispEntity**	ppElements ;

		ppElements	= (TLispEntity **)TVarbuffer_GetBuffer (&vbufVector) ;
		if (TFAILED (lispMgr_CreateVector (pLispMgr, ppElements, nElements, &pRetvalue)))
			pRetvalue	= NULL ;
	}
	if (pRetvalue != NULL)
		lispEntity_AddRef (pLispMgr, pRetvalue) ;

 exit_parse_array:
	nElements	= TVarbuffer_GetUsage (&vbufVector) ;
	if (nElements > 0) {
		TLispEntity**	ppElements ;

		ppElements	= (TLispEntity **)TVarbuffer_GetBuffer (&vbufVector) ;
		nElements	= TVarbuffer_GetUsage (&vbufVector) ;
		/*	פˤʤäƥƥ롣*/
		while (nElements > 0) {
			lispEntity_Release (pLispMgr, *ppElements ++) ;
			nElements	-- ;
		}
	}

	pParser->m_pNextStateFunc	= pOrgNextStateFunc ;

	TVarbuffer_Uninitialize (&vbufVector) ;

	return	pRetvalue ;
}

Char
lispParser_SkipSpace (
	register TLispParser* pParser)
{
	register Char	cc ;

	assert (pParser != NULL) ;

	for ( ; ; ) {
		cc	= (pParser->m_pGetChar)(pParser) ;
		if (Char_DifferenceAscii (cc, ' ')	&&
			Char_DifferenceAscii (cc, '\t')	&&
			Char_DifferenceAscii (cc, '\n')	&&
			Char_DifferenceAscii (cc, '\r'))
			break ;
	}
	return	cc ;
}

Char
lispParser_GetStringChar (
	register TLispParser*	pParser)
{
	register Char	cc ;

	if (pParser->m_src.m_string.m_iRest <= 0)
		return	(Char) EOF ;

	cc	= *pParser->m_src.m_string.m_pHead ++ ;
	pParser->m_src.m_string.m_iRest	-- ;
	return	cc ;
}

int
lispParser_UngetStringChar (
	register TLispParser*	pParser,
	register Char			cc)
{
	if (pParser->m_src.m_string.m_pHead <= pParser->m_src.m_string.m_pStringTop)
		return	-1 ;

	pParser->m_src.m_string.m_pHead	-- ;
	pParser->m_src.m_string.m_iRest	++ ;
	return	0 ;

	UNREFERENCED_PARAMETER (cc) ;
}

Char
lispParser_GetStringCharA (
	register TLispParser*	pParser)
{
	register char	cc ;

	if (pParser->m_src.m_stringA.m_iRest <= 0)
		return	(Char) EOF ;
	
	cc	= *pParser->m_src.m_stringA.m_pHead ++ ;
	pParser->m_src.m_stringA.m_iRest	-- ;
	return	Char_MakeAscii (cc) ;
}

int
lispParser_UngetStringCharA (
	register TLispParser*	pParser,
	register Char			cc)
{
	if (pParser->m_src.m_stringA.m_pHead <= pParser->m_src.m_stringA.m_pStringTop)
		return	-1 ;

	pParser->m_src.m_stringA.m_pHead	-- ;
	pParser->m_src.m_stringA.m_iRest	++ ;
	return	0 ;

	UNREFERENCED_PARAMETER (cc) ;
}

Char
lispParser_GetFileChar (
	register TLispParser*	pParser)
{
	register Char	cc ;

	if (pParser->m_src.m_file.m_iBufUsage > 0) {
		cc	= pParser->m_src.m_file.m_achBuf [-- pParser->m_src.m_file.m_iBufUsage] ;
	} else {
		cc	= KFile_Getc (pParser->m_src.m_file.m_pFile) ;
	}
	return	cc ;
}

int
lispParser_UngetFileChar (
	register TLispParser*	pParser,
	register Char			cc)
{
	if (pParser->m_src.m_file.m_iBufUsage < MAX_UNGETC_BUF) {
		pParser->m_src.m_file.m_achBuf [pParser->m_src.m_file.m_iBufUsage ++]	= cc ;
		return	0 ;
	}
	return	-1 ;
}

