/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <assert.h>
#include "lispmgrp.h"
#include "cstring.h"

static	Boolean	lispKeymap_lookupWithStringFound	(TLispManager*, TLispEntity*, const Char*, int, Boolean, Boolean, int*, TLispEntity** const) ;
static	Boolean	lispKeymap_lookupWithVectorFound	(TLispManager*, TLispEntity*, TLispEntity**, int, Boolean, Boolean, int*, TLispEntity** const) ;
static	Boolean	lispKeymap_lookupWithListFound		(TLispManager*, TLispEntity*, TLispEntity*, Boolean, Boolean, int*, TLispEntity** const) ;
static	Boolean	lispKeymap_defineSubkey				(TLispManager*, const Char*, int, TLispEntity*, TLispEntity** const) ;
static	Boolean	lispKeymap_defineSubkeyWithVector	(TLispManager*, TLispEntity**, int, TLispEntity*, TLispEntity** const) ;
static	Boolean	lispKeymap_equalChar				(Char, Char) ;

/*
 *[]
 *	̩ keymap  (keymap [nil ... nil]) η򤷤ƤΤȤ롣
 *	[nil ... nil] ϡȤȤ char-table  384 Ĥä褦˻פ
 *	뤬char-table ϼƤʤvector Ѥ롣
 */
Boolean
lispMgr_CreateKeymap (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pString,
	register TLispEntity** const	ppEntReturn)
{
	TLispEntity*	apCharTable [128] ;
	TLispEntity*	pNil ;
	TLispEntity**	ppEntity ;
	TLispEntity*	apNode [2] ;
	register Boolean	fRetval ;
	register int	nCounter, nNodes ;

	lispMgr_CreateNil (pLispMgr, &pNil) ;
	ppEntity	= apCharTable ;
	nCounter	= NELEMENTS (apCharTable) ;

	while (nCounter -- > 0) 
		*ppEntity ++	= pNil ;

	apNode [0]	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_KEYMAP) ;
	lispEntity_AddRef (pLispMgr, apNode [0]) ;
	if (TFAILED (lispMgr_CreateVector (pLispMgr, apCharTable, NELEMENTS (apCharTable), &apNode [1]))) {
		lispEntity_Release (pLispMgr, apNode [0]) ;
		return	False ;
	}
	lispEntity_AddRef (pLispMgr, apNode [1]) ;

	if (pString != NULL && TFAILED (lispEntity_Nullp (pLispMgr, pString))) {
		apNode [2]	= pString ;
		nNodes		= 3 ;
	} else {
		nNodes		= 2 ;
	}
	fRetval	= lispMgr_CreateList (pLispMgr, apNode, nNodes, ppEntReturn) ;
	lispEntity_Release (pLispMgr, apNode [1]) ;
	lispEntity_Release (pLispMgr, apNode [0]) ;
	return	fRetval ;
}

/*
 *	¤ʥޥåפؿ
 */
Boolean
lispMgr_CreateSparseKeymap (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pEntString,
	register TLispEntity** const	ppEntReturn)
{
	TLispEntity*	apEntNode [2] ;
	register int	nNode ;
	register Boolean	fRetval ;

	nNode			= 0 ;
	apEntNode [0]	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_KEYMAP) ;
	lispEntity_AddRef (pLispMgr, apEntNode [nNode ++]) ;
	if (pEntString != NULL &&
		TFAILED (lispEntity_Nullp (pLispMgr, pEntString)))
		apEntNode [nNode ++]	= pEntString ;
	fRetval	= lispMgr_CreateList (pLispMgr, apEntNode, nNode, ppEntReturn) ;
	while (nNode > 0)
		lispEntity_Release (pLispMgr, apEntNode [-- nNode]) ;
	return	fRetval ;
}

Boolean
lispMgr_CopyKeymap (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pEntKeymap,
	register TLispEntity** const	ppEntReturn)
{
	if (TFAILED (lispEntity_Keymapp (pLispMgr, pEntKeymap)))
		return	False ;
	return	lispEntity_Copy (pLispMgr, pEntKeymap, ppEntReturn) ;
}

/*
 *	ޥåפɤȽꤹؿ
 */
Boolean
lispEntity_Keymapp (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	TLispEntity*	pCar ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;

	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntity, &pCar)))
		return	False ;

	return	lispEntity_Eq (pLispMgr, pCar, lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_KEYMAP)) ;
}

/*
 *	hEntity  array Ǥ뤳ȡarray  vector  string Ǥ롣
 *	pnMatch ϲʸޥå֤ʸǲäϡphEntityRet
 *	롣ȤȤ lookup ȡ֤ˤʤ뤱ɡ줸㤢
 *	ټ֤ʡȤ
 */
Boolean
lispKeymap_Lookup (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pKeymap,
	register TLispEntity*			pEntity,
	register Boolean				fAcceptDefault,
	register Boolean				fNearMatch,
	register int*					pnMatch,
	register TLispEntity** const	ppEntReturn)
{
	int		iType ;
	
	assert (pLispMgr    != NULL) ;
	assert (pKeymap     != NULL) ;
	assert (pEntity     != NULL) ;
	assert (ppEntReturn != NULL) ;

	if (TFAILED (lispEntity_GetType (pLispMgr, pEntity, &iType)))
		return	False ;

	switch (iType) {
	case	LISPENTITY_STRING:
	{
		const Char*	pString ;
		int			nLength ;
		
		(void) lispEntity_GetStringValue (pLispMgr, pEntity, &pString, &nLength) ;
		if (pString == NULL || nLength <= 0) 
			break ;
		return	lispKeymap_LookupWithString (pLispMgr, pKeymap, pString, nLength, fAcceptDefault, fNearMatch, pnMatch, ppEntReturn) ;
	}
	
	case	LISPENTITY_VECTOR:
	{
		TLispEntity**	ppElement ;
		int				nElement ;

		(void) lispEntity_GetVectorValue (pLispMgr, pEntity, &ppElement, &nElement) ;
		if (ppElement == NULL || nElement <= 0) 
			break ;
		return	lispKeymap_LookupWithVector (pLispMgr, pKeymap, ppElement, nElement, fAcceptDefault, fNearMatch, pnMatch, ppEntReturn) ;
	}
	case	LISPENTITY_CONSCELL:
	{
		if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntity))) 
			break ;
		return	lispKeymap_LookupWithList (pLispMgr, pKeymap, pEntity, fAcceptDefault, fNearMatch, pnMatch, ppEntReturn) ;
	}
	default:
		return	False ;
	}
	*pnMatch		= 0 ;
	*ppEntReturn	= pKeymap ;
	return	True ;
}

Boolean
lispKeymap_LookupWithString (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pKeymap,
	register const Char*			pString,
	register int					nString,
	register Boolean				fAcceptDefault,
	register Boolean				fNearMatch,
	register int*					pnMatch,
	register TLispEntity** const	ppEntReturn)
{
	TLispEntity*	pEntry ;
	TLispEntity*	pList ;
	TLispEntity*	pDefaultFunc	= NULL ;
	TLispEntity*	pEntNearHit ;
	Char			cc, ccNear ;
	int				iType ;

	if (TFAILED (lispEntity_Keymapp (pLispMgr, pKeymap)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pKeymap, &pList)))
		return	False ;
	
	cc			= *pString ++ ;
	nString	-- ;
	ccNear	= (fNearMatch && Char_Charset (cc) == KCHARSET_XCHAR)? 
		Char_MakeAscii (Char_Code (cc) & 0x7F) : -1 ;
	pEntNearHit	= NULL ;
	
	while (TFAILED (lispEntity_Nullp (pLispMgr, pList))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pList, &pEntry)))
			return	False ;

		lispEntity_GetType (pLispMgr, pEntry, &iType) ;
		switch (iType) {
		case	LISPENTITY_CONSCELL:
		{
			TLispEntity*	pCar ;
			long			lValue ;
			
			(void) lispEntity_GetCar (pLispMgr, pEntry, &pCar) ;

			/*	ACCEPT-DEFAULT äƤ硣*/
			if (fAcceptDefault && TSUCCEEDED (lispEntity_Tp (pLispMgr, pCar))) {
				(void) lispEntity_GetCdr (pLispMgr, pEntry, &pDefaultFunc) ;
				assert (pDefaultFunc != NULL) ;
			}
			
			if (TSUCCEEDED (lispEntity_GetIntegerValue (pLispMgr, pCar, &lValue)) &&
				(cc == (Char) lValue || (fNearMatch && ccNear == (Char) lValue))) {
				(void) lispEntity_GetCdr (pLispMgr, pEntry, &pEntry) ;
				return	lispKeymap_lookupWithStringFound (pLispMgr, pEntry, pString, nString, fAcceptDefault, fNearMatch, pnMatch, ppEntReturn) ;
			}
			break ;
		}

		case	LISPENTITY_VECTOR:
		{
			TLispEntity**	ppElement ;
			int				nElement ;
			register int	nIndex ;
			
			(void) lispEntity_GetVectorValue (pLispMgr, pEntry, &ppElement, &nElement) ;
			nIndex	= (int) cc ;
			if (0 <= nIndex && nIndex < nElement) {
				pEntry		= *(ppElement + nIndex) ;
				return	lispKeymap_lookupWithStringFound (pLispMgr, pEntry, pString, nString, fAcceptDefault, fNearMatch, pnMatch, ppEntReturn) ;
			}
			if (fNearMatch) {
				nIndex	= (int) ccNear ;
				if (pEntNearHit == NULL && 0 <= nIndex && nIndex < nElement) 
					pEntNearHit	= *(ppElement + nIndex) ;
			}
			break ;
		}
		
		/*	ξˤϡkeymap ĤʤäƤǤϤʤΤ
		 *	פ*/
		case	LISPENTITY_SYMBOL:
			if (TSUCCEEDED (lispEntity_Keymapp (pLispMgr, pList)))
				return	lispKeymap_LookupWithString (pLispMgr, pList, pString, nString, fAcceptDefault, fNearMatch, pnMatch, ppEntReturn) ;
			break ;
			
		default:
			break ;
		}
		if (TFAILED (lispEntity_GetCdr (pLispMgr, pList, &pList)))
			return	False ;
	}

	/*	Exact Hit ˼Ԥˤ Near Hit Ȥ*/
	if (fNearMatch && pEntNearHit != NULL) 
		return	lispKeymap_lookupWithStringFound (pLispMgr, pEntNearHit, pString, nString, fAcceptDefault, fNearMatch, pnMatch, ppEntReturn) ;
	
	/*	default keymap ˺Ƥ뤾(^^;; */
	++	*pnMatch ;
	if (pDefaultFunc != NULL) {
		*ppEntReturn	= pDefaultFunc ;
	} else {
		*ppEntReturn	= pList ;
	}
	return	True ;
}

Boolean
lispKeymap_lookupWithStringFound (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pEntKeymap,
	register const Char*			pString,
	register int					nString,
	register Boolean				fAcceptDefault,
	register Boolean				fNearMatch,
	register int*					pnMatch,
	register TLispEntity** const	ppEntReturn)
{
	/*	ޤǤʳ֤ͤϷꤷƤ롣*/
	++ *pnMatch ;
	/*	ޤ򸫤Ƥߤġ*/
	if (nString > 0) {
		/*	ޥåפǤʤ False ä롣
		 *	λϡֺʳޤƤפ
		 *	Ȥˤʤ롣*/
		if (TFAILED (lispKeymap_LookupWithString (pLispMgr, pEntKeymap, pString, nString, fAcceptDefault, fNearMatch, pnMatch, ppEntReturn)))
			*ppEntReturn	= pEntKeymap ;
	} else {
		*ppEntReturn	= pEntKeymap ;
	}
	return	True ;
}

Boolean
lispKeymap_LookupWithVector (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pKeymap,
	register TLispEntity**			ppElement,
	register int					nElement,
	register Boolean				fAcceptDefault,
	register Boolean				fNearMatch,
	register int*					pnMatch,
	register TLispEntity** const	ppEntReturn)
{
	TLispEntity*	pEntry ;
	TLispEntity*	pList ;
	TLispEntity*	pDefaultFunc	= NULL ;
	TLispEntity*	pElement ;
	TLispEntity*	pEntNearHit ;
	long			lValue, lValueNear ;
	int				iType ;

	assert (pLispMgr    != NULL) ;
	assert (pnMatch     != NULL) ;
	assert (ppEntReturn != NULL) ;

	if (TFAILED (lispEntity_Keymapp (pLispMgr, pKeymap)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pKeymap, &pList)))
		return	False ;
	
	assert (ppElement != NULL && nElement > 0) ;
	pElement	= *ppElement ++ ;
	nElement	-- ;
	
	if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pElement, &lValue)))
		lValue	= -1 ;
	lValueNear	= (fNearMatch && Char_Charset ((Char) lValue) == KCHARSET_XCHAR)? 
		Char_MakeAscii (Char_Code ((Char)lValue) & 0x7F) : -1 ;
	pEntNearHit	= NULL ;

	while (TFAILED (lispEntity_Nullp (pLispMgr, pList))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pList, &pEntry)))
			return	False ;

		lispEntity_GetType (pLispMgr, pEntry, &iType) ;
		switch (iType) {
		case	LISPENTITY_CONSCELL:
		{
			TLispEntity*	pCar ;
			long			lCar ;
			
			(void) lispEntity_GetCar (pLispMgr, pEntry, &pCar) ;

			/*	ACCEPT-DEFAULT äƤ硣*/
			if (fAcceptDefault && TSUCCEEDED (lispEntity_Tp (pLispMgr, pCar))) {
				(void) lispEntity_GetCdr (pLispMgr, pEntry, &pDefaultFunc) ;
				assert (pDefaultFunc != NULL) ;
			}
			
			if (TSUCCEEDED (lispEntity_Eq (pLispMgr, pCar, pElement)) ||
				(lValue >= 0 &&
				 TSUCCEEDED (lispEntity_GetIntegerValue (pLispMgr, pCar, &lCar)) &&
				 (lValue == lCar || (fNearMatch && lValueNear == lCar)))) {
				(void) lispEntity_GetCdr (pLispMgr, pEntry, &pEntry) ;
				return	lispKeymap_lookupWithVectorFound (pLispMgr, pEntry, ppElement, nElement, fAcceptDefault, fNearMatch, pnMatch, ppEntReturn) ;
			}
			break ;
		}

		case	LISPENTITY_VECTOR:
		{
			TLispEntity**	ppLcElement ;
			int				nLcElement ;
			
			(void) lispEntity_GetVectorValue (pLispMgr, pEntry, &ppLcElement, &nLcElement) ;
			if (0 <= lValue && lValue < nLcElement) {
				pEntry		= *(ppLcElement + lValue) ;
				return	lispKeymap_lookupWithVectorFound (pLispMgr, pEntry, ppElement, nElement, fAcceptDefault, fNearMatch, pnMatch, ppEntReturn) ;
			}	
			if (fNearMatch) {
				if (pEntNearHit == NULL && 0 <= lValueNear && lValueNear < nLcElement) 
					pEntNearHit	= *(ppLcElement + lValueNear) ;
			}
			break ;
		}
		
		/*	ξˤϡkeymap ĤʤäƤǤϤʤΤ
		 *	פ*/
		case	LISPENTITY_SYMBOL:
			if (TSUCCEEDED (lispEntity_Keymapp (pLispMgr, pList)))
				return	lispKeymap_LookupWithVector (pLispMgr, pList, ppElement, nElement, fAcceptDefault, fNearMatch, pnMatch, ppEntReturn) ;
			break ;
			
		default:
			break ;
		}
		if (TFAILED (lispEntity_GetCdr (pLispMgr, pList, &pList)))
			return	False ;
	}

	if (fNearMatch && pEntNearHit != NULL)
		return	lispKeymap_lookupWithVectorFound (pLispMgr, pEntNearHit, ppElement, nElement, fAcceptDefault, fNearMatch, pnMatch, ppEntReturn) ;

	/*	default keymap ˺Ƥ뤾(^^;; */
	++	*pnMatch ;
	if (pDefaultFunc != NULL) {
		*ppEntReturn	= pDefaultFunc ;
	} else {
		*ppEntReturn	= pList ;
	}
	return	True ;
}

Boolean
lispKeymap_lookupWithVectorFound (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pEntKeymap,
	register TLispEntity**			ppElement,
	register int					nElement,
	register Boolean				fAcceptDefault,
	register Boolean				fNearMatch,
	register int*					pnMatch,
	register TLispEntity** const	ppEntReturn)
{
	/*	ޤǤʳ֤ͤϷꤷƤ롣*/
	++ *pnMatch ;
	/*	ޤ򸫤Ƥߤġ*/
	if (nElement > 0) {
		/*	ޥåפǤʤ False ä롣
		 *	λϡֺʳޤƤפ
		 *	Ȥˤʤ롣*/
		if (TFAILED (lispKeymap_LookupWithVector (pLispMgr, pEntKeymap, ppElement, nElement, fAcceptDefault, fNearMatch, pnMatch, ppEntReturn)))
			*ppEntReturn	= pEntKeymap ;
	} else {
		*ppEntReturn	= pEntKeymap ;
	}
	return	True ;
}

Boolean
lispKeymap_LookupWithList (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pKeymap,
	register TLispEntity*			pKeyList,
	register Boolean				fAcceptDefault,
	register Boolean				fNearMatch,
	register int*					pnMatch,
	register TLispEntity** const	ppEntReturn)
{
	TLispEntity*	pEntry ;
	TLispEntity*	pList ;
	TLispEntity*	pDefaultFunc	= NULL ;
	TLispEntity*	pElement ;
	TLispEntity*	pNextKeyList ;
	TLispEntity*	pEntNearHit ;
	long			lValue, lValueNear ;
	int				iType ;

	assert (pLispMgr    != NULL) ;
	assert (pnMatch     != NULL) ;
	assert (ppEntReturn != NULL) ;
	assert (pKeyList != NULL) ;

	if (TFAILED (lispEntity_Keymapp (pLispMgr, pKeymap)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pKeymap, &pList)) ||
		TSUCCEEDED (lispEntity_Nullp (pLispMgr, pKeyList)))
		return	False ;
	
	if (TFAILED (lispEntity_GetCar (pLispMgr, pKeyList, &pElement)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pKeyList, &pNextKeyList)))
		return	False ;
	pKeyList	= pNextKeyList ;
	
	if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pElement, &lValue)))
		lValue	= -1 ;
	lValueNear	= (fNearMatch && Char_Charset ((Char) lValue) == KCHARSET_XCHAR)? 
		Char_MakeAscii (Char_Code ((Char)lValue) & 0x7F) : -1 ;
	pEntNearHit	= NULL ;

	while (TFAILED (lispEntity_Nullp (pLispMgr, pList))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pList, &pEntry)))
			return	False ;

		lispEntity_GetType (pLispMgr, pEntry, &iType) ;
		switch (iType) {
		case	LISPENTITY_CONSCELL:
		{
			TLispEntity*	pCar ;
			long			lCar ;
			
			(void) lispEntity_GetCar (pLispMgr, pEntry, &pCar) ;

			/*	ACCEPT-DEFAULT äƤ硣*/
			if (fAcceptDefault && TSUCCEEDED (lispEntity_Tp (pLispMgr, pCar))) {
				(void) lispEntity_GetCdr (pLispMgr, pEntry, &pDefaultFunc) ;
				assert (pDefaultFunc != NULL) ;
			}
			
			if (TSUCCEEDED (lispEntity_Eq (pLispMgr, pCar, pElement)) ||
				(lValue >= 0 &&
				 TSUCCEEDED (lispEntity_GetIntegerValue (pLispMgr, pCar, &lCar)) &&
				 (lValue == lCar || (fNearMatch && lValueNear == lCar)))) {
				(void) lispEntity_GetCdr (pLispMgr, pEntry, &pEntry) ;
				return	lispKeymap_lookupWithListFound (pLispMgr, pEntry, pKeyList, fAcceptDefault, fNearMatch, pnMatch, ppEntReturn) ;
			}
			break ;
		}

		case	LISPENTITY_VECTOR:
		{
			TLispEntity**	ppLcElement ;
			int				nLcElement ;
			
			(void) lispEntity_GetVectorValue (pLispMgr, pEntry, &ppLcElement, &nLcElement) ;
			if (0 <= lValue && lValue < nLcElement) {
				pEntry		= *(ppLcElement + lValue) ;
				return	lispKeymap_lookupWithListFound (pLispMgr, pEntry, pKeyList, fAcceptDefault, fNearMatch, pnMatch, ppEntReturn) ;
			}
			if (fNearMatch) {
				if (pEntNearHit == NULL && 0 <= lValueNear && lValueNear < nLcElement) 
					pEntNearHit	= *(ppLcElement + lValueNear) ;
			}
			break ;
		}
		
		/*	ξˤϡkeymap ĤʤäƤǤϤʤΤ
		 *	פ*/
		case	LISPENTITY_SYMBOL:
			if (TSUCCEEDED (lispEntity_Keymapp (pLispMgr, pList)))
				return	lispKeymap_LookupWithList (pLispMgr, pList, pKeyList, fAcceptDefault, fNearMatch, pnMatch, ppEntReturn) ;
			break ;
			
		default:
			break ;
		}
		if (TFAILED (lispEntity_GetCdr (pLispMgr, pList, &pList)))
			return	False ;
	}

	if (fNearMatch && pEntNearHit != NULL)
		return	lispKeymap_lookupWithListFound (pLispMgr, pEntNearHit, pKeyList, fAcceptDefault, fNearMatch, pnMatch, ppEntReturn) ;

	/*	default keymap ˺Ƥ뤾(^^;; */
	++	*pnMatch ;
	if (pDefaultFunc != NULL) {
		*ppEntReturn	= pDefaultFunc ;
	} else {
		*ppEntReturn	= pList ;
	}
	return	True ;
}

Boolean
lispKeymap_lookupWithListFound (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pEntKeymap,
	register TLispEntity*			pEntKeyList,
	register Boolean				fAcceptDefault,
	register Boolean				fNearMatch,
	register int*					pnMatch,
	register TLispEntity** const	ppEntReturn)
{
	/*	ޤǤʳ֤ͤϷꤷƤ롣*/
	++ *pnMatch ;
	/*	ޤ򸫤Ƥߤġ*/
	if (TFAILED (lispEntity_Nullp (pLispMgr, pEntKeyList))) {
		/*	ޥåפǤʤ False ä롣
		 *	λϡֺʳޤƤפ
		 *	Ȥˤʤ롣*/
		if (TFAILED (lispKeymap_LookupWithList (pLispMgr, pEntKeymap, pEntKeyList, fAcceptDefault, fNearMatch, pnMatch, ppEntReturn)))
			*ppEntReturn	= pEntKeymap ;
	} else {
		*ppEntReturn	= pEntKeymap ;
	}
	return	True ;
}

Boolean
lispKeymap_DefineKeyWithVector (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntKeymap,
	register TLispEntity**	ppEntKeySeq,
	register int			nKeySeq,
	register TLispEntity*	pEntBind)
{
	TLispEntity*	pEntKeylist ;
	TLispEntity*	pEntKey ;
	TLispEntity*	pEntNextKeylist ;
	TLispEntity*	pEntSubkeymap ;

	pEntKeylist	= pEntKeymap ;
	if (TFAILED (lispEntity_Keymapp (pLispMgr, pEntKeylist)))
		return	False ;

	while (TFAILED (lispEntity_Nullp (pLispMgr, pEntKeylist))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pEntKeylist, &pEntKey)))
			break ;
		if (TSUCCEEDED (lispEntity_Vectorp  (pLispMgr, pEntKey)) &&
			TSUCCEEDED (lispEntity_Integerp (pLispMgr, *ppEntKeySeq))) {
			int		nLength ;
			long	nIndex ;
			lispEntity_GetLength (pLispMgr, pEntKey, &nLength) ;
			lispEntity_GetIntegerValue (pLispMgr, *ppEntKeySeq, &nIndex) ;
			if (0 <= nIndex && nIndex < nLength) {
				ppEntKeySeq	++ ;
				nKeySeq	-- ;
				if (nKeySeq == 0) 
					return	lispEntity_SetVectorElement (pLispMgr, pEntKey, nIndex, pEntBind) ;
				lispEntity_GetVectorElement (pLispMgr, pEntKey, nIndex, &pEntNextKeylist) ;
				if (TFAILED (lispEntity_Keymapp (pLispMgr, pEntNextKeylist))) {
					if (TFAILED (lispKeymap_defineSubkeyWithVector (pLispMgr, ppEntKeySeq, nKeySeq, pEntBind, &pEntSubkeymap)))
						return	False ;
					return	lispEntity_SetVectorElement (pLispMgr, pEntKey, nIndex, pEntSubkeymap) ;
				}
			}
		} else if (TSUCCEEDED (lispEntity_Consp  (pLispMgr, pEntKey))) {
			TLispEntity*	pEntKeyCode ;

			lispEntity_GetCar (pLispMgr, pEntKey, &pEntKeyCode) ;
			if (pEntKeyCode == *ppEntKeySeq) {
				ppEntKeySeq	++ ;
				nKeySeq	-- ;
				/*	ξ꤬СΥХɤ֤롣*/
				if (nKeySeq == 0) 
					return	lispEntity_SetCdr (pLispMgr, pEntKey, pEntBind) ;
				/*	ޥåפ򸫤롣*/
				lispEntity_GetCdr (pLispMgr, pEntKey, &pEntNextKeylist) ;
				if (TFAILED (lispEntity_Keymapp (pLispMgr, pEntNextKeylist)))
					break ;
				pEntKeylist	= pEntNextKeylist ;
			}
		}
		lispEntity_GetCdr (pLispMgr, pEntKeylist, &pEntNextKeylist) ;
		pEntKeylist	= pEntNextKeylist ;
	}
	/*
	 *	ʤۤɡꥹȤƬɲäȤ櫓
	 */
	lispEntity_GetCdr (pLispMgr, pEntKeymap, &pEntKeylist) ;
	if (TFAILED (lispKeymap_defineSubkeyWithVector (pLispMgr, ppEntKeySeq, nKeySeq, pEntBind, &pEntSubkeymap)))
		return	False ;
	lispEntity_AddRef (pLispMgr, pEntSubkeymap) ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pEntSubkeymap, pEntKeylist, &pEntKeylist)))
		return	False ;
	lispEntity_SetCdr (pLispMgr, pEntKeymap, pEntKeylist) ;
	lispEntity_Release (pLispMgr, pEntSubkeymap) ;
	return	True ;
}

/*
 *
 */
Boolean
lispKeymap_DefineKey (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntKeymap,
	register const Char*	pString,
	register int			nString,
	register TLispEntity*	pEntBind)
{
	TLispEntity*	pEntKeylist ;
	TLispEntity*	pEntKey ;
	TLispEntity*	pEntNextKeylist ;
	TLispEntity*	pEntSubkeymap ;

	pEntKeylist	= pEntKeymap ;
	if (TFAILED (lispEntity_Keymapp (pLispMgr, pEntKeylist)))
		return	False ;

	while (TFAILED (lispEntity_Nullp (pLispMgr, pEntKeylist))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pEntKeylist, &pEntKey)))
			break ;
		if (TSUCCEEDED (lispEntity_Vectorp (pLispMgr, pEntKey))) {
			int	nLength, nIndex ;
			lispEntity_GetLength (pLispMgr, pEntKey, &nLength) ;
			nIndex	= (int) *pString ;
			if (0 <= nIndex && nIndex < nLength) {
				pString	++ ;
				nString	-- ;
				if (nString == 0) 
					return	lispEntity_SetVectorElement (pLispMgr, pEntKey, nIndex, pEntBind) ;

				lispEntity_GetVectorElement (pLispMgr, pEntKey, nIndex, &pEntNextKeylist) ;
				if (TFAILED (lispEntity_Keymapp (pLispMgr, pEntNextKeylist))) {
					if (TFAILED (lispKeymap_defineSubkey (pLispMgr, pString, nString, pEntBind, &pEntSubkeymap)))
						return	False ;
					return	lispEntity_SetVectorElement (pLispMgr, pEntKey, nIndex, pEntSubkeymap) ;
				}
			}
		} else if (TSUCCEEDED (lispEntity_Consp  (pLispMgr, pEntKey))) {
			TLispEntity*	pEntKeyCode ;
			long			lCode ;

			lispEntity_GetCar (pLispMgr, pEntKey, &pEntKeyCode) ;
			if (TSUCCEEDED (lispEntity_GetIntegerValue (pLispMgr, pEntKeyCode, &lCode)) &&
				(Char)lCode == *pString) {
				pString	++ ;
				nString	-- ;
				/*	ξ꤬СΥХɤ֤롣*/
				if (nString == 0) 
					return	lispEntity_SetCdr (pLispMgr, pEntKey, pEntBind) ;
				/*	ޥåפ򸫤롣*/
				lispEntity_GetCdr (pLispMgr, pEntKey, &pEntNextKeylist) ;
				if (TFAILED (lispEntity_Keymapp (pLispMgr, pEntNextKeylist)))
					break ;
				pEntKeylist	= pEntNextKeylist ;
			}
		}
		lispEntity_GetCdr (pLispMgr, pEntKeylist, &pEntNextKeylist) ;
		pEntKeylist	= pEntNextKeylist ;
	}
	/*
	 *	ʤۤɡꥹȤƬɲäȤ櫓
	 */
	lispEntity_GetCdr (pLispMgr, pEntKeymap, &pEntKeylist) ;
	if (TFAILED (lispKeymap_defineSubkey (pLispMgr, pString, nString, pEntBind, &pEntSubkeymap)))
		return	False ;
	lispEntity_AddRef (pLispMgr, pEntSubkeymap) ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pEntSubkeymap, pEntKeylist, &pEntKeylist)))
		return	False ;
	lispEntity_SetCdr (pLispMgr, pEntKeymap, pEntKeylist) ;
	lispEntity_Release (pLispMgr, pEntSubkeymap) ;
	return	True ;
}

/*	private functions */

Boolean
lispKeymap_defineSubkey (
	register TLispManager*			pLispMgr,
	register const Char*			pString,
	register int					nString,
	register TLispEntity*			pEntBind,
	register TLispEntity** const	ppEntRetval)
{
	TLispEntity*	pEntKeyCode ;
	TLispEntity*	pEntKeymap ;
	TLispEntity*	pEntKeylist ;
	TLispEntity*	pEntNextKeylist ;
	const Char*	ptr ;

	assert (pString != NULL && nString > 0) ;

	ptr	= pString + nString - 1 ;
	if (TFAILED (lispMgr_CreateInteger      (pLispMgr, *ptr, &pEntKeyCode)))
		return	False ;
	lispEntity_AddRef  (pLispMgr, pEntKeyCode) ;
	if (TFAILED (lispMgr_CreateConscell     (pLispMgr, pEntKeyCode, pEntBind, &pEntKeylist)))
		return	False ;
	lispEntity_AddRef  (pLispMgr, pEntKeylist) ;
	lispEntity_Release (pLispMgr, pEntKeyCode) ;
	ptr		-- ;
	nString -- ;
	while (nString > 0) {
		if (TFAILED (lispMgr_CreateInteger      (pLispMgr, *ptr, &pEntKeyCode)))
			return	False ;
		lispEntity_AddRef  (pLispMgr, pEntKeyCode) ;
		if (TFAILED (lispMgr_CreateSparseKeymap (pLispMgr, pEntKeylist, &pEntKeymap)))
			return	False ;
		lispEntity_AddRef  (pLispMgr, pEntKeymap) ;
		if (TFAILED (lispMgr_CreateConscell     (pLispMgr, pEntKeyCode, pEntKeymap, &pEntNextKeylist)))
			return	False ;
		lispEntity_AddRef  (pLispMgr, pEntNextKeylist) ;
		lispEntity_Release (pLispMgr, pEntKeyCode) ;
		lispEntity_Release (pLispMgr, pEntKeymap) ;
		lispEntity_Release (pLispMgr, pEntKeylist) ;
		pEntKeylist	= pEntNextKeylist ;
		ptr		-- ;
		nString	-- ;
	}
	lispEntity_Release (pLispMgr, pEntKeylist) ;
	*ppEntRetval	= pEntKeylist ;
	return	True ;
}

Boolean
lispKeymap_defineSubkeyWithVector (
	register TLispManager*			pLispMgr,
	register TLispEntity**			ppEntKeySeq,
	register int					nKeySeq,
	register TLispEntity*			pEntBind,
	register TLispEntity** const	ppEntRetval)
{
	TLispEntity*	pEntKeymap ;
	TLispEntity*	pEntKeylist ;
	TLispEntity*	pEntNextKeylist ;

	assert (ppEntKeySeq != NULL && nKeySeq > 0) ;

	ppEntKeySeq	+= nKeySeq - 1 ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, *ppEntKeySeq, pEntBind, &pEntKeylist)))
		return	False ;
	lispEntity_AddRef  (pLispMgr, pEntKeylist) ;
	ppEntKeySeq	-- ;
	nKeySeq	-- ;
	while (nKeySeq > 0) {
		if (TFAILED (lispMgr_CreateSparseKeymap (pLispMgr, pEntKeylist, &pEntKeymap)))
			return	False ;
		lispEntity_AddRef  (pLispMgr, pEntKeymap) ;
		if (TFAILED (lispMgr_CreateConscell     (pLispMgr, *ppEntKeySeq, pEntKeymap, &pEntNextKeylist)))
			return	False ;
		lispEntity_AddRef  (pLispMgr, pEntNextKeylist) ;
		lispEntity_Release (pLispMgr, pEntKeymap) ;
		lispEntity_Release (pLispMgr, pEntKeylist) ;
		pEntKeylist	= pEntNextKeylist ;
		ppEntKeySeq	-- ;
		nKeySeq	-- ;
	}
	lispEntity_Release (pLispMgr, pEntKeylist) ;
	*ppEntRetval	= pEntKeylist ;
	return	True ;
}

Boolean
lispKeymap_equalChar (
	register Char		ccKey,
	register Char		ccKeymap)
{
	if (ccKey == ccKeymap)
		return	True ;
	if (Char_Charset (ccKey) == KCHARSET_XCHAR)
		ccKey	= Char_MakeAscii (Char_Code (ccKey) & 0x7F) ;
	return	ccKey == ccKeymap ;
}

