/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <assert.h>
#include <stdarg.h>
#include "lispmgrp.h"
#include "cstring.h"
#include "kanji.h"

#define	TEMPBUFSIZE	(1024)

static	Boolean	lispEntity_princStr			(TLispManager*, TLispEntity*, TVarbuffer*) ;
static	Boolean	lispEntity_princStrInteger	(TLispManager*, TLispEntity*, TVarbuffer*) ;
static	Boolean	lispEntity_princStrFloat	(TLispManager*, TLispEntity*, TVarbuffer*) ;
static	Boolean	lispEntity_princStrSymbol	(TLispManager*, TLispEntity*, TVarbuffer*) ;
static	Boolean	lispEntity_princStrString	(TLispManager*, TLispEntity*, TVarbuffer*) ;
static	Boolean	lispEntity_princStrConscell	(TLispManager*, TLispEntity*, TVarbuffer*) ;
static	Boolean	lispEntity_princStrVector	(TLispManager*, TLispEntity*, TVarbuffer*) ;
static	Boolean	lispEntity_princStrMarker	(TLispManager*, TLispEntity*, TVarbuffer*) ;
static	Boolean	lispEntity_princStrBuffer	(TLispManager*, TLispEntity*, TVarbuffer*) ;
static	Boolean	lispEntity_princStrWindow	(TLispManager*, TLispEntity*, TVarbuffer*) ;
static	Boolean	lispEntity_princStrFrame	(TLispManager*, TLispEntity*, TVarbuffer*) ;
static	Boolean	lispEntity_princStrSubr		(TLispManager*, TLispEntity*, TVarbuffer*) ;
static	Boolean	lispEntity_princStrIMClient	(TLispManager*, TLispEntity*, TVarbuffer*) ;
static	Boolean	lispEntity_princStrMutex	(TLispManager*, TLispEntity*, TVarbuffer*) ;
static	Boolean	lispEntity_princStrXEvent	(TLispManager*, TLispEntity*, TVarbuffer*) ;
static	Boolean	lispEntity_princStrRest		(TLispManager*, const char*, TLispEntity*, TVarbuffer*) ;
static	Boolean	lispEntity_print		(TLispManager*, TLispEntity*) ;
static	Boolean	lispEntity_printInteger	(TLispManager*, TLispEntity*) ;
static	Boolean	lispEntity_printFloat	(TLispManager*, TLispEntity*) ;
static	Boolean	lispEntity_printSymbol	(TLispManager*, TLispEntity*) ;
static	Boolean	lispEntity_printString	(TLispManager*, TLispEntity*) ;
static	Boolean	lispEntity_printConscell(TLispManager*, TLispEntity*) ;
static	Boolean	lispEntity_printVector	(TLispManager*r, TLispEntity*) ;
static	Boolean	lispEntity_printMarker	(TLispManager*, TLispEntity*) ;
static	Boolean	lispEntity_printBuffer	(TLispManager*, TLispEntity*) ;
static	Boolean	lispEntity_printWindow	(TLispManager*, TLispEntity*) ;
static	Boolean	lispEntity_printFrame	(TLispManager*, TLispEntity*) ;
static	Boolean	lispEntity_printSubr	(TLispManager*, TLispEntity*) ;
static	Boolean	lispEntity_printIMClient(TLispManager*, TLispEntity*) ;
static	Boolean	lispEntity_printMutex   (TLispManager*, TLispEntity*) ;
static	Boolean	lispEntity_printXEvent  (TLispManager*, TLispEntity*) ;
static	Boolean	lispEntity_printEmpty	(TLispManager*, TLispEntity*) ;
static	Boolean	lispEntity_printVoid	(TLispManager*, TLispEntity*) ;
static	Boolean	lispEntity_formatString	(TLispManager*, TLispEntity*, int, TVarbuffer*) ;
static	Boolean	lispEntity_formatChar	(TLispManager*, TLispEntity*, TVarbuffer*) ;
static	Boolean	lispEntity_formatNumber	(TLispManager*, TLispEntity*, const Char*, int, const Boolean, TVarbuffer*) ;
static	Boolean	lispEntity_formatNumberA	(TLispManager*, TLispEntity*, const char*, int, const Boolean, TVarbuffer*) ;
static	Boolean	lispEntity_formatNumberCommon	(TLispManager*, TLispEntity*, const char*, const Boolean, TVarbuffer*) ;
static	Boolean	lispEntity_copyConscell	(TLispManager*, TLispEntity*, TLispEntity**) ;
static	Boolean	lispEntity_copyVector	(TLispManager*, TLispEntity*, TLispEntity**) ;

/*
 *	LIST pEntList  CAR  pEntElt  eq ǤС CDR 
 *	֤
 */
Boolean
lispEntity_Memq (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntElt,
	register TLispEntity*	pEntList,
	register TLispEntity**	ppEntRetval)
{
	TLispEntity*	pEntCar ;
	TLispEntity*	pEntNextList ;

	assert (pLispMgr    != NULL) ;
	assert (pEntElt     != NULL) ;
	assert (pEntList    != NULL) ;

	while (TFAILED (lispEntity_Nullp (pLispMgr, pEntList))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pEntList, &pEntCar)) ||
			TFAILED (lispEntity_GetCdr (pLispMgr, pEntList, &pEntNextList)))
			return	False ;
		if (TSUCCEEDED (lispEntity_Eq (pLispMgr, pEntElt, pEntCar))) 
			break ;
		pEntList	= pEntNextList ;
	}
	if (ppEntRetval != NULL)
		*ppEntRetval	= pEntList ;
	return	True ;
}

Boolean
lispEntity_Member (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntElt,
	register TLispEntity*	pEntList,
	register TLispEntity**	ppEntRetval)
{
	TLispEntity*	pEntCar ;
	TLispEntity*	pEntNextList ;

	assert (pLispMgr    != NULL) ;
	assert (pEntElt     != NULL) ;
	assert (pEntList    != NULL) ;

	while (TFAILED (lispEntity_Nullp (pLispMgr, pEntList))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pEntList, &pEntCar)) ||
			TFAILED (lispEntity_GetCdr (pLispMgr, pEntList, &pEntNextList)))
			return	False ;
		if (TSUCCEEDED (lispEntity_Equal (pLispMgr, pEntElt, pEntCar))) 
			break ;
		pEntList	= pEntNextList ;
	}
	if (ppEntRetval != NULL)
		*ppEntRetval	= pEntList ;
	return	True ;
}

Boolean
lispEntity_Rassoc (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntKey,
	register TLispEntity*	pEntList,
	register TLispEntity**	ppEntRetval)
{
	TLispEntity*	pEntCar ;
	TLispEntity*	pEntCadr ;
	TLispEntity*	pEntNextList ;
	TLispEntity*	pEntRetval ;

	assert (pLispMgr    != NULL) ;
	assert (pEntKey     != NULL) ;
	assert (pEntList    != NULL) ;

	lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	while (TFAILED (lispEntity_Nullp (pLispMgr, pEntList))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pEntList, &pEntCar)) ||
			TFAILED (lispEntity_GetCdr (pLispMgr, pEntList, &pEntNextList)))
			return	False ;
		if (TSUCCEEDED (lispEntity_GetCdr (pLispMgr, pEntCar, &pEntCadr)) &&
			TSUCCEEDED (lispEntity_Equal  (pLispMgr, pEntKey, pEntCadr))) {
			pEntRetval	= pEntCar ;
			break ;
		}
		pEntList	= pEntNextList ;
	}
	if (ppEntRetval != NULL)
		*ppEntRetval	= pEntRetval ;
	return	True ;
}

Boolean
lispEntity_Nconc (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntList,
	register TLispEntity**	ppEntRetval)
{
	TLispEntity*	pEntRetval ;
	TLispEntity*	pEntTail ;
	TLispEntity*	pEntCar ;
	TLispEntity*	pEntCdr ;

	assert (pLispMgr    != NULL) ;
	assert (pEntList    != NULL) ;
	assert (ppEntRetval != NULL) ;

	lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	while (TFAILED (lispEntity_Nullp (pLispMgr, pEntList))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pEntList, &pEntCar)) ||
			TFAILED (lispEntity_GetCdr (pLispMgr, pEntList, &pEntCdr)))
			return	False ;
		pEntList	= pEntCdr ;
		if (TFAILED (lispEntity_Nullp  (pLispMgr, pEntCar))) {
			pEntRetval	= pEntCar ;
			lispEntity_GetLastElement (pLispMgr,  pEntCar, &pEntTail) ;
			break ;
		}
	}
	while (TFAILED (lispEntity_Nullp (pLispMgr, pEntList))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pEntList, &pEntCar)) ||
			TFAILED (lispEntity_Listp  (pLispMgr, pEntCar)) ||
			TFAILED (lispEntity_GetCdr (pLispMgr, pEntList, &pEntCdr)))
			return	False ;
		if (TFAILED (lispEntity_Nullp  (pLispMgr, pEntCar))) {
			lispEntity_SetCdr (pLispMgr, pEntTail, pEntCar) ;
			lispEntity_GetLastElement (pLispMgr, pEntCar, &pEntTail) ;
			break ;
		}
		pEntList	= pEntCdr ;
	}
	*ppEntRetval	= pEntRetval ;
	return	True ;
}

Boolean
lispEntity_Equal (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntLeft,
	register TLispEntity*	pEntRight)
{
	int			iLeftType,   iRightType ;

	assert (pLispMgr     != NULL) ;
	assert (pEntLeft  != NULL) ;
	assert (pEntRight != NULL) ;

	/*	`eq' ʤ `equal' Ω롣*/
	if (TSUCCEEDED (lispEntity_Eq (pLispMgr, pEntLeft, pEntRight)))
		return	True ;
	
	if (TFAILED (lispEntity_GetType (pLispMgr, pEntLeft,  &iLeftType))  ||
		TFAILED (lispEntity_GetType (pLispMgr, pEntRight, &iRightType)) ||
		iLeftType != iRightType) 
		return	False ;
	
	switch (iLeftType /* == iRightType */) {
	case	LISPENTITY_FLOAT:
	{
		float	fLeftValue, fRightValue ;

		lispEntity_GetFloatValue (pLispMgr, pEntLeft,  &fLeftValue) ;
		lispEntity_GetFloatValue (pLispMgr, pEntRight, &fRightValue) ;
		return	(fLeftValue == fRightValue)? True : False ;
	}

	case	LISPENTITY_CONSCELL:
	{
		TLispEntity*	pLeftCar ;
		TLispEntity*	pLeftCdr ;
		TLispEntity*	pRightCar ;
		TLispEntity*	pRightCdr ;

		lispEntity_GetCar (pLispMgr, pEntLeft,  &pLeftCar) ;
		lispEntity_GetCar (pLispMgr, pEntRight, &pRightCar) ;
		if (TFAILED (lispEntity_Equal (pLispMgr, pLeftCar, pRightCar)))
			return	False ;
		lispEntity_GetCdr (pLispMgr, pEntLeft,  &pLeftCdr) ;
		lispEntity_GetCdr (pLispMgr, pEntRight, &pRightCdr) ;
		if (TFAILED (lispEntity_Equal (pLispMgr, pLeftCdr, pRightCdr)))
			return	False ;
		return	True ;
	}
			
	case	LISPENTITY_STRING:
	{
		const Char*	pLeftString ;
		int			nLeftLength ;
		const Char*	pRightString ;
		int			nRightLength ;

		lispEntity_GetStringValue (pLispMgr, pEntLeft,  &pLeftString,  &nLeftLength) ;
		lispEntity_GetStringValue (pLispMgr, pEntRight, &pRightString, &nRightLength) ;
		if (nLeftLength != nRightLength ||
			Cstrncmp (pLeftString, pRightString, nLeftLength))
			return	False ;
		return	True ;
	}
		
	case	LISPENTITY_VECTOR:
	{
		TLispEntity**	ppLeftElement ;
		int				nLeftElement ;
		TLispEntity**	ppRightElement ;
		int				nRightElement ;

		lispEntity_GetVectorValue (pLispMgr, pEntLeft,  &ppLeftElement,  &nLeftElement) ;
		lispEntity_GetVectorValue (pLispMgr, pEntRight, &ppRightElement, &nRightElement) ;
		if (nLeftElement != nRightElement)
			return	False ;

		while (nLeftElement > 0) {
			if (TFAILED (lispEntity_Equal (pLispMgr, *ppLeftElement, *ppRightElement)))
				return	False ;
			ppLeftElement	++ ;
			ppRightElement	++ ;
			nLeftElement	-- ;
		}
		return	True ;
	}
		
	case	LISPENTITY_SYMBOL:
	case	LISPENTITY_INTEGER:
	default:
		return	False ;
	}
}

Boolean
lispEntity_GetNumberValueOrMarkerPosition (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TLispNumber*	pReturn)
{
	long			lValue ;
	float			fValue ;
	TLispEntity*	pEntBuffer ;
	int				iPos ;

	assert (pLispMgr != NULL) ;

	switch (pEntity->m_iType) {
	case	LISPENTITY_XEVENT:
	case	LISPENTITY_INTEGER:
		if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntity, &lValue)))
			return	False ;
		pReturn->m_fFloatp			= False ;
		pReturn->m_Value.m_lLong	= lValue ;
		break ;
	case	LISPENTITY_MARKER:
		if (TFAILED (lispMarker_GetBufferPosition (pLispMgr, pEntity, &pEntBuffer, &iPos)) ||
			pEntBuffer == NULL)
			return	False ;
		pReturn->m_fFloatp			= False ;
		pReturn->m_Value.m_lLong	= (long) iPos ;
		break ;
	case	LISPENTITY_FLOAT:
		(void) lispEntity_GetFloatValue (pLispMgr, pEntity, &fValue) ;
		pReturn->m_fFloatp			= True ;
		pReturn->m_Value.m_fFloat	= fValue ;
		break ;
	default:
		return	False ;
	}
	return	True ;
}

Boolean
lispEntity_GetLength (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register int*			pnLength)
{
	int			nLength ; 
	
	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (pnLength != NULL) ;

	if (TFAILED (lispEntity_Sequencep (pLispMgr, pEntity)))
		return	False ;

	switch (pEntity->m_iType) {
	case	LISPENTITY_STRING:
	{
		const Char*	pString ;
		
		(void) lispEntity_GetStringValue (pLispMgr, pEntity, &pString, &nLength) ;
		break ;
	}
		
	case	LISPENTITY_VECTOR:
	{
		TLispEntity**	ppElement ;

		(void) lispEntity_GetVectorValue (pLispMgr, pEntity, &ppElement, &nLength) ;
		break ;
	}
	
	default:
	{
		TLispEntity*	pNextEntity ;
		
		nLength	= 0 ;
		while (TFAILED (lispEntity_Nullp (pLispMgr, pEntity))) {
			if (TFAILED (lispEntity_GetCdr (pLispMgr, pEntity, &pNextEntity))) {
#if defined (DEBUG)
				fprintf (stderr, "Wrong type argument: listp, ") ;
				lispEntity_Print (pLispMgr, pEntity) ;
				fprintf (stderr, "\n") ;
#endif
				return	False ;
			}
			nLength		++ ;
			pEntity	= pNextEntity ;
		}
	}
	}
	*pnLength	= nLength ;
	return	True ;
}

Boolean
lispEntity_GetInteractive (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TLispEntity**	ppReturn)
{
	TLispEntity*	pInteractive ;
	TLispEntity*	pNextEntity ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (ppReturn != NULL) ;

	pInteractive	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_INTERACTIVE) ;
	while (TFAILED (lispEntity_Nullp (pLispMgr, pEntity))) {
		TLispEntity*	pCar ;
		TLispEntity*	pCaar ;
		if (TSUCCEEDED (lispEntity_GetCar (pLispMgr, pEntity, &pCar))  &&
			TSUCCEEDED (lispEntity_GetCar (pLispMgr, pCar,    &pCaar)) &&
			pCaar == pInteractive) {
			lispEntity_GetCadr (pLispMgr, pCar, ppReturn) ;
			return	True ;
		}
		if (TFAILED (lispEntity_GetCdr (pLispMgr, pEntity, &pNextEntity)))
			return	False ;
		pEntity	= pNextEntity ;
	}
	*ppReturn	= NULL ;
	return	True ;
}

/*
 *	(format STRING &rest OBJECTS) μΤȤʤؿ
 *
 *	(pFormat, nFormat) Ϳ줿񼰤˽ä pEntData ˤä
 *	Ϳ줿 entity  list ɽ롣
 *	񼰤б entity ְäƤ/ʤˤϥ顼֤
 */
Boolean
lispEntity_Format (
	register TLispManager*	pLispMgr,
	register const Char*	pFormat,
	register int			nFormat,
	register TLispEntity*	pEntData,
	register TLispEntity**	ppEntRetval)
{
	TVarbuffer			vbuf ;
	TLispEntity*		pEntArg ;
	TLispEntity*		pEntNextData ;
	register int			nUpper ;
	register const Char*	pSubFormat ;
	register int			nSubFormat ;

	if (TFAILED (TVarbuffer_Initialize (&vbuf, sizeof (Char))))
		return	False ;

	while (nFormat > 0) {
		if (*pFormat != '%') {
			if (TFAILED (TVarbuffer_Add (&vbuf, pFormat, 1)))
				goto	error ;
			goto	skip ;
		}
		pSubFormat	= pFormat ;
		pFormat	++ ;
		nFormat	-- ;
		if (nFormat <= 0) {
#if defined (DEBUG) || 1
			fprintf (stderr, "format error -> nFormat = 0\n") ;
#endif
			goto	error ;
		}
		nUpper	= 0 ;
		while (nFormat > 0 && '0' <= *pFormat && *pFormat <= '9') {
			nUpper	= nUpper * 10 + (nUpper - '0') ;
			pFormat	++ ;
			nFormat -- ;
		}
		if (nFormat > 0 && *pFormat == '.') {
			pFormat	++ ;
			nFormat -- ;
			while (nFormat > 0 && '0' <= *pFormat && *pFormat <= '9') {
				pFormat	++ ;
				nFormat -- ;
			}
		}
		if (nFormat <= 0) {
#if defined (DEBUG) || 1
			fprintf (stderr, "format error -> nFormat = 0\n") ;
#endif
			goto	error ;
		}
		nSubFormat	= pFormat - pSubFormat + 1 ;
		switch (*pFormat) {
		case	's':
		case	'S':
			if (TFAILED (lispEntity_GetCar (pLispMgr, pEntData, &pEntArg)) ||
				TFAILED (lispEntity_formatString (pLispMgr, pEntArg, nUpper, &vbuf))) {
#if defined (DEBUG) || 1
				fprintf (stderr, "format error -> %%s -> ") ;
				lispEntity_Print (pLispMgr, pEntArg) ;
				fprintf (stderr, "\n") ;
#endif
				goto	error ;
			}
			break ;

		case	'c':
			if (TFAILED (lispEntity_GetCar (pLispMgr, pEntData, &pEntArg)) ||
				TFAILED (lispEntity_formatChar (pLispMgr, pEntArg, &vbuf))) {
#if defined (DEBUG) || 1
				fprintf (stderr, "format error -> %%c -> ") ;
				lispEntity_Print (pLispMgr, pEntArg) ;
				fprintf (stderr, "\n") ;
#endif
				goto	error ;
			}
			break ;

		case	'd':
		case	'o':
		case	'x':
		case	'X':
			if (TFAILED (lispEntity_GetCar (pLispMgr, pEntData, &pEntArg)) ||
				TFAILED (lispEntity_formatNumber (pLispMgr, pEntArg, pSubFormat, nSubFormat, False, &vbuf))) {
#if defined (DEBUG) || 1
				fprintf (stderr, "format error -> doxX -> ") ;
				lispEntity_Print (pLispMgr, pEntArg) ;
				fprintf (stderr, "\n") ;
#endif
				goto	error ;
			}
			break ;

		case	'e':
		case	'f':
		case	'g':
			if (TFAILED (lispEntity_GetCar (pLispMgr, pEntData, &pEntArg)) ||
				TFAILED (lispEntity_formatNumber (pLispMgr, pEntArg, pSubFormat, nSubFormat, True, &vbuf))) {
#if defined (DEBUG) || 1
				fprintf (stderr, "format error -> efg -> ") ;
				lispEntity_Print (pLispMgr, pEntArg) ;
				fprintf (stderr, "\n") ;
#endif
				goto	error ;
			}
			break ;

		case	'%':
			if (TFAILED (TVarbuffer_Add (&vbuf, pFormat, 1)))
				goto	error ;
			goto	skip ;

		default:
#if defined (DEBUG) || 1
			fprintf (stderr, "format error -> default (%c), %c\n",
					 (int)*pSubFormat, (int)*pFormat) ;
#endif
			goto	error ;
		}
		if (TFAILED (lispEntity_GetCdr (pLispMgr, pEntData, &pEntNextData)))
			goto	error ;
		pEntData	= pEntNextData ;
	  skip:
		pFormat	++ ;
		nFormat	-- ;
	}

	if (TFAILED (lispMgr_CreateString (pLispMgr, TVarbuffer_GetBuffer (&vbuf), TVarbuffer_GetUsage (&vbuf), ppEntRetval)))
		goto	error ;
	return	True ;

 error:
	TVarbuffer_Uninitialize (&vbuf) ;
	return	False ;
}

/*
 *	(format STRING &rest OBJECTS) μΤȤʤؿ
 *
 *	(pFormat, nFormat) Ϳ줿񼰤˽ä pEntData ˤä
 *	Ϳ줿 entity  list ɽ롣
 *	񼰤б entity ְäƤ/ʤˤϥ顼֤
 */
Boolean
lispEntity_FormatA (
	register TLispManager*	pLispMgr,
	register const char*	pFormat,
	register int			nFormat,
	register TLispEntity*	pEntData,
	register TLispEntity**	ppEntRetval)
{
	TVarbuffer			vbuf ;
	TLispEntity*		pEntArg ;
	TLispEntity*		pEntNextData ;
	Char				cc ;
	register int			nUpper ;
	register const char*	pSubFormat ;
	register int			nSubFormat ;
	register Boolean	fRetval	= False ;

	if (TFAILED (TVarbuffer_Initialize (&vbuf, sizeof (Char))))
		return	False ;

	while (nFormat > 0) {
		if (*pFormat != '%') {
			cc	= Char_MakeAscii (*pFormat) ;
			if (TFAILED (TVarbuffer_Add (&vbuf, &cc, 1)))
				goto	error ;
			goto	skip ;
		}
		pSubFormat	= pFormat ;
		pFormat	++ ;
		nFormat	-- ;
		if (nFormat <= 0)
			goto	error ;
		nUpper	= 0 ;
		while (nFormat > 0 && '0' <= *pFormat && *pFormat <= '9') {
			nUpper	= nUpper * 10 + (nUpper - '0') ;
			pFormat	++ ;
			nFormat -- ;
		}
		if (nFormat > 0 && *pFormat == '.') {
			pFormat	++ ;
			nFormat -- ;
			while (nFormat > 0 && '0' <= *pFormat && *pFormat <= '9') {
				pFormat	++ ;
				nFormat -- ;
			}
		}
		if (nFormat <= 0)
			goto	error ;

		nSubFormat	= pFormat - pSubFormat + 1 ;
		switch (*pFormat) {
		case	's':
		case	'S':
			if (TFAILED (lispEntity_GetCar (pLispMgr, pEntData, &pEntArg)) ||
				TFAILED (lispEntity_formatString (pLispMgr, pEntArg, nUpper, &vbuf)))
				goto	error ;
			break ;

		case	'c':
			if (TFAILED (lispEntity_GetCar (pLispMgr, pEntData, &pEntArg)) ||
				TFAILED (lispEntity_formatChar (pLispMgr, pEntArg, &vbuf)))
				goto	error ;
			break ;

		case	'd':
		case	'o':
		case	'x':
		case	'X':
			if (TFAILED (lispEntity_GetCar (pLispMgr, pEntData, &pEntArg)) ||
				TFAILED (lispEntity_formatNumberA (pLispMgr, pEntArg, pSubFormat, nSubFormat, False, &vbuf)))
				goto	error ;
			break ;

		case	'e':
		case	'f':
		case	'g':
			if (TFAILED (lispEntity_GetCar (pLispMgr, pEntData, &pEntArg)) ||
				TFAILED (lispEntity_formatNumberA (pLispMgr, pEntArg, pSubFormat, nSubFormat, True, &vbuf))) 
				goto	error ;
			break ;

		case	'%':
			cc	= Char_MakeAscii (*pFormat) ;
			if (TFAILED (TVarbuffer_Add (&vbuf, &cc, 1)))
				goto	error ;
			goto	skip ;

		default:
			goto	error ;
		}
		if (TFAILED (lispEntity_GetCdr (pLispMgr, pEntData, &pEntNextData)))
			goto	error ;
		pEntData	= pEntNextData ;
	  skip:
		pFormat	++ ;
		nFormat	-- ;
	}

	if (TFAILED (lispMgr_CreateString (pLispMgr, TVarbuffer_GetBuffer (&vbuf), TVarbuffer_GetUsage (&vbuf), ppEntRetval)))
		goto	error ;
	fRetval	= True ;

 error:
	TVarbuffer_Uninitialize (&vbuf) ;
	return	fRetval ;
}

Boolean
lispEntity_PrincStr (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbuf)
{
	static const Char	chL	= '(' ;
	static const Char	chR	= ')' ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (pvbuf    != NULL) ;

	if (TSUCCEEDED (lispEntity_Consp (pLispMgr, pEntity))) {
		if (TFAILED (TVarbuffer_Add (pvbuf, &chL, 1)) ||
			TFAILED (lispEntity_princStr (pLispMgr, pEntity, pvbuf)) ||
			TFAILED (TVarbuffer_Add (pvbuf, &chR, 1)))
			return	False ;
		return	True ;
	} else {
		return	lispEntity_princStr (pLispMgr, pEntity, pvbuf) ;
	}
}

Boolean
lispEntity_princStr (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbuf)
{
	static	Boolean	(*arPrincStrFuncTbl[])(TLispManager*, TLispEntity*, TVarbuffer*) = {
		lispEntity_princStrInteger,		lispEntity_princStrFloat,
		lispEntity_princStrConscell,	lispEntity_princStrVector,
		lispEntity_princStrString,		lispEntity_princStrSymbol,
		lispEntity_princStrMarker,		lispEntity_princStrBuffer,
		lispEntity_princStrWindow,		lispEntity_princStrFrame,
		lispEntity_princStrSubr,		lispEntity_princStrIMClient,
		lispEntity_princStrMutex,		lispEntity_princStrXEvent,
		NULL,	/* empty */				NULL,	/* void */
		NULL,	/* bool-vector */		NULL,	/* char-table */
	} ;
	int		nType ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (pvbuf    != NULL) ;

	lispEntity_GetType (pLispMgr, pEntity, &nType) ;
	if (nType < 0 || nType >= MAX_LISPENTITY_TYPE)
		return	False ;
	if (arPrincStrFuncTbl [nType] != NULL) {
		return	(arPrincStrFuncTbl [nType])(pLispMgr, pEntity, pvbuf) ;
	} else {
		static const Char	sstrUnknown[]	= { '(','u','n','k','n','o','w','n',')' } ;
		return	TVarbuffer_Add (pvbuf, sstrUnknown, NELEMENTS (sstrUnknown)) ;
	}
}

/*
 *	δؿȻƤ뤬ϥǥХåѤǤ롣
 */
Boolean
lispEntity_Print (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	assert (pLispMgr != NULL) ;

	if (pEntity == NULL) {
		fprintf (stderr, "NULL") ;
		return	True ;
	}
	if (TSUCCEEDED (lispEntity_Consp (pLispMgr, pEntity))) {
		fprintf (stderr, "(") ;
		lispEntity_print (pLispMgr, pEntity) ;
		fprintf (stderr, ")") ;
		return	True ;
	} else {
		return	lispEntity_print (pLispMgr, pEntity) ;
	}
}

Boolean
lispEntity_print (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	static	Boolean	(*arPrintFuncTbl[])(TLispManager*, TLispEntity*) = {
		lispEntity_printInteger,	lispEntity_printFloat,
		lispEntity_printConscell,	lispEntity_printVector,
		lispEntity_printString,		lispEntity_printSymbol,
		lispEntity_printMarker,		lispEntity_printBuffer,
		lispEntity_printWindow,		lispEntity_printFrame,
		lispEntity_printSubr,		lispEntity_printIMClient,
		lispEntity_printMutex,		lispEntity_printXEvent,
		lispEntity_printEmpty,		lispEntity_printVoid,
		NULL, /* bool-vector */
		NULL, /* char-table */
	} ;
	int		nType ;

	assert (pLispMgr != NULL) ;

	if (pEntity == NULL) {
		fprintf (stderr, "NULL") ;
		return	True ;
	}
	lispEntity_GetType (pLispMgr, pEntity, &nType) ;
	if (nType < 0 || nType >= MAX_LISPENTITY_TYPE)
		return	False ;
	if (arPrintFuncTbl [nType] != NULL) {
		return	(arPrintFuncTbl [nType])(pLispMgr, pEntity) ;
	} else {
		fprintf (stderr, "'unknown") ;
		return	True ;
	}
}

/*	private functions */
Boolean
lispEntity_princStrInteger (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbuf)
{
	char			szBuf [64] ;
	long			lValue ;
	register int	nLength ;
	register Char*	pDest ;
	register int	nUsage ;
	
	if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntity, &lValue)))
		return	False ;
	snprintf (szBuf, sizeof (szBuf) - 1, "%ld", lValue) ;
	szBuf [sizeof (szBuf) - 1]	= '\0' ;
	nLength	= strlen (szBuf) ;
	nUsage	= TVarbuffer_GetUsage (pvbuf) ;
	if (TFAILED (TVarbuffer_Require (pvbuf, nLength)))
		return	False ;
	pDest	= (Char *)TVarbuffer_GetBuffer (pvbuf) + nUsage ;
	strtocstr (pDest, szBuf, nLength) ;
	return	True ;
}
	
Boolean
lispEntity_princStrFloat (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbuf)
{
	char			szBuf [64] ;
	float			fValue ;
	register int	nLength ;
	register Char*	pDest ;
	register int	nUsage ;

	if (TFAILED (lispEntity_GetFloatValue (pLispMgr, pEntity, &fValue)))
		return	False ;
	snprintf (szBuf, sizeof (szBuf) - 1, "%f", fValue) ;
	szBuf [sizeof (szBuf) - 1]	= '\0' ;
	nLength	= strlen (szBuf) ;
	nUsage	= TVarbuffer_GetUsage (pvbuf) ;
	if (TFAILED (TVarbuffer_Require (pvbuf, nLength)))
		return	False ;
	pDest	= (Char *)TVarbuffer_GetBuffer (pvbuf) + nUsage ;
	strtocstr (pDest, szBuf, nLength) ;
	return	True ;
}
	
Boolean
lispEntity_princStrSymbol (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbuf)
{
	const Char*	pName ;
	int			nName ;

	if (TFAILED (lispEntity_GetSymbolName (pLispMgr, pEntity, &pName, &nName)) |\
		TFAILED (TVarbuffer_Add (pvbuf, pName, nName)))
		return	False ;
	return	True ;
}

Boolean
lispEntity_princStrString (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbuf)
{
	static const Char	chDoubleQuote	= '\"' ;
	const Char*	pString ;
	int			nLength ;
	
	(void) lispEntity_GetStringValue (pLispMgr, pEntity, &pString, &nLength) ;
	if (TFAILED (TVarbuffer_Add (pvbuf, &chDoubleQuote, 1)) ||
		TFAILED (TVarbuffer_Add (pvbuf, pString, nLength)) ||
		TFAILED (TVarbuffer_Add (pvbuf, &chDoubleQuote, 1)))
		return	False ;
	return	True ;
}

Boolean
lispEntity_princStrConscell (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbuf)
{
	static const Char	chParenthesisL	= '(' ;
	static const Char	chParenthesisR	= ')' ;
	static const Char	chSpace			= ' ' ;
	TLispEntity*	pEntCar ;
	TLispEntity*	pEntCdr ;

	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntity, &pEntCar)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntity, &pEntCdr)))
		return	False ;
	if (TSUCCEEDED (lispEntity_Consp (pLispMgr, pEntCar))) {
		if (TFAILED (TVarbuffer_Add (pvbuf, &chParenthesisL, 1)) ||
			TFAILED (lispEntity_princStr (pLispMgr, pEntCar, pvbuf)) ||
			TFAILED (TVarbuffer_Add (pvbuf, &chParenthesisR, 1)))
			return	False ;
	} else {
		if (TFAILED (lispEntity_princStr (pLispMgr, pEntCar, pvbuf)))
			return	False ;
	}
	if (TFAILED (lispEntity_Listp (pLispMgr, pEntCdr))) {
		static const Char	rchSDS []	= { ' ', '.', ' ', } ;
		if (TFAILED (TVarbuffer_Add (pvbuf, rchSDS, NELEMENTS (rchSDS))) ||
			TFAILED (lispEntity_princStr (pLispMgr, pEntCdr, pvbuf)))
			return	False ;
	} else {
		if (TFAILED (lispEntity_Nullp (pLispMgr, pEntCdr)) &&
			(TFAILED (TVarbuffer_Add (pvbuf, &chSpace, 1)) ||
			 TFAILED (lispEntity_princStr (pLispMgr, pEntCdr, pvbuf))))
			return	False ;
	}
	return	True ;
}

Boolean
lispEntity_princStrVector (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbuf)
{
	static const Char	chBracketL	= '[' ;
	static const Char	chSpace		= ' ' ;
	static const Char	chBracketR	= ']' ;
	TLispEntity**	ppElement ;
	int				nElement ;

	lispEntity_GetVectorValue (pLispMgr, pEntity, &ppElement, &nElement) ;
	if (TFAILED (TVarbuffer_Add (pvbuf, &chBracketL, 1)))
		return	False ;
	if (nElement > 0) {
		do {
			if (TFAILED (lispEntity_PrincStr (pLispMgr, *ppElement ++, pvbuf)) ||
				(nElement > 1 && TFAILED (TVarbuffer_Add (pvbuf, &chSpace, 1))))
				return	False ;
			nElement	-- ;
		} while (nElement > 0) ;
	}
	return	TVarbuffer_Add (pvbuf, &chBracketR, 1) ;
}

Boolean
lispEntity_princStrMarker (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbuf)
{
	TLispEntity*	pEntBuffer ;
	int				nPos ;
	char			szBuf [64] ;
	register int	nLength ;
	register Char*	pDest ;

	lispMarker_GetBufferPosition (pLispMgr, pEntity, &pEntBuffer, &nPos) ;
	if (pEntBuffer != NULL) {
		snprintf (szBuf, sizeof (szBuf) - 1, "#<marker at %d in %lx>", nPos, (unsigned long)pEntBuffer) ;
	} else {
		snprintf (szBuf, sizeof (szBuf) - 1, "#<marker in no buffer>") ;
	}
	szBuf [sizeof (szBuf) - 1]	= '\0' ;
	nLength	= strlen (szBuf) ;
	if (TFAILED (TVarbuffer_Require (pvbuf, nLength)))
		return	False ;
	pDest	= (Char *)TVarbuffer_GetBuffer (pvbuf) + TVarbuffer_GetUsage (pvbuf) ;
	strtocstr (pDest, szBuf, nLength) ;
	return	True ;
}

Boolean
lispEntity_princStrBuffer (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity, 
	register TVarbuffer*	pvbuf)
{
	return	lispEntity_princStrRest (pLispMgr, "#<buffer %lx>", pEntity, pvbuf) ;
}

Boolean
lispEntity_princStrWindow (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbuf)
{
	return	lispEntity_princStrRest (pLispMgr, "#<window %lx>", pEntity, pvbuf) ;
}

Boolean
lispEntity_princStrFrame (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbuf)
{
	return	lispEntity_princStrRest (pLispMgr, "#<frame %lx>", pEntity, pvbuf) ;
}

Boolean
lispEntity_princStrSubr (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbuf)
{
	static const Char	rchSTR1 []	= { '#','<','s','u','b','r',' ', } ;
	static const Char	rchSTR2 []	= { '>', } ;
	const Char*	strSubrName ;

	if (TFAILED (lispSubr_GetName (pLispMgr, pEntity, &strSubrName)))
		return	False ;
	if (TFAILED (TVarbuffer_Add (pvbuf, rchSTR1, NELEMENTS (rchSTR1))) ||
		TFAILED (TVarbuffer_Add (pvbuf, strSubrName, Cstrlen (strSubrName))) ||
		TFAILED (TVarbuffer_Add (pvbuf, rchSTR2, NELEMENTS (rchSTR2))))
		return	False ;
	return	True ;
}

Boolean
lispEntity_princStrIMClient (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbuf)
{
	return	lispEntity_princStrRest (pLispMgr, "#<im-client %lx>", pEntity, pvbuf) ;
}

Boolean
lispEntity_princStrMutex (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbuf)
{
	return	lispEntity_princStrRest (pLispMgr, "#<mutex %lx>", pEntity, pvbuf) ;
}

Boolean
lispEntity_princStrXEvent (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbuf)
{
	return	lispEntity_princStrRest (pLispMgr, "#<xevent %lx>", pEntity, pvbuf) ;
}

Boolean
lispEntity_princStrRest (
	register TLispManager*	pLispMgr,
	register const char*	pFormat,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbuf)
{
	char			szBuf [64] ;
	int				nLength ;
	Char*			pDest ;
	snprintf (szBuf, sizeof (szBuf) - 1, pFormat, (unsigned long) pEntity) ;
	szBuf [sizeof (szBuf) - 1]	= '\0' ;
	nLength	= strlen (szBuf) ;
	if (TFAILED (TVarbuffer_Require (pvbuf, nLength)))
		return	False ;
	pDest	= (Char *)TVarbuffer_GetBuffer (pvbuf) + TVarbuffer_GetUsage (pvbuf) ;
	strtocstr (pDest, szBuf, nLength) ;
	return	True ;
}

Boolean
lispEntity_printInteger (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	long	lValue ;
	(void) lispEntity_GetIntegerValue (pLispMgr, pEntity, &lValue) ;
	fprintf (stderr, "%ld", lValue) ;
	return	True ;
}

Boolean
lispEntity_printFloat (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	float	fValue ;
	(void) lispEntity_GetFloatValue (pLispMgr, pEntity, &fValue) ;
	fprintf (stderr, "%f", fValue) ;
	return	True ;
}

Boolean
lispEntity_printSymbol (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	const Char*	pName ;
	int			nName ;
	int			nLength ;
	char		achBuf [TEMPBUFSIZE] ;

	(void) lispEntity_GetSymbolName (pLispMgr, pEntity, &pName, &nName) ;
	nLength	= NELEMENTS (achBuf) - 1 ;
	if (nName < nLength)
		nLength	= nName ;
	
	cstrtostr (achBuf, pName, nLength) ;
	achBuf [nLength]	= '\0' ;
	fprintf (stderr, "%s", achBuf) ;
	return	True ;
}

Boolean
lispEntity_printString (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	const Char*	pString ;
	int			nLength ;
	char		achBuf [TEMPBUFSIZE] ;
	
	(void) lispEntity_GetStringValue (pLispMgr, pEntity, &pString, &nLength) ;
	if (nLength <= 0) {
		fprintf (stderr, "\"\"") ;
		return	True ;
	}
	if (nLength  > (sizeof (achBuf) - 1))
		nLength	= sizeof (achBuf) - 1 ;
#if 1
	{
		KANJISTATEMACHINE	ksm ;
		int					n ;
		InitializeKanjiFiniteStateMachine (&ksm, KCODING_SYSTEM_ISO2022JP2) ;
		fprintf (stderr, "\"") ;
		while (nLength > 0) {
			n	= RtransferKanjiFiniteStateMachine (&ksm, *pString ++, achBuf) ;
			achBuf [n]	= '\0' ;
			fprintf (stderr, "%s", achBuf) ;
			nLength	-- ;
		}
		n	= RtransferKanjiFiniteStateMachine (&ksm, '\"', achBuf) ;
		achBuf [n]	= '\0' ;
		fprintf (stderr, "%s", achBuf) ;
		fflush (stderr) ;
	}
#else
	cstrtostr (achBuf, pString, nLength) ;
	achBuf [nLength]	= '\0' ;
	fprintf (stderr, "\"%s\"", achBuf) ;
#endif
	return	True ;
}

Boolean
lispEntity_printConscell (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	TLispEntity*	pEntCar ;
	TLispEntity*	pEntCdr ;
	
	lispEntity_GetCar (pLispMgr, pEntity, &pEntCar) ;
	lispEntity_GetCdr (pLispMgr, pEntity, &pEntCdr) ;

	if (TSUCCEEDED (lispEntity_Consp (pLispMgr, pEntCar))) {
		fprintf (stderr, "(") ;
		lispEntity_print (pLispMgr, pEntCar) ;
		fprintf (stderr, ")") ;
	} else {
		lispEntity_print (pLispMgr, pEntCar) ;
	}
	if (TFAILED (lispEntity_Listp (pLispMgr, pEntCdr))) {
		fprintf (stderr, " . ") ;
		lispEntity_print (pLispMgr, pEntCdr) ;
	} else {
		if (TFAILED (lispEntity_Nullp (pLispMgr, pEntCdr))) {
			fprintf (stderr, " ") ;
			lispEntity_print (pLispMgr, pEntCdr) ;
		}
	}
	return	True ;
}

Boolean
lispEntity_printVector (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	TLispEntity**	ppElement ;
	int				nElement ;

	lispEntity_GetVectorValue (pLispMgr, pEntity, &ppElement, &nElement) ;
	fprintf (stderr, "[") ;
	if (nElement > 0) {
		do {
			lispEntity_Print (pLispMgr, *ppElement ++) ;
			if (nElement > 1)
				fprintf (stderr, " ") ;
			nElement	-- ;
		} while (nElement > 0) ;
	}
	fprintf (stderr, "]") ;
	return	True ;
}

Boolean
lispEntity_printMarker (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	TLispEntity*	pEntBuffer ;
	int				nPos ;
	lispMarker_GetBufferPosition (pLispMgr, pEntity, &pEntBuffer, &nPos) ;
	if (pEntBuffer != NULL) {
		fprintf (stderr, "#<marker at %d in %lx>",
				 nPos,
				 (unsigned long)pEntBuffer) ;
	} else {
		fprintf (stderr, "#<marker in no buffer>") ;
	}
	return	True ;
}

Boolean
lispEntity_printBuffer (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	fprintf (stderr, "#<buffer %lx>", (unsigned long)pEntity) ;
	return	True ;
}

Boolean
lispEntity_printWindow (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	fprintf (stderr, "#<window %lx>", (unsigned long)pEntity) ;
	return	True ;
}

Boolean
lispEntity_printFrame (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	fprintf (stderr, "#<frame %lx>", (unsigned long)pEntity) ;
	return	True ;
}

Boolean
lispEntity_printSubr (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	const Char*	pName ;
	int			nName ;
	char		achBuf [TEMPBUFSIZE] ;

	fprintf (stderr, "#<subr ") ;
	lispSubr_GetName (pLispMgr, pEntity, &pName) ;
	nName	= Cstrlen (pName) ;
	if (nName >= NELEMENTS (achBuf))
		nName	= NELEMENTS (achBuf) - 1 ;
	cstrtostr (achBuf, pName, nName) ;
	achBuf [nName]	= '\0' ;
	fprintf (stderr, "%s>", achBuf) ;
	return	True ;
}

Boolean
lispEntity_printIMClient (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	fprintf (stderr, "#<im-client %lx>", (unsigned long)pEntity) ;
	return	True ;
}

Boolean
lispEntity_printMutex (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	const Char*		pString ;
	int				nLength ;
	char			achBuf [TEMPBUFSIZE] ;
	unsigned int	uLockCount ;
	const void*		pOwner ;
	
	fprintf (stderr, "#<mutex ") ;
	(void) lispEntity_GetMutexInfo (pLispMgr, pEntity, &pString, &nLength, &uLockCount, &pOwner) ;
	if (nLength <= 0) {
		fprintf (stderr, "\"\", ") ;
	} else {
		cstrtostr (achBuf, pString, nLength) ;
		achBuf [nLength]	= '\0' ;
		fprintf (stderr, "\"%s\", ", achBuf) ;
	}
	fprintf (stderr, "lock(%u), owner(0x%lx)>", uLockCount, (unsigned long)pOwner) ;
	return	True ;
}

Boolean
lispEntity_printXEvent (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	return	True ;
}

Boolean
lispEntity_printEmpty (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	fprintf (stderr, "#<empty>") ;
	return	True ;
}

Boolean
lispEntity_printVoid (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	fprintf (stderr, "#<void>") ;
	return	True ;
}

Boolean
lispEntity_formatString (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register int			nCount,
	register TVarbuffer*	pvbuf)
{
	TVarbuffer				vbuf ;
	register Boolean			fRetval	= False ;
	register const Char*	pString ;
	register int			nUsage ;

	if (TFAILED (TVarbuffer_Initialize (&vbuf, sizeof (Char))))
		return	False ;
	if (TFAILED (lispEntity_PrincStr (pLispMgr, pEntity, &vbuf))) 
		goto	error ;

	nUsage	= TVarbuffer_GetUsage (&vbuf) - 2 ;
	if (0 < nCount && nUsage < nCount) {
		register int		nSpace	= nUsage - nCount ;
		static const Char	cc		= ' ' ;
		while (nSpace > 0) {
			if (TFAILED (TVarbuffer_Add (pvbuf, &cc, 1)))
				goto	error ;
			nSpace	-- ;
		}
	}
	pString	= (const Char *)TVarbuffer_GetBuffer (&vbuf) + 1 ;
	fRetval	= TVarbuffer_Add (pvbuf, pString, nUsage) ;
  error:
	TVarbuffer_Uninitialize (&vbuf) ;
	return	fRetval ;


}

Boolean
lispEntity_formatChar (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TVarbuffer*	pvbuf)
{
	Char	cc ;

	if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntity, &cc)))
		return	False ;
	return	TVarbuffer_Add (pvbuf, &cc, 1) ;
}

Boolean
lispEntity_formatNumber (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register const Char*	pFormat,
	register int			nFormat,
	register const Boolean	fFloat,
	register TVarbuffer*	pvbuf)
{
	char				achFormat [64] ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (pvbuf    != NULL) ;

	if (nFormat	>= sizeof (achFormat))
		return	False ;
	cstrtostr (achFormat, pFormat, nFormat) ;
	achFormat [nFormat]	= '\0' ;
	return	lispEntity_formatNumberCommon (pLispMgr, pEntity, achFormat, fFloat, pvbuf) ;
}

Boolean
lispEntity_formatNumberA (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register const char*	pFormat,
	register int			nFormat,
	register const Boolean	fFloat,
	register TVarbuffer*	pvbuf)
{
	char				achFormat [64] ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (pvbuf    != NULL) ;

	if (nFormat	>= sizeof (achFormat))
		return	False ;
	memcpy (achFormat, pFormat, nFormat) ;
	achFormat [nFormat]	= '\0' ;
	return	lispEntity_formatNumberCommon (pLispMgr, pEntity, achFormat, fFloat, pvbuf) ;
}

Boolean
lispEntity_formatNumberCommon (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register const char*	pFormat,
	register const Boolean	fFloat,
	register TVarbuffer*	pvbuf)
{
	char				achBuffer [64] ;
	Char				aChBuffer [64] ;
	register char*		ptr ;
	register Char*		pPtr ;
	TLispNumber			num ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (pFormat  != NULL) ;
	assert (pvbuf    != NULL) ;

	if (TFAILED (lispEntity_GetNumberValue (pLispMgr, pEntity, &num)))
		return	False ;

	if (fFloat) {
		register double	dValue ;
		if (TSUCCEEDED (num.m_fFloatp)) {
			dValue	= num.m_Value.m_fFloat ;
		} else {
			dValue	= (float)num.m_Value.m_lLong ;
		}
		snprintf (achBuffer, NELEMENTS (achBuffer) - 1, pFormat, dValue) ;
	} else {
		register long	lValue ;
		if (TSUCCEEDED (num.m_fFloatp)) {
			lValue	= (long)num.m_Value.m_fFloat ;
		} else {
			lValue	= num.m_Value.m_lLong ;
		}
		snprintf (achBuffer, NELEMENTS (achBuffer) - 1, pFormat, lValue) ;
	}
	achBuffer [NELEMENTS (achBuffer) - 1]	= '\0' ;
	ptr		= achBuffer ;
	pPtr	= aChBuffer ;
	while (*ptr != '\0') 
		*pPtr ++	= Char_MakeAscii (*ptr ++) ;
	return	TVarbuffer_Add (pvbuf, aChBuffer, pPtr - aChBuffer) ;
}

Boolean
lispEntity_Copy (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntSrc,
	register TLispEntity**	ppEntDest)
{
	int		nType ;

	lispEntity_GetType (pLispMgr, pEntSrc, &nType) ;
	if (nType == LISPENTITY_CONSCELL) {
		return	lispEntity_copyConscell (pLispMgr, pEntSrc, ppEntDest) ;
	} else if (nType == LISPENTITY_VECTOR) {
		return	lispEntity_copyVector (pLispMgr, pEntSrc, ppEntDest) ;
	} else {
		*ppEntDest	= pEntSrc ;
		return	True ;
	}
}

Boolean
lispEntity_copyConscell (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntSrc,
	register TLispEntity**	ppEntDest)
{
	TLispEntity*	pEntSrcCar ;
	TLispEntity*	pEntSrcCdr ;
	TLispEntity*	pEntDestCar ;
	TLispEntity*	pEntDestCdr ;

	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntSrc, &pEntSrcCar)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pEntSrc, &pEntSrcCdr)))
		return	False ;
	if (TFAILED (lispEntity_Copy (pLispMgr, pEntSrcCar, &pEntDestCar)))
		return	False ;
	lispEntity_AddRef (pLispMgr, pEntDestCar) ;
	if (TFAILED (lispEntity_Copy (pLispMgr, pEntSrcCdr, &pEntDestCdr))) {
		lispEntity_Release (pLispMgr, pEntDestCar) ;
		return	False; 
	}
	lispEntity_AddRef (pLispMgr, pEntDestCdr) ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pEntDestCar, pEntDestCdr, ppEntDest))) {
		lispEntity_Release (pLispMgr, pEntDestCar) ;
		lispEntity_Release (pLispMgr, pEntDestCdr) ;
		return	False ;
	}
	lispEntity_Release (pLispMgr, pEntDestCar) ;
	lispEntity_Release (pLispMgr, pEntDestCdr) ;
	return	True ;
}

Boolean
lispEntity_copyVector (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntSrc,
	register TLispEntity**	ppEntDest)
{
	TVarbuffer				vbufEntDest ;
	TLispEntity**			ppEntSrcElm ;
	TLispEntity*			pEntDestElm ;
	int						nEntSrcElm ;
	register TLispEntity**	ppEntDests ;
	register int			nEntDests ;
	register int			i ;
	register Boolean		fRetval	= False ;

	if (TFAILED (TVarbuffer_Initialize (&vbufEntDest, sizeof (TLispEntity*))))
		return	False ;

	lispEntity_GetVectorValue (pLispMgr, pEntSrc, &ppEntSrcElm, &nEntSrcElm) ;
	for (i = 0 ; i < nEntSrcElm ; i ++) {
		if (TFAILED (lispEntity_Copy (pLispMgr, *ppEntSrcElm ++, &pEntDestElm))) 
			break ;
		if (TFAILED (TVarbuffer_Add (&vbufEntDest, &pEntDestElm, 1)))
			return	False ;
		lispEntity_AddRef (pLispMgr, pEntDestElm) ;
	}
	
	ppEntDests	= TVarbuffer_GetBuffer (&vbufEntDest) ;
	nEntDests	= TVarbuffer_GetUsage  (&vbufEntDest) ;
	if (i == nEntSrcElm) 
		fRetval	= lispMgr_CreateVector (pLispMgr, ppEntDests, nEntDests, ppEntDest) ;
	for (i = 0 ; i < nEntDests ; i ++) {
		lispEntity_Release (pLispMgr, *ppEntDests) ;
		ppEntDests	++ ;
	}
	TVarbuffer_Uninitialize (&vbufEntDest) ;
	return	fRetval ;
}


