/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#if !defined (lentity_inl_h)
#define	lentity_inl_h
#if defined (DEBUG)
#include <stdio.h>
#endif
#include "cstring.h"

struct tagLMCMDINFO ;

#define	lispEntity_GetSymbolPtr(ptr)	((TLispSymbol *)((TLispEntity *)(ptr) + 1))
#define	lispEntity_GetConsPtr(ptr)		((TLispConscell *)((TLispEntity *)(ptr) + 1))
#define	lispEntity_GetVectorPtr(ptr)	((TLispVector *)((TLispEntity *)(ptr) + 1))
#define	lispEntity_GetLongPtr(ptr)		((long *)((TLispEntity *)(ptr) + 1))
#define	lispEntity_GetFloatPtr(ptr)		((float *)((TLispEntity *)(ptr) + 1))
#define	lispEntity_GetStringPtr(ptr)	((TLispString *)((TLispEntity *)(ptr) + 1))
#define	lispEntity_GetSubrPtr(ptr)		((struct tagLMCMDINFO const **)((TLispEntity *)(ptr) + 1))
#define	lispEntity_GetBufferPtr(ptr)	((TLispBuffer *)((TLispEntity *)(ptr) + 1))
#define	lispEntity_GetWindowPtr(ptr)	((TLispWindow *)((TLispEntity *)(ptr) + 1))
#define	lispEntity_GetFramePtr(ptr)		((TLispFrame *)((TLispEntity *)(ptr) + 1))
#define	lispEntity_GetMutexPtr(ptr)		((TLispMutex *)((TLispEntity *)(ptr) + 1))
#define	lispEntity_GetXEventPtr(ptr)	((TLispXEventInt *)((TLispEntity *)(ptr) + 1))

extern inline	Boolean
lispEntity_GetType (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register int*			piType)
{
	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (piType   != NULL) ;

	*piType	= pEntity->m_iType ; 
	return	True ;
}

extern inline	Boolean
lispEntity_AddRef (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (0 <= pEntity->m_iType && pEntity->m_iType < MAX_LISPENTITY_TYPE) ;
	assert (pEntity->m_lReferCount >= 0) ;

	pEntity->m_lReferCount	++ ;
	assert (pEntity->m_lReferCount >= 0) ;
	return	True ;
}

extern inline	Boolean
lispEntity_Release (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (0 <= pEntity->m_iType && pEntity->m_iType < MAX_LISPENTITY_TYPE) ;
	assert (pEntity->m_lReferCount > 0) ;

	pEntity->m_lReferCount	-- ;
	return	True ;
}

extern inline	TLispEntity*
lispMgr_GetReservedEntity (
	register TLispManager*			pLispMgr,
	register int					nIndex)
{
	assert (pLispMgr != NULL) ;
	assert (0 <= nIndex && nIndex < LISPMGR_SIZE_RESERVED) ;
	assert (pLispMgr->m_apEntReserved [nIndex] != NULL) ;
	return	pLispMgr->m_apEntReserved [nIndex] ;
}

extern inline	Boolean
lispMgr_CreateNil (
	register TLispManager*			pLispMgr,
	register TLispEntity** const	ppEntReturn)
{
	assert (pLispMgr != NULL) ;
	assert (ppEntReturn != NULL) ;
	*ppEntReturn	= pLispMgr->m_apEntReserved [LISPMGR_INDEX_NIL] ;
	return	True ;
}

extern inline	Boolean
lispMgr_CreateT (
	register TLispManager*			pLispMgr,
	register TLispEntity** const	ppEntReturn)
{
	assert (pLispMgr != NULL) ;
	assert (ppEntReturn != NULL) ;
	*ppEntReturn	= pLispMgr->m_apEntReserved [LISPMGR_INDEX_T] ;
	return	True ;
}

extern inline	Boolean
lispMgr_CreateVoid (
	register TLispManager*			pLispMgr,
	register TLispEntity** const	ppEntReturn)
{
	assert (pLispMgr != NULL) ;
	assert (pLispMgr->m_pEntVoid != NULL) ;
	assert (ppEntReturn != NULL) ;
	
	*ppEntReturn	= pLispMgr->m_pEntVoid ;
	return	True ;
}

extern	inline	Boolean
lispMgr_CreateEmpty (
	register TLispManager*			pLispMgr,
	register TLispEntity** const	ppEntReturn)
{
	assert (pLispMgr != NULL) ;
	assert (pLispMgr->m_pEntEmpty != NULL) ;
	assert (ppEntReturn != NULL) ;
	
	*ppEntReturn	= pLispMgr->m_pEntEmpty ;
	return	True ;
}

extern inline	Boolean
lispEntity_Nullp (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	TLispEntity*	pNil ;

	(void) lispMgr_CreateNil (pLispMgr, &pNil) ;
	return	(pNil == pEntity)? True : False ;
}

extern inline	Boolean
lispEntity_Symbolp (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	int		iType ;
#if defined (DEBUG)
	Boolean	fResult ;
#endif
#if defined (DEBUG)
	fResult	= lispEntity_GetType (pLispMgr, pEntity, &iType) ;
	assert (fResult == True) ;
#else
	(void) lispEntity_GetType (pLispMgr, pEntity, &iType) ;
#endif
	return	(iType == LISPENTITY_SYMBOL)? True : False ;
}

extern inline	Boolean
lispEntity_XEventIntegerp (
	register TLispManager*		pLispMgr,
	register TLispEntity*		pEntity)
{
	register TLispXEventInt*	pEvent	= lispEntity_GetXEventPtr (pEntity) ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;

	return	(pEntity->m_iType == LISPENTITY_XEVENT && 
			 pEvent->m_pEntValue->m_iType == LISPENTITY_INTEGER) ;
}

extern inline	Boolean
lispEntity_Integerp (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	return	(pEntity->m_iType == LISPENTITY_INTEGER ||
			 lispEntity_XEventIntegerp (pLispMgr, pEntity)) ;
}

extern inline	Boolean
lispEntity_Floatp (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	int		iType ;
#if defined (DEBUG)
	Boolean	fResult ;
#endif
#if defined (DEBUG)
	fResult	= lispEntity_GetType (pLispMgr, pEntity, &iType) ;
	assert (fResult == True) ;
#else
	(void) lispEntity_GetType (pLispMgr, pEntity, &iType) ;
#endif
	return	(iType == LISPENTITY_FLOAT)? True : False ;
}

extern inline	Boolean
lispEntity_Markerp (
	register TLispManager*		pLispMgr,
	register TLispEntity*		pEntity)
{
	int		iType ;
#if defined (DEBUG)
	Boolean	fResult ;
#endif
#if defined (DEBUG)
	fResult	= lispEntity_GetType (pLispMgr, pEntity, &iType) ;
	assert (fResult == True) ;
#else
	(void) lispEntity_GetType (pLispMgr, pEntity, &iType) ;
#endif
	return	(iType == LISPENTITY_MARKER)? True : False ;
}

extern inline	Boolean
lispEntity_Numberp (
	register TLispManager*		pLispMgr,
	register TLispEntity*		pEntity)
{
	return	(lispEntity_Floatp   (pLispMgr, pEntity) ||
			 lispEntity_Integerp (pLispMgr, pEntity)) ;
}

extern inline	Boolean
lispEntity_IntegerOrMarkerp (
	register TLispManager*		pLispMgr,
	register TLispEntity*		pEntity)
{
	return	(lispEntity_Integerp (pLispMgr, pEntity) ||
			 lispEntity_Markerp  (pLispMgr, pEntity)) ;
}

extern inline	Boolean
lispEntity_Consp (
	register TLispManager*		pLispMgr,
	register TLispEntity*		pEntity)
{
	int		iType ;
#if defined (DEBUG)
	Boolean	fResult ;
#endif
#if defined (DEBUG)
	fResult	= lispEntity_GetType (pLispMgr, pEntity, &iType) ;
	assert (fResult == True) ;
#else
	(void) lispEntity_GetType (pLispMgr, pEntity, &iType) ;
#endif
	return	(iType == LISPENTITY_CONSCELL)? True : False ;
}

extern inline	Boolean
lispEntity_Listp (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	int	iType ;

	if (lispEntity_Nullp (pLispMgr, pEntity))
		return	True ;

	(void) lispEntity_GetType (pLispMgr, pEntity, &iType) ;
	return	(iType == LISPENTITY_CONSCELL)? True : False ;
}

extern inline	Boolean
lispEntity_Stringp (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	int		iType ;
#if defined (DEBUG)
	Boolean	fResult ;
#endif
#if defined (DEBUG)
	fResult	= lispEntity_GetType (pLispMgr, pEntity, &iType) ;
	assert (fResult == True) ;
#else
	(void) lispEntity_GetType (pLispMgr, pEntity, &iType) ;
#endif
	return	(iType == LISPENTITY_STRING)? True : False ;
}

extern inline	Boolean
lispEntity_Vectorp (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	int		iType ;
#if defined (DEBUG)
	Boolean	fResult ;
#endif
#if defined (DEBUG)
	fResult	= lispEntity_GetType (pLispMgr, pEntity, &iType) ;
	assert (fResult == True) ;
#else
	(void) lispEntity_GetType (pLispMgr, pEntity, &iType) ;
#endif
	return	(iType == LISPENTITY_VECTOR)? True : False ;
}

extern inline	Boolean
lispEntity_Arrayp (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	
	return	(lispEntity_Vectorp (pLispMgr, pEntity) ||
			 lispEntity_Stringp (pLispMgr, pEntity)) ;
}

/*
 *	Entity  buffer Ǥ뤫ɤȽꤹ롣
 */
extern inline	Boolean
lispEntity_Bufferp (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	int		iType ;
#if defined (DEBUG)
	Boolean	fResult ;
#endif
#if defined (DEBUG)
	fResult	= lispEntity_GetType (pLispMgr, pEntity, &iType) ;
	assert (fResult == True) ;
#else
	(void) lispEntity_GetType (pLispMgr, pEntity, &iType) ;
#endif
	return	(iType == LISPENTITY_BUFFER)? True : False ;
}

extern inline	Boolean
lispEntity_Sequencep (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	
	return	(lispEntity_Listp  (pLispMgr, pEntity) ||
			 lispEntity_Arrayp (pLispMgr, pEntity)) ;
}

extern inline	Boolean
lispEntity_Eq (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pLeftEntity,
	register TLispEntity*	pRightEntity)
{
	assert (pLispMgr     != NULL) ;
	assert (pLeftEntity  != NULL) ;
	assert (pRightEntity != NULL) ;
	return	(pLeftEntity == pRightEntity)? True : False ;

	UNREFERENCED_PARAMETER (pLispMgr) ;
}

extern inline	Boolean
lispEntity_GetCar (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pEntity, 
	register TLispEntity** const	ppRetvalue)
{
	TLispConscell*	pConscell ;

	assert (pLispMgr   != NULL) ;
	assert (pEntity    != NULL) ;
	assert (ppRetvalue != NULL) ;

	if (lispEntity_Nullp (pLispMgr, pEntity)) 
		return	lispMgr_CreateNil (pLispMgr, ppRetvalue) ;

	if (pEntity->m_iType != LISPENTITY_CONSCELL)
		return	False ;

	pConscell	= lispEntity_GetConsPtr (pEntity) ;
	*ppRetvalue	= pConscell->m_pCar ;
	return	True ;
}

extern inline	Boolean
lispEntity_GetCdr (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pEntity,
	register TLispEntity** const	ppRetvalue)
{
	register TLispConscell*	pConscell ;

	assert (pLispMgr   != NULL) ;
	assert (pEntity    != NULL) ;
	assert (ppRetvalue != NULL) ;

	if (lispEntity_Nullp (pLispMgr, pEntity)) 
		return	lispMgr_CreateNil (pLispMgr, ppRetvalue) ;

	if (pEntity->m_iType != LISPENTITY_CONSCELL)
		return	False ;

	pConscell	= lispEntity_GetConsPtr (pEntity) ;
	*ppRetvalue	= pConscell->m_pCdr ;
	return	True ;
}

extern inline	Boolean
lispEntity_GetCadr (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pEntity,
	register TLispEntity** const	ppRetvalue)
{
	TLispEntity*	pCDR ;
	TLispEntity*	pCADR ;
	
	if (TFAILED (lispEntity_GetCdr (pLispMgr, pEntity, &pCDR)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pCDR,    &pCADR)))
		return	False ;
	*ppRetvalue	= pCADR ;
	return	True ;
}

extern inline	Boolean
lispEntity_GetCaar (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pEntity,
	register TLispEntity** const	ppRetvalue)
{
	TLispEntity*	pCAR ;
	TLispEntity*	pCAAR ;
	
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntity, &pCAR)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pCAR,    &pCAAR)))
		return	False ;
	*ppRetvalue	= pCAAR ;
	return	True ;
}

extern inline	Boolean
lispEntity_GetCaddr (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pEntity,
	register TLispEntity** const	ppRetvalue)
{
	TLispEntity*	pCDR ;
	TLispEntity*	pCDDR ;
	TLispEntity*	pCADDR ;
	
	if (TFAILED (lispEntity_GetCdr (pLispMgr, pEntity, &pCDR))  ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pCDR,    &pCDDR)) ||
		TFAILED (lispEntity_GetCar (pLispMgr, pCDDR,   &pCADDR)))
		return	False ;
	*ppRetvalue	= pCADDR ;
	return	True ;
}

extern inline	Boolean
lispEntity_GetCddr (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TLispEntity** const ppRetvalue)
{
	TLispEntity*	pCDR ;
	TLispEntity*	pCDDR ;
	
	if (TFAILED (lispEntity_GetCdr (pLispMgr, pEntity, &pCDR)) ||
		TFAILED (lispEntity_GetCdr (pLispMgr, pCDR,    &pCDDR)))
		return	False ;
	*ppRetvalue	= pCDDR ;
	return	True ;
}

extern inline	Boolean
lispEntity_SetCar (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TLispEntity*	pCar)
{
	register TLispConscell*	pConscell ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (pCar     != NULL) ;

	if (pEntity->m_iType != LISPENTITY_CONSCELL)
		return	False ;

	pConscell			= lispEntity_GetConsPtr (pEntity) ;
	pConscell->m_pCar	= pCar ;
	return	True ;
}

extern inline	Boolean
lispEntity_SetCdr (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TLispEntity*	pCdr)
{
	register TLispConscell*	pConscell ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (pCdr     != NULL) ;

	if (pEntity->m_iType != LISPENTITY_CONSCELL)
		return	False ;

	pConscell			= lispEntity_GetConsPtr (pEntity) ;
	pConscell->m_pCdr	= pCdr ;
	return	True ;
}

extern inline	Boolean
lispEntity_CountArgument (
	register TLispManager*	pLispMgr,
	TLispEntity*			pEntity,
	int*					pnCount)
{
	register int		nArgument ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ; 
	assert (pnCount  != NULL) ;

	nArgument	= 0 ;
	while (!lispEntity_Nullp (pLispMgr, pEntity)) {
		if (TFAILED (lispEntity_GetCdr (pLispMgr, pEntity, &pEntity)) ||
			pEntity == NULL)
			return	False ;
		nArgument	++ ;
	}
	*pnCount	= nArgument ;
	return	True ;
}

extern inline	Boolean
lispEntity_GetSymbolName (
	register TLispManager*		pLispMgr,
	register TLispEntity*		pEntity,
	register const Char** const	ppSymName,
	register int*				piSymName)
{
	register TLispSymbol*	pSymbol ;
	
	assert (pLispMgr  != NULL) ;
	assert (pEntity   != NULL) ;
	assert (ppSymName != NULL) ;
	assert (piSymName != NULL) ;

	if (pEntity->m_iType != LISPENTITY_SYMBOL)
		return	False ;

	pSymbol		= lispEntity_GetSymbolPtr (pEntity) ;
	*ppSymName	= pSymbol->m_achName ;
	*piSymName	= pSymbol->m_nLength ;
	return	True ;
}

extern inline	Boolean
lispEntity_Tp (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	TLispEntity*	pT ;

	lispMgr_CreateT (pLispMgr, &pT) ;
	return	(pT == pEntity)? True : False ;
}

extern inline	Boolean
lispEntity_Voidp (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	assert (pLispMgr != NULL) ;
	assert (pLispMgr->m_pEntVoid != NULL) ;
	return	(pEntity == pLispMgr->m_pEntVoid) ;
}

extern	inline	Boolean
lispEntity_Emptyp (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	assert (pLispMgr != NULL) ;
	assert (pLispMgr->m_pEntEmpty != NULL) ;
	return	(pEntity == pLispMgr->m_pEntEmpty) ;
}

extern inline	Boolean
lispEntity_GetLastElement (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntList,
	register TLispEntity**	ppEntRetval)
{
	TLispEntity*	pEntNextList ;

	assert (pLispMgr != NULL) ;
	assert (pEntList != NULL) ;
	assert (ppEntRetval != NULL) ;

	while (TSUCCEEDED (lispEntity_GetCdr (pLispMgr, pEntList, &pEntNextList)) &&
		   TFAILED    (lispEntity_Nullp  (pLispMgr, pEntNextList)))
		pEntList	= pEntNextList ;
	*ppEntRetval	= pEntList ;
	return	True ;
}

/*	lnumber.c	*/
extern inline	Boolean
lispEntity_GetIntegerValue (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register long*			plValue)
{
	register long*				plInteger ;
	register TLispXEventInt*	pEvent ;

	assert (pLispMgr  != NULL) ;
	assert (pEntity   != NULL) ;
	assert (plValue   != NULL) ;
	
	switch (pEntity->m_iType) {
	case	LISPENTITY_XEVENT:
		pEvent		= lispEntity_GetXEventPtr (pEntity) ;
		assert (pEvent->m_pEntValue != NULL) ;
		pEntity		= pEvent->m_pEntValue ;
		assert (pEntity != NULL) ;
		if (pEntity->m_iType != LISPENTITY_INTEGER)
			return	False ;
		/*	fall-down */
	case	LISPENTITY_INTEGER:
		plInteger	= lispEntity_GetLongPtr (pEntity) ;
		*plValue	= *plInteger ;
		break ;
	default:
		return	False ;
	}
	return	True ;
}

extern inline	Boolean
lispEntity_GetFloatValue (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register float*			pfValue)
{
	float*	plFloat ;
	
	assert (pLispMgr  != NULL) ;
	assert (pEntity   != NULL) ;
	assert (pfValue   != NULL) ;

	if (pEntity->m_iType != LISPENTITY_FLOAT)
		return	False ;
	plFloat		= lispEntity_GetFloatPtr (pEntity) ;
	*pfValue	= *plFloat ;
	return	True ;
}

extern inline	Boolean
lispEntity_GetNumberValue (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TLispNumber*	pReturn)
{
	long	lValue ;
	float	fValue ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (pReturn  != NULL) ;

	switch (pEntity->m_iType) {
	case	LISPENTITY_INTEGER:
		(void) lispEntity_GetIntegerValue (pLispMgr, pEntity, &lValue) ;
		pReturn->m_fFloatp			= False ;
		pReturn->m_Value.m_lLong	= lValue ;
		break ;
	case	LISPENTITY_FLOAT:
		(void) lispEntity_GetFloatValue (pLispMgr, pEntity, &fValue) ;
		pReturn->m_fFloatp			= True ;
		pReturn->m_Value.m_fFloat	= fValue ;
		break ;
	default:
		return	False ;
	}
	return	True ;
}

extern inline	Boolean
lispEntity_GetNumberOrMarkerValue (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TLispNumber*	pReturn)
{
	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (pReturn  != NULL) ;

	switch (pEntity->m_iType) {
	case	LISPENTITY_INTEGER:
	{
		long	lValue ;

		(void) lispEntity_GetIntegerValue (pLispMgr, pEntity, &lValue) ;
		pReturn->m_fFloatp			= False ;
		pReturn->m_Value.m_lLong	= lValue ;
		break ;
	}
	case	LISPENTITY_FLOAT:
	{
		float	fValue ;

		(void) lispEntity_GetFloatValue (pLispMgr, pEntity, &fValue) ;
		pReturn->m_fFloatp			= True ;
		pReturn->m_Value.m_fFloat	= fValue ;
		break ;
	}
	case	LISPENTITY_MARKER:
	{
		TLispEntity*	pEntBuffer ;
		int				nPosition ;
		(void) lispMarker_GetBufferPosition (pLispMgr, pEntity, &pEntBuffer, &nPosition) ;
		if (pEntBuffer == NULL)
			return	False ;
		pReturn->m_fFloatp			= False ;
		pReturn->m_Value.m_lLong	= nPosition ;
		break ;
	}
	default:
		return	False ;
	}
	return	True ;
}

/*	lsymbol.kc */
extern inline	Boolean
lispEntity_MatchSymbolNamep (
	register TLispManager*		pLispMgr,
	register TLispEntity*		pEntity,
	register const Char*		pName,
	register int				nName)
{
	register TLispSymbol*	pSymbol ;

	if (pEntity->m_iType != LISPENTITY_SYMBOL)
		return	False ;

	pSymbol		= lispEntity_GetSymbolPtr (pEntity) ;
	if (nName != pSymbol->m_nLength ||
		Cstrncmp (pName, pSymbol->m_achName, nName))
		return	False ;
	return	True ;
}

extern inline	Boolean
lispEntity_Lambdap (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	return	lispEntity_Eq (pLispMgr, pEntity, lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_LAMBDA)) ;
}

extern inline	Boolean
lispEntity_Macrop (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	return	lispEntity_Eq (pLispMgr, pEntity, lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_MACRO)) ;
}

extern inline	Boolean
lispEntity_Optionalp (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	return	lispEntity_Eq (pLispMgr, pEntity, lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_OPTIONAL)) ;
}

extern inline	Boolean
lispEntity_Restp (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	return	lispEntity_Eq (pLispMgr, pEntity, lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_REST)) ;
}

extern inline	Boolean
lispEntity_GetVectorValue (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pEntity,
	register TLispEntity*** const	pppElement,
	register int*					pnElement)
{
	register TLispVector*	pVector ;

	assert (pLispMgr   != NULL) ;
	assert (pEntity    != NULL) ;
	assert (pppElement != NULL) ;
	assert (pnElement  != NULL) ;

	if (pEntity->m_iType != LISPENTITY_VECTOR)
		return	False ;

	pVector		= lispEntity_GetVectorPtr (pEntity) ;
	*pppElement	= pVector->m_apElement ;
	*pnElement	= pVector->m_nElement ;
	return	True ;
}

extern inline	Boolean
lispEntity_GetVectorElement (
	register TLispManager*			pLispMgr,
	register TLispEntity*			pEntity,
	register int					nIndex,
	register TLispEntity** const	ppReturn)
{
	register TLispVector*	pVector ;
	
	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (ppReturn != NULL) ;

	if (pEntity->m_iType != LISPENTITY_VECTOR)
		return	False ;

	pVector		= lispEntity_GetVectorPtr (pEntity) ;
#if defined (DEBUG)
	fprintf (stderr, "nIndex(%d), nElement(%d)\n",
			 nIndex, pVector->m_nElement) ;
#endif
	if (nIndex < 0 || pVector->m_nElement <= nIndex) 
		return	False ;

	*ppReturn	= pVector->m_apElement [nIndex] ;
	return	True ;
}

extern inline	Boolean
lispEntity_SetVectorElement (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntVector,
	register int			nIndex,
	register TLispEntity*	pEntValue)
{
	register TLispVector*	pVector ;
	
	assert (pLispMgr   != NULL) ;
	assert (pEntVector != NULL) ;
	assert (pEntValue  != NULL) ;

	if (pEntVector->m_iType != LISPENTITY_VECTOR)
		return	False ;

	pVector		= lispEntity_GetVectorPtr (pEntVector) ;
	if (nIndex < 0 || pVector->m_nElement <= nIndex) 
		return	False ;

	pVector->m_apElement [nIndex]	= pEntValue ;
	return	True ;
}

extern inline	Boolean
lispEntity_ShrinkVector (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntVector,
	register int			nShrink)
{
	register TLispVector*	pVector ;

	assert (pLispMgr   != NULL) ;
	assert (pEntVector != NULL) ;
	assert (nShrink >= 0) ;

	if (pEntVector->m_iType != LISPENTITY_VECTOR)
		return	False ;

	pVector				= lispEntity_GetVectorPtr (pEntVector) ;
	pVector->m_nElement	-= nShrink ;
	if (pVector->m_nElement < 0)
		pVector->m_nElement	= 0 ;
	return	True ;
}

extern inline	Boolean
lispEntity_GetStringValue (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	const Char** const		ppString,
	register int*			pnLength)
{
	register TLispString*	pLsString ;
	
	assert (pLispMgr  != NULL) ;
	assert (pEntity   != NULL) ;
	assert (ppString  != NULL) ;
	assert (pnLength  != NULL) ;

	if (pEntity->m_iType != LISPENTITY_STRING)
		return	False ;

	pLsString	= lispEntity_GetStringPtr (pEntity) ;
	*ppString	= pLsString->m_achString ;
	*pnLength	= pLsString->m_nLength ;
	return	True ;
}

extern inline	Boolean
lispEntity_GetStringElement (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register int			nIndex,
	register Char* const	pCh)
{
	register TLispString*	pLsString ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (pCh      != NULL) ;

	if (pEntity->m_iType != LISPENTITY_STRING)
		return	False ;

	pLsString	= lispEntity_GetStringPtr (pEntity) ;
	if (nIndex < 0 || nIndex >= pLsString->m_nLength)
		return	False ;
			
	*pCh		= pLsString->m_achString [nIndex] ;
	return	True ;
}

extern inline	Boolean
lispEntity_SetStringElement (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register int			nIndex,
	register const Char		cc)
{
	register TLispString*	pLsString ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;

	if (pEntity->m_iType != LISPENTITY_STRING)
		return	False ;

	pLsString	= lispEntity_GetStringPtr (pEntity) ;
	if (nIndex < 0 || nIndex >= pLsString->m_nLength)
		return	False ;
			
	pLsString->m_achString [nIndex]	= cc ;
	return	True ;
}

/*	lsubr.c */
extern inline	Boolean
lispEntity_Subrp (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	return	(pEntity->m_iType == LISPENTITY_SUBR) ;
}

extern inline	Boolean
lispSubr_GetProc (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntSubr,
	register struct tagLMCMDINFO const**	ppProcReturn)
{
	if (pEntSubr->m_iType != LISPENTITY_SUBR)
		return	False ;
	*ppProcReturn	= *(lispEntity_GetSubrPtr (pEntSubr)) ;
	return	True ;
}

/*	lbuffer.c */
extern inline	int
lispBuffer_GetTick (
	register TLispEntity*	pEntBuffer)
{
	register TLispBuffer*	pBuffer ;
	assert (pEntBuffer != NULL) ;
	pBuffer		= lispEntity_GetBufferPtr (pEntBuffer) ;
	return	pBuffer->m_nTick ;
}

/*	lwindow.c */
extern inline	Boolean
lispWindow_NeedUpdatep (
	register TLispManager*		pLispMgr,
	register TLispEntity*		pEntWindow)
{
	register TLispWindow*	pWindow ;
	register TLispEntity*	pEntBuffer ;
	int						nPoint ;

	assert (pEntWindow  != NULL) ;
	assert (pEntWindow->m_iType == LISPENTITY_WINDOW) ;
	pWindow		= lispEntity_GetWindowPtr (pEntWindow) ;
	pEntBuffer	= pWindow->m_pBuffer ;
	assert (pEntBuffer != NULL) ;
	if (pWindow->m_nTick != lispBuffer_GetTick (pEntBuffer))
		return	True ;
	lispBuffer_Point (pLispMgr, pEntBuffer, &nPoint) ;
	return	(nPoint != pWindow->m_nPointBack) ;
}

extern inline	void
lispWindow_Sync (
	register TLispManager*		pLispMgr,
	register TLispEntity*		pEntWindow)
{
	register TLispWindow*	pWindow ;
	register TLispEntity*	pEntBuffer ;
	int						nPoint ;

	assert (pEntWindow  != NULL) ;
	assert (pEntWindow->m_iType == LISPENTITY_WINDOW) ;
	pWindow		= lispEntity_GetWindowPtr (pEntWindow) ;
	pEntBuffer	= pWindow->m_pBuffer ;
	assert (pEntBuffer != NULL) ;
	pWindow->m_nTick 		= lispBuffer_GetTick (pEntBuffer) ;
	lispBuffer_Point (pLispMgr, pEntBuffer, &nPoint) ;
	pWindow->m_nPointBack	= nPoint ;
	return ;
}

extern inline	void
lispWindow_Desync (
	register TLispManager*		pLispMgr,
	register TLispEntity*		pEntWindow)
{
	register TLispWindow*	pWindow ;
	register TLispEntity*	pEntBuffer ;

	assert (pEntWindow  != NULL) ;
	assert (pEntWindow->m_iType == LISPENTITY_WINDOW) ;
	pWindow		= lispEntity_GetWindowPtr (pEntWindow) ;
	pEntBuffer	= pWindow->m_pBuffer ;
	assert (pEntBuffer != NULL) ;
	pWindow->m_nTick 		= lispBuffer_GetTick (pEntBuffer) - 1 ;
	return ;
}

extern inline	Boolean
lispEntity_XEventp (
	register TLispManager*		pLispMgr,
	register TLispEntity*		pEntity)
{
	assert (pEntity != NULL) ;
	return	(pEntity->m_iType == LISPENTITY_XEVENT) ;
}

extern inline	Boolean
lispEntity_GetXEvent (
	register TLispManager*		pLispMgr,
	register TLispEntity*		pEntity,
	register XEvent**			ppEvent)
{
	register TLispXEventInt*	pEvent ;

	assert (pEntity != NULL) ;
	assert (ppEvent != NULL) ;

	if (pEntity->m_iType != LISPENTITY_XEVENT)
		return	False ;

	pEvent		= lispEntity_GetXEventPtr (pEntity) ;
	*ppEvent	= &pEvent->m_evX ;
	return	True ;
}

#endif
