/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <assert.h>
#include "lispmgrp.h"
#include "cstring.h"

static Boolean	lispBind_destroyTree (TLispManager*, TLispBind** const) ;

inline	int		lispBindTable_createHashkey (
	register TLispEntity*		pEntSymbol,
	register int				nTableSize)
{
	return	((unsigned)(pEntSymbol) * 13) % nTableSize ;
}

Boolean
lispBind_Create (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntSymbol,
	register TLispBind**	ppRetBind)
{
	register TLispBind*	pBind ;

	assert (pEntSymbol != NULL) ;
	assert (ppRetBind  != NULL);

	pBind		= MALLOC (sizeof (TLispBind)) ;
	if (pBind == NULL)
		return	False ;

	pBind->m_pEntValue	= NULL ;
	pBind->m_pEntSymbol	= pEntSymbol ;
	lispEntity_AddRef (pLispMgr, pEntSymbol) ;
	pBind->m_pLeft		= NULL ;
	pBind->m_pRight		= NULL ;
	*ppRetBind			= pBind ;
	return	True ;
}

Boolean
lispBind_Destroy (
	register TLispManager*	pLispMgr,
	register TLispBind*		pBind)
{
	/*	Unregister Ϥʤ*/
	if (pBind->m_pEntSymbol != NULL) {
		lispEntity_Release (pLispMgr, pBind->m_pEntSymbol) ;
		pBind->m_pEntSymbol	= NULL ;
	}
	if (pBind->m_pEntValue != NULL) {
		lispEntity_Release (pLispMgr, pBind->m_pEntValue) ;
		pBind->m_pEntValue	= NULL ;
	}
	pBind->m_pLeft		= NULL ;
	pBind->m_pRight		= NULL ;
	FREE (pBind) ;
	return	True ;
}

Boolean
lispBind_GetValue (
	register TLispBind*				pBind,
	register TLispEntity** const	ppReturn)
{
	assert (pBind    != NULL) ;
	assert (ppReturn != NULL) ;

	/*	Ǥ reference count 䤵ʤƤӽФ¦Ǥ
	 *	Ĥȡ*/
	*ppReturn	= pBind->m_pEntValue ;
	return	True ;
}

Boolean
lispBind_SetValue (
	register TLispManager*	pLispMgr,
	register TLispBind*		pBind,
	register TLispEntity*	pValue)
{
	assert (pLispMgr != NULL) ;
	assert (pBind    != NULL);
	assert (pValue   != NULL) ;

	if (pBind->m_pEntValue != NULL) 
		lispEntity_Release (pLispMgr, pBind->m_pEntValue) ;

	pBind->m_pEntValue	= pValue ;
	if (pValue != NULL)
		lispEntity_AddRef (pLispMgr, pValue) ;
	return	True ;
}

static	Boolean
lispBind_Search (
	register TLispBind*			pNode,
	register TLispEntity*		pSymbol,
	register TLispBind** const	ppReturn,
	register TLispBind** const	ppParent)
{
	register TLispBind*	pRoot ;
	register TLispBind*	pRetvalue ;
	register int		nCompare ;

	assert (pSymbol != NULL) ;
	assert (ppReturn != NULL) ;

	pRetvalue	= NULL ;
	pRoot		= NULL ;

	while (pNode != NULL) {
		nCompare	= pNode->m_pEntSymbol - pSymbol ;
		if (nCompare == 0) {
			pRetvalue	= pNode ;
			break ;
		} else if (nCompare < 0) {
			pRoot		= pNode ;
			pNode		= pNode->m_pRight ;
		} else {
			pRoot		= pNode ;
			pNode		= pNode->m_pLeft ;
		}
	}
	if (ppParent != NULL)
		*ppParent	= pRoot ;
	*ppReturn	= pRetvalue ;
	return	True ;
}

Boolean
lispBind_Register (
	register TLispBind** const	ppRootOfTree,
	register TLispBind*			pNewBind)
{
	register TLispBind*		pNode ;
	register TLispEntity*	pEntSymbol ;
	register int			nCompare ;

	assert (ppRootOfTree != NULL) ;
	assert (pNewBind     != NULL) ;

	pNode	= *ppRootOfTree ;
	if (pNode == NULL) {
		*ppRootOfTree	= pNewBind ;
		return	True ;
	} 
	pEntSymbol	= pNewBind->m_pEntSymbol ;
	while (pNode != NULL) {
		nCompare	= pNode->m_pEntSymbol - pEntSymbol ;
		if (nCompare == 0) {
			return	False ;
		} else if (nCompare < 0) {
			if (pNode->m_pRight == NULL) {
				pNode->m_pRight	= pNewBind ;
				return	True ;
			} else {
				pNode			= pNode->m_pRight ;
			}
		} else {
			if (pNode->m_pLeft  == NULL) {
				pNode->m_pLeft	= pNewBind ;
				return	True ;
			} else {
				pNode			= pNode->m_pLeft ;
			}
		}
	}
	return	False ;
}

Boolean
lispBind_Unregister (
	register TLispBind** const	ppRootOfTree,
	register TLispBind*			pBind)
{
	register TLispBind*		pChild ;
	register TLispEntity*	pSymbol ;
	TLispBind*	pReturn ;
	TLispBind*	pParent ;

	assert (ppRootOfTree != NULL) ;
	assert (pBind != NULL) ;

	pSymbol	= pBind->m_pEntSymbol ;
	if (TFAILED (lispBind_Search (*ppRootOfTree, pSymbol, &pReturn, &pParent)))
		return	False ;
	assert (pReturn == pBind) ;
	
	if (pBind->m_pLeft == NULL) {
		pChild	= pBind->m_pRight ;
	} else if (pBind->m_pRight == NULL) {
		pChild	= pBind->m_pLeft ;
	} else {
		register TLispBind*	pNode ;
		pChild	= pBind->m_pLeft ;
		assert (pChild != NULL) ;
		pNode	= pChild ;
		while (pNode->m_pRight != NULL) 
			pNode		= pNode->m_pRight ;
		pNode->m_pRight	= pBind->m_pRight ;
	}
	pBind->m_pLeft	= NULL ;
	pBind->m_pRight	= NULL ;
	
	if (pParent == NULL) {
		*ppRootOfTree	= pChild ;
	} else {
		if (pParent->m_pLeft == pBind) {
			pParent->m_pLeft	= pChild ;
		} else {
			assert (pParent->m_pRight == pBind) ;
			pParent->m_pRight	= pChild ;
		}
	}
	return	True ;
}

Boolean
lispBind_destroyTree (
	register TLispManager*		pLispMgr,
	register TLispBind** const	ppTable)
{
	register TLispBind*	pBind ;

	assert (pLispMgr != NULL) ;
	assert (ppTable  != NULL) ;

	pBind	= *ppTable ;
	if (pBind == NULL)
		return	True ;
	if (pBind->m_pLeft != NULL) 
		lispBind_destroyTree (pLispMgr, &pBind->m_pLeft) ;
	assert (pBind->m_pLeft == NULL) ;
	if (pBind->m_pRight != NULL) 
		lispBind_destroyTree (pLispMgr, &pBind->m_pRight) ;
	assert (pBind->m_pRight == NULL) ;

	lispBind_Destroy (pLispMgr, pBind) ;
	*ppTable	= NULL ;
	return	True ;
}

Boolean
lispBindTable_Destroy (
	register TLispManager*		pLispMgr,
	register TLispBind** const	ppTable,
	register int				nTableSize)
{
	register int	i ;
	for (i = 0 ; i < nTableSize ; i ++)
		lispBind_destroyTree (pLispMgr, &ppTable [i]) ;
	return	True ;
}

/*
 *	Symbol - Value  Bind Table ˿ Symbol (Entry) ɲä롣
 */
Boolean
lispBindTable_MakeEntry (
	register TLispManager*		pLispMgr,
	register TLispBind** const	ppTable,
	register int				nTableSize,
	register TLispEntity*		pSymbol,
	register TLispBind** const	ppBindReturn)
{
	TLispBind*		pBind ;
	TLispEntity*	pVoid ;
	register int	iHashkey ;

	assert (pLispMgr != NULL) ;
	assert (ppTable  != NULL && nTableSize > 0) ;
	assert (pSymbol  != NULL) ;
	
	if (TFAILED (lispBindTable_SearchEntry (pLispMgr, ppTable, nTableSize, pSymbol, &pBind)))
		return	False ;
	if (TFAILED (lispBind_Create (pLispMgr, pSymbol, &pBind)))
		return	False ;

	iHashkey	= lispBindTable_createHashkey (pSymbol, nTableSize) ;
	assert (0 <= iHashkey && iHashkey < nTableSize) ;
	if (TFAILED (lispBind_Register (&ppTable [iHashkey], pBind)))
		return	False ;

	/*	ͤ void ˤ롣*/
	(void) lispMgr_CreateVoid (pLispMgr, &pVoid) ;
	lispBind_SetValue (pLispMgr, pBind, pVoid) ;

	if (ppBindReturn != NULL)
		*ppBindReturn	= pBind ;
	return	True ;
}

Boolean
lispBindTable_SetEntryValue (
	register TLispManager*		pLispMgr,
	register TLispBind** const	ppTable,
	register int				nTableSize,
	register TLispEntity*		pSymbol,
	register TLispEntity*		pValue)
{
	TLispBind*	pBind ;

	if (TFAILED (lispBindTable_SearchEntry (pLispMgr, ppTable, nTableSize, pSymbol, &pBind)) ||
		pBind == NULL)
		return	False ;
	(void) lispBind_SetValue (pLispMgr, pBind, pValue) ;
	return	True ;
}

Boolean
lispBindTable_GetEntryValue (
	register TLispManager*			pLispMgr,
	register TLispBind** const		ppTable,
	register int					nTableSize,
	register TLispEntity*			pSymbol,
	register TLispEntity** const	ppValueReturn)
{
	TLispBind*		pBind ;
	TLispEntity*	pReturn ;

	assert (pLispMgr != NULL) ;
	assert (ppTable  != NULL && nTableSize > 0) ;
	assert (pSymbol  != NULL) ;
	assert (ppValueReturn != NULL) ;
	/*assert (hValue  != NULL) ;*/

	if (TFAILED (lispBindTable_SearchEntry (pLispMgr, ppTable, nTableSize, pSymbol, &pBind)) ||
		pBind == NULL)
		return	False ;

	if (TFAILED (lispBind_GetValue (pBind, &pReturn)) ||
		pReturn == NULL)
		return	False ;
	
	*ppValueReturn	= pReturn ;
	return	True ;
}

Boolean
lispBindTable_SearchEntry (
	register TLispManager*		pLispMgr,
	register TLispBind** const	ppTable,
	register int				nTableSize,
	register TLispEntity*		pSymbol,
	register TLispBind** const	ppReturn)
{
	register int			iHashkey ;

	iHashkey	= lispBindTable_createHashkey (pSymbol, nTableSize) ;
	assert (0 <= iHashkey && iHashkey < nTableSize) ;
	return	lispBind_Search (*(ppTable + iHashkey), pSymbol, ppReturn, NULL) ;
}

