/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "sysdep.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if defined (__linux__)
#include <sys/param.h>
#endif
#include <assert.h>
#include "filebuf.h"
#include "varbuf.h" 

/*
 *	ޥ
 */
#define	NELEMENTS(array)	(sizeof(array) / sizeof(array[0]))

/*========================================================================*
 *	
 *========================================================================*/

#define	STR_MARKER	"CSTR("
#define	CHR_MARKER	"CCHR("

enum {
	STATE_NORMAL	= 0,
	STATE_STRING,
	STATE_QUOTE,
	STATE_IGNORE_NEXT,
	STATE_COMMENT,
	STATE_STR_MARKER,
	STATE_CHR_MARKER,
	STATE_ENDOFFILE,
} ;

enum {
	ERR				= -1,
	NOERR			= 0,
} ;

#define	MAX_LABEL_NUM	(256)
#define	TMPBUFSIZE		(1024)

/*========================================================================*
 *	ץȥ
 *========================================================================*/

static	int		Kpp_Initialize (int* piArgc, char* pArgv []) ;
static	void	Kpp_ShowUsage (void) ;
static	int		Kpp_ForeachFile (const char* pFileName, FILE* fpHeader, FILE* fpSource) ;
static	int		Kpp_ParseTargetFile (const char* pTargetFile, FILE* fpHeader, FILE* fpSource) ;
static	int		Kpp_ParseFile (const char* pFileName, FILE* fpHeader, FILE* fpSource) ;
static	void	Kpp_CreateLabel (const char* pString) ;
static	int		Kpp_MainLoop (PFILEBUF pfbuf, FILE* fpHeader, FILE* fpSource) ;
static	int		Kpp_StateNormal (PFILEBUF pfbuf, FILE* fpHeader, FILE* fpSource) ;
static	int		Kpp_StateString (PFILEBUF pfbuf, FILE* fpHeader, FILE* fpSource) ;
static	int		Kpp_StateQuote (PFILEBUF pfbuf, FILE* fpHeader, FILE* fpSource) ;
static	int		Kpp_StateComment (PFILEBUF pfbuf, FILE* fpHeader, FILE* fpSource) ;
static	int		Kpp_StateStrMarker (PFILEBUF pfbuf, FILE* fpHeader, FILE* fpSource) ;
static	int		Kpp_StateChrMarker (PFILEBUF pfbuf, FILE* fpHeader, FILE* fpSource) ;
static	int		Kpp_GetNextStateNormal (PFILEBUF pfbuf) ;
static	int		Kpp_GetNextStateString (PFILEBUF pfbuf) ;
static	int		Kpp_GetNextStateQuote (PFILEBUF pfbuf) ;
static	int		Kpp_GetNextStateComment (PFILEBUF pfbuf) ;
static	void	Kpp_putchar (Char cc, FILE* fp) ;
static	unsigned	short	j_char_jis_to_sjis (unsigned short cc) ;

/*========================================================================*
 *	Хѿ(Υ static)
 *========================================================================*/
/*
 *	߹ԡԥɤɤ߹ 1 䤵롣
 */
static	long	slLineNumber				= 1 ;

/*
 *	ֹ̤档
 */
static	long	slTotalNumber				= 1 ;

/*
 *	٥롣Ѥ롣
 */
static	char	sachLabel [MAX_LABEL_NUM]	= "" ;
static	BOOL	sfNoLabel					= TRUE ;

/*
 *	ϥإåե롣
 */
static	char	sachHeader [MAX_PATH + 1]	= "a.h" ;

/*
 *	ϥե롣
 */
static	char	sachSource [MAX_PATH + 1]	= "a.c" ;

/*========================================================================*
 *	Хؿ
 *========================================================================*/

int
main (int iArgc, char* pArgv [])
{
	FILE*	fpHeader ;
	FILE*	fpSource ;
	int		i ;
	int		iRetval ;

	iRetval			= EXIT_SUCCESS ;

	if (Kpp_Initialize (&iArgc, pArgv)) {
		Kpp_ShowUsage () ;
		return	1 ;
	}
	if (!sachLabel [0]) {
		sfNoLabel	= TRUE ;
	} else {
		sfNoLabel	= FALSE ;
	}
	fpHeader	= fopen (sachHeader, "a") ;
	if (fpHeader == NULL) {
		fprintf (stderr, "Cannot create header file: \"%s\".\n", sachHeader) ;
		return	EXIT_FAILURE ;
	}
	fpSource	= fopen (sachSource, "a") ;
	if (fpSource == NULL) {
		fprintf (stderr, "Cannot create source file: \"%s\".\n", sachSource) ;
		fclose (fpHeader) ;
		return	EXIT_FAILURE ;
	}
	if (iArgc == 1) {
		FILEBUF	fbuf ;
		FileBuf_Init (&fbuf, stdin, KCODING_SYSTEM_SHIFTJIS) ;
		if (sfNoLabel) 
			Kpp_CreateLabel ("a_out") ;
		Kpp_MainLoop (&fbuf, fpHeader, fpSource) ;
	} else {
		for (i = 1 ; i < iArgc ; i ++) {
			if (Kpp_ParseFile (pArgv [i], fpHeader, fpSource) != NOERR) {
				iRetval	= EXIT_FAILURE ;
				break ;
			}
		}
	}
	fclose (fpHeader) ;
	fclose (fpSource) ;
	return	iRetval ;
}

/*========================================================================*
 *	ؿ
 *========================================================================*/

/*
 *	ץΥåʤɤԤ
 */
int
Kpp_Initialize (int* piArgc, char* pArgv [])
{
	char*	pString ;
	char*	pTemp ;
	int		i ;
	int		iArgc ;

	assert (piArgc != NULL) ;
	assert (pArgv != NULL) ;

	iArgc	= *piArgc ;
	i		= 1 ;
	/*
	 *	ץβϤԤ
	 *	ϤäȤȽ񤫤ʤȤʤɡ
	 */
	while (i < iArgc) {
		if (*pArgv [i] != '-') {
			i	++ ;
			continue ;
		}
		pString	= pArgv [i] + 1 ;
		switch (*pString) {
			/*
			 *	ϥեꡣ
			 */
		case	'h':
			pTemp	= sachHeader ;
			goto	file_common ;

		case	's':
			pTemp	= sachSource ;

		file_common:
			pString	++ ;
			if (*pString != '\0')
				goto	invalid_argument ;
			if ((i + 1) >= iArgc) {
				fprintf (stderr, "Lack of argument: \"%s\"\n", pArgv [i]) ;
				return	-1 ;
			}
			pArgv [i ++]	= NULL ;
			strncpy (pTemp, pArgv [i], MAX_PATH) ;
			pTemp [MAX_PATH]	= '\0' ;
			pArgv [i ++]	= NULL ;
			break ;

			/*
			 *	٥ꡣ
			 */
		case	'l':
			pString	++ ;
			if (*pString != '\0')
				goto	invalid_argument ;
			if ((i + 1) >= iArgc) {
				fprintf (stderr, "Lack of argument: \"%s\"\n", pArgv [i]) ;
				return	-1 ;
			}
			pArgv [i ++]	= NULL ;
			strncpy (sachLabel, pArgv [i], MAX_LABEL_NUM - 1) ;
			sachLabel [MAX_LABEL_NUM - 1]	= '\0' ;
			pArgv [i ++]	= NULL ;
			break ;

			/*
			 *	ʳΥץϥ顼ˤʤ롣
			 */
		default:
		invalid_argument:
			fprintf (stderr, "Invalid argument: \"%s\"\n", pArgv [i]) ;
			return	-1 ;
		}
	}

	i	= 0 ;
	while (i < iArgc) {
		if (pArgv [i] == NULL) {
			if ((i + 1) < iArgc ) {
				memmove (pArgv + i, pArgv + (i + 1), (iArgc - (i + 1)) * sizeof (char*)) ;
			} else {
				break ;
			}
			iArgc	-- ;
		} else {
			i	++ ;
		}
	}
	*piArgc	= iArgc ;
	return	0 ;
}

void
Kpp_ShowUsage (void)
{
	fprintf (stderr, "[usage] kpp.exe [-l label] [-h header] [-s source] [file1 file2 ... fileN]\n") ;
	return ;
}

int
Kpp_ParseFile (const char* pFileName, FILE* fpHeader, FILE* fpSource)
{
#if defined (WIN32)
	char			achPathBuffer [_MAX_PATH] ;
	char			achDrive [_MAX_DRIVE] ;
	char 			achDir [_MAX_DIR] ;
	HANDLE			hFile ;
	WIN32_FIND_DATA	wfd ;
	int				iRetval	= NOERR ;

	assert (pFileName != NULL) ;
	
	if (*pFileName == '@') {
		return	Kpp_ParseTargetFile (pFileName + 1, fpHeader, fpSource) ;
	}

	_splitpath (pFileName, achDrive, achDir, NULL, NULL) ;
	hFile		= FindFirstFile (pFileName, &wfd) ;
	if (hFile == INVALID_HANDLE_VALUE) {
		return	ERR ;
	}
	for ( ; ; ){
		if ((wfd.dwFileAttributes & FILE_ATTRIBUTE_SYSTEM) == 0 &&
			(wfd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) == 0 &&
			(wfd.dwFileAttributes & FILE_ATTRIBUTE_HIDDEN) == 0) {
			_makepath (achPathBuffer, achDrive, achDir, wfd.cFileName, NULL) ;
#if defined (DEBUG)
			fprintf (stderr, "%s\n", achPathBuffer) ;
#endif
			if (Kpp_ForeachFile (achPathBuffer, fpHeader, fpSource) == ERR) {
				iRetval	= ERR ;
				break ;
			}
		}
		if (FindNextFile (hFile, &wfd) == FALSE) {
			if (GetLastError () != ERROR_NO_MORE_FILES)
				iRetval	= ERR ;
			break ;
		}
	}
	FindClose (hFile) ;
	return	iRetval ;
#else
	if (*pFileName == '@') 
		return	Kpp_ParseTargetFile (pFileName + 1, fpHeader, fpSource) ;
	return	Kpp_ForeachFile (pFileName, fpHeader, fpSource) ;
#endif
}

int
Kpp_ParseTargetFile (const char* pTargetFile, FILE* fpHeader, FILE* fpSource)
{
	FILE*		pTargetFp ;
	int			iRetval	= NOERR ;
	char		achBuffer [TMPBUFSIZE] ;
	const char*	ptr ;

	pTargetFp	= fopen (pTargetFile, "r") ;
	if (pTargetFp == NULL)
		return	ERR ;
	while (!feof (pTargetFp)) {
		int		nLen ;
		fgets (achBuffer, NELEMENTS (achBuffer), pTargetFp) ;

		nLen	= strlen (achBuffer) ;
		if (nLen <= 0)
			continue ;
		if (achBuffer [nLen - 1] != '\n') {
			int	cc ;
			while (cc = fgetc (pTargetFp), cc != EOF && cc != '\n')
				;
			continue ;
		}
		achBuffer [nLen - 1]	= '\0' ;
		
		ptr		= achBuffer ;
		while (*ptr != '\0' && (*ptr == ' ' || *ptr == '\t'))
			ptr	++ ;
		if (*ptr == '\0' || *ptr == '\n' || *ptr == '#')
			continue ;
		iRetval	= Kpp_ForeachFile (ptr, fpHeader, fpSource) ;
		if (iRetval != NOERR)
			break ;
	}
	fclose (pTargetFp) ;
	return	iRetval ;
}

int
Kpp_ForeachFile (const char* pFileName, FILE* fpHeader, FILE* fpSource)
{
	FILE*	fpSrc ;
	FILEBUF	fbuf ;
	char	achBuffer [16384] ;
	int		nRead, iKanjiCode, iNewlineType ;
	
	slLineNumber	= 1 ;
	slTotalNumber	= 1 ;

	fpSrc	= fopen (pFileName, "r") ;
	if (fpSrc == NULL) {
		fprintf (stderr, "No such file: \"%s\".\n", pFileName) ;
		return	ERR ;
	}
	nRead		= fread (achBuffer, 1, NELEMENTS (achBuffer), fpSrc) ;
	iKanjiCode	= DetectKanjiCodingSystem (achBuffer, nRead, &iNewlineType) ;
	rewind (fpSrc) ;
#if defined (DEBUG)
	fprintf (stderr, "(%d, %d, %d)\n", nRead, iKanjiCode, iNewlineType) ;
#endif
	
	FileBuf_Init (&fbuf, fpSrc, iKanjiCode) ;
	if (sfNoLabel) 
		Kpp_CreateLabel (pFileName) ;
	Kpp_MainLoop (&fbuf, fpHeader, fpSource) ;
	fclose (fpSrc) ;
	
	return	NOERR ;
}

void
Kpp_CreateLabel (const char* pString)
{
	int	i ;
#if defined (WIN32)
	char	achFname [_MAX_FNAME] ;
	char	achExt [_MAX_EXT] ;
#else
	const char*	pHead ;
	const char*	ptr ;
#endif
	assert (pString != NULL) ;

#if defined (WIN32)
	_splitpath (pString, NULL, NULL, achFname, achExt) ;
	strncpy (sachLabel, achFname, MAX_LABEL_NUM - 1) ;
	sachLabel [MAX_LABEL_NUM - 1]	= '\0' ;
	strncat (sachLabel, achExt, MAX_LABEL_NUM - strlen (sachLabel) - 1) ;
	sachLabel [MAX_LABEL_NUM - 1]	= '\0' ;
#else
	pHead	= pString ;
	ptr		= pString ;
	while (*ptr != '\0') {
		if (*ptr == '\\') {
			ptr	++ ;
		} else if (*ptr == '/') {
			pHead	= ptr + 1 ;
		}
		ptr	++ ;
	}
	assert (pHead < ptr) ;

	strncpy (sachLabel, pHead, MAX_LABEL_NUM - 1) ;
	sachLabel [MAX_LABEL_NUM - 1]	= '\0' ;
#endif

	for (i = 0 ; i < MAX_LABEL_NUM && sachLabel [i] != '\0' ; i ++) {
		/*
		 *	C ѿ̾ȤƻȤʤΤϤΤޤޤˤơܤʤ
		 *	ϡ``_'' ֤롣
		 */
		if ('a' <= sachLabel [i] && sachLabel [i] <= 'z')
			continue ;
		if ('A' <= sachLabel [i] && sachLabel [i] <= 'Z')
			continue ;
		if (i > 0 && '0' <= sachLabel [i] && sachLabel [i] <= '9')
			continue ;
		if (sachLabel [i] == '_')
			continue ;

		sachLabel [i]	= '_' ;
	}
	return ;
}


/*
 *	ѡΥᥤ롼ס		
 */
int
Kpp_MainLoop (PFILEBUF pfbuf, FILE* fpHeader, FILE* fpSource)
{
	int		iCurrentState ;
	int		iNextState ;

	iCurrentState	= STATE_NORMAL ;

	while (iCurrentState != STATE_ENDOFFILE && iCurrentState >= 0) {
		switch (iCurrentState) {
		case	STATE_NORMAL:
			iNextState	= Kpp_StateNormal (pfbuf, fpHeader, fpSource) ;
			break ;

		case	STATE_STRING:
			iNextState	= Kpp_StateString (pfbuf, fpHeader, fpSource) ;
			break ;

		case	STATE_QUOTE:
			iNextState	= Kpp_StateQuote (pfbuf, fpHeader, fpSource) ;
			break ;

		case	STATE_STR_MARKER:
			iNextState	= Kpp_StateStrMarker (pfbuf, fpHeader, fpSource) ;
			break ;

		case	STATE_COMMENT:
			iNextState	= Kpp_StateComment (pfbuf, fpHeader, fpSource) ;
			break ;

		case	STATE_CHR_MARKER:
			iNextState	= Kpp_StateChrMarker (pfbuf, fpHeader, fpSource) ;
			break ;

		default:
			fprintf (stderr, "Unkown state (%d).\n", iCurrentState) ;
			return	ERR ;
		}
		iCurrentState	= iNextState ;
	}
	return	NOERR ;
}

int
Kpp_StateNormal (PFILEBUF pfbuf, FILE* fpHeader, FILE* fpSource)
{
	int		iNextState ;
	Char	cc ;

#if defined (DEBUG)
	fprintf (stderr, "Enter(%ld) [normal]\n", slLineNumber) ;
#endif

	for ( ; ; ) {
		iNextState	= Kpp_GetNextStateNormal (pfbuf) ;

		if (iNextState != STATE_NORMAL) 
			break ;

		cc	= FileBuf_Getc (pfbuf) ;
		if (cc == EOF) {
			iNextState	= STATE_ENDOFFILE ;
			break ;
		}
		Kpp_putchar (cc, stdout) ;

		if (cc == '\n')
			slLineNumber	++ ;
	}
#if defined (DEBUG)
	fprintf (stderr, "Leave(%ld) [normal]\n", slLineNumber) ;
#endif
	return	iNextState ;

	UNREFERENCED_PARAMETER (fpHeader) ;
	UNREFERENCED_PARAMETER (fpSource) ;
}

int
Kpp_StateComment (PFILEBUF pfbuf, FILE* fpHeader, FILE* fpSource)
{
	Char	cc ;
	int		iNextState ;
	
#if defined (DEBUG)
	fprintf (stderr, "Enter(%ld) [comment]\n", slLineNumber) ;
#endif

	cc	= FileBuf_Getc (pfbuf) ;
	assert (cc == '/') ;
	Kpp_putchar (cc, stdout) ;
	cc	= FileBuf_Getc (pfbuf) ;
	assert (cc == '*') ;
	Kpp_putchar (cc, stdout) ;

	for ( ; ; ) {
		iNextState	= Kpp_GetNextStateComment (pfbuf) ;

		if (iNextState != STATE_COMMENT) {
			cc	= FileBuf_Getc (pfbuf) ;
			assert (cc == '*') ;
			Kpp_putchar (cc, stdout) ;
			cc	= FileBuf_Getc (pfbuf) ;
			assert (cc == '/') ;
			Kpp_putchar (cc, stdout) ;
			break ;
		}

		cc	= FileBuf_Getc (pfbuf) ;
#if defined (DEBUG)
		fprintf (stderr, "comment = %08lX\n", cc) ;
#endif
		Kpp_putchar (cc, stdout) ;

		if (cc == '\n')
			slLineNumber	++ ;
	}

#if defined (DEBUG)
	fprintf (stderr, "Leave(%ld) [comment]\n", slLineNumber) ;
#endif
	return	iNextState ;

	UNREFERENCED_PARAMETER (fpHeader) ;
	UNREFERENCED_PARAMETER (fpSource) ;
}

/*
 *	double-quote ǰϤޤ줿ɤФؿ
 */
int
Kpp_StateString (
	register PFILEBUF	pfbuf,
	register FILE*		fpHeader,
	register FILE*		fpSource)
{
	int		iNextState ;
	int		cc ;

#if defined (DEBUG)
	fprintf (stderr, "Enter(%ld) [string]\n", slLineNumber) ;
#endif

	/*
	 *	褿ʳǰʸɤȤʸ󳫻Ϥ double-quote
	 *	ǤʤФʤʤ
	 */
	cc	= FileBuf_Getc (pfbuf) ;
	assert (cc == 0x22) ;
	fprintf (stdout, "%c", 0x22) ;

	/*
	 *	ʸüޤɤФλbackslash ˤ double-quote
	 *	̵դ롣
	 */
	for ( ; ; ) {
		iNextState	= Kpp_GetNextStateString (pfbuf) ;

		cc	= FileBuf_Getc (pfbuf) ;
 		if (cc == EOF) {
			fprintf (stderr, "Unexpected end of file.\n") ;
			return	-1 ;
		}

		if (iNextState == STATE_IGNORE_NEXT) {
			/*
			 *	̵뤹Ȥ̣ backslash ˺줺˽
			 *	롣
			 */
			Kpp_putchar (cc, stdout) ;
			/*
			 *	̵뤵ʸϤ롣
			 */
			cc	= FileBuf_Getc (pfbuf) ;
			Kpp_putchar (cc, stdout) ;
			if (cc == '\n')
				slLineNumber	++ ;
			continue ;
		}
		Kpp_putchar (cc, stdout) ;

		if (cc == '\n')
			slLineNumber	++ ;

		if (iNextState != STATE_STRING) 
			break ;
	}
#if defined (DEBUG)
	fprintf (stderr, "Leave(%ld) [string]\n", slLineNumber) ;
#endif
	return	iNextState ;

	UNREFERENCED_PARAMETER (fpHeader) ;
	UNREFERENCED_PARAMETER (fpSource) ;
}

/*
 *	quote ɤФؿ
 *	quote  double-quote о줹ǽäơ
 *	ʸγϤȻפȵư
 */
int
Kpp_StateQuote (PFILEBUF pfbuf, FILE* fpHeader, FILE* fpSource)
{
	int		iNextState ;
	int		cc ;

#if defined (DEBUG)
	fprintf (stderr, "Enter(%ld) [quote]\n", slLineNumber) ;
#endif

	/*
	 *	褿ʳǰʸɤȤʸ󳫻Ϥ double-quote
	 *	ǤʤФʤʤ
	 */
	cc	= FileBuf_Getc (pfbuf) ;
	assert (cc == 0x27) ;
	fprintf (stdout, "%c", 0x27) ;

	/*
	 *	ʸüޤɤФλbackslash ˤ double-quote
	 *	̵դ롣
	 */
	for ( ; ; ) {
		iNextState	= Kpp_GetNextStateQuote (pfbuf) ;

		cc	= FileBuf_Getc (pfbuf) ;
 		if (cc == EOF) {
			fprintf (stderr, "Unexpected end of file.\n") ;
			return	-1 ;
		}

		if (iNextState == STATE_IGNORE_NEXT) {
			/*
			 *	̵뤹Ȥ̣ backslash ˺줺˽
			 *	롣
			 */
			Kpp_putchar (cc, stdout) ;
			/*
			 *	ΰ̵̣뤵ʸϤ롣
			 */
			cc	= FileBuf_Getc (pfbuf) ;
			Kpp_putchar (cc, stdout) ;
			if (cc == '\n')
				slLineNumber	++ ;
			continue ;
		}
		Kpp_putchar (cc, stdout) ;

		if (cc == '\n')
			slLineNumber	++ ;

		if (iNextState != STATE_QUOTE) 
			break ;
	}
#if defined (DEBUG)
	fprintf (stderr, "Leave(%ld) [quote]\n", slLineNumber) ;
#endif
	return	iNextState ;

	UNREFERENCED_PARAMETER (fpHeader) ;
	UNREFERENCED_PARAMETER (fpSource) ;
}

int
Kpp_StateStrMarker (PFILEBUF pfbuf, FILE* fpHeader, FILE* fpSource)
{
	Char	cc ;
	VARBUF	varbuf ;
	long	lValue ;
	long	lNextValue ;
	int		i ;
	Char*	ptr ;

#if defined (DEBUG)
	fprintf (stderr, "Enter(%ld) [STR_MARKER]\n", slLineNumber) ;
#endif

	/*
	 *	褿ʳǡɤ߹4ʸ STR_MARKER Ʊ
	 *	ȷޤäƤ롣
	 */
	cc	= FileBuf_Getc (pfbuf) ;
	assert (cc == 'C') ;
	cc	= FileBuf_Getc (pfbuf) ;
	assert (cc == 'S') ;
	cc	= FileBuf_Getc (pfbuf) ;
	assert (cc == 'T') ;
	cc	= FileBuf_Getc (pfbuf) ;
	assert (cc == 'R') ;
	cc	= FileBuf_Getc (pfbuf) ;
	assert (cc == '(') ;

	/*
	 *	ʸγϵõSTR_MARKER ʸγϵ
	 *	Ǥˤϡ٤¸ߤƤޤʤ
	 */
	cc	= FileBuf_Getc (pfbuf) ;
	while (cc == ' ' || cc == '\t' || cc == '\r' || cc == '\n') {
		if (cc == '\n')
			slLineNumber	++ ;
		cc	= FileBuf_Getc (pfbuf) ;
	}

	/*
	 *	ʸγϤǤä
	 */
	if (cc != 0x22) {
		/*
		 *	ԤޤSTR_MARKER ȻפäΤϱΤ褦Ǥ
		 */
		fprintf (stdout, "CSTR(") ;
		FileBuf_Ungetc (pfbuf, cc) ;
		return	STATE_NORMAL ;
	}

	VarBuf_Init (&varbuf, sizeof (Char)) ;

	/*
	 *	ĹХåեʸáࡣ
	 */
	for ( ; ; ) {
		cc	= FileBuf_Getc (pfbuf) ;
		switch (cc) {
			/*
			 *	double-quote դ顢ʸνü
			 */
		case	0x22:
			goto	exit_loop ;

			/*
			 *	backslash 褿ա
			 */
		case	0x5C:
			/*
			 *	ʸɤ߹ࡣ
			 */
			cc	= FileBuf_Getc (pfbuf) ;
			switch (cc) {
			case	'b':
				cc	= Char_MakeAscii (0x02) ;
				VarBuf_Puts (&varbuf, &cc, 1) ;
				break ;

			case	'n':
				cc	= Char_MakeAscii ('\n') ;
				VarBuf_Puts (&varbuf, &cc, 1) ;
				break ;

			case	'r':
				cc	= Char_MakeAscii ('\r') ;
				VarBuf_Puts (&varbuf, &cc, 1) ;
				break ;

			case	't':
				cc	= Char_MakeAscii ('\t') ;
				VarBuf_Puts (&varbuf, &cc, 1) ;
				break ;

			case	'x':
				lValue	= 0 ;
				for (i = 0 ; i < 2 ; i ++) {
					cc	= FileBuf_Getc (pfbuf) ;
					if ('0' <= cc && cc <= '9') {
						lNextValue	= lValue * 16 + (cc - '0') ;
					} else if ('a' <= Char_ToLower (cc) && Char_ToLower (cc) <= 'f') {
						lNextValue	= lValue * 16 + (Char_ToLower (cc) - 'a') ;
					} else {
						FileBuf_Ungetc (pfbuf, cc) ;
						break ;
					}
					if (lNextValue >= 128 || lNextValue < 0) {
						FileBuf_Ungetc (pfbuf, cc) ;
						break ;
					}
					lValue	= lNextValue ;
				}
				cc	= Char_MakeAscii ((char)lValue) ;
				VarBuf_Puts (&varbuf, &cc, 1) ;
				break ;

			default:
				/*
				 *	ä硣
				 */
				if ('0' <= cc && cc <= '7') {
					long	lValue ;
					lValue	= (cc - '0') ;
					for (i = 0 ; i < 2 ; i ++) {
						cc	= FileBuf_Getc (pfbuf) ;
						if (cc < '0' || '7' < cc) {
							FileBuf_Ungetc (pfbuf, cc) ;
							break ;
						}
						lNextValue	= lValue * 8 + (cc - '0') ;
/*
						if (lNextValue >= 128 || lNextValue < 0) {
							FileBuf_Ungetc (pfbuf, cc) ;
							break ;
						}
*/
						lValue	= lNextValue ;
					}
					cc	= Char_MakeAscii ((char)lValue) ;
				}
				VarBuf_Puts (&varbuf, &cc, 1) ;
				break ;
			}
			break ;

		default:
			if (cc == EOF) {
				fprintf (stderr, "Unexpected end of file.\n") ;
				return	-1 ;
			}
			VarBuf_Puts (&varbuf, &cc, 1) ;
			if (cc == '\n')
				slLineNumber	++ ;
			break ;
		}
	}

 exit_loop:
	/*
	 *	ɤФ
	 */
	cc	= FileBuf_Getc (pfbuf) ;
	while (cc == ' ' || cc == '\t' || cc == '\n' || cc == '\r') {
		if (cc == '\n')
			slLineNumber	++ ;
		cc	= FileBuf_Getc (pfbuf) ;
	}

	/*
	 *	STR_MARKER üƤ뤳Ȥǧ롣
	 */
	if (cc != ')') {
		fprintf (stderr, "Suspicious STR_MARKER\n") ;
		return	-1 ;
	}
	
	fprintf (stdout, "%s_l%ld_n%ld", sachLabel, slLineNumber, slTotalNumber) ;

	/*
	 *	إå¦ؤνϡ
	 */
	fprintf (fpHeader, "extern\tChar\t%s_l%ld_n%ld [] ; \n", sachLabel, slLineNumber, slTotalNumber) ;

	/*
	 *	¦ؤνϡ
	 */
	fprintf (fpSource, "Char\t%s_l%ld_n%ld [] = {\n", sachLabel, slLineNumber, slTotalNumber) ;

	ptr	= (Char*)VarBuf_GetBuffer (&varbuf) ;
	for (i = 0 ; i < VarBuf_GetUsage (&varbuf) ; i ++) {
		if (i > 0 && (i % 8) == 0)
			fprintf (fpSource, "\n") ;
		if ((i % 8) == 0)
			fprintf (fpSource, "\t") ;
		fprintf (fpSource, "0x%08lx, ", *ptr) ;
		ptr	++ ;
	}
	if (i > 0 && (i % 8) == 0) 
		fprintf (fpSource, "\n") ;
	if ((i % 8) == 0)
		fprintf (fpSource, "\t") ;
	fprintf (fpSource, "0x00000000, \n} ; \n") ;

	VarBuf_Uninit (&varbuf) ;

	slTotalNumber	++ ;

#if defined (DEBUG)
	fprintf (stderr, "Leave(%ld) [STR_MARKER]\n", slLineNumber) ;
#endif

	return	STATE_NORMAL ;
}

int
Kpp_StateChrMarker (PFILEBUF pfbuf, FILE* fpHeader, FILE* fpSource)
{
	Char	cc ;
	VARBUF	varbuf ;
	long	lValue ;
	long	lNextValue ;
	int		i ;

#if defined (DEBUG)
	fprintf (stderr, "Enter(%ld) [CHR_MARKER]\n", slLineNumber) ;
#endif

	/*
	 *	褿ʳǡɤ߹4ʸ CHR_MARKER Ʊ
	 *	ȷޤäƤ롣
	 */
	cc	= FileBuf_Getc (pfbuf) ;
	assert (cc == 'C') ;
	cc	= FileBuf_Getc (pfbuf) ;
	assert (cc == 'C') ;
	cc	= FileBuf_Getc (pfbuf) ;
	assert (cc == 'H') ;
	cc	= FileBuf_Getc (pfbuf) ;
	assert (cc == 'R') ;
	cc	= FileBuf_Getc (pfbuf) ;
	assert (cc == '(') ;

	/*
	 *	ʸγϵõCHR_MARKER ʸγϵ
	 *	Ǥˤϡ٤¸ߤƤޤʤ
	 */
	cc	= FileBuf_Getc (pfbuf) ;
	while (cc == ' ' || cc == '\t' || cc == '\r' || cc == '\n') {
		if (cc == '\n')
			slLineNumber	++ ;
		cc	= FileBuf_Getc (pfbuf) ;
	}

	/*
	 *	Single-Quote 
	 */
	if (cc != 0x27) {
		/*
		 *	 CCHR ѴɬפϤʤȻפ
		 */
		fprintf (stdout, "CCHR(") ;
		FileBuf_Ungetc (pfbuf, cc) ;
		return	STATE_NORMAL ;
	}

	VarBuf_Init (&varbuf, sizeof (Char)) ;

	/*
	 *	ĹХåեʸáࡣ
	 */
	for ( ; ; ) {
		cc	= FileBuf_Getc (pfbuf) ;
		switch (cc) {
			/*
			 *	single-quote դ顢ʸνü
			 */
		case	0x27:
			goto	exit_loop ;

			/*
			 *	backslash 褿ա
			 */
		case	0x5C:
			/*
			 *	ʸɤ߹ࡣ
			 */
			cc	= FileBuf_Getc (pfbuf) ;
			switch (cc) {
			case	't':
				cc	= Char_MakeAscii ('\t') ;
				VarBuf_Puts (&varbuf, &cc, 1) ;
				break ;

			case	'r':
				cc	= Char_MakeAscii ('\r') ;
				VarBuf_Puts (&varbuf, &cc, 1) ;
				break ;

			case	'n':
				cc	= Char_MakeAscii ('\n') ;
				VarBuf_Puts (&varbuf, &cc, 1) ;
				break ;

			case	'x':
				lValue	= 0 ;
				for (i = 0 ; i < 2 ; i ++) {
					cc	= FileBuf_Getc (pfbuf) ;
					if ('0' <= cc && cc <= '9') {
						lNextValue	= lValue * 16 + (cc - '0') ;
					} else if ('a' <= Char_ToLower (cc) && Char_ToLower (cc) <= 'f') {
						lNextValue	= lValue * 16 + (Char_ToLower (cc) - 'a') ;
					} else {
						/*	Unexpected character */
						FileBuf_Ungetc (pfbuf, cc) ;
						break ;
					}
					if (lNextValue >= 128 || lNextValue < 0) {
						FileBuf_Ungetc (pfbuf, cc) ;
						break ;
					}
					lValue	= lNextValue ;
				}
				cc	= Char_MakeAscii ((char)lValue) ;
				VarBuf_Puts (&varbuf, &cc, 1) ;
				break ;

			default:
				/*
				 *	ä硣
				 */
				if ('0' <= cc && cc <= '7') {
					long	lValue ;
					lValue	= (cc - '0') ;
					for (i = 0 ; i < 2 ; i ++) {
						cc	= FileBuf_Getc (pfbuf) ;
						if (cc < '0' || '7' < cc) {
							FileBuf_Ungetc (pfbuf, cc) ;
							break ;
						}
						lNextValue	= lValue * 8 + (cc - '0') ;
						if (lNextValue >= 128 || lNextValue < 0) {
							FileBuf_Ungetc (pfbuf, cc) ;
							break ;
						}
						lValue	= lNextValue ;
					}
					cc	= Char_MakeAscii ((char)lValue) ;
				}
				VarBuf_Puts (&varbuf, &cc, 1) ;
				break ;
			}
			break ;

		default:
			if (cc == EOF) {
				fprintf (stderr, "Unexpected end of file.\n") ;
				return	-1 ;
			}
			VarBuf_Puts (&varbuf, &cc, 1) ;
			if (cc == '\n')
				slLineNumber	++ ;
			break ;
		}
	}

 exit_loop:
	/*
	 *	ɤФ
	 */
	cc	= FileBuf_Getc (pfbuf) ;
	while (cc == ' ' || cc == '\t' || cc == '\n' || cc == '\r') {
		if (cc == '\n')
			slLineNumber	++ ;
		cc	= FileBuf_Getc (pfbuf) ;
	}

	if ( VarBuf_GetUsage (&varbuf) != 1) {
		fprintf (stderr, "Strange CCHR (line: %ld).\n", slLineNumber) ;
		VarBuf_Uninit (&varbuf) ;
		return	-1 ;
	}

	/*
	 *	CHR_MARKER üƤ뤳Ȥǧ롣
	 */
	if (cc != ')') {
		fprintf (stderr, "Suspicious CHR_MARKER\n") ;
		VarBuf_Uninit (&varbuf) ;
		return	-1 ;
	}
	
	fprintf (stdout, "(Char)0x%lx", *(Char *)VarBuf_GetBuffer (&varbuf)) ;
	VarBuf_Uninit (&varbuf) ;

#if defined (DEBUG)
	fprintf (stderr, "Leave(%ld) [CHR_MARKER]\n", slLineNumber) ;
#endif

	return	STATE_NORMAL ;

	UNREFERENCED_PARAMETER (fpHeader) ;
	UNREFERENCED_PARAMETER (fpSource) ;
}

/*
 *	ʸ(ˤäƤʸ)ɤ߹ɤξ֤ܤ
 *	Τ롣
 *	δؿϡ֤ NORMAL λΤΤǤ롣
 */
int
Kpp_GetNextStateNormal (PFILEBUF pfbuf)
{
	char*	ptr ;
	Char	cc ;
	int		iRetvalue ;
	int		iHead ;

	iHead		= FileBuf_GetHead (pfbuf) ;
	cc			= FileBuf_BufferedGetc (pfbuf) ;
	iRetvalue	= STATE_NORMAL ;

	switch (cc) {
	case	'/':
		cc	= FileBuf_BufferedGetc (pfbuf) ;
		if (cc == '*') 
			iRetvalue	= STATE_COMMENT ;
		break ;

		/*
		 *	double-quote νdouble-quote ˶ޤ줿 CSTR ̵Ǥ롣
		 */
	case	0x22:
		iRetvalue		= STATE_STRING ;
		break ;

		/*
		 *	quote դνquote  double-quote 򶴤⤢
		 *	Τ̵뤹뤳ȤϤǤʤ
		 */
	case	0x27:
		iRetvalue		= STATE_QUOTE ;
		break ;

	case	EOF:
		iRetvalue		= STATE_ENDOFFILE ;
		break ;

	default:
		ptr	= STR_MARKER ;
		while (cc != EOF && !Char_DifferenceAscii (cc, *ptr) && *ptr != '\0') {
			cc	= FileBuf_BufferedGetc (pfbuf) ;
			ptr	++ ;
		}
		if (!*ptr) {
			iRetvalue	= STATE_STR_MARKER ;
#if 1
		} else {
			FileBuf_MoveHead (pfbuf, iHead) ;
			ptr	= CHR_MARKER ;
			cc	= FileBuf_BufferedGetc (pfbuf) ;
			while (cc != EOF && !Char_DifferenceAscii (cc, *ptr) && *ptr != '\0') {
				cc	= FileBuf_BufferedGetc (pfbuf) ;
				ptr	++ ;
			}
			if (!*ptr)
				iRetvalue	= STATE_CHR_MARKER ;
#endif
		}
		break ;
	}
	FileBuf_MoveHead (pfbuf, iHead) ;
	return	iRetvalue ;
}

/*
 *	ʸ(ˤäƤʸ)ɤ߹ɤξ֤ܤ
 *	Τ롣
 *	δؿϡ֤ STATE_STRING λΤΤǤ롣
 */
int
Kpp_GetNextStateString (PFILEBUF pfbuf)
{
	int	cc ;
	int	iRetvalue ;
	int	iHead ;

	iHead		= FileBuf_GetHead (pfbuf) ;
	cc			= FileBuf_BufferedGetc (pfbuf) ;
	iRetvalue	= STATE_STRING ;

	switch (cc) {
		/*
		 *	backslash ա³ doublequote 
		 *	̵뤵뤫顣
		 */
	case	0x5c:
		iRetvalue	= STATE_IGNORE_NEXT ;
		break ;

		/*
		 *	doublequote ʸϽü롣
		 */
	case	0x22:
		iRetvalue	= STATE_NORMAL ;
		break ;

	case	EOF:
		iRetvalue	= STATE_ENDOFFILE ;
		break ;

	default:
		iRetvalue	= STATE_STRING ;
		break ;
	}

	FileBuf_MoveHead (pfbuf, iHead) ;

	return	iRetvalue ;
}

/*
 *	ʸ(ˤäƤʸ)ɤ߹ɤξ֤ܤ
 *	Τ롣
 *	δؿϡ֤ STATE_QUOTE λΤΤǤ롣
 */
int
Kpp_GetNextStateQuote (PFILEBUF pfbuf)
{
	int	cc ;
	int	iRetvalue ;
	int	iHead ;

	iHead		= FileBuf_GetHead (pfbuf) ;
	cc			= FileBuf_BufferedGetc (pfbuf) ;
	iRetvalue	= STATE_STRING ;

	switch (cc) {
		/*
		 *	backslash ա³ doublequote 
		 *	̵뤵뤫顣
		 */
	case	0x5c:
		iRetvalue	= STATE_IGNORE_NEXT ;
		break ;

		/*
		 *	quote ʸϽü롣
		 */
	case	0x27:
		iRetvalue	= STATE_NORMAL ;
		break ;

		/*
		 *	եκǸ夬褿ˤܰդʤ顢EOF 
		 *	֤STATE_QUOTE  EOF Τư
		 *	ϻפʤ
		 */
	case	EOF:
		iRetvalue	= STATE_ENDOFFILE ;
		break ;

	default:
		iRetvalue	= STATE_QUOTE ;
		break ;
	}

	FileBuf_MoveHead (pfbuf, iHead) ;

	return	iRetvalue ;
}

int
Kpp_GetNextStateComment (PFILEBUF pfbuf)
{
	int	cc ;
	int	iRetvalue ;
	int	iHead ;

	iHead		= FileBuf_GetHead (pfbuf) ;
	cc			= FileBuf_BufferedGetc (pfbuf) ;
	iRetvalue	= STATE_COMMENT ;

	if (cc == '*') {
		cc		= FileBuf_BufferedGetc (pfbuf) ;
		if (cc == '/') {
			iRetvalue	= STATE_NORMAL ;
		}
	}
	FileBuf_MoveHead (pfbuf, iHead) ;

	return	iRetvalue ;
}

void
Kpp_putchar (Char cc, FILE* fp)
{
	unsigned short		chsj ;

	/*
	 *
	 */
	if (Char_IsAscii (cc)) {
		fputc (cc & 0x7F, fp) ;
		return ;
	}

	/*
	 *	ξˤ ShiftJis Ѵ롣Windows Ǥġ
	 */
	switch (CHAR_CHARSET (cc)) {
	case	KCHARSET_JISX0201_1976:
		fputc ((CHAR_CODE (cc) & 0xFF) | 0x80, fp) ;
		break ;

	case	KCHARSET_JISX0208_1978:
	case	KCHARSET_JISX0208_1983:
		/*
		 *	ShiftJis Ѵʤܡ
		 */
		chsj	= j_char_jis_to_sjis (CHAR_CODE (cc)) ;
		fputc ((chsj >> 8) & 0xFF, fp) ;
		fputc ((chsj     ) & 0xFF, fp) ;
		break ;

	default:
		/*
		 *	ʸɽǤʤ餺
		 */
		break ;
	}
	return ;
}

unsigned short
j_char_jis_to_sjis (unsigned short cc)
{
	unsigned jh, jl, sh, sl ;

	jh	= (unsigned) ((cc >> 8) & 0x007F) ;
	jl	= (unsigned) ((cc >> 0) & 0x007F) ;
	if (jh & 0x01) {
		jh	= jh >> 1 ;
		if (jh < 0x2F) {
			sh	= jh + 0x71 ;
		} else {
			sh	= jh - 0x4F ;
		}
		if (jl > 0x5F) {
			sl	= jl + 0x20 ;
		} else {
			sl	= jl + 0x1F ; 
		}
	} else {
		jh	= jh >> 1 ;
		if (jh < 0x2F) {
			sh	= jh + 0x70 ;
		} else {
			sh	= jh - 0x50 ;
		}
		sl	= jl + 0x7E ;
	}
#if defined (DEBUG)
	fprintf (stderr, "(%04x) -> (%x, %x)\n", cc, sh, sl) ;
#endif
	return	(((unsigned short)sh & 0x00FF) << 8) | ((unsigned short)sl & 0x00FF) ;
}

