/* Copyright (c) 2020-2021 The Creators of Simphone

   See the file COPYING.LESSER.txt for copying permission.
*/

#ifndef _PARAM_H_
#define _PARAM_H_

#include "simtypes.h"

simbool param_check_set (const char *param); /* check if non-default value has been set */

/* get configuration parameters (values from configuration parameters file) */
extern simtype param_table, param_login_table;

#if SIM_TYPE_CHECKS || ! defined(SIM_TYPE_CHECKS)
simtype _param_get (const char *param, int type, const char *file, unsigned line);
simtype _param_get_any (const char *param, const char *file, unsigned line);
#define param_get(param, type) _param_get (param, type, __FUNCTION__, __LINE__)
#define param_get_any(param) _param_get_any (param, __FUNCTION__, __LINE__)
#define param_get_number(param) ((int) param_get (param, SIMNUMBER).num)
#define param_get_string(param) param_get (param, SIMSTRING)
#define param_get_strings(param) param_get (param, SIMARRAY_STRING)
int _param_get_type (const char *param); /* return type of configuration parameter */
#define param_get_type(param) _param_get_type (param)
#else
#define param_get_any(param) table_get (param_table, param)
#define param_get_number(param) ((int) table_get_number (param_table, param))
#define param_get_string(param) table_get_string (param_table, param)
#define param_get_strings(param) table_get_array_string (param_table, param)
#define param_get_type(param) table_get (param_table, param).typ
#endif
#define param_get_pointer(param) param_get_string (param).ptr

/* return default, minimal and maximal allowed value of configuration parameter */
int param_get_limits (const char *param, simtype *def, int *min, int *max, const char **description);

simtype param_get_default (const char *param, const simtype def);
int param_get_min (const char *param, int min);
int param_get_max (const char *param, int max);

int param_get_number_default (const char *param, int def); /* if value is non-positive, return def */
#define param_get_default_number(param, def) (int) param_get_default (param, number_new (def)).num
int param_get_number_min (const char *param, int min); /* if value is positive, return no less than min */
int param_get_number_max (const char *param, int max); /* if value is positive, return no more than max */

/* internal use only */
simtype sim_param_parse_strings (const char **values);
int sim_param_set_error (const char *param, const simtype value, int error);

/* set configuration parameters. permanent is SIM_PARAM_xxx */
int param_set_number (const char *param, int number, int permanent);
int param_set_string (const char *param, const simtype string, int permanent);
int param_set_strings (const char *param, const simtype array, int permanent);
int param_unset (const char *param, int permanent);

/* save configuration parameters file */
int param_save (void);

/* load login configuration parameters file */
int param_init (simbool init);

/* load user configuration parameters file */
int param_open (void);

/* close configuration parameters file (without saving) */
void param_close (void);

/* free configuration parameters */
int param_uninit (void);

#endif
