require 'test/unit/testcase'
require 'amrita/vm'
require 'amrita/parser'
require 'amrita/compiler'
require 'amrita/testsupport'

$DUMP = ENV["DUMP"]
$BENCH = ENV["BENCH"]

class TestVM < Test::Unit::TestCase
  include Amrita

  def setup
    #GC.disable
  end

  def teardown
    #GC.start
  end

  def check_vm1(template, data, answer)
    t = Time.now
    vm = VirtualMachine.new
    node = HtmlParser::parse_text template
    compiler = Compiler.new
    bytecode = node.convert_to_bytecode(compiler)
    yield(vm)
    vm.load_bytecode(bytecode)
    vm.dump_bytecode if $DUMP
    vm.register = data
    vm.go
    assert_equal_node(answer, vm.out)
    t2 = Time.now
    20.times { vm.go } if $BENCH
  rescue 
    print "error happend in "
    puts template
    puts "vm.optimize_bytecode=#{vm.optimize_bytecode} vm.use_compiler=#{vm.use_compiler}"
    raise
  ensure
    if $BENCH
      p vm.opt
      p Time.now - t
      p Time.now - t2
    end
  end

  def check_vm(template, data, answer)
    # debug_mode
    check_vm1(template, data, answer) do |vm|
      vm.debug = true
      vm.debug_out = VirtualMachine::NullOut.new unless $DUMP
    end

    # execute normal
    check_vm1(template, data, answer) do |vm|
      vm.use_compiler = false
    end


    # execute with optimize and compiler
    check_vm1(template, data, answer) do |vm|
      vm.use_compiler = true
    end

    # execute with optimize and compiler
    check_vm1(template, data, answer) do |vm|
      vm.use_compiler = true
      vm.optimize_bytecode = true
    end
  end
  
  def test_noid
    check_vm("\n<html>\n <h1>aaaa</h1><p>xxxx</p></html>\n", 
             true,
             "<html><h1>aaaa</h1><p>xxxx</p></html>")
    check_vm("<html><h1>aaaa</h1><p>xxxx</p></html>", 
             true,
             "<html><h1>aaaa</h1><p>xxxx</p></html>")
  end

  def test_scalar
    check_vm("***@{test1}***", 
             { :test1 => "amrita virtual machine" },
             "***amrita virtual machine***")
    check_vm("<!-- amrita_pragma: ScalarData --><span id=test1></span>", 
             { :test1 => "amrita virtual machine" },
             "amrita virtual machine")
    check_vm('<span id=test1 class="xxx"></span>', 
             { :test1 => "amrita virtual machine" },
             '<span class="xxx">amrita virtual machine</span>')
    check_vm("<div id=test1></div>", 
             { :test1 => "amrita virtual machine" },
             "<div>amrita virtual machine</div>")
    check_vm("aaa <div id=test1></div> bbb", 
             { :test1 => "amrita virtual machine" },
             "aaa <div>amrita virtual machine</div> bbb")

  end

  def test_hash
    check_vm("<span id=test1><span id=xxx></span></span>", 
             { :test1 => { :xxx => "amrita virtual machine" } },
             "amrita virtual machine")
    check_vm("<!-- amrita_pragma: HashData --><span id=test1><!-- amrita_pragma: ScalarData --><span id=aaa></span></span>", 
             { :test1 => { :aaa => "amrita virtual machine" } },
             "amrita virtual machine")
  end

  def test_attrarray
    check_vm("<!-- amrita_pragma: AttrData --><imgxxx id=test1>", 
             { :test1 => a(:src=>"xxx.gif") },
             '<imgxxx src="xxx.gif"></imgxxx>')
    check_vm("<!-- amrita_pragma: AttrData --><a id=test1>aaa</a>", 
             { :test1 => a(:href=>"http://www.walrus-ruby.org/") { "amrita site" } },
             '<a href="http://www.walrus-ruby.org/">amrita site</a>')
    check_vm("<!-- amrita_pragma: AttrData[ScalarData] --><a id=test1>aaa</a>", 
             { :test1 => a(:href=>"http://www.walrus-ruby.org/") { "amrita site" } },
             '<a href="http://www.walrus-ruby.org/">amrita site</a>')
    check_vm("<imgxxx id=test1>", 
             { :test1 => a(:src=>"xxx.gif") },
             '<imgxxx src="xxx.gif"></imgxxx>')
  end

  def test_expand_attr
    check_vm('<xxx id=aaa yyy="@bbb">@{ccc}</xxx>', 
             { :aaa=> { :bbb=>"bbbb" , :ccc=>"cccc"} },
             '<xxx yyy="bbbb">cccc</xxx>')
    check_vm('<a id=site_a href="@url">@{name}</a><a id=site_b href="@url">@{name}</a>', 
             { :site_a => { :name => "amrita" , :url=>"http://www.walrus-ruby.org/" },
               :site_b => { :name => "ruby" , :url=>"http://www.ruby-lang.org/" } },
             '<a href="http://www.walrus-ruby.org/">amrita</a><a href="http://www.ruby-lang.org/">ruby</a>')
    check_vm('<a name="@{label}:@{seq}"><span id="text"></span></a>', 
             { :label=>'label', :seq=>1, :text=>"labeled text" },
             '<a name="label:1">labeled text</a>')
    check_vm('<a name="label:@{seq}"><span id="text"></span></a>', 
             { :seq=>1, :text=>"labeled text" },
             '<a name="label:1">labeled text</a>')
    check_vm('<a name="label:@{seq}_xx"><span id="text"></span></a>', 
             { :seq=>1, :text=>"labeled text" },
             '<a name="label:1_xx">labeled text</a>')
    check_vm('<span id="data"><a name="@number"><span id="text">header</span></a></span>',
             { :data=> { :number=>1, :text=>'aaa' } },
             '<a name="1">aaa</a>')
    check_vm('<x aaa="@aa" bbb="@bb" ccc=cc>', 
             { :aa=>"111", :bb=>"222" },
             '<x bbb="222" aaa="111" ccc="cc">')
    check_vm('<xxx src="@url">', 
             { :url=>"xxx.gif" },
             '<xxx src="xxx.gif">')
    check_vm('<imgxxx src="@url">', 
             { :url=>"xxx.gif" },
             '<imgxxx src="xxx.gif">')
  end

  def test_array
    check_vm("<!-- amrita_pragma: EnumerableData --><li id=test1></li>", 
             { :test1 => [1,2,3] },
             "<li>1</li><li>2</li><li>3</li>")
    check_vm("<li id=test1></li>", 
             { :test1 => [1,2,3] },
             "<li>1</li><li>2</li><li>3</li>")
    check_vm("<!-- amrita_pragma: EnumerableData[ScalarData] --><li id=test1></li>", 
             { :test1 => [1,2,3] },
             "<li>1</li><li>2</li><li>3</li>")
  end
  
  def test_arrayofhash
    data = [1,2,3].collect do |x|
      { :aaa => x , :bbb => x*2}
    end
    check_vm("<!-- amrita_pragma: EnumerableData[HashData] --><li id=test1><span id=aaa></span> <span id=bbb></span></li>", 
             { :test1=>data },
             "<li>1 2</li><li>2 4</li><li>3 6</li>")
    check_vm("<!-- amrita_pragma: EnumerableData --><li id=test1><span id=aaa></span> <span id=bbb></span></li>", 
             { :test1=>data },
             "<li>1 2</li><li>2 4</li><li>3 6</li>")
    check_vm("<li id=test1><span id=aaa></span> <span id=bbb></span></li>", 
             { :test1=>data },
             "<li>1 2</li><li>2 4</li><li>3 6</li>")
    data = [1,2,3].collect do |x|
      { :aaa => x , :bbb => x*2}
    end
    check_vm("<li id=test1><x>xxx</x> \n<span id=aaa></span> <span id=bbb></span></li>", 
             { :test1=>data },
             "<li><x>xxx</x> \n1 2</li><li><x>xxx</x> \n2 4</li><li><x>xxx</x> \n3 6</li>")
  end

  def test_proc
    check_vm("<!-- amrita_pragma: ProcData --><div id=test1 xxx='xxx'>aaa</div>", 
             { 
               :test1 => proc do |reg, stack, out, vm| 
                 reg[:xxx] = reg[:xxx] + 'yyy' 
                 reg
               end
             },
             '<div xxx="xxxyyy">aaa</div>')
    check_vm("<div id=test1 xxx='xxx'>aaa</div>", 
             { 
               :test1 => proc do |reg, stack, out, vm| 
                 reg[:xxx] = reg[:xxx] + 'yyy' 
                 reg
               end
             },
             '<div xxx="xxxyyy">aaa</div>')
  end

  def test_allownil
    check_vm("<!-- amrita_pragma: AllowNil[ScalarData] --><span id=test1></span>", 
             { :test1 => "amrita virtual machine" },
             "amrita virtual machine")
    check_vm("<!-- amrita_pragma: AllowNil[ScalarData] --><span id=test1></span>", 
             { :test1 =>nil},
             "")
    check_vm("<!-- amrita_pragma: AllowNil[EnumerableData] --><span id=test1></span>", 
             { :test1 => 1..5 },
             "12345")
    check_vm("<!-- amrita_pragma: AllowNil[EnumerableData] --><span id=test1></span>", 
             { :test1 =>nil},
             "")
  end

  def test_any
    check_vm("<div id=test1>aaa</div>", 
             { :test1 => a(:xxx=>999) { 'xxx' }},
             '<div xxx="999">xxx</div>')
    check_vm("<div id=test1></div>", 
             { :test1 => 1 },
             "<div>1</div>")
    check_vm("<div id=test1>aaaa</div>", 
             { :test1 => true },
             "<div>aaaa</div>")
    check_vm("<div id=test1>aaaa</div>", 
             { :test1 => false },
             "")
    check_vm("<div id=test1></div>", 
             { :test1 => "1" },
             "<div>1</div>")
    check_vm("<div id=test1></div>", 
             { :test1 => [1,2,3] },
             "<div>1</div><div>2</div><div>3</div>")
    check_vm("<div id=test1></div>", 
             { :test1 => 1..3 },
             "<div>1</div><div>2</div><div>3</div>")
    check_vm("<div id=test1>aaa</div>", 
             { :test1 => a(:xxx=>999) },
             '<div xxx="999">aaa</div>')
  end

  def test_any2
    check_vm("<div id=test1><span id=aaa>aaaa</span></div>", 
             { :test1 => { :aaa=>1...5 } },
             "<div>1234</div>")
    check_vm("<div id=test1><span id=aaa>aaaa</span></div>", 
             { :test1 => { :aaa=>a(:class=>'cls11') { 'xxx' }} },
             '<div><span class="cls11">xxx</span></div>')

    check_vm("<div id=test1><span id=aaa>aaaa</span></div>", 
             { :test1 => { :aaa=>a(:class=>'cls11') } },
             '<div><span class="cls11">aaaa</span></div>')

    check_vm("<div id=test1><span id=aaa>aaaa</span></div>", 
             { :test1 => { :aaa=>nil } },
             "<div></div>")

    check_vm("<div id=test1><span id=aaa>aaaa</span></div>", 
             { :test1 => { :aaa=>a(:class=>'cls11') } },
             '<div><span class="cls11">aaaa</span></div>')
    check_vm("<div id=test1><span id=aaa>aaaa</span></div>", 
             { :test1 => { :aaa=>true } },
             "<div>aaaa</div>")

    check_vm("<div id=test1><span id=aaa>aaaa</span></div>", 
             { :test1 => { :aaa=>true } },
             "<div>aaaa</div>")

    s1 = Struct.new(:test1)
    s2 = Struct.new(:aaa, :bbb)
    d = s2.new('666', '777')
    d.extend Amrita::ExpandByMember
    data = s1.new(d)
    data.extend Amrita::ExpandByMember
    check_vm("<div id=test1><span id=aaa>aaaa</span></div>", 
             data,
             "<div>666</div>")
  end

  def test_any3
    tmpl = <<END
<table border="1">
  <tr id=table1>
    <td><a id="webpage"></a></td>
  </tr>
</table>
END

    data = {
    :table1=>[ 
    { 
          :webpage=> a(:href=>"http://www.ruby-lang.org/") { "Ruby Home Page" },
    },
   ] 
    }
    
    check_vm(tmpl, data,
             '<table border="1">
  <tr>
    <td><a href="http://www.ruby-lang.org/">Ruby Home Page</a></td>
  </tr>
</table>
')
  end

  def test_any4
    tmpl = <<END
<table border="1">
  <tr><th>name</th><th>author</th><th>webpage</tr>
  <tr id=table1>
    <td id="name"></td>
    <td id="author"></td>
    <td><a id="webpage"></a></td>
  </tr>
</table>
END

    data = {
    :table1=>[ 
    { 
      :name=>"Ruby", 
      :author=>"matz" , 
      :webpage=> a(:href=>"http://www.ruby-lang.org/") { "Ruby Home Page" },
    },
    { 
      :name=>"python", 
      :author=>"Guido van Rossum" ,
      :webpage=> a(:href=>"http://www.python.org/") { "Python Language Website" },
    },
    { 
      :name=>"perl", 
      :author=>"Larry Wall" ,
      :webpage=> a(:href=>"http://www.perl.com/") { "Perl.com" },
    },
   ] 
    }
    
    check_vm1(tmpl, data,
             '<table border="1">
  <tr><th>name</th><th>author</th><th>webpage</th></tr>
  <tr>
    <td>Ruby</td>
    <td>matz</td>
    <td><a href="http://www.ruby-lang.org/">Ruby Home Page</a></td>
  </tr><tr>
    <td>python</td>
    <td>Guido van Rossum</td>
    <td><a href="http://www.python.org/">Python Language Website</a></td>
  </tr><tr>
    <td>perl</td>
    <td>Larry Wall</td>
    <td><a href="http://www.perl.com/">Perl.com</a></td>
  </tr>
</table>
')   {   }
  end

  def check_marshal(template, data)
    vm = VirtualMachine.new
    node = HtmlParser::parse_text template
    compiler = Compiler.new
    bytecode = node.convert_to_bytecode(compiler)
    yield(vm) if block_given?
    vm.load_bytecode(bytecode)
    vm.dump_bytecode if $DUMP
    vm.register = data
    vm.out = ""
    vm.go
    answer = vm.out

    stream = Marshal::dump(vm.iset)
    bytecode = Marshal::load(stream)
    yield(vm) if block_given?
    vm.load_bytecode(bytecode)
    vm.debug_out = STDOUT
    vm.dump_bytecode if $DUMP
    vm.register = data
    vm.out = ""
    vm.go
    assert_equal_node(answer, vm.out)
    bytecode
  end
  
  def test_marshal   
    return if RUBY_VERSION > "1.8" # marshal is not supported now with 1.8
    i = ByteCode::NullInstruction[]
    assert_equal(ByteCode::NullInstruction, i.class)

    i = ByteCode::PrintStaticNode[ e(:xx) { "yyy" }]
    stream = Marshal::dump(i)
    ii = Marshal::load(stream)
    assert_equal(ByteCode::PrintStaticNode, ii.class)
    assert_equal(e(:xx) { "yyy" }, ii.node)

    node = HtmlParser::parse_text 'aaa'
    compiler = Compiler.new
    bytecode = node.convert_to_bytecode(compiler)
    vm = VirtualMachine.new
    vm.load_bytecode(bytecode)
    #bytecode.dump(STDOUT)
    stream = Marshal::dump(bytecode)
    ii = Marshal::load(stream)
    assert_equal(ByteCode::PrintStaticNode, ii.class)
    assert_equal_node('aaa' , ii.node)
  end

  def test_marshal2
    return if RUBY_VERSION > "1.8" # marshal is not supported now with 1.8
    check_marshal('<!-- amrita_pragma: ScalarData --><span id=xxx></span>', :xxx=>123) 
    check_marshal('<!-- amrita_pragma: AllowNil[ScalarData] --><span id=xxx></span>', :xxx=>123) 
    check_marshal('<span id=xxx></span>', :xxx=>123) 
  end

  def test_marshal3
    return if RUBY_VERSION > "1.8" # marshal is not supported now with 1.8
    vm = VirtualMachine.new
    bytecode = check_marshal('<span id=xxx></span>', :xxx=>123) do |vm|
      vm.partial_compile = false
    end

    vm.out = "" ; vm.register = { :xxx=>[4,5,6] } ; vm.go
    assert_equal("456", vm.out)

    bytecode = check_marshal('<span id=xxx></span>', :xxx=>123) do |vm|
      vm.use_compiler = true
      vm.partial_compile = false
    end

    vm.out = "" ; vm.register = { :xxx=>[4,5,6] } ; vm.go
    assert_equal("456", vm.out)

    bytecode = check_marshal('<span id=xxx></span>', :xxx=>123) do |vm|
      vm.use_compiler = true
      vm.partial_compile = true
    end

    vm.out = "" ; vm.register = { :xxx=>[4,5,6] } ; vm.go
    assert_equal("456", vm.out)
    vm.out = "" ; vm.register = { :xxx=>proc { |e|'456'} } ; vm.go
    assert_equal("456", vm.out)
    vm.out = "" ; vm.register = { :xxx=>nil } ; vm.go
    assert_equal("", vm.out)
    bytecode = Marshal::load(Marshal::dump(bytecode))
    vm.out = "" ; vm.register = { :xxx=>true } ; vm.go
    assert_equal("", vm.out)
    vm.out = "" ; vm.register = { :xxx=>'abc' } ; vm.go
    bytecode = Marshal::load(Marshal::dump(bytecode))
    assert_equal("abc", vm.out)
    vm.out = "" ; vm.register = { :xxx=>%w(4 5 6) } ; vm.go
    assert_equal("456", vm.out)
  end

end


#--- main program ----
if __FILE__ == $0
  require 'test/unit/ui/console/testrunner'

  if ARGV.size == 0
    Test::Unit::UI::Console::TestRunner.run(TestVM, Test::Unit::UI::Console::TestRunner::VERBOSE)
    require 'amrita/accel'
    Test::Unit::UI::Console::TestRunner.run(TestVM)
  else
    #require 'amrita/accel'
    ARGV.each do |method|
      Test::Unit::UI::Console::TestRunner.run(TestVM.new(method), Test::Unit::UI::Console::TestRunner::VERBOSE)
    end
  end
end

