require 'test/unit/testcase'

require 'amrita/parser'
require 'amrita/testsupport'

class TestParser < Test::Unit::TestCase
  include Amrita

  Tag = HtmlScanner::TagInline

  def scan(text)
    begin
      ret = []
      HtmlScanner.scan_text(text) do | state, token|
	ret << token
      end
    end
    ret
  end

  def parse(text)
    HtmlParser.parse_text(text)
  end

  def test_scanner
    assert_equal(Tag.new("html"), Tag.new("html"))
    assert_equal([Tag.new("em"), "xxx", Tag.new("/em")], scan("<em>xxx</em>"))
    assert_equal([Tag.new("img", [["src", "xxx.gif"]])], scan("<img src=xxx.gif>"))
    assert_equal([Tag.new("a",[["href", "http://www.brain-tokyo.co.jp/"]])],
		 scan('<a href="http://www.brain-tokyo.co.jp/">'))
    h = <<END
<table border="1" align="center" width="95%" bgcolor="#CCFFCC">
  <tr>
     <td>
        <a name="28077">[2:3]    
        <b>mailtoΥƥ</b>
        <dl>
            <dt>1̾ <a href="mailto:tnaka@brain-tokyo.com"></a>   Fri Jan 26 19:00:25 JST 2001<dd><br></dd></dt>
            <dt>2̾ <a href="mailto:">ʤʤ</a>   Fri Jan 26 19:00:33 JST 2001<dd> <br></dd></dt>
            <dt>3̾ <a href="mailto:">ʤʤ</a>   Fri Jan 26 19:00:44 JST 2001<dd>줿ߤ <br>
        </dl>
     </td>
  </tr>
</table>
END
    if RUBY_VERSION > "1.8.0"
      assert_equal(
		 [
		   Tag.new("table", [["border", "1"], ["align", "center"], ["width", "95%"], ["bgcolor", "#CCFFCC"]]), 
		   "\n     ", 
		   Tag.new("a", [["name", "28077"]])
		 ],
		 scan(h).select(0,3,6))
    else
      assert_equal(
		 [
		   Tag.new("table", [["border", "1"], ["align", "center"], ["width", "95%"], ["bgcolor", "#CCFFCC"]]), 
		   "\n     ", 
		   Tag.new("a", [["name", "28077"]])
		 ],
		 scan(h).indexes(0,3,6))
    end
  end

  def parse_and_print(t)
    n = parse(t)
    puts ""
    puts n.to_ruby
    puts "\n------------------------------"
    print M { n  }
    puts "\n------------------------------"
  end

  def test_parser
    assert_equal('e(:p) { "abc" }',parse("<p>abc</p>").to_ruby)

    assert_equal('e(:head) { e(:title) { "abc" } }',
		 parse("<head><title>abc</title></head>").to_ruby)

    assert_equal('e(:html) { [ e(:head) { e(:title) { "abc" } }, e(:body) { e(:p) { "abc" } } ] }',
		 parse("<html><head><title>abc</title></head><body><p>abc</p></body></html>").to_ruby)
    assert_equal_node(e(:x,:a=> "1",:b=>nil) { " " },
		 parse("<x a = 1 b > </x>"))
    assert_equal('e(:span,:id=>"xyz") ',parse("@{xyz}").to_ruby)

    h = <<END
  <tr>
     <td>
        <a name="28077">[2:3]    </a>
        <b>mailtoΥƥ</b>
        <dl>
            <dt>3̾ <a href="mailto:">ʤʤ</a>  Fri Jan 26 19:00:44 JST 2001줿ߤ <br></dt>
            <dt>3̾ <a href="mailto:">ʤʤ</a>  Fri Jan 26 19:00:44 JST 2001줿ߤ <br></dt>
        </dl>
     </td>
  </tr>
</table>
END
    #parse_and_print(h)

    assert_equal_node('<!-- comment -->',
		 parse('<!-- comment -->'))
    assert_equal_node('<!-- comment-with-dash -->',
                 parse('<!-- comment-with-dash -->'))
    assert_equal_node('<!-- <x> -->',
		 parse('<!-- <x> -->'))
#    assert_equal('<?xml version="1.0" encoding="Shift_JIS"?>',
#		 parse('<?xml version="1.0" encoding="Shift_JIS"?>').to_s)
    assert_equal_node('<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">',
		 parse('<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">'))

    assert_equal_node('<meta http-equiv="Content-Type" content="text/html;CHARSET=EUC-JP">',
		 parse('<meta http-equiv="Content-Type" content="text/html;CHARSET=EUC-JP">'))
  end

  def test_parser2
    assert_equal(e(:ul) { e(:li) { "abc" } },parse("<ul><li>abc</li></ul>"))
    assert_equal(e(:ul) { [ e(:li) { "abc" }, e(:li) { "efg" }, e(:li) { "xyz" } ] },
		 parse("<ul><li>abc</li><li>efg</li><li>xyz</li></ul>"))
    assert_equal(e(:ul) { [ e(:li) { "abc" }, e(:li) { "efg" }, e(:li) { "xyz" } ] },
		 parse("<ul><li>abc</li><li>efg</li><li>xyz</ul>"))
    assert_equal(e(:ul) { [ e(:li) { "abc" }, e(:li) { "efg" }, e(:li) { "xyz" } ] },
		 parse("<ul><li>abc<li>efg</li><li>xyz</ul>"))
    assert_equal(e(:ul) { [ e(:li) { "abc" }, e(:li) { "efg" }, e(:li) { "xyz" } ] },
		 parse("<ul><li>abc<li>efg<li>xyz</ul>"))
    assert_equal(Node::to_node([ e(:p) { "a" }, e(:p) { "b" } ]),
		 parse("<p>a<p>b"))
    assert_equal(e(:dl) { [ e(:dt) { "a" }, e(:dd) { "b" }, e(:dd) { "c" } ] },
		 parse("<dl><dt>a<dd>b<dd>c"))

    #assert_equal('[ e(:p) { [ "a", e(:em) { "b" }, "c" ] }, e(:br) , e(:hr) , e(:p) { "x" } ]',

    assert_equal(Node::to_node([ e(:p) { [ "a", e(:em) { "b" }, "c", e(:br)  ] }, e(:hr) , e(:p) { "x" } ]),
		 parse("<p>a<em>b</em>c<br><hr><p>x"))
    assert_equal(e(:table) { e(:tr) { e(:td) { e(:p) { "a" } } } },
		 parse("<table><tr><TD><P>a</table>"))
    assert_equal(e(:table) { e(:tbody) { e(:tr) { e(:td) { e(:p) { "a" } } } } },
		 parse("<table><tbody><tr><TD><P>a</table>"))

    template_text = <<END
<!-- start of template ------------->
<html>
<head>
  <title id="title">title</title>
</head>
<body>
<h1 id="title">title</h1>
  <ul>
  <li id=list> </li>
  </ul>
</body>
</html>
END
      assert_equal(Node::to_node([ special_tag("!--", " start of template -----------") , "\n", 
                   e(:html) { [ "\n", e(:head) { [ "\n  ", e(:title, :id=>"title") { "title" }, "\n" ] }, "\n",  
                   e(:body) { [ "\n", e(:h1, :id=>"title") { "title" }, "\n  ", e(:ul) { [ "\n  ", e(:li,:id=>"list") { " " }, "\n  " ] }, "\n" ] }, "\n" ] }, "\n" ]),
                   parse(template_text))
  end

  def test_parser_error
    assert_raises(HtmlParseError) { parse('<x></x></y>') }
    assert_raises(HtmlParseError) { parse('<x></x></y>') }
    assert_raises(HtmlParseError) { parse('<ul><ul>a</ul><li></ul></ul>')}
    assert_raises(HtmlParseError) { parse('<table><tr></ul><li></ul></ul>')}
    assert_raises(HtmlParseError) { parse('<p>a<a>aa</p><p>') }
  end

  def test_dt
    return 
    template = HtmlParser.parse_inline <<-END
    <dl>
      <dt>1<dd>2<dt>3<dd>4<dt>5<dd>6
    </dl> 
    END

    assert_equal_node(" <dl> <dt>1</dt><dd>2</dd><dt>3</dt><dd>4</dd><dt>5</dt><dd>6 </dd></dl> ", template)
  end


  def test_escaped_char
    assert_equal('<>&"',parse("&lt;&gt;&amp;&quot;").to_s)
  end

  def test_pragma
    p = parse("<!-- amrita_pragm(global): test -->")
    assert_equal(SpecialElement, p.class)

    p = parse("<!-- amrita_pragma: test -->")
    assert_equal(AmritaPragma, p.class)
    assert_equal(:local, p.scope)
    assert_equal("test", p.pragma_body.strip)

    p = parse("<!-- amrita_pragma(global): test -->")
    assert_equal(AmritaPragma, p.class)
    assert_equal(:global, p.scope)
    assert_equal("test", p.pragma_body.strip)

    p = parse("<!-- amrita_pragma(local): test -->")
    assert_equal(AmritaPragma, p.class)
    assert_equal(:local, p.scope)
    assert_equal("test", p.pragma_body.strip)
  end
end


#--- main program ----
if __FILE__ == $0
  require 'test/unit/ui/console/testrunner'

  if ARGV.size == 0
    Test::Unit::UI::Console::TestRunner.run(TestParser)
    require 'amrita/accel'
    Test::Unit::UI::Console::TestRunner.run(TestParser)
  else
    require 'amrita/accel'
    ARGV.each do |method|
      Test::Unit::UI::Console::TestRunner.run(TestParser.new(method))
    end
  end
end

