require 'test/unit'
require 'amrita/node'
require 'amrita/testsupport'

class TestNode < Test::Unit::TestCase
  include Amrita

  def test_attr

    a = Attr.new("key", "value")
    assert_equal("value", a.value)
    assert_equal("key", a.key)
    assert_equal(:key, a.key_symbol)
    assert_equal("value", a.value)
    assert_equal(a, eval(a.to_ruby))

    a = Attr.new(:key, 123)
    assert_equal(:key, a.key_symbol)
    assert_equal("key", a.key)
    assert_equal("123", a.value)

    assert_equal(true, a == a)
    assert_equal(true, a ==  Attr.new(:key, 123))
    assert_equal(true, a ==  Attr.new(:key, "123"))
    assert_equal(true, a ==  Attr.new(:key, 100+23))
    assert_equal(false, a ==  Attr.new(:key, "v"))
    a.value = "v"
    assert_equal(true, a ==  Attr.new(:key, "v"))

    assert_equal(true, a ==  a.dup)
    assert_equal(true, a ==  a.clone)
    x = a.dup
    GC.start
    assert_equal(a, x)
  end

  def test_init
    e = Element.new("x")
    assert_equal("x", e.tag)

    e = Element.new("x", :y=>"z")
    assert_equal("x", e.tag)
    assert_equal("z", e[:y])

    e = Element.new("x", Attr.new(:y,"z"))
    assert_equal("x", e.tag)
    assert_equal("z", e[:y])

    e = Element.new("x", :y=>"z") { "body" }
    assert_equal("x", e.tag)
    assert_equal("z", e[:y])
    assert_equal('"body"', e.body.to_ruby)
  end

  def test_element
    e = Element.new(:x)
    e = Element.new("x")
    assert_equal("x", e.tag)
    e.set_tag(:xx)
    assert_equal("xx", e.tag)
    assert_equal(:xx, e.tag_symbol)
    assert_nil(e.amrita_id)
    e[:id] = "1234"
    assert_equal("1234", e.amrita_id)
    assert_nil(e.tagclass)
    e[:class] = "4567"
    assert_equal("4567", e.tagclass)

    e = (Element.new("x") << Attr.new(:id, 1234) << Attr.new(:class, 4567))
    assert_equal("x", e.tag)
    assert_equal("1234", e.amrita_id)
    assert_equal("4567", e.tagclass)

    assert_equal(false, e.include_attr?(:xxx))
    assert_equal(true, e.include_attr?(:class))

    e.delete_attr!(:class)
    assert_nil(e.tagclass)

    e.set_text("abc")
    assert_equal('e(:x,:id=>"1234") { "abc" }', e.to_ruby)

    assert_equal(false, e.no_child?)
    assert_equal(true, e(:x).no_child?)

    assert(e(:span, :id=>'xxx').simple_span?)
    assert(e(:span, :id=>'xxx'){'yyy'}.simple_span?)
    assert_equal(false, (e(:span, :aaa=>'xxx').simple_span?))
    assert_equal(false, (e(:span, :id=>'xxx'){ e(:x) }.simple_span?))
    assert_equal(false, (e(:span, :id=>'aaa', :aaa=>'xxx').simple_span?))
    assert_equal(false, (e(:div, :id=>'xxx').simple_span?))
  end

  def test_elementarray
    c1 = Element.new("child1")
    c2 = Element.new("child2")
    p = Element.new(:parent) { [ c1, c2] }
    assert_equal('e(:parent) { [ e(:child1) , e(:child2)  ] }', p.to_ruby)
    a = p.body + Element.new("child3")
    assert_equal('[ e(:child1) , e(:child2) , e(:child3)  ]', a.to_ruby)
    assert_equal('e(:parent) { [ e(:child1) , e(:child2)  ] }', p.to_ruby)
    p.body << Element.new("child3")
    assert_equal('e(:parent) { [ e(:child1) , e(:child2) , e(:child3)  ] }', p.to_ruby)

    x = [];  p.each_element { |e| x << e.tag_symbol }
    assert_equal([:parent, :child1, :child2, :child3], x)
  end

  def test_attrarray
    a = a(:xxx=>111)
    assert_equal(1, a.size)
    assert_equal(a, eval(a.to_ruby))
    assert_equal(a, a.dup)
    assert_equal(a, a.clone)
    assert_equal(Null, a.body)

    a = AttrArray.new
    assert_equal(0, a.size)
    a.add(Attr.new(:k, 'val'))
    assert_equal(:k, a[0].key_symbol)
    assert_equal('val', a[0].value)

    a = AttrArray.new (Attr.new(:a, 'aaa'), Attr.new(:b, 'bbb'))
    assert_equal(2, a.size)
    assert_equal(:a, a[0].key_symbol)
    assert_equal('aaa', a[0].value)
    assert_equal(:b, a[1].key_symbol)
    assert_equal('bbb', a[1].value)
    a.each do |x|
      GC.start
    end
    assert_equal(%w(aaa bbb), a.collect {|attr| attr.value} )
    assert_equal('aaa', a.value_by_key(:a))
    assert_equal('bbb', a.value_by_key(:b))

    a = AttrArray.new(:xxx=>123, :yyy=>456)
    a2 = a.clone
    a.attr_by_key(:xxx).value = 789
    assert_equal('789', a.value_by_key(:xxx))
    assert_equal('123', a2.value_by_key(:xxx))
  end

  def test_attrarray2
    a1 = a(:xx=>11)
    assert_equal('a(:xx=>"11")', a1.to_ruby)
    assert_equal('a(:xx=>"11")', eval(a1.to_ruby).to_ruby)
    a1 = a(:xx=>11, :yy=>22)
    assert_equal(2, a1.size())
    a1.each do |a|
      case a.key_symbol
      when :xx
        assert_equal("11", a.value)
      when :yy
        assert_equal("22", a.value)
      else
        raise
      end
    end

    a1 = a(:xx=>11, :yy=>22)
    assert_equal(a(:xx=>"11", :yy=>"22"), eval(a1.to_ruby))

    a1 = a(a(:xx=>11), a(:yy=>22))
    a2 = a(:xx=>11, :yy=>22)
    assert_equal(true, a1 == a1)
    assert_equal(true, a1 == a2)
    assert_equal(a1, a2)

    assert_equal({:xx=>"11", :yy=>"22"}, a1.to_hash)
  end

  def test_has_id_element
    e1 = Element.new(:x) { Element.new(:withid, :id=>"aaa") { "text" } }
    assert_equal(true, e1.has_id_element?)
    e2 = Element.new(:x) { Element.new(:y) { "text" } }
    assert_equal(false, e2.has_id_element?)

    p = Element.new(:parent) { [ e1, e2] }
    assert_equal(true, p.has_id_element?)
    p = Element.new(:parent) { [ e2, e2.clone] }
    assert_equal(false, p.has_id_element?)
    
  end

  def test_equal
    e1 = (Element.new("x") << Attr.new(:id, 1234) << Attr.new(:class, 4567))
    e2 = (Element.new("x") << Attr.new(:class, 4567) << Attr.new(:id, 1234) )
    e3 = e1.clone
    assert_equal(e1, e1)
    assert_equal(true, e1 == e1)
    assert_equal(e1, e2)
    assert_equal(true, e1 == e2)
    assert_equal(true, e3 == e1)
    assert_equal(e1, e3)
    assert_equal(e3, e2)
    assert_equal(true, e3 == e2)

    e1[:id] = 9876
    assert_equal(true, e1 != e2)
    assert_equal(false, e1 == e2)
    e2[:id] = 9876
    assert_equal(false, e1 != e2)
    assert_equal(true, e1 == e2)
    assert_equal(e1, e2)

    e1.set_tag(:y)
    assert_equal(true, e1 != e2)
    assert_equal(false, e1 == e2)
    e2.set_tag(:y)
    assert_equal(false, e1 != e2)
    assert_equal(true, e1 == e2)
    assert_equal(e1, e2)

    e1.init_body { "zzzz" }
    assert_equal(false, e1 == e2)
    assert_equal(true, e1 != e2)
    e2.init_body { "zzzz" }
    assert_equal(true, e1 == e2)
    assert_equal(false, e1 != e2)
    assert_equal(e1, e2)
  end

  def test_copy_on_write
    e1 = (Element.new("x") << Attr.new(:id, 1234) << Attr.new(:class, 4567))
    e2 = (Element.new("x") << Attr.new(:class, 4567) << Attr.new(:id, 1234) )
    assert_equal(false, e1.attrs.shared)
    e3 = e1.clone
    e4 = e3.clone
    assert_equal(true, e1.attrs.shared)
    assert_equal(e3.attrs.id, e1.attrs.id)
    assert_equal(e4.attrs.id, e1.attrs.id)
    e3[:id] = 9876
    assert_equal(true, e1.attrs.shared)
    assert_equal(false, e3.attrs.shared)
    assert(e3.attrs.id != e1.attrs.id)
    assert_equal(e1, e1)
    assert(e1 == e1)
    assert_equal(e1, e2)
    assert(e1 == e2)
    assert(e1 != e3)
    e1[:id] = 9876
    assert(e1 == e3)

    assert_equal(e2, e4)
  end

  def test_children
    assert_equal([], Null.children)
    assert_equal([], Node::to_node("aaa").children)
    assert_equal([], e(:x).children)
    assert_equal([ e(:y) ], e(:x) { e(:y) } .children)
    assert_equal([ TextElement.new("1"), TextElement.new("2"), TextElement.new("3")],
                 Node::to_node([1,2,3]).children)
  end

  def test_element_with_id
    e1 = e(:x, :id=>1)
    e2 = e(:x, :id=>2)
    a = [] ; e1.each_element_with_id { |e| a << e.amrita_id }
    assert_equal([ "1" ], a)
    a = [] ; (e1 + e2 + e1).each_element_with_id { |e| a << e.amrita_id }
    assert_equal([ "1", "2", "1" ], a)
    a = [] ; (e(:x, :id=>1) { e(:y, :id=>2) } ).each_element_with_id { |e| a << e.amrita_id }
    assert_equal([ "1" ], a)
    a = [] ; (e(:x, :id=>1) { e(:y, :id=>2) } ).each_element_with_id(true) { |e| a << e.amrita_id }
    assert_equal([ "1", "2" ], a)

    a = [] ; (e(:x, :id=>1) { e(:y, :id=>2) } * 2 + e1).each_element_with_id(true) { |e| a << e.amrita_id }
    assert_equal([ "1", "2" , "1", "2", "1" ], a)

    a = [] ; (e(:x) { e(:y, :id=>2) } ).each_element_with_id { |e| a << e.amrita_id }
    assert_equal([ "2" ], a)

    e3 = e(:x) { e1 }
    a = [] ; (e3 + e2 + e1).each_element_with_id { |e| a << e.amrita_id }
    assert_equal([ "1", "2", "1" ], a)

    e4 = e(:x) { e3 + e2 + e1 }
    a = [] ; (e4 + e3 + e2 + e1).each_element_with_id { |e| a << e.amrita_id }
    assert_equal([ "1", "2", "1","1", "2", "1" ], a)

  end

  def test_assign
    x = e(:xxx, :id=>"111")
    y = e(:yyy, :id=>"222")
    x[:id] = y[:id] = nil
    assert_equal(e(:xxx), x)
    assert_equal(e(:yyy), y)
  end

  def test_elementprocessor
    ep = ElementProcessor.new
    x = e(:xxx, :id=>"111") 
    assert_equal('<xxx>', ep.format_start_tag(x))
    assert_equal("</xxx>", ep.format_end_tag(x))

    x = e(:xxx, :id=>"111", :aaa=>111, :bbb=>"222") { "yyy" }
    assert_equal_node('<xxx aaa="111" bbb="222">', ep.format_start_tag(x))
    assert_equal('</xxx>', ep.format_end_tag(x))

    x = e(:xxx, :zzz=>"abc") 
    assert_equal('<xxx zzz="abc">', ep.format_start_tag(x))
    x = e(:xxx, :zzz=>"<>&'") 
    assert_equal('<xxx zzz="&lt;&gt;&amp;&#39;">', ep.format_start_tag(x))
    x = e(:a, :href=>"javascript:") 
    assert_equal('<a href="">', ep.format_start_tag(x))
    x = e(:aa, :href=>"javascript:") 
    assert_equal('<aa href="javascript:">', ep.format_start_tag(x))
    x = e(:xxx, :id=>"111", :aaa=>111, :bbb=>"222") { "<>" }
    assert_equal_node('<xxx aaa="111" bbb="222">&lt;&gt;</xxx>', ep.format_node(x))

  end

  def test_elementprocessor2
    ep = ElementProcessor.new
    ep.amrita_id = 'amrita_id'
    x = ep.generate_element(:xxx, :amrita_id=>"111") 
    assert_equal_node(e(:xxx, :id=>"111"), x)
    assert_equal_node('<xxx>', ep.format_node(x))
    assert_equal("111", x.amrita_id)

    x = ep.generate_element(:xxx, :id=>"111") 
    assert_equal_node(e(:xxx, :__id__=>"111"), x)
    assert_equal_node('<xxx id="111">', ep.format_node(x))
    assert_nil(x.amrita_id)

    x = ep.generate_element(:xxx, :id=>"111", :amrita_id=>"222") 
    assert_equal_node(e(:xxx, :__id__=>"111", :id=>"222"), x)
    assert_equal_node('<xxx id="111">', ep.format_node(x))
    assert_equal("222", x.amrita_id)

    ep.amrita_id = nil
    ep.escaped_id = '__id'
    x = ep.generate_element(:xxx, :__id=>"111") 
    assert_equal_node(e(:xxx, :__id__=>"111"), x)
    assert_equal_node('<xxx id="111">', ep.format_node(x))
    x = ep.generate_element(:xxx, :id=>"222") 
    assert_equal_node(e(:xxx, :id=>"222"), x)
    assert_equal("222", x.amrita_id)
    assert_equal_node('<xxx>', ep.format_node(x))

    x = ep.generate_element(:xxx, :__id=>111, :id=>"222") 
    assert_equal_node(e(:xxx, :__id__=>111, :id=>"222"), x)
    assert_equal("222", x.amrita_id)
    assert_equal_node('<xxx id=111>', ep.format_node(x))
  end

  def test_elementprocessor3
    ep = ElementProcessor.new
    ep.delete_id = false
    x = ep.generate_element(:xxx, :id=>"111") 
    assert_equal_node(e(:xxx, :id=>"111"), x)
    assert_equal_node('<xxx id="111">', ep.format_node(x))
    assert_equal("111", x.amrita_id)

    ep.do_copy do
      assert_equal_node(e(:xxx, :id=>"111"), x)
      assert_equal_node('<xxx>', ep.format_node(x))
      assert_equal("111", x.amrita_id)
    end

    ep.delete_id_on_copy = false
    ep.do_copy do
      assert_equal_node(e(:xxx, :id=>"111"), x)
      assert_equal_node('<xxx id="111">', ep.format_node(x))
      assert_equal("111", x.amrita_id)
    end

    ep.amrita_id = 'amrita_id'
    x = ep.generate_element(:xxx, :amrita_id=>"111") 
    assert_equal_node(e(:xxx, :id=>"111"), x)
    assert_equal_node('<xxx amrita_id="111">', ep.format_node(x))
    assert_equal("111", x.amrita_id)
  end
end

#--- main program ----
if __FILE__ == $0
  require 'test/unit/ui/console/testrunner'

  if ARGV.size == 0
    Test::Unit::UI::Console::TestRunner.run(TestNode)
    require 'amrita/accel'
    Test::Unit::UI::Console::TestRunner.run(TestNode)
  else
    require 'amrita/accel'
    ARGV.each do |method|
      Test::Unit::UI::Console::TestRunner.run(TestNode.new(method))
    end
  end
end

