require 'test/unit/testcase'
require 'amrita/vm'
require 'amrita/parser'
require 'amrita/compiler'
require 'amrita/testsupport'

$DUMP = ENV["DUMP"]
$BENCH = ENV["BENCH"]

class TestByteCode < Test::Unit::TestCase
  include Amrita
  def check_instruction(i,  register=nil, stack=[], out="", opts=nil, &assert_block)
    result = ""
    opts = [ {:debug=>true}, {},  { :optimize=>true,:debug=>true },  { :optimize=>true }, 
        { :precompile=>true },{ :precompile=>true, :optimize=>true }]
    opts.each do |opt|
      vm = VirtualMachine.new
      vm.out = result
      vm.load_bytecode(i)
      r, s, o = register, stack.dup, out.dup
      r = vm.iset.execute(r, s, o, vm)
      yield(opt, i, r, s, o)
    end
  end

  def test_null
    check_instruction(ByteCode::NullInstruction[]) do |opt, i,r,s,o|
      assert_equal(nil, r)
      assert_equal([], s)
      assert_equal("", o)
    end
  end

  def test_getdatabykey
    check_instruction(ByteCode::GetDataByKey["xxx"],
                      { :xxx=>1 }
                      ) do |opt, i,r,s,o|
      assert_equal(1, r)
      assert_equal([{ :xxx=>1 }], s)
      assert_equal("", o)
    end
  end

  def test_popdata
    check_instruction(ByteCode::PopData[],
                      nil, [1]
                      ) do |opt, i,r,s,o|
      assert_equal(1, r)
      assert_equal([], s)
      assert_equal("", o)
    end

    check_instruction(ByteCode::PopData[],
                      111, [9,8,7]
                      ) do |opt, i,r,s,o|
      assert_equal(7, r)
      assert_equal([9,8], s)
      assert_equal("", o)
    end
  end

  def test_swapdata
    check_instruction(ByteCode::SwapData[],
                      1, [2]
                      ) do |opt, i,r,s,o|
      assert_equal(2, r)
      assert_equal([1], s)
      assert_equal("", o)
    end
  end

  def test_printstatic
    check_instruction(ByteCode::PrintStaticText[ "xxx" ]
                      ) do |opt, i,r,s,o|
      assert_equal("xxx", o)
    end

    check_instruction(ByteCode::PrintStaticNode[ e(:xx) ]
                      ) do |opt, i,r,s,o|
      assert_equal_node("<xx></xx>", o)
    end

    check_instruction(ByteCode::PrintStaticNode[ e(:xx) { "yyy" }]
                      ) do |opt, i,r,s,o|
      assert_equal("<xx>yyy</xx>", o)
    end
  end

  def test_printregister
    check_instruction(ByteCode::PrintRegister[],
                      1, []
                      ) do |opt, i,r,s,o|
      assert_equal("1", o)
    end

    check_instruction(ByteCode::PrintRegister[],
                      "<>", []
                      ) do |opt, i,r,s,o|
      assert_equal("&lt;&gt;", o)
    end

    check_instruction(ByteCode::PrintRegister[],
                      Amrita::SanitizedString["<>"], []
                      ) do |opt, i,r,s,o|
      assert_equal("<>", o)
    end

    check_instruction(ByteCode::PrintRegister[],
                      e(:x), []
                      ) do |opt, i,r,s,o|
      assert_equal_node("<x></x>", o)
    end
  end

  def test_mergeelement
    check_instruction(ByteCode::MergeElement[e(:x)],
                      { :aaa=>"bbb" }
                      ) do |opt, i,r,s,o|
      assert_equal_node(e(:x, :aaa=>"bbb"), r)
      assert_equal("", o)
    end

    check_instruction(ByteCode::MergeElement[e(:x, :aaa=>"xxx")],
                      { :aaa=>"bbb" }
                      ) do |opt, i,r,s,o|
      assert_equal(e(:x, :aaa=>"bbb"), r)
      assert_equal("", o)
    end

    check_instruction(ByteCode::MergeElement[e(:x, :ccc=>"ddd")],
                      { :aaa=>"bbb" }
                      ) do |opt, i,r,s,o|
      assert_equal(e(:x, :aaa=>"bbb", :ccc=>"ddd"), r)
      assert_equal("", o)
    end
  end

  def test_getdatafromattrarray
    check_instruction(ByteCode::GetDataFromAttrArray[],
                      a(:xx=>"yy")
                      ) do |opt, i,r,s,o|
      assert_equal(Amrita::Null, s[-1])
      assert_equal({:xx=>"yy"}, r)
      assert_equal("", o)
    end

    check_instruction(ByteCode::GetDataFromAttrArray[],
                      a(:xx=>"yy") { "zzz" }
                      ) do |opt, i,r,s,o|
      assert_equal("zzz", s[-1])
      assert_equal({:xx=>"yy"}, r)
      assert_equal("", o)
    end
  end

  def test_extractdata
    check_instruction(ByteCode::ExtractData[ :xxx=>:aaa ],
                      { :aaa=>'111', :bbb=>'222' }
                      ) do |opt, i,r,s,o|
      assert_equal({:xxx=>"111"}, r)
      assert_equal({ :aaa=>'111', :bbb=>'222' }, s[-1])
      assert_equal("", o)
    end
    check_instruction(ByteCode::ExtractData[ :xxx=>[:aaa, "+++", :bbb]],
                      { :aaa=>'111', :bbb=>'222' }
                      ) do |opt, i,r,s,o|
      assert_equal({:xxx=>"111+++222"}, r)
      assert_equal({ :aaa=>'111', :bbb=>'222' }, s[-1])
      assert_equal("", o)
    end
  end

  def test_simplespan
    check_instruction(ByteCode::SimpleSpan[:xx, 'abcde'],
                      { :xx=>'<>&' }
                      ) do |opt, i,r,s,o|
      assert_equal("&lt;&gt;&amp;", o)
    end
    check_instruction(ByteCode::SimpleSpan[:xx, 'abcde'],
                      { :xx=>true }
                      ) do |opt, i,r,s,o|
      assert_equal({:xx=>true}, r)
      assert_equal([], s)
      assert_equal("abcde", o)
    end
    check_instruction(ByteCode::SimpleSpan[:xx, 'abcde'],
                      { :xx=>'12345' }
                      ) do |opt, i,r,s,o|
      assert_equal({:xx=>'12345'}, r)
      assert_equal([], s)
      assert_equal("12345", o)
    end
    check_instruction(ByteCode::SimpleSpan[:xx, 'abcde'],
                      { :xx=>1..5 }
                      ) do |opt, i,r,s,o|
      assert_equal({:xx=>1..5}, r)
      assert_equal([], s)
      assert_equal("12345", o)
    end
    check_instruction(ByteCode::SimpleSpan[:xx, 'abcde'],
                      { :xx=>a(:aaa=>123, :ccc=>456) }
                      ) do |opt, i,r,s,o|
      assert_equal({ :xx=>a(:aaa=>123, :ccc=>456) }, r)
      assert_equal([], s)
      assert_equal_node('<span aaa="123"ccc="456">abcde</span>', o)
    end
    check_instruction(ByteCode::SimpleSpan[:xx, 'abcde'],
                      { :xx=>a(:aaa=>123, :ccc=>456) { 'zzz'} }
                      ) do |opt, i,r,s,o|
      assert_equal_node('<span aaa="123"ccc="456">zzz</span>', o)
    end
    check_instruction(ByteCode::SimpleSpan[:xx, 'abcde'],
                      { :xx=>nil }
                      ) do |opt, i,r,s,o|
      assert_equal("", o)
    end
    check_instruction(ByteCode::SimpleSpan[:xx, 'abcde'],
                      { :xx=>e(:qqq, :rrr=>'ssss') { 'yyy'} }
                      ) do |opt, i,r,s,o|
      assert_equal('<qqq rrr="ssss">yyy</qqq>', o)
    end
    check_instruction(ByteCode::SimpleSpan[:xx, 'abcde'],
                      { :xx=>proc { |elem, *args| elem[:rrr] = 'sss'; elem.set_text('yyy'); elem + elem } }
                      ) do |opt, i,r,s,o|
      assert_equal('<span rrr="sss">yyy</span><span rrr="sss">yyy</span>', o)
    end
    s = Struct.new(:xx, :yy)
    data = s.new(1,2)
    check_instruction(ByteCode::SimpleSpan[:xx, 'abcde'],
                      data
                      ) do |opt, i,r,s,o|
      assert_equal("1", o)
    end
  end    

  def test_sequence
    i = ByteCode::Sequence[
      ByteCode::PrintStaticText[ "xxx" ],
      ByteCode::PrintStaticText[ "yyy" ],
      ByteCode::PrintStaticText[ "zzz" ]
    ]
    check_instruction(i) do |opt, i,r,s,o|
      assert_equal("xxxyyyzzz", o)
    end
  end

  def test_sequence2
    i = ByteCode::Sequence[]
    i << ByteCode::GetDataFromAttrArray[]
    i << ByteCode::MergeElement[ e(:x) ]
    i << ByteCode::PrintRegister[]
    check_instruction(i, a(:yyy=>123, :zzz=>456)) do |opt, i,r,s,o|
      #assert_equal('<x zzz="456" yyy="123"></x>', o)
      assert_equal_node('<x yyy="123" zzz="456"></x>', o)
    end

    i = ByteCode::Sequence[
      ByteCode::GetDataFromAttrArray[],
      ByteCode::MergeElement[ e(:x) ],
      ByteCode::PrintRegister[]
    ]
    check_instruction(i, a(:yyy=>123, :zzz=>456)) do |opt, i,r,s,o|
      #assert_equal('<x zzz="456" yyy="123"></x>', o)
      assert_equal_node('<x yyy="123" zzz="456"></x>', o)
    end
  end

  def test_printdynamicelement
    i = ByteCode::Sequence[
      ByteCode::GetDataFromAttrArray[],
      ByteCode::MergeElement[ e(:x) ],
      ByteCode::PrintDynamicElement[
        ByteCode::PrintStaticText[ "abcd" ]
      ]
    ]
    check_instruction(i, a(:yyy=>123, :zzz=>456)) do |opt, i,r,s,o|
       assert_equal_node('<x yyy="123" zzz="456">abcd</x>', o)
    end
  end

  def test_loop
    i = ByteCode::Loop[
      ByteCode::Sequence[
        ByteCode::PrintRegister[],
        ByteCode::PrintStaticText[ "abcd" ]
      ]
    ]
    check_instruction(i, [1,2,3]) do |opt, i,r,s,o|
      assert_equal('1abcd2abcd3abcd', o)
    end
    check_instruction(i, 1..3) do |opt, i,r,s,o|
      assert_equal('1abcd2abcd3abcd', o)
    end
  end

  def test_selectbytype
    i = ByteCode::SelectByType[]
    i[NilClass] =  ByteCode::PrintStaticText[ "abcd" ]
    i.else_ = ByteCode::PrintStaticText[ "efgh" ]

    # i.dump(STDOUT) 
    check_instruction(i) do |opt, i,r,s,o|
      assert_equal('abcd', o)
    end

    check_instruction(i, true) do |opt, i,r,s,o|
      assert_equal('efgh', o)
    end
  end
end

#--- main program ----
if __FILE__ == $0
  require 'test/unit/ui/console/testrunner'

  if ARGV.size == 0
    Test::Unit::UI::Console::TestRunner.run(TestByteCode, Test::Unit::UI::Console::TestRunner::VERBOSE)
    require 'amrita/accel'
    Test::Unit::UI::Console::TestRunner.run(TestByteCode)
  else
    require 'amrita/accel'
    ARGV.each do |method|
      Test::Unit::UI::Console::TestRunner.run(TestByteCode.new(method), Test::Unit::UI::Console::TestRunner::VERBOSE)
    end
  end
end
