/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin.tags.beans;

import shohaku.core.lang.Eval;
import shohaku.core.lang.ObjectCreationProxy;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.tags.AbstractValueTag;
import shohaku.ginkgo.type.ExpressionValue;
import shohaku.sugina.dynbind.BindArgumentDesc;
import shohaku.sugina.dynbind.BindArgumentRule;

/**
 * 引数を拡張された規則で束縛するデータの値タグを提供します。shohaku.sugina.binder.BindArgumentDesc 型のデータを生成します。<br>
 * 子のエレメントが定義されない場合は、テキストを組み込み式として解析します、デフォルトではOGDL式が使用されます。
 */
public class BindArgumentDescTag extends AbstractValueTag {

    /* null と未設定を分別する為の参照識別子 */
    private static final Object NULL = new Object();

    /* 単一の要素データを格納する */
    private Object singleton = NULL;

    /* テキストから自動変換されたデータを格納する */
    private ExpressionValue textValue = null;

    /* 引数のデータ型 */
    private Class type = null;

    private boolean isFinal = false;

    /**
     * 値を生成して返却します。
     * 
     * @return 生成された値
     */
    protected Object generateValue() {

        // argument name
        final String argumentName = getName();
        if (Eval.isBlank(argumentName)) {
            throw new GinkgoException("name is blank." + argumentName);
        }

        // 双方未設定またはテキストか要素（相互排他の為、必ず一方のみ）
        Object argumentValue = (textValue != null) ? textValue.getResultValue() : ((singleton != NULL) ? singleton : null);

        boolean isDefaultNull = (singleton == NULL && textValue == null);

        // get argument type
        Class argumentType = type;
        if (argumentType == null) {
            // get argument type
            if (argumentValue == null) {
                argumentType = Object.class;
            } else if (argumentValue instanceof ObjectCreationProxy) {
                argumentType = ((ObjectCreationProxy) argumentValue).getInstanceType();
            } else {
                argumentType = argumentValue.getClass();
            }
        }

        // get update rule
        BindArgumentRule updateRule = BindArgumentRule.FINAL;
        if (isFinal && isDefaultNull) {
            throw new GinkgoException("default value is empty");// final の場合はデフォルト値必須
        }
        if (isFinal) {
            updateRule = BindArgumentRule.FINAL;
        } else {
            if (!isDefaultNull) {
                updateRule = BindArgumentRule.OVERWRITE;
            } else {
                updateRule = BindArgumentRule.REQUIRED;
            }
        }

        // generate BindArgumentDesc
        return new BindArgumentDesc(argumentType, argumentValue, argumentName, updateRule);

    }

    /**
     * final 属性を格納します。
     * 
     * @param isFinal
     *            final属性
     */
    public void setFinal(boolean isFinal) {
        this.isFinal = isFinal;
    }

    /**
     * type を格納します。
     * 
     * @param type
     *            type属性
     */
    public void setType(Class type) {
        this.type = type;
    }

    /**
     * type 属性を返却します。
     * 
     * @return type 属性
     */
    public boolean getFinal() {
        return this.isFinal;
    }

    /**
     * type 属性を返却します。
     * 
     * @return type 属性
     */
    public Class getType() {
        return this.type;
    }

    /**
     * 組み込み式としてテキストを変換して格納します。
     * 
     * @param textValue
     *            変換されたテキスト
     */
    public void setTextTransferValue(ExpressionValue textValue) {
        this.textValue = textValue;
    }

    /**
     * エレメントを追加します。
     * 
     * @param element
     *            エレメント
     */
    public void addElementTransferValue(Object element) {
        if (singleton == NULL) {
            this.singleton = element;
        } else {
            throw new GinkgoException("element size is one.");
        }
    }

}
