/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin.resourceset;

import java.util.AbstractCollection;
import java.util.AbstractMap;
import java.util.AbstractSet;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import shohaku.core.collections.decorator.UnmodifiableIterator;
import shohaku.shoin.ResourceSet;

/**
 * リソース集合の変更不可能なマップ表現を提供します。
 */
public class ResourceSetMapView extends AbstractMap {

    /* 基のリソース集合。 */
    final ResourceSet resourceSet;

    /**
     * リソース集合を保管して初期化します。
     * 
     * @param resourceSet
     *            リソース集合
     */
    public ResourceSetMapView(ResourceSet resourceSet) {
        this.resourceSet = resourceSet;
    }

    /**
     * 基のリソース集合を返却します。
     * 
     * @return 基のリソース集合
     */
    protected ResourceSet getResourceSet() {
        return resourceSet;
    }

    public boolean containsKey(Object key) {
        return resourceSet.containsKey(key);
    }

    public Object get(Object key) {
        return resourceSet.getObject(key);
    }

    public boolean isEmpty() {
        return (resourceSet.size() == 0);
    }

    public int size() {
        return resourceSet.size();
    }

    /*
     * view
     */

    transient volatile Set keySet = null;

    transient volatile Set entrySet = null;

    transient volatile Collection values = null;

    public Set keySet() {
        Set e = keySet;
        return (e != null) ? e : (keySet = new KeySet());
    }

    public Set entrySet() {
        Set e = entrySet;
        return (e != null) ? e : (entrySet = new EntrySet());
    }

    public Collection values() {
        Collection c = values;
        return (c != null) ? c : (values = new Values());
    }

    /*
     * Unsupported
     */

    public void clear() {
        throw new UnsupportedOperationException();
    }

    public void putAll(Map m) {
        throw new UnsupportedOperationException();
    }

    public Object remove(Object key) {
        throw new UnsupportedOperationException();
    }

    /*
     * class
     */

    /**
     * keySet()
     */
    class KeySet extends AbstractSet {
        public Iterator iterator() {
            return resourceSet.getKeys();
        }

        public int size() {
            return ResourceSetMapView.this.size();
        }
    }

    /**
     * values()
     */
    class Values extends AbstractCollection {
        public Iterator iterator() {
            return new Valuesterator(resourceSet.getKeys());
        }

        public int size() {
            return ResourceSetMapView.this.size();
        }
    }

    /**
     * values().Iterator()
     */
    class Valuesterator extends UnmodifiableIterator {
        protected Valuesterator(Iterator i) {
            super(i);
        }

        public Object next() {
            return ResourceSetMapView.this.get(i.next());
        }
    }

    /**
     * entrySet()
     */
    class EntrySet extends AbstractSet {
        public Iterator iterator() {
            return new EntryIterator(resourceSet.getKeys());
        }

        public int size() {
            return ResourceSetMapView.this.size();
        }
    }

    /**
     * entrySet() element
     */
    class MapEntry implements Map.Entry {
        protected final Object key;

        protected MapEntry(Object key) {
            this.key = key;
        }

        public Object getKey() {
            return key;
        }

        public Object getValue() {
            return ResourceSetMapView.this.get(key);
        }

        public Object setValue(Object value) {
            throw new UnsupportedOperationException();
        }
    }

    /**
     * entrySet().Iterator()
     */
    class EntryIterator extends UnmodifiableIterator {
        protected EntryIterator(Iterator i) {
            super(i);
        }

        public boolean hasNext() {
            return i.hasNext();
        }

        public Object next() {
            return new MapEntry(i.next());
        }
    }

}
