/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.shoin.resourceset;

import java.io.File;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.URI;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.regex.Pattern;

import shohaku.shoin.ResourceSet;

/**
 * 多様な型でリソースを取得する機能を持つラッパーオブジェクトを提供します。
 */
public class MultiTypeResourceSet implements ResourceSet {

    /** 保管するリソース集合。 */
    protected ResourceSet lookup;

    /**
     * デフォルトコンストラクタ。
     */
    public MultiTypeResourceSet() {
        super();
    }

    /**
     * リソース集合を保管して初期化します。
     * 
     * @param resources
     *            リソース集合
     */
    public MultiTypeResourceSet(ResourceSet resources) {
        setResourceSet(resources);
    }

    /**
     * 基のリソース集合を返却します。
     * 
     * @return 基のリソース集合
     */
    public ResourceSet getResourceSet() {
        return lookup;
    }

    /**
     * 基のリソース集合を格納します。
     * 
     * @param resources
     *            基のリソース集合
     */
    public void setResourceSet(ResourceSet resources) {
        if (resources == null) {
            throw new NullPointerException("ShoinResources as null");
        }
        lookup = resources;
    }

    public Iterator getKeys() {
        return lookup.getKeys();
    }

    public Object getObject(Object key) {
        return lookup.getObject(key);
    }

    public Object getObject(Object key, Object defaultValue) {
        return lookup.getObject(key, defaultValue);
    }

    public boolean containsKey(Object key) {
        return lookup.containsKey(key);
    }

    public int size() {
        return lookup.size();
    }

    public Map getMapView() {
        return lookup.getMapView();
    }

    /**
     * リソースの文字列表現(Object#toString())を取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値の文字列表現
     * @throws NullPointerException
     *             key が null の場合
     */
    public String getToString(Object key) {
        Object o = getObject(key);
        if (o == null) {
            return null;
        }
        return o.toString();
    }

    /**
     * リソースの文字列表現(Object#toString())を取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値の文字列表現
     * @throws NullPointerException
     *             key が null の場合
     */
    public String getToString(Object key, Object defaultValue) {
        Object o = getObject(key, defaultValue);
        if (o == null) {
            return null;
        }
        return o.toString();
    }

    /**
     * リソースを String 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public String getString(Object key) {
        return (String) getObject(key);
    }

    /**
     * リソースを String 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public String getString(Object key, Object defaultValue) {
        return (String) getObject(key, defaultValue);
    }

    /**
     * リソースを Boolean 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Boolean getBoolean(Object key) {
        return (Boolean) getObject(key);
    }

    /**
     * リソースを Boolean 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Boolean getBoolean(Object key, Object defaultValue) {
        return (Boolean) getObject(key, defaultValue);
    }

    /**
     * リソースを Character 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Character getCharacter(Object key) {
        return (Character) getObject(key);
    }

    /**
     * リソースを Character 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Character getCharacter(Object key, Object defaultValue) {
        return (Character) getObject(key, defaultValue);
    }

    /**
     * リソースを Number 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Number getNumber(Object key) {
        return (Number) getObject(key);
    }

    /**
     * リソースを Number 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Number getNumber(Object key, Object defaultValue) {
        return (Number) getObject(key, defaultValue);
    }

    /**
     * リソースを Byte 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Byte getByte(Object key) {
        return (Byte) getObject(key);
    }

    /**
     * リソースを Byte 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Byte getByte(Object key, Object defaultValue) {
        return (Byte) getObject(key, defaultValue);
    }

    /**
     * リソースを Short 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Short getShort(Object key) {
        return (Short) getObject(key);
    }

    /**
     * リソースを Short 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Short getShort(Object key, Object defaultValue) {
        return (Short) getObject(key, defaultValue);
    }

    /**
     * リソースを Integer 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Integer getInteger(Object key) {
        return (Integer) getObject(key);
    }

    /**
     * リソースを Integer 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Integer getInteger(Object key, Object defaultValue) {
        return (Integer) getObject(key, defaultValue);
    }

    /**
     * リソースを Long 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Long getLong(Object key) {
        return (Long) getObject(key);
    }

    /**
     * リソースを Long 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Long getLong(Object key, Object defaultValue) {
        return (Long) getObject(key, defaultValue);
    }

    /**
     * リソースを Float 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Float getFloat(Object key) {
        return (Float) getObject(key);
    }

    /**
     * リソースを Float 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Float getFloat(Object key, Object defaultValue) {
        return (Float) getObject(key, defaultValue);
    }

    /**
     * リソースを Double 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Double getDouble(Object key) {
        return (Double) getObject(key);
    }

    /**
     * リソースを Double 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Double getDouble(Object key, Object defaultValue) {
        return (Double) getObject(key, defaultValue);
    }

    /**
     * リソースを BigInteger 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public BigInteger getBigInteger(Object key) {
        return (BigInteger) getObject(key);
    }

    /**
     * リソースを BigInteger 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public BigInteger getBigInteger(Object key, Object defaultValue) {
        return (BigInteger) getObject(key, defaultValue);
    }

    /**
     * リソースを BigDecimal 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public BigDecimal getBigDecimal(Object key) {
        return (BigDecimal) getObject(key);
    }

    /**
     * リソースを BigDecimal 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public BigDecimal getBigDecimal(Object key, Object defaultValue) {
        return (BigDecimal) getObject(key, defaultValue);
    }

    /**
     * リソースを java.util.Date 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public java.util.Date getDate(Object key) {
        return (java.util.Date) getObject(key);
    }

    /**
     * リソースを java.util.Date 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public java.util.Date getDate(Object key, Object defaultValue) {
        return (java.util.Date) getObject(key, defaultValue);
    }

    /**
     * リソースを java.io.File 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public File getFile(Object key) {
        return (File) getObject(key);
    }

    /**
     * リソースを java.io.File 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public File getFile(Object key, Object defaultValue) {
        return (File) getObject(key, defaultValue);
    }

    /**
     * リソースを java.util.regex.Pattern 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Pattern getRegex(Object key) {
        return (Pattern) getObject(key);
    }

    /**
     * リソースを java.util.regex.Pattern 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Pattern getRegex(Object key, Object defaultValue) {
        return (Pattern) getObject(key, defaultValue);
    }

    /**
     * リソースを java.net.URI 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public URI getURI(Object key) {
        return (URI) getObject(key);
    }

    /**
     * リソースを java.net.URI 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public URI getURI(Object key, Object defaultValue) {
        return (URI) getObject(key, defaultValue);
    }

    /**
     * リソースを java.util.Properties 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Properties getProperties(Object key) {
        return (Properties) getObject(key);
    }

    /**
     * リソースを java.util.Properties 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Properties getProperties(Object key, Object defaultValue) {
        return (Properties) getObject(key, defaultValue);
    }

    /**
     * リソースを java.util.Collection 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Collection getCollection(Object key) {
        return (Collection) getObject(key);
    }

    /**
     * リソースを java.util.Collection 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Collection getCollection(Object key, Object defaultValue) {
        return (Collection) getObject(key, defaultValue);
    }

    /**
     * 配列型で定義された値を java.lang.Object 型で取得します。 <br>
     * 取得された値が配列型でない場合 ClassCastException が発生します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が配列型でない場合に
     */
    public Object getArrays(Object key) {
        Object o = getObject(key);
        if (o == null) {
            return null;
        }
        if (o.getClass().isArray()) {
            return o;
        } else {
            throw new ClassCastException("Not Array Object");
        }

    }

    /**
     * 配列型で定義された値を java.lang.Object 型で取得します。 <br>
     * 取得された値が配列型でない場合 ClassCastException が発生します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が配列型でない場合に
     */
    public Object getArrays(Object key, Object defaultValue) {
        Object o = getArrays(key);
        if (o == null) {
            o = defaultValue;
        }
        return o;
    }

    /**
     * リソースを java.util.List 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public List getList(Object key) {
        return (List) getObject(key);
    }

    /**
     * リソースを java.util.List 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public List getList(Object key, Object defaultValue) {
        return (List) getObject(key, defaultValue);
    }

    /**
     * リソースを java.util.Set 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Set getSet(Object key) {
        return (Set) getObject(key);
    }

    /**
     * リソースを java.util.Set 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Set getSet(Object key, Object defaultValue) {
        return (Set) getObject(key, defaultValue);
    }

    /**
     * リソースを java.util.Map 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Map getMap(Object key) {
        return (Map) getObject(key);
    }

    /**
     * リソースを java.util.Map 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Map getMap(Object key, Object defaultValue) {
        return (Map) getObject(key, defaultValue);
    }

    /**
     * クラス型で定義された値を java.lang.Class 型で取得します。 <br>
     * 指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            キー
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Class getType(Object key) {
        return (Class) getObject(key);
    }

    /**
     * クラス型で定義された値を java.lang.Class 型で取得します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Class getType(Object key, Object defaultValue) {
        return (Class) getObject(key, defaultValue);
    }

    /**
     * ラップされたリソース集合の toString() へ転送します。
     * 
     * @return 文字列表現
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return lookup.toString();
    }

}
