/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin.resourceset;

import java.util.Iterator;
import java.util.Map;

import shohaku.shoin.ResourceSet;

/**
 * 複数のリソース集合を束ねるリソース集合を提供します。
 */
public class CompositeResourceSet implements ResourceSet {

    /* 基のリソース集合の配列 */
    private final ResourceSet[] resourceSets;

    /**
     * リソース集合の配列を格納して初期化します。
     * 
     * @param resourceSets
     *            リソース集合の配列
     */
    public CompositeResourceSet(ResourceSet[] resourceSets) {
        this.resourceSets = (ResourceSet[]) resourceSets.clone();
    }

    public Iterator getKeys() {
        return new CompositeResourceSetIterator(resourceSets);
    }

    public Object getObject(Object key) {
        if (key == null) {
            throw new NullPointerException("resource key is null");
        }
        for (int i = (resourceSets.length - 1); i >= 0; i--) {
            if (resourceSets[i].containsKey(key)) {
                return resourceSets[i].getObject(key);
            }
        }
        return null;
    }

    public Object getObject(Object key, Object defaultValue) {
        if (key == null) {
            throw new NullPointerException("resource key is null");
        }
        Object o = getObject(key);
        if (o != null) {
            o = defaultValue;
        }
        return o;
    }

    public boolean containsKey(Object key) {
        if (key == null) {
            throw new NullPointerException("resource key is null");
        }
        for (int i = (resourceSets.length - 1); i >= 0; i--) {
            if (resourceSets[i].containsKey(key)) {
                return true;
            }
        }
        return false;
    }

    public int size() {
        int size = 0;
        for (int i = 0; i < resourceSets.length; i++) {
            size += resourceSets[i].size();
        }
        return size;
    }

    public Map getMapView() {
        return new ResourceSetMapView(this);
    }

    /**
     * 基のリソース集合の配列を返却します。
     * 
     * @return 基のリソース集合の配列
     */
    public ResourceSet[] getResourceSets() {
        return (ResourceSet[]) resourceSets.clone();
    }

    /**
     * 全ての要素からマップ書式で文字列を構築して返却します。
     * 
     * @return 文字列表現
     * @see java.lang.Object#toString()
     */
    public String toString() {
        StringBuffer buf = new StringBuffer();
        buf.append("{");

        Iterator i = getKeys();
        boolean hasNext = i.hasNext();
        while (hasNext) {
            Object key = (i.next());
            Object value = getObject(key);
            buf.append(key);
            buf.append('=');
            buf.append(value);
            hasNext = i.hasNext();
            if (hasNext) {
                buf.append(", ");
            }
        }

        buf.append("}");
        return buf.toString();
    }

}
