/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin.proxy;

import shohaku.shoin.ResourceSet;
import shohaku.shoin.ResourceSetCreationException;

/**
 * 指定されている時間が経過していると新たなリソース集合を返却する生成プロキシを提供します。
 */
public class TimerResourceSetFactoryProxy extends AbstractResourceSetFactoryProxy {

    /* 更新する間隔をミリ秒 */
    private long periodTimeMillis;

    /* リソースの最終更新時刻 */
    private long lastModified;

    /**
     * プロパティを初期値で初期化します。
     */
    public TimerResourceSetFactoryProxy() {
        super();
    }

    /**
     * 指定されている時間が経過していると新たなリソース集合を返却します。<br>
     * ただし初回は経過時間に関わりなく、新たなリソース集合を生成します。<br>
     * create フラグが true が指定されている場合も新規に生成されます。
     * 
     * @param create
     *            true が指定されている場合は新規に生成する。
     * @return リソース集合
     * @throws ResourceSetCreationException
     *             リソース集合の生成に失敗した場合
     * @see shohaku.shoin.ResourceSetFactoryProxy#getResourceSet(boolean)
     */
    public ResourceSet getResourceSet(boolean create) throws ResourceSetCreationException {
        synchronized (this) {
            if (create || resourceSet == null || isModified()) {
                return (resourceSet = createResourceSet());
            }
            return resourceSet;
        }
    }

    private boolean isModified() {
        return System.currentTimeMillis() < (lastModified + periodTimeMillis);
    }

    /**
     * 更新する間隔をミリ秒を格納します。
     * 
     * @return 更新する間隔のミリ秒
     */
    public long getPeriodTimeMillis() {
        synchronized (this) {
            return periodTimeMillis;
        }
    }

    /**
     * 更新する間隔をミリ秒を返却します。
     * 
     * @param periodTimeMillis
     *            更新する間隔のミリ秒
     */
    public void setPeriodTimeMillis(long periodTimeMillis) {
        synchronized (this) {
            this.periodTimeMillis = periodTimeMillis;
        }
    }

    /**
     * 最終更新時刻を格納します。
     * 
     * @return 最終更新時刻
     */
    public long getLastModified() {
        synchronized (this) {
            return lastModified;
        }
    }

    /**
     * 最終更新時刻を格納します。
     * 
     * @param lastModified
     *            最終更新時刻
     */
    public void setLastModified(long lastModified) {
        synchronized (this) {
            this.lastModified = lastModified;
        }
    }

}
