/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin.proxy;

import shohaku.shoin.ObjectResourceSetFactory;
import shohaku.shoin.ResourceSet;
import shohaku.shoin.ResourceSetCreationException;
import shohaku.shoin.ResourceSetFactoryProxy;

/**
 * Java オブジェクトからリソース集合を生成するプロキシの抽象実装を提供します。
 */
public class AbstractObjectResourceSetFactoryProxy implements ResourceSetFactoryProxy {

    /** リソース集合のキャッシュ */
    protected ResourceSet resourceSet;

    /* リソース集合生成機能 */
    private ObjectResourceSetFactory objectResourceSetFactory;

    /**
     * プロパティを初期値で初期化します。
     */
    public AbstractObjectResourceSetFactoryProxy() {
        super();
    }

    /**
     * create フラグが true の場合以外は、常に同一のリソース集合を返却します。
     * 
     * @param create
     *            再生成を要求する場合は true
     * @return リソース集合
     * @throws ResourceSetCreationException
     *             リソース集合の生成に失敗した場合
     * @see shohaku.shoin.ResourceSetFactoryProxy#getResourceSet(boolean)
     */
    public ResourceSet getResourceSet(boolean create) throws ResourceSetCreationException {
        synchronized (this) {
            if (create || resourceSet == null) {
                return (resourceSet = createResourceSet());
            }
            return resourceSet;
        }
    }

    /**
     * リソース集合を生成して返却します。拡張ポイントです。
     * 
     * @return 生成されたリソース集合
     * @throws ResourceSetCreationException
     *             リソース集合の生成に失敗した場合
     */
    protected ResourceSet createResourceSet() throws ResourceSetCreationException {
        return objectResourceSetFactory.getResourceSet();
    }

    /**
     * リソース集合生成機能を返却します。
     * 
     * @return リソース集合生成機能
     */
    public ObjectResourceSetFactory getObjectResourceSetFactory() {
        synchronized (this) {
            return objectResourceSetFactory;
        }
    }

    /**
     * リソース集合生成機能を格納します。
     * 
     * @param factory
     *            リソース集合生成機能
     */
    public void setObjectResourceSetFactory(ObjectResourceSetFactory factory) {
        synchronized (this) {
            this.objectResourceSetFactory = factory;
            this.resourceSet = null;
        }
    }

}
