/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.shoin.factory;

import java.util.Iterator;
import java.util.Map;

import shohaku.core.lang.Concat;
import shohaku.core.lang.Eval;
import shohaku.ginkgo.ContainTag;
import shohaku.ginkgo.TagNode;
import shohaku.ginkgo.ValueNode;
import shohaku.shoin.ResourceKeyPrefixSupport;

/**
 * XMLで定義された、’/’ で区切られた階層名を持つプロパティセットを生成するファクトリを提供します。<br>
 * <p>
 * 例：
 * 
 * <pre>
 * &lt;xml-properties&gt;
 *     &lt;content&gt;         
 *         &lt;byte name="byte"&gt;-123&lt;/byte&gt;
 *         &lt;double name="double"&gt;+1,521,414.012411&lt;/double&gt;
 *         &lt;int name="int"&gt;100&lt;/int&gt;
 *         &lt;string name="string"&gt;文字列&lt;/string&gt;
 *         &lt;ns name="foo"&gt;
 *             &lt;list id="season" desc="四季"&gt;
 *                 &lt;string&gt;春&lt;/ref&gt;
 *                 &lt;string&gt;夏&lt;/ref&gt;
 *                 &lt;string&gt;秋&lt;/ref&gt;
 *                 &lt;string&gt;冬&lt;/ref&gt;
 *             &lt;/list&gt;
 *         &lt;/ns"&gt;
 *     &lt;/content&gt;
 * &lt;/xml-properties&gt;
 * </pre>
 * 
 * は以下の様に '/' で区切られた名前階層パスにマッピングされ生成されます。 <br>
 * 
 * <pre>
 * /byte   = -123
 * /double = 1521414.012411
 * /int    = 100
 * /string = "文字列"
 * /foo/season = ["春", "夏", "秋", "冬"]
 * </pre>
 * 
 * 有効な階層の深さは定義ファイルで変更出来ますが、デフォルトは４階層です。<br>
 * 
 * <pre>
 * 一般的なファイルの読み取りとコンポーネント取得の手順は以下の様に為ります。 
 * XMLProperties factory = new XMLProperties();
 * IOResource inres = FeatureFactory.getLoader().getIOResource("properties.xml");
 * factory.setIOResources(new IOResource[] { inres });
 * 
 * ResourceSet resources = factory.getResourceSet();
 *
 * List list = (List) resources.getObject("/foo/season");
 * System.out.println("リスト=" + list.toString());
 *
 * ＞＞リスト=["春", "夏", "秋", "冬"]
 * </pre>
 */
public class XMLProperties extends AbstractGinkgoResourceSetFactory implements ResourceKeyPrefixSupport {

    /* ResourceKeyPrefixSupport Mix-in */
    private final ResourceKeyPrefixMixIn prefixsMixIn = new ResourceKeyPrefixMixIn();

    /**
     * プロパティを初期値で初期化します。
     */
    public XMLProperties() {
        super();
    }

    /* プロパティの初期化および登録をします。 */
    protected void initValues(Map lookup, TagNode root, int index) {
        final String prefix = prefixsMixIn.findPrefix(index);
        Iterator i = root.getTagContext().elementIterator("content");
        while (i.hasNext()) {
            TagNode content = (TagNode) i.next();
            initValues(lookup, prefix, content, content);
            break;
        }
    }

    /* 再帰的にノードを読み取り値を登録します。 */
    private void initValues(Map lookup, String prefix, TagNode root, TagNode content) {

        // 子のノードを全登録
        Iterator i = content.getTagContext().ownerIterator();
        while (i.hasNext()) {
            TagNode t = (TagNode) i.next();
            if (t instanceof ContainTag) {
                // 再起呼出
                initValues(lookup, prefix, root, t);

            } else if (t instanceof ValueNode) {
                String path = getPath(root, t);
                if (path != null) {
                    saveValue(lookup, prefix, path, (ValueNode) t);
                }
            }

        }
    }

    /* 値を登録します。 */
    private void saveValue(Map lookup, String prefix, Object path, ValueNode node) {
        lookup.put(Concat.get(prefix, path), node.getNodeValue());
    }

    /* このプロパティのパスを返す。 */
    private String getPath(TagNode root, TagNode tag) {

        // 名前階層へ参加する情報は親情報から再帰処理で構築する
        StringBuffer sb = new StringBuffer(80);
        TagNode n = tag;
        while (n != root && n != null) {
            String name = n.getName();
            if (Eval.isBlank(name)) {
                return null;
            }
            sb.insert(0, name);
            sb.insert(0, '/');
            n = n.getTagContext().getParent();
        }
        if (tag instanceof ContainTag) {
            sb.append('/');
        }
        return sb.toString();
    }

    /*
     * ResourceKeyPrefixSupport
     */

    public String getPrefix() {
        return prefixsMixIn.getPrefix();
    }

    public void setPrefix(String prefix) {
        prefixsMixIn.setPrefix(prefix);
    }

    public String[] getSourcesPrefix() {
        return prefixsMixIn.getSourcesPrefix();
    }

    public void setSourcesPrefix(String[] prefixs) {
        prefixsMixIn.setSourcesPrefix(prefixs);
    }

}
