/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin.factory;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import shohaku.core.resource.IOResource;
import shohaku.shoin.ResourceKeyPrefixSupport;
import shohaku.shoin.ResourceSet;
import shohaku.shoin.ResourceSetCreationException;
import shohaku.shoin.resourceset.MapResourceSet;

/**
 * java.util.Properties を使用するシンプルなプロパティセットを格納するファクトリを提供します。<br>
 * <p>
 * 例：
 * 
 * <pre>
 * # java.util.Properties と同じです。
 * name=松柏
 * </pre>
 * 
 * <pre>
 * 一般的なファイルの読み取りとコンポーネント取得の手順は以下の様に為ります。 
 * SimpleProperties factory = new SimpleProperties();
 * IOResource inres = FeatureFactory.getLoader().getIOResource("simple.properties");
 * factory.setIOResources(new IOResource[] { inres });
 * 
 * ResourceSet resources = factory.getResourceSet();
 *
 * String name = (String) resources.getObject("name");
 * System.out.println("名前=" + name);
 *
 * ＞＞名前=松柏
 * </pre>
 */
public class SimpleProperties extends AbstractIOResourceSetFactory implements ResourceKeyPrefixSupport {

    /* ResourceKeyPrefixSupport Mix-in */
    private final ResourceKeyPrefixMixIn prefixsMixIn = new ResourceKeyPrefixMixIn();

    /**
     * プロパティを初期値で初期化します。
     */
    public SimpleProperties() {
        super();
    }

    /**
     * IOリソースから、リソース集合を生成し返却します。<br>
     * IOリソースが設定されていない場合は、生成を行わずにリソース集合を返却します。 <br>
     * 入力ストリームからキーと要素が対になったプロパティセットを読み込みます。 <br>
     * ストリームは ISO 8859-1 文字エンコーディングを使用しているとみなされます。 <br>
     * このエンコーディングに直接表示できない文字には Unicode escapes が使用されます。 <br>
     * ただし、エスケープシーケンスでは 1 文字の「u」だけが使用可能です。 <br>
     * 他の文字エンコーディングとプロパティファイルを変換する場合 native2ascii ツールを使用できます。
     * 
     * @return リソース集合
     * @throws ResourceSetCreationException
     *             リソース集合の生成に失敗した場合
     */
    public ResourceSet getResourceSet() throws ResourceSetCreationException {

        IOResource[] inreses = getIOResources();
        if (null == inreses || inreses.length == 0) {
            return newResourceSetObject(newResourceSetMap());
        }
        // load resource
        final Map lookup = newResourceSetMap();
        load(lookup, inreses);
        return newResourceSetObject(lookup);
    }

    /*
     * protected
     */

    /**
     * リソース集合を格納するマップを生成して返却します。拡張ポイントです。
     * 
     * @return リソース集合を格納するマップ
     */
    protected Map newResourceSetMap() {
        return new HashMap();
    }

    /**
     * マップからリソース集合を生成して返却します。拡張ポイントです。
     * 
     * @param lookup
     *            リソースを格納するマップ
     * @return リソース集合
     */
    protected ResourceSet newResourceSetObject(final Map lookup) {
        return new MapResourceSet(lookup);
    }

    /**
     * IOリソースからリソースを読み込みます。
     * 
     * @param lookup
     *            リソースを格納するマップ
     * @param inreses
     *            IOリソースの配列
     * @throws ResourceSetCreationException
     *             リソース集合の生成に失敗した場合
     */
    protected void load(Map lookup, IOResource[] inreses) throws ResourceSetCreationException {
        for (int i = 0; i < inreses.length; i++) {
            load(lookup, inreses[i], i);
        }
    }

    /**
     * IOリソースからリソースを読み込みます。
     * 
     * @param lookup
     *            リソースを格納するマップ
     * @param inres
     *            IOリソース
     * @param index
     *            IOリソースのインデックス
     * @throws ResourceSetCreationException
     *             リソース集合の生成に失敗した場合
     */
    protected void load(Map lookup, IOResource inres, int index) throws ResourceSetCreationException {

        final Properties properties = new Properties();
        InputStream stream = null;
        try {

            stream = new BufferedInputStream(inres.getInputStream());
            properties.load(stream);
            prefixsMixIn.putAll(lookup, properties, index);

        } catch (IOException e) {
            throw new ResourceSetCreationException("load resource error.", e);
        } finally {
            if (null != stream) {
                try {
                    stream.close();
                } catch (Exception e) {
                    // no op
                }
            }
        }
    }

    /*
     * ResourceKeyPrefixSupport
     */

    public String getPrefix() {
        return prefixsMixIn.getPrefix();
    }

    public void setPrefix(String prefix) {
        prefixsMixIn.setPrefix(prefix);
    }

    public String[] getSourcesPrefix() {
        return prefixsMixIn.getSourcesPrefix();
    }

    public void setSourcesPrefix(String[] prefixs) {
        prefixsMixIn.setSourcesPrefix(prefixs);
    }

}
