/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.shoin.factory;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.LinkedHashMap;
import java.util.Map;

import shohaku.core.resource.IOResource;
import shohaku.ogdl.OgdlContext;
import shohaku.ogdl.OgdlHelper;
import shohaku.shoin.ResourceKeyPrefixSupport;
import shohaku.shoin.ResourceSet;
import shohaku.shoin.ResourceSetCreationException;
import shohaku.shoin.resourceset.MapResourceSet;
import shohaku.sugina.util.OgdlXProperties;

/**
 * プロパティ値を OGDL式として認識するプロパティセットを格納するファクトリを提供します。<br>
 * <p>
 * 例：
 * 
 * <pre>
 * # &#64; で始まると処理の宣言と見なされます。
 * &#64;import=java.sql.Time　　　//クラスをOGDLにインポートします。
 * 
 * # $ で始まると、OGDL変数の宣言と為ります。
 * $sunday="日曜"　　　//sundayが変数名、値をOGDL式として評価
 * 
 * #コメント
 * season.map={ 1="春", 2="夏", 3="秋", 4="冬" }　//プロパティを定義します。キーは文字列、値をOGDL式として評価
 * </pre>
 * 
 * コメントや区切り文字等の仕様は java.util.Properties と同等です。<br>
 * OGDLの構文に関しては、以下の <a href="http://shohaku.sourceforge.jp/projects/ogdl/reference/reference.html" target="_blank">OGDL リファレンス</a> を参照してください。<br>
 * <br>
 * 
 * <pre>
 * 一般的なファイルの読み取りとコンポーネント取得の手順は以下の様に為ります。 
 * OgdlProperties factory = new OgdlProperties();
 * IOResource inres = FeatureFactory.getLoader().getIOResource("ognl.properties");
 * factory.setIOResources(new IOResource[] { inres });
 * 
 * ResourceSet resources = factory.getResourceSet();
 *
 * Map m = (Map) resources.getObject("season.map");
 * System.out.println("季節名のマッピング=" + m.toString());
 *
 * ＞＞季節名のマッピング={1=春,2=夏,3=秋,4=冬}
 * </pre>
 */
public class OgdlProperties extends AbstractIOResourceSetFactory implements ResourceKeyPrefixSupport {

    /* ResourceKeyPrefixSupport Mix-in */
    private final ResourceKeyPrefixMixIn prefixsMixIn = new ResourceKeyPrefixMixIn();

    /* リソースの生成に使用するクラスローダ */
    private ClassLoader classLoader;

    /**
     * プロパティを初期値で初期化します。
     */
    public OgdlProperties() {
        super();
    }

    /**
     * IOリソースから、リソース集合を生成し返却します。<br>
     * IOリソースが設定されていない場合は、生成を行わずにリソース集合を返却します。 <br>
     * 入力ストリームからキーと要素が対になったプロパティセットを読み込みます。 <br>
     * ストリームは ISO 8859-1 文字エンコーディングを使用しているとみなされます。 <br>
     * このエンコーディングに直接表示できない文字には Unicode escapes が使用されます。 <br>
     * ただし、エスケープシーケンスでは 1 文字の「u」だけが使用可能です。 <br>
     * 他の文字エンコーディングとプロパティファイルを変換する場合 native2ascii ツールを使用できます。
     * 
     * @return リソース集合
     * @throws ResourceSetCreationException
     *             リソース集合の生成に失敗した場合
     */
    public ResourceSet getResourceSet() throws ResourceSetCreationException {

        IOResource[] inreses = getIOResources();
        if (null == inreses || inreses.length == 0) {
            return newResourceSetObject(newResourceSetMap());
        }
        ClassLoader _loader = findClassLoader();
        OgdlContext context = OgdlHelper.getOgdlContext();

        // load resource
        final Map lookup = newResourceSetMap();
        load(lookup, context, _loader, inreses);
        return newResourceSetObject(lookup);
    }

    /**
     * リソースの生成に使用するクラスローダを返却します。
     * 
     * @return クラスローダ
     */
    public ClassLoader getClassLoader() {
        return classLoader;
    }

    /**
     * リソースの生成に使用するクラスローダを格納します。
     * 
     * @param classLoader
     *            クラスローダ
     */
    public void setClassLoader(ClassLoader classLoader) {
        this.classLoader = classLoader;
    }

    /*
     * protected
     */

    /**
     * 使用するクラスローダを検索して返却します。
     * 
     * @return クラスローダ
     */
    protected ClassLoader findClassLoader() {
        ClassLoader loader = getClassLoader();
        if (null == loader) {
            loader = this.getClass().getClassLoader();
        }
        return loader;
    }

    /**
     * リソース集合を格納するマップを生成して返却します。拡張ポイントです。
     * 
     * @return リソース集合を格納するマップ
     */
    protected Map newResourceSetMap() {
        return new LinkedHashMap();
    }

    /**
     * マップからリソース集合を生成して返却します。拡張ポイントです。
     * 
     * @param lookup
     *            リソースを格納するマップ
     * @return リソース集合
     */
    protected ResourceSet newResourceSetObject(final Map lookup) {
        return new MapResourceSet(lookup);
    }

    /**
     * IOリソースからリソースを読み込みます。
     * 
     * @param lookup
     *            リソースを格納するマップ
     * @param loader
     *            クラスローダ
     * @param context
     *            式コンテキスト
     * @param inreses
     *            IOリソースの配列
     * @throws ResourceSetCreationException
     *             リソース集合の生成に失敗した場合
     */
    protected void load(Map lookup, OgdlContext context, ClassLoader loader, IOResource[] inreses) throws ResourceSetCreationException {
        for (int i = 0; i < inreses.length; i++) {
            load(lookup, context, loader, inreses[i], i);
        }
    }

    /**
     * IOリソースからリソースを読み込みます。
     * 
     * @param lookup
     *            リソースを格納するマップ
     * @param loader
     *            クラスローダ
     * @param context
     *            式コンテキスト
     * @param inres
     *            IOリソース
     * @param index
     *            IOリソースのインデックス
     * @throws ResourceSetCreationException
     *             リソース集合の生成に失敗した場合
     */
    protected void load(Map lookup, OgdlContext context, ClassLoader loader, IOResource inres, int index) throws ResourceSetCreationException {

        final OgdlXProperties properties = new OgdlXProperties(context, loader);
        InputStream stream = null;
        try {

            stream = new BufferedInputStream(inres.getInputStream());
            properties.load(stream);
            prefixsMixIn.putAll(lookup, properties.toMap(), index);

        } catch (IOException e) {
            throw new ResourceSetCreationException("load resource error.", e);
        } finally {
            if (null != stream) {
                try {
                    stream.close();
                } catch (Exception e) {
                    // no op
                }
            }
        }
    }

    /*
     * class
     */

    public String getPrefix() {
        return prefixsMixIn.getPrefix();
    }

    public void setPrefix(String prefix) {
        prefixsMixIn.setPrefix(prefix);
    }

    public String[] getSourcesPrefix() {
        return prefixsMixIn.getSourcesPrefix();
    }

    public void setSourcesPrefix(String[] prefixs) {
        prefixsMixIn.setSourcesPrefix(prefixs);
    }

}
