/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin.factory;

import java.util.LinkedHashMap;
import java.util.Map;

import shohaku.shoin.ResourceKeyPrefixSupport;
import shohaku.shoin.ResourceSet;
import shohaku.shoin.ResourceSetCreationException;
import shohaku.shoin.resourceset.MapResourceSet;

/**
 * 指定されたマップをプロパティセットとして生成するファクトリを提供します。<br>
 * 基となるオブジェクトはマップ型である必要があります。
 * <p>
 * 例：
 * 
 * <pre>
 * 一般的なファイルの読み取りとコンポーネント取得の手順は以下の様に為ります。 
 * MapProperties factory = new MapProperties();
 * HashMap map = new HashMap();
 * map.put("foo", new Integer(1234));
 * factory.setSources(new Object[] { map });
 * 
 * ResourceSet resources = factory.getResourceSet();
 *
 * Integer foo = (Integer) resources.getObject("foo");
 * System.out.println("foo=" + foo);
 *
 * ＞＞foo=1234
 * </pre>
 */
public class MapProperties extends AbstractObjectResourceSetFactory implements ResourceKeyPrefixSupport {

    /* ResourceKeyPrefixSupport Mix-in */
    private final ResourceKeyPrefixMixIn prefixsMixIn = new ResourceKeyPrefixMixIn();

    /**
     * プロパティを初期値で初期化します。
     */
    public MapProperties() {
        super();
    }

    public ResourceSet getResourceSet() throws ResourceSetCreationException {
        try {
            final Map lookup = new LinkedHashMap();
            final Object[] sources = getSources();
            for (int i = 0; i < sources.length; i++) {
                final Map source = (Map) sources[i];
                prefixsMixIn.putAll(lookup, source, i);
            }
            return new MapResourceSet(lookup);
        } catch (ClassCastException e) {
            throw new ResourceSetCreationException("source is not Map.", e);
        }
    }

    /*
     * ResourceKeyPrefixSupport
     */

    public String getPrefix() {
        return prefixsMixIn.getPrefix();
    }

    public void setPrefix(String prefix) {
        prefixsMixIn.setPrefix(prefix);
    }

    public String[] getSourcesPrefix() {
        return prefixsMixIn.getSourcesPrefix();
    }

    public void setSourcesPrefix(String[] prefixs) {
        prefixsMixIn.setSourcesPrefix(prefixs);
    }

}
