/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin.factory;

import java.util.LinkedHashMap;
import java.util.Map;

import shohaku.core.helpers.HBeans;
import shohaku.core.lang.IntrospectionBeansException;
import shohaku.shoin.ResourceKeyPrefixSupport;
import shohaku.shoin.ResourceSet;
import shohaku.shoin.ResourceSetCreationException;
import shohaku.shoin.resourceset.MapResourceSet;

/**
 * 指定されたクラスのパブリックな定数（public static final なフィールド）をプロパティセットとして生成するファクトリを提供します。<br>
 * 基となるオブジェクトがクラス型の場合はそのまま使用されます、以外の場合はそのオブジェクトのクラス型が使用されます。
 * <p>
 * 例：
 * 
 * <pre>
 * 一般的なファイルの読み取りとコンポーネント取得の手順は以下の様に為ります。 
 * ConstantsProperties factory = new ConstantsProperties();
 * factory.setSources(new Object[] { Integer.class });
 * 
 * ResourceSet resources = factory.getResourceSet();
 *
 * Integer max = (Integer) resources.getObject("MAX_VALUE");
 * System.out.println("int の最大値=" + max);
 *
 * ＞＞int の最大値=2147483647
 * </pre>
 */
public class ConstantsProperties extends AbstractObjectResourceSetFactory implements ResourceKeyPrefixSupport {

    /* ResourceKeyPrefixSupport Mix-in */
    private final ResourceKeyPrefixMixIn prefixsMixIn = new ResourceKeyPrefixMixIn();

    /**
     * プロパティを初期値で初期化します。
     */
    public ConstantsProperties() {
        super();
    }

    public ResourceSet getResourceSet() throws ResourceSetCreationException {
        try {
            final Map lookup = new LinkedHashMap();
            final Object[] sources = getSources();
            for (int i = 0; i < sources.length; i++) {
                Class source = (sources[i] instanceof Class) ? (Class) sources[i] : sources[i].getClass();
                prefixsMixIn.putAll(lookup, HBeans.getConstantFieldMap(source), i);
            }
            return new MapResourceSet(lookup);
        } catch (IntrospectionBeansException e) {
            throw new ResourceSetCreationException("class introspection error.", e);
        }
    }

    /*
     * ResourceKeyPrefixSupport
     */

    public String getPrefix() {
        return prefixsMixIn.getPrefix();
    }

    public void setPrefix(String prefix) {
        prefixsMixIn.setPrefix(prefix);
    }

    public String[] getSourcesPrefix() {
        return prefixsMixIn.getSourcesPrefix();
    }

    public void setSourcesPrefix(String[] prefixs) {
        prefixsMixIn.setSourcesPrefix(prefixs);
    }

}
