/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin.factory;

import java.util.Iterator;
import java.util.Map;

import shohaku.core.lang.Concat;
import shohaku.ginkgo.TagNode;
import shohaku.ginkgo.ValueNode;
import shohaku.shoin.ResourceKeyPrefixSupport;

/**
 * XMLデータで定義された情報を基に、メソッドを特殊な規則で拘束する機能を生成するファクトリを提供します。<br>
 * このクラスは shohaku.sugina.binder.BindMethod をXMLデータから生成し、識別子でマッピングします。
 * <p>
 * 例：
 * 
 * <pre>
 * //関連づけるクラス
 * public class Convert {
 *     public static Number toDecimal(String num, String pattern) {
 *         final DecimalFormat format = new DecimalFormat(pattern);
 *         return format.parse(num);
 *     }
 * }
 *
 * //定義ファイル bind-method-factory.xml
 * &lt;component-factory&gt;
 *  
 * &lt;bind-method-factory&gt;
 *  
 *     &lt;bind-method-mapping groupId="convert"&gt;
 *      
 *         &lt;bind-method id="decimal" name="toDecimal" class="Convert"&gt;
 *             &lt;bind-arg name="value" type="String"/&gt;
 *             &lt;bind-arg name="pattern" type="String" final="true"&gt;
 *                 &lt;string&gt;###,###,###&lt;/string&gt;
 *             &lt;/bind-arg&gt;
 *         &lt;/bind-method&gt;
 *         
 *     &lt;/bind-method-mapping&gt;
 *  
 * &lt;/bind-method-factory&gt;
 * 
 * BindMethodFactory factory = new BindMethodFactory();
 * final IOResource inres = new ClassPathIOResource("bind-method-factory.xml");
 * factory.setIOResources(new IOResource[] { inres });
 * 
 * ResourceSet rset = factory.getResourceSet();
 * BindMethod bindMethod = (BindMethod) rset.getObject("convert:decimal");　// groupId:methodId の識別子で取得する
 * Number num = (Number) bindMethod.invoke(Collections.singletonMap("value", "123,456,789")); //名前で値を関連付けメソッドを実行する
 * 
 * System.out.println("変換結果=" + num);
 * 
 * ＞＞変換結果=123456789
 *
 * </pre>
 */
public class BindMethodFactory extends AbstractGinkgoResourceSetFactory implements ResourceKeyPrefixSupport {

    /* ResourceKeyPrefixSupport Mix-in */
    private final ResourceKeyPrefixMixIn prefixsMixIn = new ResourceKeyPrefixMixIn();

    /**
     * プロパティを初期値で初期化します。
     */
    public BindMethodFactory() {
        super();
    }

    protected void initValues(Map lookup, TagNode root, int index) {
        initValues(root, prefixsMixIn.findPrefix(index), lookup);
    }

    /* 再起的にリソースを登録します。 */
    private void initValues(TagNode parent, String prefix, Map output) {
        for (Iterator i = parent.getTagContext().elementIterator("bind-method-mapping"); i.hasNext();) {
            TagNode tag = (TagNode) i.next();
            String groupId = tag.getTagContext().getAttribute("groupId");
            initValue(tag, prefix, Concat.get(groupId, ":"), output);
        }
    }

    /* リソースを評価して登録します。 */
    private void initValue(TagNode parent, String prefix, String groupId, Map output) {
        for (Iterator i = parent.getTagContext().elementIterator("bind-method"); i.hasNext();) {
            TagNode node = (TagNode) i.next();
            String id = node.getId();
            output.put(Concat.get(prefix, groupId, id), ((ValueNode) node).getNodeValue());
        }
    }

    /*
     * ResourceKeyPrefixSupport
     */

    public String getPrefix() {
        return prefixsMixIn.getPrefix();
    }

    public void setPrefix(String prefix) {
        prefixsMixIn.setPrefix(prefix);
    }

    public String[] getSourcesPrefix() {
        return prefixsMixIn.getSourcesPrefix();
    }

    public void setSourcesPrefix(String[] prefixs) {
        prefixsMixIn.setSourcesPrefix(prefixs);
    }
}
