/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin.factory;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.LinkedHashMap;
import java.util.Map;

import shohaku.core.resource.IOResource;
import shohaku.ginkgo.Document;
import shohaku.ginkgo.DocumentCompositeRule;
import shohaku.ginkgo.Ginkgo;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.NodeCompositeRule;
import shohaku.ginkgo.TagNode;
import shohaku.shoin.GinkgoResourceSetFactory;
import shohaku.shoin.ResourceSet;
import shohaku.shoin.ResourceSetCreationException;
import shohaku.shoin.ShoinResourceLoader;
import shohaku.shoin.resourceset.MapResourceSet;

/**
 * Ginkgo API 用いたリソース集合を生成するファクトリの抽象実装を提供します。
 */
public abstract class AbstractGinkgoResourceSetFactory extends AbstractIOResourceSetFactory implements GinkgoResourceSetFactory {

    /* ノード構成ルール */
    private NodeCompositeRule nodeCompositeRule;

    /* 親のドキュメント */
    private Document parentDocument;

    /* ドキュメント構成ルール */
    private DocumentCompositeRule documentCompositeRule;

    /* 直前に解析したドキュメント */
    private Document parseDocument;

    /* リソースの生成に使用するクラスローダ */
    private ClassLoader classLoader;

    /**
     * プロパティを初期値で初期化します。
     */
    public AbstractGinkgoResourceSetFactory() {
        super();
    }

    /*
     * impl
     */

    public ResourceSet getResourceSet() throws ResourceSetCreationException {

        IOResource[] inreses = getIOResources();
        if (null == inreses || inreses.length == 0) {
            return newResourceSetObject(newResourceSetMap());
        }

        final Ginkgo ginkgo = new Ginkgo();

        ClassLoader loader = findClassLoader();
        ginkgo.setClassLoader(loader);

        NodeCompositeRule rule = getNodeCompositeRule();
        if (null == rule) {
            rule = getDefaultNodeCompositeRule();
        }
        ginkgo.setNodeCompositeRule(rule);

        if (null != getDocumentCompositeRule()) {
            ginkgo.setDocumentCompositeRule(getDocumentCompositeRule());
        }
        if (null != getParentDocument()) {
            ginkgo.setParentDocument(getParentDocument());
        }

        // load resource
        final Map lookup = newResourceSetMap();
        load(ginkgo, lookup, inreses);
        return newResourceSetObject(lookup);

    }

    public NodeCompositeRule getNodeCompositeRule() {
        return nodeCompositeRule;
    }

    public void setNodeCompositeRule(NodeCompositeRule nodeCompositeRule) {
        this.nodeCompositeRule = nodeCompositeRule;
    }

    public DocumentCompositeRule getDocumentCompositeRule() {
        return documentCompositeRule;
    }

    public void setDocumentCompositeRule(DocumentCompositeRule documentCompositeRule) {
        this.documentCompositeRule = documentCompositeRule;
    }

    public Document getParentDocument() {
        return parentDocument;
    }

    public void setParentDocument(Document parentDocument) {
        this.parentDocument = parentDocument;
    }

    public Document getParseDocument() {
        return parseDocument;
    }

    public ClassLoader getClassLoader() {
        return classLoader;
    }

    public void setClassLoader(ClassLoader classLoader) {
        this.classLoader = classLoader;
    }

    /*
     * protected
     */

    /**
     * 使用するクラスローダを検索して返却します。
     * 
     * @return クラスローダ
     */
    protected ClassLoader findClassLoader() {
        ClassLoader loader = getClassLoader();
        if (null == loader) {
            loader = this.getClass().getClassLoader();
        }
        return loader;
    }

    /**
     * リソース集合を格納するマップを生成して返却します。拡張ポイントです。
     * 
     * @return リソース集合を格納するマップ
     */
    protected Map newResourceSetMap() {
        return new LinkedHashMap();
    }

    /**
     * マップからリソース集合を生成して返却します。拡張ポイントです。
     * 
     * @param lookup
     *            リソースを格納するマップ
     * @return リソース集合
     */
    protected ResourceSet newResourceSetObject(final Map lookup) {
        return new MapResourceSet(lookup);
    }

    /**
     * 直前に解析したドキュメントを格納します。
     * 
     * @param parseDoc
     *            直前に解析したドキュメント
     */
    protected void setParseDocument(Document parseDoc) {
        this.parseDocument = parseDoc;
    }

    /**
     * IOリソースからリソースを読み込みます。
     * 
     * @param ginkgo
     *            Ginkgo
     * @param lookup
     *            リソースを格納するマップ
     * @param inreses
     *            IOリソースの配列
     * @throws ResourceSetCreationException
     *             リソース集合の生成に失敗した場合
     */
    protected void load(Ginkgo ginkgo, Map lookup, IOResource[] inreses) throws ResourceSetCreationException {
        for (int i = 0; i < inreses.length; i++) {
            load(ginkgo, lookup, inreses[i], i);
        }
    }

    /**
     * IOリソースからリソースを読み込みます。
     * 
     * @param ginkgo
     *            Ginkgo
     * @param lookup
     *            リソースを格納するマップ
     * @param inres
     *            IOリソース
     * @param index
     *            IOリソースのインデックス
     * @throws ResourceSetCreationException
     *             リソース集合の生成に失敗した場合
     */
    protected void load(Ginkgo ginkgo, Map lookup, IOResource inres, int index) throws ResourceSetCreationException {
        // XMLの解析
        InputStream stream = null;
        try {

            stream = new BufferedInputStream(inres.getInputStream());
            ginkgo.parse(stream);
            TagNode root = ginkgo.getDocument().getContext().getRoot();
            setParseDocument(ginkgo.getDocument());
            initValues(lookup, root, index);

        } catch (GinkgoException e) {
            throw new ResourceSetCreationException("Ginkgo parsing error.", e);
        } catch (IOException e) {
            throw new ResourceSetCreationException("resource stream error.", e);
        } finally {
            if (null != stream) {
                try {
                    stream.close();
                } catch (Exception e) {
                    // no op
                }
            }
        }
    }

    /**
     * デフォルトの構成ルールを返却します。
     * 
     * @return デフォルトの構成ルール
     */
    protected NodeCompositeRule getDefaultNodeCompositeRule() {
        return ShoinResourceLoader.getDefaultNodeCompositeRule(this.getClass(), findClassLoader());
    }

    /*
     * abstract
     */

    /**
     * リソースの初期化および登録します。
     * 
     * @param lookup
     *            リソースを格納するマップ
     * @param root
     *            ルートタグ
     * @param index
     *            IOリソースのインデックス
     */
    abstract protected void initValues(Map lookup, TagNode root, int index);

}
