/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.shoin.bundle;

import java.util.MissingResourceException;

import shohaku.core.resource.IOResourceLoader;
import shohaku.ginkgo.Document;
import shohaku.shoin.ResourceSet;
import shohaku.shoin.XResourceBundle;
import shohaku.shoin.XResourceBundleBase;
import shohaku.shoin.XResourceBundleCache;

/**
 * {@link shohaku.shoin.factory.XMLProperties} を用いた階層化リソースバンドルを提供します。
 * <p>
 * <b>XMLリソースバンドル固有の名前規約 </b>
 * <p>
 * XMLリソースバンドルには唯一 java.util.ResourceBundle とは別の名前規約が有ります。 <br>
 * XMLリソースバンドルでは基底名を基にして複数のファイルが読取の対象となるため、同一の基底名に属すファイルは同一の解析ルールが摘要される必要があります。 <br>
 * よって基底名単位で解析ルールを指定する為の以下の配置規約が定義されています。
 * <ul type=square>
 * <li>基底名単位で解析ルールを指定する場合のファイル名は '基底名 + _rule.xml' で指定します。
 * </ul>
 * <br>
 * 上記位置に配置しておけばシステムが自動で読み取ります, 配置されていなければデフォルトが使用されます。
 * </p>
 * <p>
 * <b>XMLリソースバンドル固有の問題 </b>
 * <p>
 * このバンドルは複数のデータ型に対応する一方で、 java.util.ResourceBundle と同様に上位階層から値が継承されるモデルを採用しています。<br>
 * そのため上位階層で定義される型とは、交換性の無い型で、下位の階層が同一名のプロパティを定義する事が可能です。 <br>
 * この特徴はデータ型の管理を難しくする可能性があります。 <br>
 * この特徴を踏まえて、データ型の設計および管理を行うことが推奨されます。
 * </p>
 */
public class XMLResourceBundle extends AbstractGinkgoXResourceBundle {

    /* 拡張リソースバンドルのキャッシュ機能のシングルトンインスタンスを格納します。 */
    static final class XMLResourceBundleCache {
        // new Singleton Instance
        static final XResourceBundleCache instance = new XResourceBundleCache(new XMLResourceBundleCreater());
    }

    /**
     * 拡張リソースバンドルを初期化します。
     * 
     * @param parent
     *            親バンドル
     * @param bundleBase
     *            束縛基準
     * @param resources
     *            リソース集合
     * @param document
     *            ドキュメント
     */
    XMLResourceBundle(XResourceBundle parent, XResourceBundleBase bundleBase, ResourceSet resources, Document document) {
        super(parent, bundleBase, resources, document);
    }

    /*
     * static
     */

    /* キャッシュ機能のインスタンスを返却します。 */
    static XResourceBundleCache getXResourceBundleCache() {
        return XMLResourceBundleCache.instance;
    }

    /*
     * factory
     */

    /**
     * 指定された基底名、束縛基準、および呼び出し側のクラスローダを使用して、リソースバンドルを取得します。 <br>
     * このメソッドを呼び出すことは、以下を呼び出すことと同じです。 <br>
     * getBundle(baseName, locale, getDefaultLoader())
     * 
     * @param baseName
     *            基底名
     * @param bundleBase
     *            束縛基準
     * @return 指定された基底名と束縛基準の拡張リソースバンドル
     * @throws MissingResourceException
     *             指定された基底名のリソースバンドルが見つからない場合
     */
    public static XMLResourceBundle getBundle(String baseName, XResourceBundleBase bundleBase) {
        return getBundle(baseName, bundleBase, null, null);
    }

    /**
     * 指定された基底名、束縛基準、クラスローダを使用して、拡張リソースバンドルを取得します。
     * 
     * @param baseName
     *            基底名
     * @param bundleBase
     *            束縛基準
     * @param irLoader
     *            IOリソース生成機能
     * @param classLoader
     *            クラスローダ
     * @return 指定された基底名と束縛基準の拡張リソースバンドル
     * @throws MissingResourceException
     *             指定された基底名のリソースバンドルが見つからない場合
     */
    public static XMLResourceBundle getBundle(String baseName, XResourceBundleBase bundleBase, IOResourceLoader irLoader, ClassLoader classLoader) {
        GinkgoXResourceBundleEvent bundleEvent = new GinkgoXResourceBundleEvent(XMLResourceBundle.class);
        bundleEvent.setBaseName(baseName);
        bundleEvent.setBundleBase(bundleBase);
        bundleEvent.setIOResourceLoader(irLoader);
        bundleEvent.setClassLoader(classLoader);
        return (XMLResourceBundle) getXResourceBundleCache().getBundle(bundleEvent);
    }

}
