/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin.bundle;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;

import shohaku.core.collections.Cache;
import shohaku.core.collections.cache.AccessOrderHashCache;
import shohaku.core.functor.FFactory;
import shohaku.core.lang.Concat;
import shohaku.core.lang.Eval;
import shohaku.core.lang.EvalSet;
import shohaku.shoin.XResourceBundleBase;

/**
 * 拡張リソースバンドルをロケールに応じて階層化する束縛基準を提供します。<br>
 * <p>
 * <b>ロケール束縛基準の特徴 </b>
 * </p>
 * <p>
 * ロケール束縛基準には、ロケール固有の情報に基づき拡張リソースバンドルを束縛する機能を提供しています。<br>
 * プログラムでロケール固有のリソースが必要なときは、ユーザのロケールに合った拡張リソースバンドルからロードできます。 <br>
 * このように、ロケール固有の情報のその大部分を切り離すことで、ロケールにはほとんど依存しないプログラムコードを書くことができます。
 * </p>
 * これにより、以下の特徴を持つプログラムを書くことが可能になります。
 * <UL type=SQUARE>
 * <LI>異なる言語への地域対応、または翻訳が容易
 * <LI>複数のロケールを同時に処理可能
 * <LI>将来サポートするロケールを追加する際の修正が容易
 * </UL>
 * <p>
 * この特徴は java.util.ResourceBundle と同様のものであり、 <br>
 * ロケール束縛基準の設計は java.util.ResourceBundle の国際化モデルを継承しています。<br>
 * 大きな相違点は java.util.ResourceBundle がデータモデルを制限するに対して、<br>
 * ロケール束縛基準は拡張リソースバンドルの実装するデータモデルを選択的に使用できる点です。
 * </p>
 * この機能により java.util.ResourceBundle の柔軟性を保ち、<br>
 * ユーザの問題領域に応じた階層データモデルを利用して情報を外部ファイル化する事が出来ます。
 * <p>
 * ロケール束縛基準の基底名と各ロケールをファミリとして定義する名前規約に関しては java.util.ResourceBundle を参照してください。
 * </p>
 */
public class LocaleXResourceBundleBase implements XResourceBundleBase {

    /* ルートの束縛基準 */
    static final LocaleXResourceBundleBase ROOT = new LocaleXResourceBundleBase();

    /* 最大キャッシュサイズ */
    static final int CACHE_MAXIMUM_SIZE = 255;

    /* 束縛基準をロケールを識別子としてキャッシュする */
    private static Cache cache = new AccessOrderHashCache(32, 0.75f, CACHE_MAXIMUM_SIZE);
    static {
        cache.setFactory(new FFactory() {
            public Object create(Object o) {
                return new LocaleXResourceBundleBase((Locale) o);
            }
        });
    }

    /* ロケール */
    private final Locale locale;

    /* 識別子 */
    private final String bundleId;

    /* 接尾辞 */
    private final String resourceSuffix;

    /* 上位の束縛基準 */
    private final List parents;

    /**
     * ルートの束縛規準を初期化します。
     */
    LocaleXResourceBundleBase() {
        this.locale = new Locale("", "", "");
        this.bundleId = candidateBundleId(this.locale);
        this.resourceSuffix = candidateResourceSuffix(this.locale);
        this.parents = Collections.EMPTY_LIST;
    }

    /**
     * ロケールを元に束縛規準を初期化します。
     * 
     * @param locale
     *            ロケール
     */
    LocaleXResourceBundleBase(Locale locale) {
        this.locale = locale;
        this.bundleId = candidateBundleId(locale);
        this.resourceSuffix = candidateResourceSuffix(locale);
        this.parents = candidateParents(locale);
    }

    /**
     * ルートの束縛基準である場合に true を返却します。
     * 
     * @return ルートの束縛基準である場合に true
     */
    public boolean isRoot() {
        return (ROOT.equals(this));
    }

    /**
     * ルートの束縛基準を返却します。
     * 
     * @return ルートの束縛基準
     */
    public XResourceBundleBase getRoot() {
        return ROOT;
    }

    /**
     * この束縛基準の識別子を返却します。
     * 
     * @return この束縛基準の識別子
     */
    public String getBundleId() {
        return bundleId;
    }

    /**
     * リソースに対する、この束縛基準の接尾辞を返却します。
     * 
     * @return この束縛基準の接尾辞
     */
    public String getResourceSuffix() {
        return resourceSuffix;
    }

    /**
     * この束縛基準の上位の束縛基準をルートをトップとする階層順で返却します。
     * 
     * @return 上位の束縛基準のリスト
     */
    public List getParents() {
        return parents;
    }

    /**
     * この束縛基準のハッシュコード値を返却します。<br>
     * ハッシュコードはこの束縛基準のロケールから算出します。<br>
     * これにより束縛基準 t1 と t2 で t1.equals(t2) であれば、t1.hashCode()==t2.hashCode() となることが保証されます。
     * 
     * @return 束縛基準のハッシュコード値
     * @see java.lang.Object#hashCode()
     */
    public int hashCode() {
        return (locale.getLanguage().hashCode() + locale.getCountry().hashCode() + locale.getVariant().hashCode());
    }

    /**
     * 引数がこの束縛基準と同値の束縛基準として認識できるか検証します。<br>
     * 引数が LocaleXResourceBundleBase 型であり同時に同一のロケールコードを保有する場合は同値とします。<br>
     * これにより束縛基準 t1 と t2 で t1.equals(t2) であれば、t1.hashCode()==t2.hashCode() となることが保証されます。
     * 
     * @param o
     *            比較するオブジェクト
     * @return 引数がこの束縛基準と同値の束縛基準の場合は true
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof LocaleXResourceBundleBase)) {
            return false;
        }
        Locale l1 = this.locale;
        Locale l2 = ((LocaleXResourceBundleBase) o).locale;
        if (l1.getLanguage().equals(l2.getLanguage()) && l1.getCountry().equals(l2.getCountry()) && l1.getVariant().equals(l2.getVariant())) {
            return true;
        }
        return false;
    }

    /*
     * static
     */

    /**
     * ルートの束縛基準を返却します。
     * 
     * @return ルートの束縛基準
     */
    public static LocaleXResourceBundleBase forBundleBase() {
        return ROOT;
    }

    /**
     * 言語コードの示すロケールの束縛基準を生成して返却します。
     * 
     * @param language
     *            言語コード
     * @return 言語コードの示すロケールの束縛基準
     */
    public static LocaleXResourceBundleBase forBundleBase(String language) {
        return forBundleBase(new Locale(language, "", ""));
    }

    /**
     * 言語、国の示すロケールの束縛基準を生成して返却します。
     * 
     * @param language
     *            言語コード
     * @param country
     *            国コード
     * @return 言語、国の示すロケールの束縛基準
     */
    public static LocaleXResourceBundleBase forBundleBase(String language, String country) {
        return forBundleBase(new Locale(language, country, ""));
    }

    /**
     * 言語、国、バリアントの示すロケールの束縛基準を生成して返却します。
     * 
     * @param language
     *            言語コード
     * @param country
     *            国コード
     * @param variant
     *            ベンダーとブラウザに固有のコード
     * @return 言語、国、バリアントの示すロケールの束縛基準
     */
    public static LocaleXResourceBundleBase forBundleBase(String language, String country, String variant) {
        return forBundleBase(new Locale(language, country, variant));
    }

    /**
     * ロケールの束縛基準を生成して返却します。
     * 
     * @param locale
     *            ロケール
     * @return ロケールの束縛基準
     */
    public static LocaleXResourceBundleBase forBundleBase(Locale locale) {
        return (LocaleXResourceBundleBase) cache.get(locale);
    }

    /*
     * private
     */

    /* 束縛基準の接尾辞を算出して返却します */
    private static String candidateResourceSuffix(Locale locale) {
        final String language = locale.getLanguage();
        final String country = locale.getCountry();
        final String variant = locale.getVariant();
        if (EvalSet.isAndBlank(language, country, variant)) {
            return "";// The locale is "", "", "".
        }
        final StringBuffer sb = new StringBuffer();
        sb.append('_');
        sb.append(language);
        if (!EvalSet.isAndBlank(country, variant)) {
            sb.append('_');
            sb.append(country);
            if (!Eval.isBlank(variant)) {
                sb.append('_');
                sb.append(variant);
            }
        }
        return sb.toString().toLowerCase();
    }

    /* 束縛基準の識別子を算出して返却します */
    private static String candidateBundleId(Locale locale) {
        final String language = locale.getLanguage();
        final String country = locale.getCountry();
        final String variant = locale.getVariant();
        if (EvalSet.isAndBlank(language, country, variant)) {
            return "root";// The locale is "", "", "".
        }
        final StringBuffer sb = new StringBuffer();
        sb.append(language);
        if (!EvalSet.isAndBlank(country, variant)) {
            sb.append('_');
            sb.append(country);
            if (!Eval.isBlank(variant)) {
                sb.append('_');
                sb.append(variant);
            }
        }
        return sb.toString().toLowerCase();
    }

    /* 上位の束縛基準を算出して返却します */
    private static List candidateParents(Locale locale) {

        if (Eval.isBlank(locale.getLanguage())) {
            return Collections.EMPTY_LIST;
        }

        final List result = new ArrayList(3);
        final String language = locale.getLanguage();
        final String country = locale.getCountry();
        final String variant = locale.getVariant();

        result.add(ROOT);
        if (!Eval.isBlank(country)) {
            result.add(LocaleXResourceBundleBase.forBundleBase(new Locale(language)));
            if (!Eval.isBlank(variant)) {
                result.add(LocaleXResourceBundleBase.forBundleBase(new Locale(language, country)));
            }
        }

        return Collections.unmodifiableList(result);
    }

    public String toString() {
        return Concat.get(LocaleXResourceBundleBase.class.getName(), "[", locale + "]");
    }

}
