/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin.bundle;

import shohaku.ginkgo.Document;
import shohaku.ginkgo.DocumentCompositeRule;
import shohaku.ginkgo.NodeCompositeRule;
import shohaku.ginkgo.TagNode;
import shohaku.shoin.ShoinResourceLoader;
import shohaku.shoin.XResourceBundle;
import shohaku.shoin.XResourceBundleCreater;
import shohaku.shoin.XResourceBundleEvent;

/**
 * Ginkgo API 用いた階層化リソースバンドルの生成機能の抽象実装を提供します。
 */
public abstract class AbstractGinkgoXResourceBundleCreater extends XResourceBundleCreater {

    /**
     * プロパティを初期値で初期化します。
     */
    public AbstractGinkgoXResourceBundleCreater() {
        super();
    }

    /**
     * getXMLResourceBundleClass() で返されるクラスのクラスローダを返却します。
     * 
     * @return getXMLResourceBundleClass() で返されるクラスのクラスローダ
     */
    public ClassLoader getDefaultClassLoader() {
        return getGinkgoResourceBundleClass().getClassLoader();
    }

    /**
     * AbstractGinkgoXResourceBundle 実装を返却します。
     * 
     * @param event
     *            バンドル生成イベント
     * @return AbstractGinkgoXResourceBundle の実装インスタンス
     */
    public XResourceBundle createXResourceBundle(XResourceBundleEvent event) {

        DocumentCompositeRule docRule = getDocumentCompositeRule(event);
        NodeCompositeRule nodeRule = getNodeCompositeRule(event);
        Document parentDoc = (event.getTargetParent() == null) ? null : ((AbstractGinkgoXResourceBundle) event.getTargetParent()).getDocument();

        GinkgoXResourceBundleEvent ginkgoEvent = (GinkgoXResourceBundleEvent) event;
        ginkgoEvent.setTargetDocumentCompositeRule(docRule);
        ginkgoEvent.setTargetNodeCompositeRule(nodeRule);
        ginkgoEvent.setTargetParentDocument(parentDoc);
        return createGinkgoResourceBundle(ginkgoEvent);

    }

    /*
     * protected
     */

    /**
     * 接尾辞文字列 ".xml" を返却します。
     * 
     * @return ".xml"
     */
    protected String getResourceUrlSuffix() {
        return ".xml";
    }

    /**
     * 基底名の拡張リソースバンドルが利用する構成ルールを検索して返却します。 <br>
     * カスタマイズされた構成ルールが配置されていない場合デフォルトを返却します。
     * 
     * @param event
     *            バンドル生成イベント
     * @return 構成ルール
     */
    protected NodeCompositeRule getNodeCompositeRule(XResourceBundleEvent event) {
        return ShoinResourceLoader.getDefaultNodeCompositeRule(getGinkgoResourceBundleClass(), event.getClassLoader());
    }

    /**
     * 束縛基準の識別子をドキュメントIDとして返すドキュメント構成ルールを生成して返却します。
     * 
     * @param event
     *            バンドル生成イベント
     * @return ドキュメント構成ルール
     */
    protected DocumentCompositeRule getDocumentCompositeRule(XResourceBundleEvent event) {
        final String id = event.getTargetBundleBase().getBundleId();
        DocumentCompositeRule rule = new DocumentCompositeRule() {
            public String getDocumentId(Document document, TagNode root) {
                return id;
            }
        };
        return rule;
    }

    /*
     * abstract
     */

    /**
     * Ginkgo API 用いた階層化リソースバンドルを生成して返却します。
     * 
     * @param event
     *            Ginkgo API 用のバンドル生成イベント
     * @return 階層化リソースバンドルの新規インスタンス
     */
    protected abstract AbstractGinkgoXResourceBundle createGinkgoResourceBundle(GinkgoXResourceBundleEvent event);

    /**
     * Ginkgo API 用いた階層化リソースバンドルのクラスを返します。<br>
     * getXMLResourceBundle() で返されるクラスと同じ AbstractGinkgoXResourceBundle の実装クラスである必要があります。
     * 
     * @return Ginkgo API 用いた階層化リソースバンドルのクラス
     */
    protected abstract Class getGinkgoResourceBundleClass();

}