/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin;

import java.io.IOException;
import java.net.URISyntaxException;

import shohaku.core.lang.Concat;
import shohaku.core.lang.feature.FeatureFactory;
import shohaku.core.resource.IOResource;
import shohaku.core.resource.IOResourceLoader;

/**
 * 拡張可能な階層化リソースバンドルを生成する機能を提供します。<br>
 * <br>
 * このオブジェクトはスレッドセーフである必要が有ります。<br>
 * 基本的に可変のフィールドは保有せずに各メソッド内部で処理が完結するべきです。
 */

public abstract class XResourceBundleCreater {

    /**
     * プロパティを初期値で初期化します。
     */
    public XResourceBundleCreater() {
        super();
    }

    /**
     * バンドルの実装の新規インスタンスを生成して返却します。<br>
     * インスタンス生成に失敗した場合は null を返します。
     * 
     * @param event
     *            バンドル生成イベント
     * @return バンドルの実装インスタンス
     */
    public XResourceBundle getXResourceBundle(XResourceBundleEvent event) {

        if (event.getIOResourceLoader() == null) {
            event.setIOResourceLoader(FeatureFactory.getLoader().getIOResourceLoader());
        }
        if (event.getClassLoader() == null) {
            event.setClassLoader(getDefaultClassLoader());
        }

        IOResource ioResource = getBundleIOResource(event);
        if (ioResource != null && ioResource.exists()) {
            event.setTargetIOResource(ioResource);
            return createXResourceBundle(event);
        }
        return null;
    }

    /*
     * abstract
     */

    /**
     * バンドルの実装の新規インスタンスを生成して返却します。<br>
     * IOリソースが存在する場合は呼び出されます。<br>
     * インスタンス生成に失敗した場合は null を返します。
     * 
     * @param event
     *            バンドル生成イベント
     * @return バンドルの実装インスタンス
     */
    abstract protected XResourceBundle createXResourceBundle(XResourceBundleEvent event);

    /*
     * protected
     */

    /**
     * デフォルトのクラスローダを返却します。<br>
     * デフォルトではインスタンスのクラスローダが返却されます。
     * 
     * @return クラスローダ
     */
    protected ClassLoader getDefaultClassLoader() {
        return this.getClass().getClassLoader();
    }

    /**
     * デフォルトのIOリソース生成機能を返却します。<br>
     * デフォルトでは FeatureFactory.getLoader().getIOResourceLoader() が返却されます。
     * 
     * @return IOリソース生成機能
     */
    protected IOResourceLoader getDefaultIOResourceLoader() {
        return FeatureFactory.getLoader().getIOResourceLoader();
    }

    /**
     * リソースファイルのIOリソースを生成して返却します。 <br>
     * リソースファイルを発見出来なかった場合 null を返却します。
     * 
     * @param event
     *            バンドル生成イベント
     * @return リソースファイルのIOリソース
     */
    protected IOResource getBundleIOResource(XResourceBundleEvent event) {
        final String url = getResourceUrl(event.getTargetBundleName());
        return getIOResource(event, url);
    }

    /**
     * URL文字列からIOリソースを生成して返却します。<br>
     * リソースファイルを発見出来なかった場合 null を返却します。
     * 
     * @param event
     *            バンドル生成イベント
     * @param url
     *            リソースアドレスのURL文字列表現
     * @return IOリソース
     */
    protected IOResource getIOResource(XResourceBundleEvent event, String url) {
        try {
            IOResourceLoader ioResourceLoader = event.getIOResourceLoader();
            ioResourceLoader.setClassLoader(event.getClassLoader());
            return ioResourceLoader.getIOResource(url);
        } catch (IOException e) {
            // no op
        } catch (URISyntaxException e) {
            // no op
        }
        return null;
    }

    /**
     * リソースのアドレスをURL文字列で表現して返却します。<br>
     * デフォルトではプレフィックスは付加されず、文字 "." を "/" に変換します。<br>
     * 戻り値は IOResourceLoader へ渡されます。
     * 
     * @param bundleName
     *            生成対象のバンドル名
     * @return リソースアドレスのURL文字列表現
     */
    protected String getResourceUrl(String bundleName) {
        return Concat.get(bundleName.replace('.', '/'), getResourceUrlSuffix());
    }

    /**
     * ファイルパスを構築する際に、バンドル名に追加する接尾辞文字列を返却します、".xml" や ".properties" 等です。<br>
     * デフォルトでは".properties"を返します。
     * 
     * @return バンドル名に追加する接尾辞文字列
     */
    protected String getResourceUrlSuffix() {
        return ".properties";
    }

}
