/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin;

import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.MissingResourceException;

import shohaku.core.helpers.HLog;
import shohaku.core.lang.Concat;
import shohaku.core.lang.EvalSet;

/**
 * 拡張可能な階層化リソースバンドルのキャッシュ機能を提供します。
 */
public class XResourceBundleCache {

    /* キャシュの初期容量。 */
    private static final int INITIAL_CACHE_SIZE = 25;

    /* キャシュの負荷係数。 */
    private static final float CACHE_LOAD_FACTOR = 1.0f;

    /* バンドルのインスタンスをキャシュします。 */
    private final Hashtable bundleCache = new Hashtable(INITIAL_CACHE_SIZE, CACHE_LOAD_FACTOR);

    /* 候補となるバンドル名の単位でロックを掛ける為のObjectを保管します。 */
    private final Hashtable lockObjects = new Hashtable(INITIAL_CACHE_SIZE, CACHE_LOAD_FACTOR);

    /* 拡張リソースバンドルの生成機能。 */
    private final XResourceBundleCreater bundleCreater;

    /**
     * 拡張リソースバンドルの生成機能を格納してキャシュ機能を初期化します。
     * 
     * @param creater
     *            拡張リソースバンドルの生成機能
     */
    public XResourceBundleCache(XResourceBundleCreater creater) {
        this.bundleCreater = creater;
    }

    /**
     * 指定された基底名、束縛基準、クラスローダを使用して、拡張リソースバンドルを取得します。
     * 
     * @param event
     *            バンドル生成イベント
     * @return 指定された基底名と束縛基準の拡張リソースバンドル
     * @throws MissingResourceException
     *             指定された基底名のリソースバンドルが見つからない場合
     */
    public XResourceBundle getBundle(XResourceBundleEvent event) {
        return getBundleImpl(event);
    }

    /**
     * デフォルトのクラスローダを返却します。
     * 
     * @return デフォルトのクラスローダ
     */
    public ClassLoader getDefaultLoader() {
        return this.bundleCreater.getDefaultClassLoader();
    }

    /*
     * protected
     */

    /* 拡張リソースバンドルを取得する処理を実装します。 */
    protected XResourceBundle getBundleImpl(XResourceBundleEvent event) {

        if (EvalSet.isOrNull(event.getBaseName(), event.getBundleBase())) {
            throw new NullPointerException("baseName or XResourceBundleBase is null");
        }

        XResourceBundle bundle = findBundle(event);
        if (bundle == null) {
            String msg = HLog.list("can't find bundle for base name. ", event.getBaseName(), event.getBundleBase());
            throw new MissingResourceException(msg, event.getBaseName(), event.getBundleBase().toString());
        }
        return bundle;
    }

    /* 拡張リソースバンドルを検索して返却します。 拡張リソースバンドルを発見できなかった場合 null を返却します。 */
    protected XResourceBundle findBundle(XResourceBundleEvent event) {
        List parents = event.getBundleBase().getParents();
        return loadBundle(event, parents);
    }

    /* バンドル名の候補の階層順にリソースを読み取り、処理対象となるリソースを返却します、発見できなかった場合 null を返却します。 */
    protected XResourceBundle loadBundle(XResourceBundleEvent event, List parents) {

        if (event.getBundleBase().isRoot()) {
            // get root
            return loadBundle(event, event.getBaseName(), event.getBundleBase(), null);
        }

        XResourceBundle parent = null;
        XResourceBundle target = null;
        XResourceBundle bundle = null;

        // ルートから下位バンドル順に生成する
        String baseName = event.getBaseName();
        for (Iterator i = parents.iterator(); i.hasNext();) {
            XResourceBundleBase bundleBase = (XResourceBundleBase) i.next();
            String bundleName = Concat.get(baseName, bundleBase.getResourceSuffix());
            bundle = loadBundle(event, bundleName, bundleBase, parent);
            if (bundle != null) {
                parent = bundle;
                target = bundle;
            }
        }

        // イベントのバンドルを生成する
        String bundleName = Concat.get(baseName, event.getBundleBase().getResourceSuffix());
        bundle = loadBundle(event, bundleName, event.getBundleBase(), parent);
        if (bundle != null) {
            target = bundle;
        }

        return target;
    }

    /* 個々のバンドル名の候補からリソースを検索して返却します、発見できなかった場合 null を返却します。 */
    protected XResourceBundle loadBundle(XResourceBundleEvent event, String bundleName, XResourceBundleBase bundleBase, XResourceBundle parent) {

        XResourceBundle bundle = null;

        // 候補となるバンドル名単位でロックを行う
        synchronized (getLockObject(bundleName)) {
            bundle = (XResourceBundle) findBundleInCache(bundleName);
            if (bundle == null) {
                bundle = createBundle(event, bundleName, bundleBase, parent);
                if (bundle != null) {
                    putBundleInCache(bundleName, bundle);
                }
            }
        }

        return bundle;
    }

    /* バンドルを生成して返却します、生成出来なかった場合は null を返却します。 */
    protected XResourceBundle createBundle(XResourceBundleEvent event, String bundleName, XResourceBundleBase bundleBase, XResourceBundle parent) {

        // set target bundle
        event.setTargetBundleName(bundleName);
        event.setTargetBundleBase(bundleBase);
        event.setTargetParent(parent);

        // return XResourceBundle or null
        return bundleCreater.getXResourceBundle(event);

    }

    /* リソース単位でロックを掛ける為の Object を返却します。 */
    protected Object getLockObject(String bundleName) {
        synchronized (lockObjects) {
            Object key = lockObjects.get(bundleName);
            if (key == null) {
                key = new Object();
                lockObjects.put(bundleName, key);
            }
            return key;
        }
    }

    /* キャシュからバンドル名の示す XResourceBundle を返却します。 */
    protected Object findBundleInCache(String bundleName) {
        synchronized (bundleCache) {
            return bundleCache.get(bundleName);
        }
    }

    /* バンドル名の示す XResourceBundle をキャシュします。 */
    protected void putBundleInCache(String bundleName, Object value) {
        synchronized (bundleCache) {
            bundleCache.put(bundleName, value);
        }
    }

}
