/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin;

import java.util.Collections;
import java.util.Map;

import shohaku.core.lang.Concat;
import shohaku.core.lang.SystemUtils;

/**
 * 松韻APIのシステム情報を提供します。
 */
public class ShoinSystem {

    /**
     * 松韻APIのバージョンを返却します。
     * 
     * @return 松韻APIのバージョン
     */
    public static String getLibraryVersion() {
        return SystemUtils.getLibraryVersion(ShoinSystem.class);
    }

    /**
     * 松韻ライブラリ固有のプロパティセットを返却します。<br>
     * 返されるマップは普遍マップです。
     * 
     * @return 松韻ライブラリ固有のプロパティセット
     */
    public static Map getLibraryProperties() {
        return PropertiesCache.libraryProperties;
    }

    /**
     * キーの示す松韻ライブラリ固有のプロパティを返却します。<br>
     * キーに対応するプロパティが無い場合は null が返されます。
     * 
     * @param key
     *            キー
     * @return 松韻ライブラリ固有のプロパティ
     */
    public static Object getLibraryProperty(String key) {
        return getLibraryProperties().get(key);
    }

    /**
     * '@' で始まるクラスと関連づけられたキーの示す松韻ライブラリ固有のプロパティを返却します。<br>
     * キーに対応するプロパティが無い場合は null が返されます。
     * 
     * @param clazz
     *            クラス
     * @param key
     *            キー
     * @return 松韻ライブラリ固有のプロパティ
     */
    public static Object getLibraryClassProperty(Class clazz, String key) {
        return getLibraryProperties().get(Concat.get("@", clazz.getName(), ".", key));
    }

    /**
     * '@' で始まるパッケージと関連づけられたキーの示す松韻ライブラリ固有のプロパティを返却します。<br>
     * キーに対応するプロパティが無い場合は null が返されます。
     * 
     * @param pkg
     *            パッケージ
     * @param key
     *            キー
     * @return 松韻ライブラリ固有のプロパティ
     */
    public static Object getLibraryPackageProperty(Package pkg, String key) {
        return getLibraryProperties().get(Concat.get("@", pkg.getName(), ".", key));
    }

    /* 松韻ライブラリ固有のプロパティを保持するクラス */
    private static final class PropertiesCache {
        static final Map libraryProperties;
        static {
            final Map properties = SystemUtils.getLibraryProperties(ShoinSystem.class);
            libraryProperties = Collections.unmodifiableMap(properties);
        }
    }

}
