/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin;

import java.util.Iterator;
import java.util.Map;

/**
 * 識別子で対応づけられるリソースの集合を表現する機能を提供します。
 */
public interface ResourceSet {

    /**
     * 全てのリソースキーの変更不可の反復子を返却します。
     * 
     * @return 全てのリソースキーの反復子
     */
    Iterator getKeys();

    /**
     * 指定されたキーが示す値を返却します。 <br>
     * 指定された識別子が存在しない場合は null を返却します。
     * 
     * @param key
     *            キー
     * @return キーが示す値
     * @throws ResourceCreationException
     *             リソース生成に失敗した場合
     * @throws NullPointerException
     *             key が null の場合
     */
    Object getObject(Object key);

    /**
     * 指定されたキーが示す値を返却します。 <br>
     * 指定されたキーが存在しない場合は defaultValue を返却します。
     * 
     * @param key
     *            キー
     * @param defaultValue
     *            デフォルト値
     * @return キーが示す値
     * @throws ResourceCreationException
     *             リソース生成に失敗した場合
     * @throws NullPointerException
     *             key が null の場合
     */
    Object getObject(Object key, Object defaultValue);

    /**
     * 指定されたキーがリソースに含まれている場合に true を返却します。
     * 
     * @param key
     *            キー
     * @return 指定されたキーが含まれている場合は true
     * @throws NullPointerException
     *             key が null の場合
     */
    boolean containsKey(Object key);

    /**
     * リソースの数を返却します。
     * 
     * @return リソースの数
     */
    int size();

    /**
     * リソース集合の変更不可能なマップ表現を返却します。
     * 
     * @return リソース集合のマップ表現
     */
    Map getMapView();

}