/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.util;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Serializable;
import java.nio.charset.Charset;
import java.util.Calendar;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import shohaku.core.collections.IteratorUtils;
import shohaku.core.helpers.HCoder;
import shohaku.core.helpers.HEval;
import shohaku.core.lang.Predicate;

/**
 * 拡張可能なプロパティセットを提供します。<br>
 * 拡張が容易あり、直接文字エンコーディングを指定する機能をもつ、以外は標準のプロパティセットと同等の仕様を持ちます。<br>
 * また Map インターフェースを実装していません。これはプロパティセットは Map のメンバーとは判断出来なかった為です。<br>
 * 異論は有りそうですが、Map 表現を考えるならば、ラッパーによりマップビューを提供する戦略のほうが妥当と考えます。
 */
public class XProperties implements Serializable {

    /* serialVersionUID */
    private static final long serialVersionUID = -6330260201160781417L;

    /** デフォルトで使用する文字セット(ISO 8859-1). */
    protected static final Charset DEFAULT_CHARSET = Charset.forName("8859_1");

    /** キーと値の以外の文字。 */
    protected static final char[] keyValueSeparators = "=: \t\r\n\f".toCharArray();

    /** キーと値の区切り文字。 */
    protected static final char[] strictKeyValueSeparators = "=:".toCharArray();

    /** コメント文を示す文字。 */
    protected static final char[] commentChars = "#!".toCharArray();

    /** コメント文を示す文字。 */
    protected static final String commentPrefix = "#";

    /** スペースとして扱う文字。 */
    protected static final char[] whiteSpaceChars = " \t\r\n\f".toCharArray();

    /** 行の継続を示す文字。 */
    protected static final char continueLineChar = '\\';

    /** エスケープシーケンス文字 */
    protected static final char escapeChar = '\\';

    /** プロパティを保管します。 */
    protected final Map lookup;

    /** デフォルトプロパティを保管します。 */
    protected XProperties defaults;

    /**
     * 空のプロパティセットを初期化します。
     */
    public XProperties() {
        this(null, new LinkedHashMap());
    }

    /**
     * デフォルトのプロパティセットを格納して初期化します。
     * 
     * @param defaults
     *            デフォルトのプロパティセット
     */
    public XProperties(XProperties defaults) {
        this(defaults, new LinkedHashMap());
    }

    /**
     * 拡張ポイントのコンストラクタ。
     * 
     * @param defaults
     *            デフォルトのプロパティセット
     * @param lookup
     *            プロパティを格納するマップ
     */
    protected XProperties(XProperties defaults, Map lookup) {
        this.defaults = defaults;
        this.lookup = lookup;
    }

    /**
     * 入力ストリームからキーと要素が対になったプロパティセットを読み込みます。 <br>
     * ストリームはデフォルトの ISO 8859-1 文字エンコーディングを使用しているとみなされます。 <br>
     * このエンコーディングに直接表示できない文字には Unicode escapes が使用されます。 <br>
     * ただし、エスケープシーケンスでは 1 文字の「u」だけが使用可能です。 <br>
     * 他の文字エンコーディングとプロパティファイルを変換する場合 native2ascii ツールを使用できます。
     * 
     * @param inStream
     *            入力ストリーム
     * @throws IOException
     *             IO例外
     */
    public void load(InputStream inStream) throws IOException {
        loadImpl(inStream, DEFAULT_CHARSET, true);
    }

    /**
     * 入力ストリームからキーと要素が対になったプロパティセットを読み込みます。 <br>
     * ストリームは、引数 charset で指定された文字エンコーディングを使用しているとみなされます。
     * 
     * @param inStream
     *            入力ストリーム
     * @param charset
     *            文字エンコーディング
     * @throws IOException
     *             IO例外
     */
    public void load(InputStream inStream, Charset charset) throws IOException {
        loadImpl(inStream, charset, false);
    }

    /**
     * 出力ストリームにプロパティセットを書き込みます。<br>
     * ストリームはデフォルトの ISO 8859-1 文字エンコーディングを使用しているとみなされます。
     * 
     * @param outStream
     *            出力ストリーム
     * @param header
     *            ヘッダー
     * @throws IOException
     *             IO例外
     */
    public void store(OutputStream outStream, String header) throws IOException {
        storeImpl(outStream, header, DEFAULT_CHARSET, true);
    }

    /**
     * 出力ストリームにプロパティセットを書き込みます。<br>
     * ストリームは、引数 charset で指定された文字エンコーディングを使用しているとみなされます。
     * 
     * @param outStream
     *            出力ストリーム
     * @param charset
     *            文字エンコーディング
     * @param header
     *            ヘッダー
     * @throws IOException
     *             IO例外
     */
    public void store(OutputStream outStream, Charset charset, String header) throws IOException {
        storeImpl(outStream, header, charset, false);
    }

    /**
     * このプロパティにあるすべてのキーの反復子を返却します。
     * 
     * @return プロパティにあるすべてのキーの反復子
     */
    public Iterator getKeys() {
        if (defaults != null) {
            Iterator[] is = new Iterator[2];
            is[0] = lookup.keySet().iterator();
            is[1] = IteratorUtils.predicateIterator(defaults.getKeys(), new Predicate() {
                public boolean evaluate(Object o) {
                    return !lookup.containsKey(o);
                }
            });
            return IteratorUtils.compositeIterator(is);
        }
        return lookup.keySet().iterator();
    }

    /**
     * プロパティセットをマップにコピーして返却します。
     * 
     * @return プロパティセットをコピーしたマップ
     */
    public Map toMap() {
        return toMap(new LinkedHashMap());
    }

    /**
     * プロパティセットをマップにコピーして返却します。
     * 
     * @param map
     *            格納先のマップ
     * @return プロパティセットをコピーした引数のマップ
     */
    public Map toMap(Map map) {
        if (defaults != null) {
            defaults.toMap(map);
        }
        map.putAll(lookup);
        return map;
    }

    /*
     * get Property
     */

    /**
     * 指定されたキーを持つプロパティを、プロパティから探します。 <br>
     * プロパティキーがない場合は null が返されます。
     * 
     * @param key
     *            プロパティキー
     * @return プロパティキーが示す値
     */
    public Object getProperty(Object key) {
        if (lookup.containsKey(key)) {
            return lookup.get(key);
        }
        return (defaults != null) ? defaults.getProperty(key) : null;
    }

    /**
     * 指定されたキーを持つプロパティを、プロパティから探します。 <br>
     * プロパティキーがない場合は、デフォルト値の引数が返されます。
     * 
     * @param key
     *            プロパティキー
     * @param defaultValue
     *            デフォルト値
     * @return プロパティキーが示す値または defaultValue
     */
    public Object getProperty(Object key, Object defaultValue) {
        Object o = getProperty(key);
        return (o != null) ? o : defaultValue;
    }

    /**
     * プロパティを追加します。
     * 
     * @param key
     *            プロパティキー
     * @param value
     *            プロパティ値
     * @return 既存のプロパティ値、既存プロパティが無い場合は null
     */
    public Object setProperty(Object key, Object value) {
        return lookup.put(key, value);
    }

    /**
     * 指定されたキーがプロパティに含まれている場合に true を返却します。
     * 
     * @param key
     *            プロパティキー
     * @return 指定されたキーが含まれている場合は true
     */
    public boolean containsKey(Object key) {
        return (lookup.containsKey(key) || (defaults != null && defaults.containsKey(key)));
    }

    /**
     * プロパティの文字列表現を返却します。
     * 
     * @return プロパティの文字列表現
     * @see java.lang.Object#toString()
     */
    public String toString() {
        StringBuffer buf = new StringBuffer();
        buf.append("{");

        Iterator i = lookup.entrySet().iterator();
        boolean hasNext = i.hasNext();
        while (hasNext) {
            Map.Entry e = (Map.Entry) (i.next());
            Object key = e.getKey();
            Object value = e.getValue();
            buf.append(key);
            buf.append('=');
            buf.append(value);
            hasNext = i.hasNext();
            if (hasNext) {
                buf.append(", ");
            }
        }

        buf.append("}");
        return buf.toString();
    }

    /*
     * protected
     */

    /**
     * 入力ストリームからキーと要素が対になったプロパティを読み込みます。 <br>
     * ストリームは、引数 charset で指定された文字エンコーディングを使用しているとみなされます。
     * 
     * @param inStream
     *            入力ストリーム
     * @param charset
     *            文字エンコーディング
     * @param isEscapes
     *            ユニコードエスケープが必要な場合は true
     * @throws IOException
     *             入力ストリームからの読み込み中にエラーが発生した場合
     */
    protected synchronized void loadImpl(InputStream inStream, Charset charset, boolean isEscapes) throws IOException {

        final BufferedReader in = new BufferedReader(new InputStreamReader(inStream, charset));
        while (true) {
            String line = in.readLine();
            if (line == null) {
                return;
            }
            if (line.length() > 0) {
                int len = line.length();

                // キーの開始位置の算出
                int keyStart;
                // 先頭の空白をスキップ
                for (keyStart = 0; keyStart < len; keyStart++) {
                    if (!isWhiteSpaceChars(line.charAt(keyStart))) {
                        break;
                    }
                }

                // 空行はスキップする
                if (keyStart == len) {
                    continue;
                }

                // 行の開始文字がコメント文字の場合は行をスキップ
                char firstChar = line.charAt(keyStart);
                if (isNotCommentChars(firstChar)) {

                    // 複数行を結合する
                    while (continueLine(line)) {
                        String nextLine = in.readLine();
                        if (nextLine == null) {
                            nextLine = "";
                        }
                        String loppedLine = line.substring(0, len - 1);
                        int startIndex;
                        // 行頭の空白をスキップ
                        for (startIndex = 0; startIndex < nextLine.length(); startIndex++) {
                            if (!isWhiteSpaceChars(nextLine.charAt(startIndex))) {
                                break;
                            }
                        }
                        nextLine = nextLine.substring(startIndex, nextLine.length());
                        line = new String(loppedLine + nextLine);
                        len = line.length();
                    }

                    // 区切り文字の算出
                    int separatorIndex;
                    for (separatorIndex = keyStart; separatorIndex < len; separatorIndex++) {
                        char currentChar = line.charAt(separatorIndex);
                        if (isEscapeChar(currentChar)) {
                            separatorIndex++;
                        } else if (isKeyValueSeparators(currentChar)) {
                            break;
                        }
                    }

                    // 値の算出
                    int valueIndex;
                    // 値の前方の空白をスキップ
                    for (valueIndex = separatorIndex; valueIndex < len; valueIndex++) {
                        if (!isWhiteSpaceChars(line.charAt(valueIndex))) {
                            break;
                        }
                    }
                    // 区切り文字が複数ある場合の値開始位置の算出
                    if (valueIndex < len) {
                        if (isKeyValueSeparators(line.charAt(valueIndex))) {
                            valueIndex++;
                        }
                    }
                    // 再度値の前方の空白をスキップ
                    while (valueIndex < len) {
                        if (!isWhiteSpaceChars(line.charAt(valueIndex))) {
                            break;
                        }
                        valueIndex++;
                    }

                    // キーと値の抽出と保存
                    final String key = line.substring(keyStart, separatorIndex);
                    final String value = (separatorIndex < len) ? line.substring(valueIndex, len) : "";
                    putProperty(key, value, isEscapes);
                }
            }
        }
    }

    /**
     * 出力ストリームにプロパティを書き込みます。 <br>
     * ストリームは、引数 charset で指定された文字エンコーディングを使用します。
     * 
     * @param outStream
     *            出力ストリーム
     * @param header
     *            プロパティのヘッダ
     * @param charset
     *            文字エンコーディング
     * @param isEscapes
     *            ユニコードエスケープが必要な場合は true
     * @throws IOException
     *             出力ストリームからの書き込み中にエラーが発生した場合
     */
    protected synchronized void storeImpl(OutputStream outStream, String header, Charset charset, boolean isEscapes) throws IOException {

        final PrintWriter writer = new PrintWriter(new OutputStreamWriter(outStream, charset));
        if (header != null) {
            writer.print(getCommentPrefix());
            writer.println(header);
        }
        writer.print(getCommentPrefix());
        writer.println(Calendar.getInstance().getTime());

        final Iterator i = lookup.entrySet().iterator();
        final StringBuffer buf = new StringBuffer();
        while (i.hasNext()) {
            Map.Entry e = (Map.Entry) i.next();
            Object key = e.getKey();
            Object value = e.getValue();
            storeProperty(buf, key, value, isEscapes);
            writer.println(buf);
            buf.setLength(0);
        }

        writer.flush();

    }

    /**
     * プロパティを登録します。
     * 
     * @param key
     *            プロパティキー
     * @param value
     *            プロパティ値
     * @param isEscapes
     *            Unicode escapes を行う場合は true
     * @throws IOException
     *             プロパティの登録中にエラーが発生した場合
     */
    protected void putProperty(String key, String value, boolean isEscapes) throws IOException {
        String k = key;
        String v = value;
        if (isEscapes) {
            k = HCoder.decodePropertiesEscapes(k);
            v = HCoder.decodePropertiesEscapes(v);
        } else {
            k = deleteEscapeChar(k);
        }
        lookup.put(k, v);
    }

    /**
     * バッファにプロパティを出力します。
     * 
     * @param buf
     * @param key
     *            プロパティキー
     * @param value
     *            プロパティ値
     * @param isEscapes
     *            Unicode escapes を行う場合は true
     * @throws IOException
     *             プロパティの出力中にエラーが発生した場合
     */
    protected void storeProperty(StringBuffer buf, Object key, Object value, boolean isEscapes) throws IOException {
        Object k = key;
        Object v = value;
        if (isEscapes) {
            k = HCoder.encodePropertiesEscapes(String.valueOf(k));
            v = HCoder.encodePropertiesEscapes(String.valueOf(v));
        } else {
            k = appendEscapeChar(k);
        }
        buf.append(k);
        buf.append('=');
        buf.append(v);
    }

    /**
     * 次行を現在の行の継続として扱う場合は true を返却します。 <br>
     * (行末が '\' である場合は次行を現在の行の継続として扱う)
     * 
     * @param line
     *            検証する行文字列
     * @return 継続として扱う場合は true
     */
    protected boolean continueLine(String line) {
        int slashCount = 0;
        int index = line.length() - 1;
        while ((index >= 0) && (line.charAt(index--) == continueLineChar)) {
            slashCount++;
        }
        return (slashCount % 2 == 1);
    }

    /**
     * コメント列のプレフィックスを返却します。
     * 
     * @return コメント列のプレフィックス
     */
    protected String getCommentPrefix() {
        return commentPrefix;
    }

    /**
     * コメント列を示す文字の以外の場合は true を返却します。
     * 
     * @param c
     *            検証する文字
     * @return コメント列を示す文字の以外の場合は true
     */
    protected boolean isNotCommentChars(char c) {
        return (!HEval.isContains(commentChars, c));
    }

    /**
     * キーと値の区切り文字の場合は true を返却します。
     * 
     * @param c
     *            検証すう文字
     * @return キーと値の区切り文字の場合は true
     */
    protected boolean isKeyValueSeparators(char c) {
        return (HEval.isContains(keyValueSeparators, c));
    }

    /**
     * スペース文字の場合は true を返却します。
     * 
     * @param c
     *            検証すう文字
     * @return スペース文字の場合は true
     */
    protected boolean isWhiteSpaceChars(char c) {
        return (HEval.isContains(whiteSpaceChars, c));
    }

    /**
     * エスケープシーケンス文字の以外の場合は true を返却します。
     * 
     * @param c
     *            検証する文字
     * @return エスケープシーケンス文字の以外の場合は true
     */
    protected boolean isEscapeChar(char c) {
        return (c == escapeChar);
    }

    /**
     * エスケープシーケンス文字を追加した文字列を返却します。
     * 
     * @param o
     *            変換元のオブジェクト
     * @return エスケープシーケンス文字を追加した文字列
     */
    protected String appendEscapeChar(Object o) {
        return String.valueOf(o).replaceAll("([=:\\])", "\\$1");
    }

    /**
     * エスケープシーケンス文字を削除した文字列を返却します。
     * 
     * @param s
     *            変換元の文字列
     * @return エスケープシーケンス文字を削除した文字列
     */
    protected String deleteEscapeChar(String s) {
        return s.replaceAll("\\\\([^\\\\])", "$1");
    }

}
